package org.maachang.dbm.engine ;

import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.FileChannel.MapMode;
import java.util.Arrays;

import org.maachang.util.ConvertParam;
import org.maachang.util.FileUtil;

/**
 * Hash管理.
 * 
 * @version 2008/06/05
 * @author masahito suzuki
 * @since MaachangDBM 1.12
 */
public class M2Hash {
    
    /**
     * Hashサイズ.
     */
    public static final int MAX_HASH_SIZE = MDbmEnv.HASH ;
    
    /**
     * フラグ管理サイズ.
     */
    private static final int FLAG_POS_OFFSET = M2HashFlag.convertRect( MDbmEnv.HASH ) ;
    
    /**
     * Hashマスク.
     */
    public static final int MASK_HASH = MAX_HASH_SIZE - 1 ;
    
    /**
     * シーケンスID最大数.
     */
    public static final int MAX_SEQUENCE_ID = MDbmDefine.MAX_SEQUENCE_ID ;
    
    /**
     * 1Hashに格納するサイズ.
     */
    public static final int ONE_HASH = 8 ;
    
    /**
     * ファイル.
     */
    private RandomAccessFile fp = null ;
    
    /**
     * mapFile.
     */
    private MappedByteBuffer map = null ;
    
    /**
     * ファイル名.
     */
    private String filename = null ;
    
    /**
     * データサイズ.
     */
    private int size = -1 ;
    
    /**
     * Hashポジション管理.
     */
    private M2HashFlag posMan = null ;
    
    /**
     * データ確保用.
     */
    private final byte[] tmp = new byte[ ONE_HASH ] ;
    
    /**
     * サイズ確保用.
     */
    private final byte[] tmpSize = new byte[ 4 ] ;
    
    /**
     * コンストラクタ.
     */
    private M2Hash() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * ファイル名を設定して、オブジェクトを生成します.
     * <BR>
     * @param filename 対象のファイル名を設定します.
     * @exception Exception 例外.
     */
    public M2Hash( String filename ) throws Exception {
        if( filename == null || ( filename = filename.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "ファイル名は不正です" ) ;
        }
        filename = FileUtil.getFullPath( filename ) ;
        boolean isFile = FileUtil.isFileExists( filename ) ;
        RandomAccessFile fp = new RandomAccessFile( filename,"rwd" ) ;
        if( isFile == false ) {
            initFile( fp ) ;
        }
        FileChannel channel = fp.getChannel();
        MappedByteBuffer map = channel.map( MapMode.READ_WRITE,0,fp.length() );
        M2HashFlag pflg = new M2HashFlag( map,MAX_HASH_SIZE ) ;
        this.fp = fp ;
        this.map = map ;
        this.filename = filename ;
        this.posMan = pflg ;
    }
    
    /**
     * デストラクタ.
     * <BR>
     * @exception Exception 例外.
     */
    protected void finalize() throws Exception {
        this.destroy() ;
    }
    
    /**
     * オブジェクト破棄.
     */
    public synchronized void destroy() {
        if( this.map != null ) {
            try {
                this.map.force() ;
            } catch( Exception e ) {
            }
        }
        if( this.fp != null ) {
            try {
                this.fp.close() ;
            } catch( Exception e ) {
            }
        }
        this.fp = null ;
        this.map = null ;
        this.filename = null ;
        this.posMan = null ;
    }
    
    /**
     * 強制書き込み.
     * <BR>
     * @exception Exception 例外.
     */
    public synchronized void flush() throws Exception {
        if( this.map != null ) {
            this.map.force() ;
        }
    }
    
    /**
     * Hashポジションを設定.
     * <BR>
     * @param hash 対象のHashコードを設定します.
     * @param pos 対象のポジション値を設定します.
     * @param fileNo 対象のファイルNoを設定します.
     * @exception Exception 例外.
     */
    public synchronized void put( int hash,int pos,int fileNo ) throws Exception {
        if( check() == false ) {
            throw new IOException( "既に破棄されています" ) ;
        }
        if( hash <= -1 || hash >= MAX_HASH_SIZE ) {
            throw new IllegalArgumentException( "指定Hashコード("+hash+")は範囲外です" ) ;
        }
        this.map.position( FLAG_POS_OFFSET + ( hash * ONE_HASH ) ) ;
        ConvertParam.convertInt( tmp,0,pos ) ;
        ConvertParam.convertInt( tmp,4,fileNo ) ;
        this.map.put( tmp ) ;
        if( pos == -1 || fileNo == -1 ) {
            this.posMan.removePos( hash ) ;
        }
        else {
            this.posMan.setPos( hash ) ;
        }
    }
    
    /**
     * Hashポジションを削除.
     * <BR>
     * @param hash 対象のHashコードを設定します.
     * @exception Exception 例外.
     */
    public synchronized void remove( int hash ) throws Exception {
        this.put( hash,-1,-1 ) ;
    }
    
    /**
     * Hashポジションを取得.
     * <BR>
     * @param hash 対象のHashコードを設定します.
     * @return int 対象のポジション値が返されます.<BR>
     *              [-1]の場合、無効です.
     * @exception Exception 例外.
     */
    public synchronized long get( int hash ) throws Exception {
        if( check() == false ) {
            throw new IOException( "既に破棄されています" ) ;
        }
        if( hash <= -1 || hash >= MAX_HASH_SIZE ) {
            throw new IllegalArgumentException( "指定Hashコード("+hash+")は範囲外です" ) ;
        }
        this.map.position( FLAG_POS_OFFSET + ( hash * ONE_HASH ) ) ;
        byte[] b = tmp ;
        this.map.get( b ) ;
        return ConvertParam.convertLong( 0,b ) ;
    }
    
    /**
     * 次の有効Hash位置を取得.
     * @param pos 検索開始位置を設定します.
     * @return int 有効なHash位置が返されます.<BR>
     *             [-1]の場合、終端に到達しました.
     * @exception Exception 例外.
     */
    public synchronized int useHash( int pos ) throws Exception {
        if( check() == false ) {
            throw new IOException( "既に破棄されています" ) ;
        }
        return this.posMan.useNextPos( pos ) ;
    }
    
    /**
     * ファイル名を取得.
     * <BR>
     * @return String ファイル名が返されます.
     */
    public synchronized String getFileName() {
        if( check() == false ) {
            return null ;
        }
        return this.filename ;
    }
    
    /**
     * シーケンスオフセット値.
     */
    private static final int SEQUENCE_OFFSET = FLAG_POS_OFFSET + ( MAX_HASH_SIZE * ONE_HASH ) ;
    
    /**
     * 新しいシーケンスIDを取得.
     * <BR><BR>
     * 新しいシーケンスIDを取得します.
     * <BR>
     * @param no シーケンスNoを設定します.<BR>
     *           [0-63]まで利用可能です.
     * @return long 新しいシーケンスIDが返されます.
     * @exception Exception 例外.
     */
    public synchronized long sequenceId( int no ) throws Exception {
        if( check() == false ) {
            throw new IOException( "既に破棄されています" ) ;
        }
        if( no < 0 || no >= MAX_SEQUENCE_ID ) {
            return -1L ;
        }
        byte[] b = new byte[ 8 ] ;
        int off = SEQUENCE_OFFSET + ( no * 8 ) ;
        this.map.position( off ) ;
        this.map.get( b ) ;
        long ret = ConvertParam.convertLong( 0,b ) ;
        b = null ;
        if( ret >= Long.MAX_VALUE ) {
            ret = 0L ;
        }
        else {
            ret += 1L ;
        }
        this.map.position( off ) ;
        this.map.put( ConvertParam.convertLong( ret ) ) ;
        return ret ;
    }
    
    /**
     * サイズを１インクリメント.
     * @exception Exception 例外.
     */
    public synchronized void addOne() throws Exception {
        if( this.size <= -1 ) {
            readSize() ;
        }
        this.size ++ ;
        writeSize() ;
    }
    
    /**
     * サイズを１デクリメント.
     * @exception Exception 例外.
     */
    public synchronized void deleteOne() throws Exception {
        if( this.size <= -1 ) {
            readSize() ;
        }
        this.size -- ;
        if( this.size <= 0 ) {
            this.size = 0 ;
        }
        writeSize() ;
    }
    
    /**
     * サイズを取得.
     * @exception Exception 例外.
     */
    public synchronized int getSize() throws Exception {
        if( this.size <= -1 ) {
            readSize() ;
        }
        return this.size ;
    }
    
    /**
     * このオブジェクトが有効かチェック.
     * <BR>
     * @return boolean [true]の場合、有効です.
     */
    public synchronized boolean isUse() {
        return check() ;
    }
    
    private boolean check() {
        if( this.fp == null || this.map == null ) {
            return false ;
        }
        return true ;
    }
    
    /**
     * ファイル初期化.
     */
    private static final void initFile( RandomAccessFile fp ) throws Exception {
        int off = ( MAX_HASH_SIZE * ONE_HASH ) + FLAG_POS_OFFSET ;
        byte[] b = new byte[ off + ( 8 * MAX_SEQUENCE_ID ) + 4 ] ;
        Arrays.fill( b,( byte )0xff ) ;
        for( int i = 0 ; i < FLAG_POS_OFFSET ; i ++ ) {
            b[ i ] = 0 ;
        }
        for( int i = 0 ; i < MAX_SEQUENCE_ID ; i ++ ) {
            b[ off ] = 0 ;
            b[ off+1 ] = 0 ;
            b[ off+2 ] = 0 ;
            b[ off+3 ] = 0 ;
            b[ off+4 ] = 0 ;
            b[ off+5 ] = 0 ;
            b[ off+6 ] = 0 ;
            b[ off+7 ] = 0 ;
            off += 8 ;
        }
        fp.seek( 0L ) ;
        fp.write( b ) ;
    }
    
    /**
     * データ格納オフセット.
     */
    private static final int DATA_SIZE_OFFSET = FLAG_POS_OFFSET + ( MAX_HASH_SIZE * ONE_HASH ) + ( MAX_SEQUENCE_ID * 8 ) ;
    
    /**
     * データサイズを取得.
     */
    private void readSize() throws Exception {
        this.map.position( DATA_SIZE_OFFSET ) ;
        this.map.get( tmpSize ) ;
        this.size = ConvertParam.convertInt( 0,tmpSize ) ;
        if( this.size <= -1 ) {
            this.size = 0 ;
        }
    }
    
    /**
     * データサイズを書き込む.
     */
    private void writeSize() throws Exception {
        ConvertParam.convertInt( tmpSize,0,this.size ) ;
        this.map.position( DATA_SIZE_OFFSET ) ;
        this.map.put( tmpSize ) ;
    }
}

/**
 * Hashポジション管理.
 */
class M2HashFlag {
    protected static final int ETC_MASK = 0x0000001f ;
    protected static final int FSS_MASK = ~ETC_MASK ;
    protected static final int RIGHT_SHIFT = 5 ;
    private MappedByteBuffer map = null ;
    private int size = -1 ;
    private final byte[] tmp = new byte[ 4 ] ;
    
    private M2HashFlag() {
        
    }
    
    public M2HashFlag( MappedByteBuffer map,int size )
        throws Exception {
        this.map = map ;
        this.size = size ;
    }
    
    public static final int convertRect( int size ) {
        return ( ( ( size & FSS_MASK ) >> RIGHT_SHIFT ) + ( ( ( size & ETC_MASK ) != 0 ) ? 1 : 0 ) ) * 4 ;
    }
    
    public void setPos( int pos ) throws Exception {
        int thisPos = ( ( pos & FSS_MASK ) >> RIGHT_SHIFT ) ;
        int innerPos = ( pos & ETC_MASK ) ;
        int target = mfArray( thisPos ) ;
        if( ( ( ( 1 << innerPos ) ) & target ) != 0 ) {
            throw new IOException( "指定項番["+pos+"]は既にONです" ) ;
        }
        target = ( 1 << innerPos ) | target ;
        mfArray( thisPos,target ) ;
    }
    
    public void removePos( int pos ) throws Exception {
        int thisPos = ( ( pos & FSS_MASK ) >> RIGHT_SHIFT ) ;
        int innerPos = ( pos & ETC_MASK ) ;
        int target = mfArray( thisPos ) ;
        if( ( ( ( 1 << innerPos ) ) & target ) == 0 ) {
            throw new IOException( "指定項番["+pos+"]は既にOFFです" ) ;
        }
        target = ( ~( 1 << innerPos ) ) & target ;
        mfArray( thisPos,target ) ;
    }
    
    public int useNextPos( int pos ) throws Exception {
        int one = 0 ;
        pos ++ ;
        if( ( pos & ETC_MASK ) != 0 ) {
            one = mfArray( ( ( pos & FSS_MASK ) >> RIGHT_SHIFT ) ) ;
        }
        for( ;; ) {
            if( pos >= size ) {
                return -1 ;
            }
            if( ( pos & ETC_MASK ) == 0 ) {
                one = mfArray( ( ( pos & FSS_MASK ) >> RIGHT_SHIFT ) ) ;
                if( one == 0 ) {
                    pos += 32 ;
                    continue ;
                }
            }
            if( targetPos( pos,one ) == true ) {
                return pos ;
            }
            pos ++ ;
        }
    }
    
    private int mfArray( int no ) throws Exception {
        this.map.position( ( no * 4 ) ) ;
        byte[] b = tmp ;
        this.map.get( b ) ;
        return ConvertParam.convertInt( 0,b ) ;
    }
    
    private void mfArray( int no,int n ) throws Exception {
        this.map.position( ( no * 4 ) ) ;
        this.map.put( ConvertParam.convertInt( n ) ) ;
    }
    
    private boolean targetPos( int pos,int one ) throws Exception {
        return ( ( ( 1 << ( pos & ETC_MASK ) ) ) & one ) != 0 ;
    }
}
