package org.maachang.dbm.engine ;

import org.maachang.util.ConvertParam;

/**
 * キー情報.
 * 
 * @version 2008/06/06
 * @author masahito suzuki
 * @since MaachangDBM 1.12
 */
public class M2Key {
    
    /**
     * ハッシュ管理.
     */
    private M2Hash hash = null ;
    
    /**
     * セクタ管理.
     */
    private M2Sector sector = null ;
    
    /**
     * コンストラクタ.
     */
    private M2Key() {
        
    }
    
    /**
     * コンストラクタ.
     * @param hash ハッシュを設定します.
     * @param sector セクタを設定します.
     */
    public M2Key( M2Hash hash,M2Sector sector ) {
        this.hash = hash ;
        this.sector = sector ;
    }
    
    /**
     * デストラクタ.
     */
    protected void finalize() throws Exception {
        hash = null ;
        sector = null ;
    }
    
    /**
     * 新しいキーを追加.
     * @param code 対象のHashコードを設定します.
     * @param key 対象のキーを設定します.
     * @param no 要素のセクタNoを設定します.
     * @param fileNo 要素のファイルNoを設定します.
     * @exception Exception 例外.
     */
    public void add( int code,byte[] key,int no,int fileNo )
        throws Exception {
        M2KeyChild ch = new M2KeyChild() ;
        // 検索結果、一致する内容が存在する場合.
        if( search( ch,code,key ) == true ) {
            // 前に設定されている要素は削除.
            sector.removeAll( ch.getElementPos(),ch.getElementFileNo() ) ;
            // 次の要素に対して、ポジションを上書き.
            ch.setElementPos( no ) ;
            ch.setElementFileNo( fileNo ) ;
            M2SectorData data = ch.header() ;
            sector.writeOne( data,ch.getStartPos(),ch.getStartFileNo() ) ;
        }
        // Hash管理には、何もセットされていない新規状態.
        else if( ch.getStartPos() == -1 ) {
            // 新しい領域で生成.
            ch.clear() ;
            ch.setElementPos( no ) ;
            ch.setElementFileNo( fileNo ) ;
            ch.setHashCode( code ) ;
            ch.setData( key ) ;
            byte[] b = ch.save() ;
            ch = null ;
            // セクタに書き込む.
            long p = sector.writeAll( b,0,b.length ) ;
            // ハッシュにキー位置の内容を書き込む.
            hash.put( mask( code ),
                ( int )( p & 0x00000000ffffffffL ),
                 ( int )( ( p & 0xffffffff00000000L ) >> 32L ) ) ;
            // Key数を１インクリメント.
            hash.addOne() ;
        }
        // Hash管理に条件が存在するので、次の領域に付加する.
        else {
            // 新しい領域を保存する.
            M2KeyChild newCh = new M2KeyChild() ;
            newCh.setBefPos( ch.getStartPos() ) ;
            newCh.setBefFileNo( ch.getStartFileNo() ) ;
            newCh.setElementPos( no ) ;
            newCh.setElementFileNo( fileNo ) ;
            newCh.setHashCode( code ) ;
            newCh.setData( key ) ;
            byte[] b = newCh.save() ;
            newCh = null ;
            // 一番最後の領域に対して、新しい領域と連結させる.
            long p = sector.writeAll( b,0,b.length ) ;
            ch.setNextPos( ( int )( p & 0x00000000ffffffffL ) ) ;
            ch.setNextFileNo( ( int )( ( p & 0xffffffff00000000L ) >> 32L ) ) ;
            M2SectorData data = ch.header() ;
            sector.writeOne( data,ch.getStartPos(),ch.getStartFileNo() ) ;
            // Key数を１インクリメント.
            hash.addOne() ;
        }
    }
    
    /**
     * 指定キーを削除.
     * また、キーに対する要素も削除されます.
     * @param code 対象のHashコードを設定します.
     * @param key 削除対象のKeyを設定します.
     * @exception Exception 例外.
     */
    public void remove( int code,byte[] key ) throws Exception {
        M2KeyChild ch = new M2KeyChild() ;
        // 検索結果、一致する内容が存在する場合.
        if( search( ch,code,key ) == true ) {
            // 前に設定されている要素は削除.
            sector.removeAll( ch.getElementPos(),ch.getElementFileNo() ) ;
            sector.removeAll( ch.getStartPos(),ch.getStartFileNo() ) ;
            // Key数を１デクリメント.
            hash.deleteOne() ;
            // 削除内容に対して、前後の条件が存在する場合、
            // その内容を修正する.
            int befPos = ch.getBefPos() ;
            int befFileNo = ch.getBefFileNo() ;
            int nextPos = ch.getNextPos() ;
            int nextFileNo = ch.getNextFileNo() ;
            // 前後データ存在確認.
            if( nextPos <= -1 || nextFileNo <= -1 ) {
                nextPos = -1 ;
                nextFileNo = -1 ;
            }
            if( befPos <= -1 || befFileNo <= -1 ) {
                befPos = -1 ;
                befFileNo = -1 ;
            }
            // 前後の条件が存在しない場合.
            if( nextPos <= -1 && nextFileNo <= -1 &&
                befPos <= -1 && befFileNo <= -1 ) {
                // Hash情報を破棄.
                hash.remove( mask( code ) ) ;
            }
            else {
                // 削除前の条件が存在する場合.
                if( befPos >= 0 && befFileNo >= 0 ) {
                    M2SectorData data = new M2SectorData() ;
                    sector.readOne( data,befPos,befFileNo ) ;
                    ch.create( data,befPos,befFileNo ) ;
                    ch.setNextPos( nextPos ) ;
                    ch.setNextFileNo( nextFileNo ) ;
                    ch.header( data ) ;
                    sector.writeOne( data,befPos,befFileNo ) ;
                }
                // 削除後の条件が存在する場合.
                if( nextPos >= 0 && nextFileNo >= 0 ) {
                    M2SectorData data = new M2SectorData() ;
                    sector.readOne( data,nextPos,nextFileNo ) ;
                    ch.create( data,nextPos,nextPos ) ;
                    ch.setBefPos( befPos ) ;
                    ch.setBefFileNo( befFileNo ) ;
                    ch.header( data ) ;
                    sector.writeOne( data,nextPos,nextFileNo ) ;
                    // 先頭の条件の場合.
                    if( befPos <= -1 || befFileNo <= -1 ) {
                        // Hash管理に直接書き込む.
                        hash.put( ( mask( code ) ),nextPos,nextFileNo ) ;
                    }
                }
            }
        }
    }
    
    /**
     * 指定キーの要素を取得.
     * @param code 対象のHashコードを設定します.
     * @param key 取得対象のKeyを設定します.
     * @return long 要素に対する項番が返されます.<BR>
     *              [-1L]の場合、対象キーは存在しません.
     * @exception Exception 例外.
     */
    public long get( int code,byte[] key ) throws Exception {
        M2KeyChild ch = new M2KeyChild() ;
        // 検索結果、一致する内容が存在する場合.
        if( search( ch,code,key ) == true ) {
            return ( long )( ( ( long )ch.getElementPos() & 0x00000000ffffffffL ) |
                ( ( ( long )ch.getElementFileNo() & 0x00000000ffffffffL ) << 32L ) ) ;
        }
        return -1L ;
    }
    
    /**
     * キー一覧を取得.
     * @param next 対象のNextKeyを設定します.
     * @return M2NextKey 次のNextKeyが返されます.
     * @exception Exception 例外.
     */
    public M2NextKey nextKey( M2NextKey next )
        throws Exception {
        if( next == null ) {
            next = new M2NextKey() ;
        }
        if( next.getCount() >= hash.getSize() ) {
            return null ;
        }
        boolean result = false ;
        int pos = -1 ;
        int fno = -1 ;
        for( ;; ) {
            if( next.getPos() <= -1 && next.getFileNo() <= 0 ) {
                int code = hash.useHash( next.getHashNo() ) ;
                if( code <= -1 ) {
                    return null ;
                }
                next.setHashNo( code ) ;
                long pf = hash.get( code ) ;
                if( pf <= -1L ) {
                    continue ;
                }
                pos = ( int )( pf & 0x00000000ffffffffL ) ;
                fno = ( int )( ( pf & 0xffffffff00000000L ) >> 32L ) ;
                result = true ;
            }
            else {
                pos = next.getPos() ;
                fno = next.getFileNo() ;
                result = true ;
            }
            if( result == true ) {
                byte[] b = sector.readAll( pos,fno ) ;
                M2KeyChild ch = new M2KeyChild( b,pos,fno,-1,-1 ) ;
                next.setPos( ch.getNextPos() ) ;
                next.setFileNo( ch.getNextFileNo() ) ;
                next.setKey( ch.getData() ) ;
                next.addCount() ;
                return next ;
            }
        }
    }
    
    /**
     * キーサイズを取得.
     * @return int キーサイズが返されます.
     * @exception Exception 例外.
     */
    public int size() throws Exception {
        return hash.getSize() ;
    }
    
    /**
     * Hashオブジェクトを取得.
     * @return M2Hash Hashオブジェクトが返されます.
     */
    public M2Hash getHash() {
        return hash ;
    }
    
    /**
     * 指定Key情報で検索.
     */
    private boolean search( M2KeyChild ch,int code,byte[] key ) throws Exception {
        ch.clear() ;
        int hcd = mask( code ) ;
        long startPos = hash.get( hcd ) ;
        if( startPos <= -1L ) {
            return false ;
        }
        int pos = ( int )( startPos & 0x00000000ffffffffL ) ;
        int fno = ( int )( ( startPos & 0xffffffff00000000L ) >> 32L ) ;
        M2SectorData data = new M2SectorData() ;
        for( ;; ) {
            if( pos <= -1 || fno <= -1 ) {
                return false ;
            }
            sector.readOne( data,pos,fno ) ;
            if( data.getLength() <= -1 ) {
                return false ;
            }
            if( useChild( ch,data,pos,fno,code,key ) ) {
                return true ;
            }
            pos = ch.getNextPos() ;
            fno = ch.getNextFileNo() ;
        }
    }
    
    /**
     * １つのKeyが正しいか確認.
     */
    private boolean useChild( M2KeyChild ch,M2SectorData data,int pos,int fno,int code,byte[] key ) throws Exception {
        ch.create( data,pos,fno ) ;
        if( ch.getHashCode() != code && ch.getLength() != key.length ) {
            return false ;
        }
        for( ;; ) {
            if( data.getNextNo() <= -1 || data.getNextFileNo() <= -1 ) {
                break ;
            }
            sector.readOne( data,data.getNextNo(),data.getNextFileNo() ) ;
            ch.addData( data ) ;
        }
        int len = key.length ;
        byte[] b = ch.getData() ;
        for( int i = 0 ; i < len ; i ++ ) {
            if( key[ i ] != b[ i ] ) {
                return false ;
            }
        }
        return true ;
    }
    
    /**
     * Hashコードのマスク値を計算.
     */
    private static final int mask( int code ) {
        return code & M2Hash.MASK_HASH ;
        //return 1 ;
    }
    
}

/**
 * Key内容.
 */
class M2KeyChild {
    private static final int HEADER = 32 ;
    private int startPos = -1 ;
    private int startFileNo = -1 ;
    private int sectorNextPos = -1 ;
    private int sectorNextFileNo = -1 ;
    private int befPos = -1 ;
    private int befFileNo = -1 ;
    private int nextPos = -1 ;
    private int nextFileNo = -1 ;
    private int hashCode = -1 ;
    private int elementPos = -1 ;
    private int elementFileNo = -1 ;
    private int length = -1 ;
    private byte[] data = null ;
    
    private int position = 0 ;
    
    public M2KeyChild() {
        
    }
    public M2KeyChild( M2SectorData data,int pos,int fno ) throws Exception {
        this.create( data,pos,fno ) ;
    }
    
    public M2KeyChild( byte[] b,int pos,int fno,int sectorNextNo,int sectorNextFno ) throws Exception {
        this.create( b,pos,fno,sectorNextNo,sectorNextFno ) ;
    }
    
    public void create( M2SectorData data,int pos,int fno ) throws Exception {
        clear() ;
        this.startPos = pos ;
        this.startFileNo = fno ;
        this.sectorNextPos = data.getNextNo() ;
        this.sectorNextFileNo = data.getNextFileNo() ;
        byte[] b = data.getData() ;
        this.befPos = ConvertParam.convertInt( 0,b ) ;
        this.befFileNo = ConvertParam.convertInt( 4,b ) ;
        this.nextPos = ConvertParam.convertInt( 8,b ) ;
        this.nextFileNo = ConvertParam.convertInt( 12,b ) ;
        this.hashCode = ConvertParam.convertInt( 16,b ) ;
        this.elementPos = ConvertParam.convertInt( 20,b ) ;
        this.elementFileNo = ConvertParam.convertInt( 24,b ) ;
        this.length = ConvertParam.convertInt( 28,b ) ;
        
        this.data= new byte[ this.length ] ;
        this.position = data.getLength()-HEADER ;
        System.arraycopy( b,HEADER,this.data,0,this.position ) ;
    }
    
    public void create( byte[] b,int pos,int fno,int sectorNextNo,int sectorNextFno ) throws Exception {
        clear() ;
        this.startPos = pos ;
        this.startFileNo = fno ;
        this.sectorNextPos = sectorNextNo ;
        this.sectorNextFileNo = sectorNextFno ;
        this.befPos = ConvertParam.convertInt( 0,b ) ;
        this.befFileNo = ConvertParam.convertInt( 4,b ) ;
        this.nextPos = ConvertParam.convertInt( 8,b ) ;
        this.nextFileNo = ConvertParam.convertInt( 12,b ) ;
        this.hashCode = ConvertParam.convertInt( 16,b ) ;
        this.elementPos = ConvertParam.convertInt( 20,b ) ;
        this.elementFileNo = ConvertParam.convertInt( 24,b ) ;
        this.length = ConvertParam.convertInt( 28,b ) ;
        
        this.data = new byte[ this.length ] ;
        System.arraycopy( b,HEADER,this.data,0,this.length ) ;
        this.position = this.length ;
    }
    
    public void clear() {
        startPos = -1 ;
        startFileNo = -1 ;
        sectorNextPos = -1 ;
        sectorNextFileNo = -1 ;
        befPos = -1 ;
        befFileNo = -1 ;
        nextPos = -1 ;
        nextFileNo = -1 ;
        hashCode = -1 ;
        elementPos = -1 ;
        elementFileNo = -1 ;
        length = -1 ;
        data = null ;
        position = 0 ;
    }
    
    public void addData( M2SectorData data ) {
        int len = data.getLength() ;
        System.arraycopy( data.getData(),0,this.data,position,len ) ;
        position += len ;
    }
    
    public void setStartPos( int startPos ) {
        this.startPos = startPos ;
    }
    
    public int getStartPos() {
        return startPos ;
    }
    
    public void setStartFileNo( int startFileNo ) {
        this.startFileNo = startFileNo ;
    }
    
    public int getStartFileNo() {
        return startFileNo ;
    }
    
    public void setSectorNextPos( int sectorNextPos ) {
        this.sectorNextPos = sectorNextPos ;
    }
    
    public int getSectorNextPos() {
        return sectorNextPos ;
    }
    
    public void setSectorNextFileNo( int sectorNextFileNo ) {
        this.sectorNextFileNo = sectorNextFileNo ;
    }
    
    public int getSectorNextFileNo() {
        return sectorNextFileNo ;
    }
    
    public void setBefPos( int befPos ) {
        this.befPos = befPos ;
    }
    
    public int getBefPos() {
        return befPos ;
    }
    
    public void setBefFileNo( int befFileNo ) {
        this.befFileNo = befFileNo ;
    }
    
    public int getBefFileNo() {
        return befFileNo ;
    }
    
    public void setNextPos( int nextPos ) {
        this.nextPos = nextPos ;
    }
    
    public int getNextPos() {
        return nextPos ;
    }
    
    public void setNextFileNo( int nextFileNo ) {
        this.nextFileNo = nextFileNo ;
    }
    
    public int getNextFileNo() {
        return nextFileNo ;
    }
    
    public void setHashCode( int hashCode ) {
        this.hashCode = hashCode ;
    }
    
    public int getHashCode() {
        return hashCode ;
    }
    
    public void setElementPos( int elementPos ) {
        this.elementPos = elementPos ;
    }
    
    public int getElementPos() {
        return elementPos ;
    }
    
    public void setElementFileNo( int elementFileNo ) {
        this.elementFileNo = elementFileNo ;
    }
    
    public int getElementFileNo() {
        return elementFileNo ;
    }
    
    public int getLength() {
        return length ;
    }
    
    public void setData( byte[] data ) {
        this.data = data ;
        this.length = data.length ;
    }
    
    public byte[] getData() {
        return data ;
    }
    
    public byte[] save() {
        byte[] ret = new byte[ HEADER + length ] ;
        int p = 0 ;
        ConvertParam.convertInt( ret,p,befPos ) ;
        p += 4 ;
        ConvertParam.convertInt( ret,p,befFileNo ) ;
        p += 4 ;
        ConvertParam.convertInt( ret,p,nextPos ) ;
        p += 4 ;
        ConvertParam.convertInt( ret,p,nextFileNo ) ;
        p += 4 ;
        ConvertParam.convertInt( ret,p,hashCode ) ;
        p += 4 ;
        ConvertParam.convertInt( ret,p,elementPos ) ;
        p += 4 ;
        ConvertParam.convertInt( ret,p,elementFileNo ) ;
        p += 4 ;
        ConvertParam.convertInt( ret,p,length ) ;
        p += 4 ;
        System.arraycopy( data,0,ret,p,length ) ;
        return ret ;
    }
    
    public M2SectorData header() {
        M2SectorData ret = new M2SectorData() ;
        header( ret ) ;
        return ret ;
    }
    
    public void header( M2SectorData data ) {
        int len = ( this.length >= MDbmEnv.SECTOR ) ?
            MDbmEnv.SECTOR : this.length ;
        byte[] b = new byte[ len + HEADER ] ;
        int p = 0 ;
        ConvertParam.convertInt( b,p,this.befPos ) ;
        p += 4 ;
        ConvertParam.convertInt( b,p,this.befFileNo ) ;
        p += 4 ;
        ConvertParam.convertInt( b,p,this.nextPos ) ;
        p += 4 ;
        ConvertParam.convertInt( b,p,this.nextFileNo ) ;
        p += 4 ;
        ConvertParam.convertInt( b,p,this.hashCode ) ;
        p += 4 ;
        ConvertParam.convertInt( b,p,this.elementPos ) ;
        p += 4 ;
        ConvertParam.convertInt( b,p,this.elementFileNo ) ;
        p += 4 ;
        ConvertParam.convertInt( b,p,this.length ) ;
        p += 4 ;
        System.arraycopy( this.data,0,b,HEADER,len ) ;
        data.setData( b ) ;
        data.setLength( b.length ) ;
        data.setNextNo( this.sectorNextPos ) ;
        data.setNextFileNo( this.sectorNextFileNo ) ;
    }
    
    public String toString() {
        return new StringBuilder().
            append( " startPos:" ).append( startPos ).
            append( " startFileNo:" ).append( startFileNo ).
            append( " sectorNextPos:" ).append( sectorNextPos ).
            append( " sectorNextFileNo:" ).append( sectorNextFileNo ).
            append( " befPos:" ).append( befPos ).
            append( " befFileNo:" ).append( befFileNo ).
            append( " nextPos:" ).append( nextPos ).
            append( " nextFileNo:" ).append( nextFileNo ).
            append( " hashCode:" ).append( hashCode ).
            append( " elementPos:" ).append( elementPos ).
            append( " elementFileNo:" ).append( elementFileNo ).
            append( " length:" ).append( length ).toString() ;
    }
}
