package org.maachang.dbm.engine ;

import java.io.IOException;
import java.nio.ByteBuffer;

import org.maachang.util.ConvertParam;
import org.maachang.util.FileUtil;

/**
 * １ファイルでのセクタ管理を行う.
 * 
 * @version 2008/06/05
 * @author masahito suzuki
 * @since MaachangDBM 1.12
 */
class M2OneFileSector {
    
    /**
     * 1ファイルでの最大セクタ数.
     */
    public static final int MAX_ONE_FILE = MDbmDefine.MAX_ONE_FILE_SECTOR ;
    
    /**
     * 非同期最大書込み数.
     */
    public static final int FULL_SYNC_SIZE = MDbmEnv.FULL_SYNC_SIZE ;
    
    /**
     * セクタ拡張子.
     */
    public static final String SECTOR_PLUS = ".sct2" ;
    
    /**
     * １ファイルのセクタ数.
     */
    private static final int ONE_SECTOR = MDbmEnv.SECTOR ;
    
    /**
     * セクタオフセット.
     */
    private static final long SECTOR_OFFSET = MDbmDefine.SECTOR_HEADER ;
    
    /**
     * タイムアウト値.
     */
    private static final long READ_TIMEOUT = 15000L ;
    private static final long WRITE_TIMEOUT = 30000L ;
    
    /**
     * 最初のファイルセクタ数.
     */
    private static final int FIRST_SECTOR = 8192 ;
    
    /**
     * 追加係数.
     */
    private static final double CONST_ADD = 0.75f ;
    
    /**
     * 空きフラグ管理用.
     */
    private M2Flag flags = null ;
    
    /**
     * ファイルアクセス用.
     */
    private M2Rand randIo = null ;
    
    /**
     * ファイルNo.
     */
    private int fileNo = -1 ;
    
    /**
     * セクタ開始位置.
     */
    private long startSectorPos = -1L ;
    
    /**
     * 読み込みバッファ.
     */
    private M2Buffers readBuf = null ;
    
    /**
     * 書込みバッファ.
     */
    private M2Buffers writeBuf = null ;
    
    /**
     * 現在ファイルサイズ.
     */
    private long nowFileSize = 0 ;
    
    /**
     * 付加条件長.
     */
    private int nextAdd = 0 ;
    
    /**
     * データ書込みロック.
     */
    private M2WriteSync m2WriteSync = null ;
    
    /**
     * 書込み同期.
     */
    private Object syncWrite = null ;
    
    /**
     * コンストラクタ.
     */
    private M2OneFileSector() {
        
    }
    
    /**
     * コンストラクタ.
     * @param fileNo 対象のファイルNoを設定します.
     * @param dir 対象のディレクトリ名を設定します.
     * @param size 管理セクタ数を設定します.
     * @param syncWrite 書込み同期オブジェクトを設定します.
     * @param readBuf 読み込みバッファを設定します.
     * @param writeBuf 書込みバッファを設定します.
     * @exception Exception 例外.
     */
    public M2OneFileSector( int fileNo,String dir,int size,
        Object syncWrite,M2Buffers readBuf,M2Buffers writeBuf ) throws Exception {
        if( fileNo <= -1 || dir == null || ( dir = dir.trim() ).length() <= 0 ||
            size <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        if( FileUtil.isDirExists( dir ) == false ) {
            throw new IllegalArgumentException( "指定ディレクトリ[" + dir + "]は存在しません" ) ;
        }
        if( size >= MAX_ONE_FILE ) {
            size = MAX_ONE_FILE ;
        }
        if( dir.endsWith( "/" ) == false && dir.endsWith( "\\" ) == false ) {
            dir += "/" ;
        }
        String name = new StringBuilder().append( dir ).append( fileNo ).append( SECTOR_PLUS ).toString() ;
        boolean newFile = FileUtil.isFileExists( name ) == false ;
        long startPos = M2Flag.convertRect( size ) ;
        M2Rand rnd = null ;
        if( newFile ) {
            rnd = new M2Rand( name,( long )( ( ONE_SECTOR + SECTOR_OFFSET ) * FIRST_SECTOR ) + startPos ) ;
        }
        else {
            rnd = new M2Rand( name ) ;
        }
        M2Flag flg = new M2Flag( newFile,rnd.getRandomAccessFile(),( int )0,size ) ;
        this.flags = flg ;
        this.randIo = rnd ;
        this.fileNo = fileNo ;
        this.startSectorPos = startPos ;
        this.readBuf = readBuf ;
        this.writeBuf = writeBuf ;
        this.nowFileSize = this.randIo.getLength() ;
        this.nextAdd = nextAddSize( this.nowFileSize ) ;
        this.m2WriteSync = new M2WriteSync( size ) ;
        this.syncWrite = syncWrite ;
    }
    
    /**
     * デストラクタ.
     */
    protected void finalize() throws Exception {
        destroy() ;
    }
    
    /**
     * オブジェクト破棄.
     */
    public synchronized void destroy() {
        if( flags != null ) {
            flags.clear() ;
        }
        flags = null ;
        if( randIo != null ) {
            randIo.destroy() ;
        }
        if( m2WriteSync != null ) {
            m2WriteSync.destroy() ;
        }
        randIo = null ;
        fileNo = -1 ;
        startSectorPos = -1L ;
        readBuf = null ;
        writeBuf = null ;
        nowFileSize = -1 ;
        nextAdd = -1 ;
        m2WriteSync = null ;
    }
    
    /**
     * １つのセクタを予約.
     * @return int [-1]の場合、予約できませんでした.
     * @exception Exception 例外.
     */
    public int getPos() throws Exception {
        return flags.usePosBySet( 0 ) ;
    }
    
    /**
     * １つのセクタを削除.
     * @param no 削除対象のセクタを設定します.
     * @exception Exception 例外.
     */
    public void removePos( int no ) throws Exception {
        flags.removePos( no ) ;
    }
    
    /**
     * 指定内容のデータを読み込む.
     * @param outHeader 対象のヘッダを設定します.
     * @param out 読み込まれたデータが格納されます.
     * @param no 読み込み位置を設定します.
     * @param offset 対象のオフセット値を設定します.
     * @exception Exception 例外.
     */
    public void read( M2SectorHeader outHeader,byte[] out,int no ) throws Exception {
        read( outHeader,out,no,0 ) ;
    }
    
    /**
     * 指定内容のデータを読み込む.
     * @param header 対象のヘッダを設定します.
     * @param out 読み込まれたデータが格納されます.
     * @param no 読み込み位置を設定します.
     * @param offset 対象のオフセット値を設定します.
     * @exception Exception 例外.
     */
    public void read( M2SectorHeader outHeader,byte[] out,int no,int offset ) throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        outHeader.clear() ;
        if( no >= 0 ) {
            long seek = ( long )( ( ONE_SECTOR + SECTOR_OFFSET ) * no ) + startSectorPos ;
            ByteBuffer rbuf = getUseBuffer( readBuf ) ;
            if( rbuf != null ) {
                try {
                    rbuf.clear() ;
                    syncWrite( false,no ) ;
                    if( isUse() == false ) {
                        throw new IOException( "オブジェクトは既に破棄されています" ) ;
                    }
                    int len = randIo.read( rbuf,seek ) ;
                    if( len != ONE_SECTOR + SECTOR_OFFSET ) {
                        throw new IOException( "指定項番[" + no + "]は不正です" ) ;
                    }
                    byte[] temp = new byte[ 4 ] ;
                    rbuf.get( temp,0,4 ) ;
                    outHeader.setNextNo( ConvertParam.convertInt( 0,temp ) ) ;
                    rbuf.get( temp,0,4 ) ;
                    outHeader.setNextFileNo( ConvertParam.convertInt( 0,temp ) ) ;
                    rbuf.get( temp,0,4 ) ;
                    outHeader.setLength( ConvertParam.convertInt( 0,temp ) ) ;
                    if( outHeader.getLength() > 0 ) {
                        rbuf.get( out,offset,outHeader.getLength() ) ;
                    }
                } finally {
                    if( isUse() ) {
                        rbuf.clear() ;
                        readBuf.append( rbuf ) ;
                    }
                }
            }
        }
    }
    
    /**
     * 指定内容の書き込み処理を行う.
     * @param mode [true]の場合、キャッシュからの書込みです.
     * @param in 書き込み内容を設定します.
     * @param header 書き込みセクタヘッダを設定します.
     * @param no 書き込み位置を設定します.
     * @exception Exception 例外.
     */
    public void writeTo( boolean mode,M2SectorHeader header,byte[] in,int no ) throws Exception {
        writeTo( mode,header,in,no,0 ) ;
    }
    
    /**
     * 指定内容の書き込み処理を行う.
     * @param mode [true]の場合、キャッシュからの書込みです.
     * @param in 書き込み内容を設定します.
     * @param header 書き込みセクタヘッダを設定します.
     * @param no 書き込み位置を設定します.
     * @param offset 対象のオフセット値を設定します.
     * @exception Exception 例外.
     */
    public void writeTo( boolean mode,M2SectorHeader header,byte[] in,int no,int offset ) throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        if( no >= 0 ) {
            long seek = ( long )( ( ONE_SECTOR + SECTOR_OFFSET ) * no ) + startSectorPos ;
            ByteBuffer wbuf = getUseBuffer( writeBuf ) ;
            if( wbuf != null ) {
                try {
                    wbuf.clear() ;
                    plus( no ) ;
                    wbuf.put( ConvertParam.convertInt( header.getNextNo() ) ) ;
                    wbuf.put( ConvertParam.convertInt( header.getNextFileNo() ) ) ;
                    wbuf.put( ConvertParam.convertInt( header.getLength() ) ) ;
                    if( header.getLength() > 0 ) {
                        wbuf.put( in,offset,header.getLength() ) ;
                    }
                    randIo.write( wbuf,seek,syncWrite ) ;
                } finally {
                    if( isUse() ) {
                        if( mode == true ) {
                            try {
                                m2WriteSync.off( no ) ;
                            } catch( Exception e ) {
                            }
                        }
                        wbuf.clear() ;
                        writeBuf.append( wbuf ) ;
                    }
                }
            }
        }
    }
    
    /**
     * 書込み領域同期.
     */
    protected boolean syncWrite( boolean mode,int no ) throws Exception {
        if( no >= 0 && isUse() == true ) {
            long time = System.currentTimeMillis() ;
            // 読み込み用同期.
            if( mode == false ) {
                while( true ) {
                    if( isUse() == false ) {
                        throw new IOException( "オブジェクトは既に破棄されています" ) ;
                    }
                    if( m2WriteSync != null && m2WriteSync.flag( no ) ) {
                        if( time + READ_TIMEOUT <= System.currentTimeMillis() ) {
                            throw new M2SyncTimeoutException( "timeout" ) ;
                        }
                        Thread.sleep( 1L ) ;
                    }
                    else {
                        return true ;
                    }
                }
            }
            // 書込み用同期.
            else {
                while( true ) {
                    if( isUse() == false ) {
                        throw new IOException( "オブジェクトは既に破棄されています" ) ;
                    }
                    if( m2WriteSync != null && m2WriteSync.flag( no ) ) {
                        if( time + WRITE_TIMEOUT <= System.currentTimeMillis() ) {
                            throw new M2SyncTimeoutException( "timeout" ) ;
                        }
                        Thread.sleep( 1L ) ;
                    }
                    else if( m2WriteSync.size() >= FULL_SYNC_SIZE ) {
                        return false ;
                    }
                    else {
                        return true ;
                    }
                }
            }
        }
        throw new IOException( "オブジェクトは既に破棄されているか、条件が不正です["+no+"]" ) ;
    }
    
    /**
     * 書込み領域をONにセット.
     */
    protected void onSync( int no ) throws Exception {
        if( no >= 0 && isUse() == true ) {
            m2WriteSync.on( no ) ;
        }
    }
    
    /**
     * 書込み領域をOFFにセット.
     */
    protected void offSync( int no ) throws Exception {
        if( no >= 0 && isUse() == true ) {
            m2WriteSync.off( no ) ;
        }
    }
    
    /**
     * ファイル名を取得.
     * @return String ファイル名が返されます.
     */
    public String getName() {
        return randIo.getName() ;
    }
    
    /**
     * ファイル項番を取得.
     * @return int ファイル項番が返されます.
     */
    public synchronized int getFileNo() {
        return fileNo ;
    }
    
    /**
     * 現在有効件数を取得.
     * @return int 有効セクタ件数が返されます.
     * @exception Exception 例外.
     */
    public int getSize() throws Exception {
        return flags.size() ;
    }
    
    /**
     * 最大件数を取得.
     * @return int 最大件数が返されます.
     * @exception Exception 例外.
     */
    public int getMax() throws Exception {
        return flags.maxSize() ;
    }
    
    /**
     * 次に増加させるサイズを取得.
     */
    private int nextAddSize( long now ) {
        return ( int )( ( double )( ( now - this.startSectorPos ) / ( long )( ONE_SECTOR + SECTOR_OFFSET ) ) * CONST_ADD ) ;
    }
    
    /**
     * ファイルサイズ付与条件の場合、任意のサイズ分増やす.
     */
    private void plus( int no ) throws Exception {
        if( nextAdd >= flags.maxSize() ) {
            return ;
        }
        //if( flags.size() >= nextAdd ) {
        if( no >= nextAdd ) {
            nextAdd = nextAdd * 2 ;
            if( nextAdd >= flags.maxSize() ) {
                nextAdd = flags.maxSize() ;
            }
            int now = ( int )( ( nowFileSize - startSectorPos ) / ( long )( ONE_SECTOR + SECTOR_OFFSET ) ) ;
            if( now < nextAdd ) {
                randIo.expansion( (( long )( nextAdd - now )*( long )( ONE_SECTOR + SECTOR_OFFSET )),syncWrite ) ;
                nowFileSize = randIo.getLength() ;
            }
        }
    }
    
    /**
     * オブジェクトチェック.
     */
    private synchronized boolean isUse() {
        return randIo != null ;
    }
    
    /**
     * 有効バッファを取得.
     */
    private ByteBuffer getUseBuffer( M2Buffers buf ) throws Exception {
        ByteBuffer ret = null ;
        for( ;; ) {
            if( isUse() == false ) {
                throw new IOException( "オブジェクトは既に破棄されています" ) ;
            }
            if( ( ret = buf.getQueue() ) != null ) {
                return ret ;
            }
            Thread.sleep( 1L ) ;
        }
    }
}
