package org.maachang.dbm.engine ;

import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;

import org.maachang.util.FileUtil;

/**
 * ランダムアクセス.
 * 
 * @version 2008/06/05
 * @author masahito suzuki
 * @since MaachangDBM 1.12
 */
class M2Rand {
    
    /**
     * オープンファイル名.
     */
    private String name = null ;
    
    /**
     * ファイルアクセス.
     */
    private RandomAccessFile rfp = null ;
    
    /**
     * ファイルチャネル.
     */
    private FileChannel channel = null ;
    
    /**
     * ファイルサイズ.
     */
    private long length = -1L ;
    
    /**
     * コンストラクタ.
     */
    private M2Rand() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 指定条件を設定して既存ファイルを読み込みます.
     * <BR>
     * @param name 読み込み対象のファイル名を設定します.
     * @exception Exception 例外.
     */
    public M2Rand( String name ) throws Exception {
        if( name == null || ( name = name.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        if( FileUtil.isFileExists( name ) == false ) {
            throw new IOException( "指定ファイル名[" + name + "]は存在しません" ) ;
        }
        this.rfp = new RandomAccessFile( name,"rwd" ) ;
        this.channel = this.rfp.getChannel() ;
        this.name = name ;
        this.length = ( int )this.rfp.length() ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 指定条件を設定して新しいファイルを生成します.
     * <BR>
     * @param name 対象のファイル名を設定します.
     * @param size 対象のファイルサイズを設定します.
     * @exception Exception 例外.
     */
    public M2Rand( String name,long size )
        throws Exception {
        if( size <= 0L ||
            name == null || ( name = name.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        this.rfp = new RandomAccessFile( name,"rwd" ) ;
        if( FileUtil.isFileExists( name ) == true ) {
            this.rfp.setLength( 0L ) ;
        }
        this.rfp.setLength( size ) ;
        this.channel = this.rfp.getChannel() ;
        this.name = name ;
        this.length = ( int )this.rfp.length() ;
    }
    
    /**
     * デストラクタ.
     */
    protected void finalize() throws Exception {
        this.destroy() ;
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    public synchronized void destroy() {
        if( rfp != null ) {
            try {
                channel.close() ;
            } catch( Exception e ) {
            }
            try {
                rfp.close() ;
            } catch( Exception e ) {
            }
        }
        rfp = null ;
        channel = null ;
        name = null ;
        length = -1L ;
    }
    
    /**
     * 空き容量を増やす.
     * <BR><BR>
     * 新しい空き容量を増やします.
     * <BR>
     * @param size 新しく増やすサイズを設定します.
     * @param syncWrite 書込み同期オブジェクトを設定します.
     * @exception Exception 例外.
     */
    public synchronized void expansion( long size,Object syncWrite ) throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは破棄されています" ) ;
        }
        if( size <= 0 ) {
            return ;
        }
        long newLen = this.length + size ;
        this.length = newLen ;
        synchronized( syncWrite ) {
            this.rfp.setLength( newLen ) ;
        }
    }
    
    /**
     * データ読み込み.
     * @param buf 対象のバッファを設定します.
     * @param seek 読み込み開始位置を設定します.
     * @return int 受信された内容が返されます.
     * @exception Exception 例外.
     */
    public int read( ByteBuffer buf,long seek )
        throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは破棄されています" ) ;
        }
        int ret = channel.read( buf,seek ) ;
        buf.flip() ;
        return ret ;
    }
    
    /**
     * データ書き込み.
     * @param buf 対象のバッファを設定します.
     * @param seek 書き込み開始位置を設定します.
     * @param syncWrite 書込み同期オブジェクトを設定します.
     * @exception Exception 例外.
     */
    public void write( ByteBuffer buf,long seek,Object syncWrite )
        throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは破棄されています" ) ;
        }
        buf.flip() ;
        synchronized( syncWrite ) {
            channel.write( buf,seek ) ;
        }
    }
    
    /**
     * ファイル名を取得.
     * <BR><BR>
     * ファイル名が返されます.
     * <BR>
     * @return String ファイル名が返されます.
     */
    public synchronized String getName() {
        return name ;
    }
    
    /**
     * ファイル名長を取得.
     * <BR><BR>
     * 現在のファイル名長が返されます.
     * <BR>
     * @return long 現在のファイル名長が返されます.
     */
    public synchronized long getLength() {
        return length ;
    }
    
    /**
     * ランダムアクセスファイルオブジェクトを取得.
     * <BR><BR>
     * ランダムアクセスファイルオブジェクトを取得します.
     * <BR>
     * @return RandomAccessFile ランダムアクセスファイルオブジェクトが返されます.
     */
    public synchronized RandomAccessFile getRandomAccessFile() {
        return rfp ;
    }
    
    /**
     * オブジェクトが有効かチェック.
     */
    protected synchronized boolean isUse() {
        return ( channel != null ) ;
    }
}
