package org.maachang.dbm.engine ;


/**
 * 非同期書き込みオブジェクト.
 * 
 * @version 2008/06/05
 * @author masahito suzuki
 * @since MaachangDBM 1.12
 */
class M2Write {
    
    /**
     * 最小値.
     */
    private static final int MIN = MDbmDefine.MIN_ASYNC_WRITE ;
    
    /**
     * 最大値.
     */
    private static final int MAX = MDbmDefine.MAX_ASYNC_WRITE ;
    
    /**
     * 書き込みキュー.
     */
    private M2WriteQueue queue = null ;
    
    /**
     * 書き込みスレッド群.
     */
    private M2WriteThread[] threads = null ;
    
    /**
     * コンストラクタ.
     */
    private M2Write() {
        
    }
    
    /**
     * コンストラクタ.
     * @param size 非同期書き込みスレッド数を設定します.
     * @exception Exception 例外.
     */
    public M2Write( int size ) throws Exception {
        if( size <= MIN ) {
            size = MIN ;
        }
        else if( size >= MAX ) {
            size = MAX ;
        }
        M2WriteQueue q = new M2WriteQueue() ;
        M2WriteThread[] t = new M2WriteThread[ size ] ;
        for( int i = 0 ; i < size ; i ++ ) {
            t[ i ] = new M2WriteThread( q ) ;
        }
        this.queue = q ;
        this.threads = t ;
    }
    
    /**
     * デストラクタ.
     */
    protected void finalize() throws Exception {
        destroy() ;
    }
    
    /**
     * オブジェクト破棄.
     */
    public synchronized void destroy() {
        if( threads != null ) {
            if( queue != null ) {
                for( ;; ) {
                    if( queue.size() <= 0 ) {
                        break ;
                    }
                    try { Thread.sleep( 30 ) ; } catch( Exception e ) {}
                }
            }
            int len = threads.length ;
            for( int i = 0 ; i < len ; i ++ ) {
                if( threads[ i ] != null ) {
                    threads[ i ].destroy() ;
                }
                threads[ i ] = null ;
            }
        }
        threads = null ;
        queue = null ;
    }
    
    /**
     * データ書き込み処理.
     * @param sectorObject 書き込み先セクタオブジェクトを設定します.
     * @param header 書き込みセクタ情報を設定します.
     * @param data 書き込みデータを設定します.
     * @param no 書き込みセクタ位置を設定します.
     * @exception Exception 例外.
     */
    public void write( M2OneFileSector sectorObject,M2SectorHeader header,
        byte[] data,int no ) throws Exception {
        write( sectorObject,header,data,no,0 ) ;
    }
    
    /**
     * データ書き込み処理.
     * @param sectorObject 書き込み先セクタオブジェクトを設定します.
     * @param header 書き込みセクタ情報を設定します.
     * @param data 書き込みデータを設定します.
     * @param no 書き込みセクタ位置を設定します.
     * @param offset 書き込みセクタオフセット値を設定します.
     * @exception Exception 例外.
     */
    public void write( M2OneFileSector sectorObject,M2SectorHeader header,
        byte[] data,int no,int offset ) throws Exception {
        if( isUse() ) {
            if( sectorObject != null && header != null ) {
                // キャッシュサイズが一定範囲内の場合.
                // キャッシュ書込みで処理する.
                if( sectorObject.syncWrite( true,no ) == true ) {
                    // 位置ロック.
                    sectorObject.onSync( no ) ;
                    // ヘッダをコピー.
                    M2SectorHeader insert = new M2SectorHeader() ;
                    header.copyObject( insert ) ;
                    // データをコピー.
                    byte[] insertData = new byte[ insert.getLength() ] ;
                    System.arraycopy( data,offset,insertData,0,insert.getLength() ) ;
                    // データセット.
                    queue.append( new M2WriteElement( sectorObject,insert,insertData,no,0 ) ) ;
                }
                // キャッシュサイズが越している場合.
                else {
                    // 直接書込み処理を行う.
                    sectorObject.writeTo( false,header,data,no,offset ) ;
                }
            }
        }
    }
    
    /**
     * 終了チェック.
     */
    private synchronized boolean isUse() {
        return ( queue != null ) ;
    }
}
