package org.maachang.comet;

import org.maachang.comet.httpd.engine.HttpdEngine;
import org.maachang.comet.net.NetService;
import org.maachang.dao.dbms.RecordFactory;
import org.maachang.manager.GlobalManager;
import org.maachang.rawio.Cpu;
import org.maachang.util.atomic.AtomicBOOL;
import org.maachang.util.atomic.AtomicDOUBLE;
import org.maachang.util.atomic.AtomicINT;
import org.maachang.util.atomic.AtomicLONG;

/**
 * MaachangCometシステム情報.
 * 
 * @version 2007/08/26
 * @author masahito suzuki
 * @since MaachangComet 1.27
 */
public class MSystem {
    
    /** ネットサービスを取得. */
    private static final NetService netService() {
        return ( ( HttpdEngine )GlobalManager.getInstance().get( ServiceDef.MANAGER_BY_HTTPD ) ).getNetService() ;
    }
    
    /** レコードファクトリを取得. */
    private static final RecordFactory recordFactory() {
        return ( RecordFactory )GlobalManager.getInstance().get( ServiceDef.MANAGER_BY_DBMS_POOL ) ;
    }
    
    /**
     * ネットワーク有効プーリングスレッド数を取得.
     * @return int 有効プーリングスレッド数が返されます.
     */
    public static int getNetUsePooling() {
        return netService().getNioService().getUsePooling() ;
    }
    
    /**
     * ネットワークActiveプーリングスレッド数を取得.
     * @return int Activeプーリングスレッド数が返されます.
     */
    public static int getNetActivePooling() {
        return netService().getNioService().getActivePooling() ;
    }
    
    /**
     * DBコネクション最大プーリング数を取得. <BR>
     * <BR>
     * 最大プーリング数を取得します. <BR>
     * 
     * @return int 管理されている最大プーリング数が返されます.
     */
    public static int getDbMax() {
        return recordFactory().getConnectionManager().getMax() ;
    }

    /**
     * DBコネクション現在アクティブであるプーリングコネクション数を取得. <BR>
     * <BR>
     * 現在アクティブであるプーリングコネクション数を取得します. <BR>
     * 
     * @return int 現在アクティブなプーリングコネクション数が返されます.
     */
    public static int getDbActives() {
        return recordFactory().getConnectionManager().getActives() ;
    }
    
    /**
     * DBコネクション現在アイドルであるプーリングコネクション数を取得. <BR>
     * <BR>
     * 現在アイドルであるプーリングコネクション数を取得します. <BR>
     * 
     * @return int 現在アイドルなプーリングコネクション数が返されます.
     */
    public static int getDbIdle() {
        return recordFactory().getConnectionManager().getIdle() ;
    }
    
    /**
     * CPU使用率を取得.
     * @return double CPU使用率が返されます.
     */
    public static final double useCpu() {
        return MSystem.getInstance()._useCpu() ;
    }
    
    /**
     * プロセスIDを取得.
     * @return int プロセスIDが返されます.
     */
    public static final int getProcessId() {
        return MSystem.getInstance()._getProcessId() ;
    }
    
    // 内部用.
    private static final long NEXT_TIME = 3000L ;
    private final AtomicDOUBLE useCpu = new AtomicDOUBLE( 0.0f ) ;
    private final AtomicINT processId = new AtomicINT( -1 ) ;
    private final AtomicBOOL processIdFlag = new AtomicBOOL( false ) ;
    private final AtomicLONG next = new AtomicLONG( -1L ) ;
    private static final MSystem SNGL = new MSystem() ;
    
    protected static final MSystem getInstance() {
        return SNGL ;
    }
    private int _getProcessId() {
        if( processIdFlag.get() ) {
            return processId.get() ;
        }
        processId.set( Cpu.getProcessId() ) ;
        processIdFlag.set( true ) ;
        return processId.get() ;
    }
   private double _useCpu() {
        long t = System.currentTimeMillis() ;
        if( t >= next.get() + NEXT_TIME ) {
            useCpu.set( Cpu.useCpu() ) ;
            next.set( t ) ;
        }
        return useCpu.get() ;
    }
}
