package org.maachang.comet.httpd.engine;

import java.util.ArrayList;
import java.util.HashMap;

import org.maachang.comet.httpd.HttpdHeaders;
import org.maachang.comet.httpd.HttpdRequest;
import org.maachang.comet.httpd.HttpdResponse;
import org.maachang.util.StringUtil;

/**
 * HTTP上のCookieを扱う.
 * 
 * @version 2007/08/26
 * @author masahito suzuki
 * @since MaachangComet 1.00
 */
public class HttpdCookie {
    
    /**
     * Cookie取得ヘッダ.
     */
    public static final String GET_COOKIE = "Cookie" ;
    
    /**
     * Cookie設定ヘッダ.
     */
    public static final String SET_COOKIE = "Set-Cookie" ;
    
    /**
     * Cookieを発行.
     * <BR><BR>
     * 新しいCookieを発行します.
     * <BR>
     * @param key 対象のキー名を設定します.
     * @param value 対象の要素を設定します.
     * @parma path 指定Cookieが有効となるPathを設定します.
     * @param expire このCookieを削除する時間を設定します.
     * @return String Set-Cookieヘッダ情報が返されます.
     * @exception Exception 例外.
     */
    public static final String createCookie( String key,String value,String path )
        throws Exception {
        return createCookie( key,value,path,Long.MIN_VALUE ) ;
    }
    
    /**
     * Cookieを発行.
     * <BR><BR>
     * 新しいCookieを発行します.
     * <BR>
     * @param key 対象のキー名を設定します.
     * @param value 対象の要素を設定します.
     * @parma path 指定Cookieが有効となるPathを設定します.
     * @param expire このCookieを削除する時間を設定します.
     * @return String Set-Cookieヘッダ情報が返されます.
     * @exception Exception 例外.
     */
    public static final String createCookie( String key,String value,String path,long expire )
        throws Exception {
        if( key == null || ( key = key.trim() ).length() <= 0 ||
            value == null || ( value = value.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "key及びvalueは必須条件です" ) ;
        }
        else if( key.indexOf( "=" ) != -1 ) {
            throw new IllegalArgumentException( "key名[" + key + "]には不正な文字が格納されています" ) ;
        }
        StringBuilder buf = new StringBuilder() ;
        buf.append( key ).append( "=" ).append( value ).append( ";" ) ;
        if( path != null && ( path = path.trim() ).length() > 0 ) {
            buf.append( " path=" ).append( path ).append( ";" ) ;
        }
        if( expire != Long.MIN_VALUE ) {
            buf.append( " expires=" ).append( HttpdTimestamp.getTimestamp( true,expire ) ).append( ";" ) ;
        }
        return buf.toString() ;
    }
    
    /**
     * 指定Cookie情報を解析.
     * <BR><BR>
     * 指定されたCookie情報を解析します.
     * <BR>
     * @param cookie 対象のCookieヘッダ内容を設定します.
     * @return HashMap<String,String> 解析された内容が返されます.
     * @exception Exception 例外.
     */
    public static final HashMap<String,String> analysisCookie( String cookie )
        throws Exception {
        if( cookie == null || ( cookie = cookie.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "Cookie情報は不正です" ) ;
        }
        ArrayList<String> lst = StringUtil.cutString( true,false,cookie,":; =" ) ;
        if( lst == null || lst.size() <= 0 ) {
            return null ;
        }
        HashMap<String,String> ret = new HashMap<String,String>() ;
        int len = lst.size() ;
        for( int i = 0 ; i < len ; i += 2 ) {
            String key = lst.get( i ).trim() ;
            String value = lst.get( i+1 ) ;
            ret.put( key,value ) ;
        }
        return ret ;
    }
    
    /**
     * レスポンスヘッダに、Cookie情報を設定.
     * <BR><BR>
     * レスポンスヘッダに、Cookie情報を設定します.
     * <BR>
     * @param response 追加対象のレスポンスを設定します.
     * @param cookie 対象のCookie情報を設定します.
     */
    public static final void putCookie( HttpdResponse response,String cookie ) {
        if( response == null ||
            cookie == null || ( cookie = cookie.trim() ).length() <= 0 ) {
            return ;
        }
        HttpdHeaders header = response.getHeader() ;
        header.addHeader( SET_COOKIE,cookie ) ;
    }
    
    /**
     * リクエストヘッダから、Cookie情報を取得.
     * <BR><BR>
     * リクエストヘッダから、Cookie情報を取得します.
     * <BR>
     * @param request 対象のHttpdRequest情報を設定します.
     * @return String 対象のCookie情報が返されます.
     */
    public static final String getCookie( HttpdRequest request ) {
        HttpdHeaders header = request.getHeader() ;
        return header.getHeader( GET_COOKIE ) ;
    }
    
    
}
