package org.maachang.comet.net;

import java.io.InputStream;



/**
 * HTTPリクエスト受信用バッファ.
 * 
 * @version 2008/11/12
 * @author masahito suzuki
 * @since MaachangComet 1.29
 */
public class RequestBuffer {
    // 受信Body格納用.
    private static final int BUFFER = 8192 ;
    private byte[] binary = null ;
    private int length = 0 ;
    
    // 一時受信領域.
    private static final int TEMP_BINARY = 1024 ;
    private final byte[] tempBinary = new byte[ TEMP_BINARY ]  ;
    private int tempPos = 0 ;
    private int tempLen = 0 ;
    
    /**
     * コンストラクタ.
     */
    public RequestBuffer() {
        this.binary = new byte[ BUFFER ] ;
        this.length = 0 ;
    }
    
    protected void finalize() throws Exception {
        this.destroy() ;
    }
    
    /**
     * オブジェクト破棄.
     */
    protected void destroy() {
        this.binary = null ;
        this.length = 0 ;
    }
    
    /**
     * 内容リセット.
     */
    public void reset() {
        if( this.binary.length == BUFFER ) {
            this.length = 0 ;
        }
        else {
            this.binary = new byte[ BUFFER ] ;
            this.length = 0 ;
        }
    }
    
    /**
     * テンポラリ領域クリア.
     */
    public void clearTemp() {
        tempPos = 0 ;
        tempLen = 0 ;
    }
    
    /**
     * データ書き込み.
     * @param b 書き込み対象のデータを設定します.
     */
    public void write( int b ) {
        addLimit( 1 ) ;
        binary[ length ] = ( byte )( b & 0x000000ff ) ;
        length ++ ;
    }
    
    /**
     * データ書き込み.
     * @param bin 対象のバイナリを設定します.
     */
    public void write( byte[] bin ) {
        write( bin,-1 ) ;
    }
    
    /**
     * データ書き込み.
     * @param bin 対象のバイナリを設定します.
     * @param len 書き込み対象長を設定します.
     */
    public void write( byte[] bin,int len ) {
        if( bin == null || bin.length <= 0 ) {
            return ;
        }
        if( len <= 0 || bin.length <= len ) {
            len = bin.length ;
        }
        addLimit( len ) ;
        System.arraycopy( bin,0,binary,length,len ) ;
        length += len ;
    }
    
    /**
     * 生バイナリ情報を取得.
     * @return byte[] 生バイナリ情報が返されます.
     */
    public byte[] getRawBinary() {
        return binary ;
    }
    
    /**
     * 指定範囲のバイナリを取得.
     * @param length 取得対象の指定範囲を設定します.
     * @return byte[] 指定範囲のバイナリ情報が返されます.
     */
    public byte[] getBinary( int length ) {
        if( length <= 0 ) {
            length = this.length ;
        }
        if( this.length < length ) {
            length = this.length ;
        }
        byte[] ret = new byte[ length ] ;
        System.arraycopy( this.binary,0,ret,0,length ) ;
        return ret ;
    }
    
    /**
     * 現在のバイナリ長を取得.
     * @return int バイナリ長が返されます.
     */
    public int getLength() {
        return length ;
    }
    
    /**
     * テンポラリ領域から、情報を取得.
     * @param in 対象のInputStreamを設定します.
     * @return int １バイトデータが返されます.
     * @exception Exception 例外.
     */
    public int readRequest( InputStream in ) throws Exception {
        if( tempLen <= tempPos ) {
            tempLen = in.read( tempBinary,0,TEMP_BINARY ) ;
            tempPos = 0 ;
        }
        int ret = ( int )( tempBinary[ tempPos ] & 0x000000ff ) ;
        tempPos ++ ;
        return ret ;
    }
    
    /**
     * テンポラリ読み込み残り長を取得.
     * @return int テンポラリ読み込み残り長が返されます.
     */
    public int tempLength() {
        return tempLen - tempPos ;
    }
    
    /** リミット時にバイナリ長を増加. **/
    private void addLimit( int len ) {
        if( binary.length <= length + len ) {
            byte[] t = binary ;
            int iLen = ( int )( ( double )t.length * 1.5f ) + len ;
            binary = new byte[ iLen ] ;
            System.arraycopy( t,0,binary,0,t.length ) ;
            t = null ;
        }
    }
}
