package org.maachang.dao ;

import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.sql.Blob;
import java.sql.ResultSet;
import java.sql.Types;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.maachang.dao.dbms.DbUtil;
import org.maachang.dao.dbms.MetaColumn;
import org.maachang.dao.dbms.MetaFactory;
import org.maachang.dao.dbms.Record;
import org.maachang.dao.dbms.ResultUtil;
import org.maachang.dao.dbms.kind.SupportKind;

/**
 * Dao実行オブジェクト.
 * 
 * @version 2007/10/18
 * @author masahito suzuki
 * @since MaachangDao 1.00
 */
public class ExecutionDao {
    
    /**
     * SQL文[insert].
     */
    private static final String INSERT_BY_SQL = "insert" ;
    
    /**
     * シーケンス付加対象パラメータ.
     */
    public static final String SEQ_COLUMN = "id" ;
    
    /**
     * テーブルカラム生成日付付加パラメータ.
     */
    public static final String CREATE_TIME = "create_time" ;
    
    /**
     * テーブルカラム更新日付付加パラメータ.
     */
    public static final String UPDATE_TIME = "update_time" ;
    
    /**
     * 楽観的ロック対象バージョンカラム.
     */
    public static final String OPTIMISTIC_LOCK_COLUMN = "optimistic_lock" ;
    
    /**
     * テーブル１対他結合名.
     */
    private static final String MANY_JOIN_NAME = "$" ;
    
    /**
     * SQL実行.
     * <BR><BR>
     * 指定SQLを実行します.
     * <BR>
     * @param outId insert文の実行時のシーケンスIDが返されます.
     * @param record SQL実行レコードを設定します.
     * @param model テーブル名を設定します.
     * @param meta このテーブルのメタデータを設定します.
     * @param sql 実行対象のSQL文を設定します.
     * @param params パラメータ群を設定します.
     * @param size 対象のパラメータ長を設定します.
     * @return int 実行件数が返されます.
     * @exception Exception 例外.
     */
    public static final int executionSql( Long[] outId,Record record,String model,MetaColumn meta,
        String sql,Object[] params,int size )
        throws Exception {
        if( record == null ) {
            return 0 ;
        }
        if( sql == null || ( sql = sql.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "SQL文は不正です" ) ;
        }
        SupportKind kind = record.getSupportKind() ;
        if( kind == null ) {
            throw new IOException( "サポート外のアダプタです" ) ;
        }
        String lower = sql.toLowerCase() ;
        boolean insertFlag = lower.startsWith( INSERT_BY_SQL ) ;
        boolean seqFlag = isSequence( meta ) ;
        Long seq = null ;
        // 最初にシーケンスIDを取得.
        if( outId != null && outId.length >= 1 ) {
            // JavaシーケンスでID発行できる場合.
            if( DaoSessionFactory.getInstance().isJavaSequence() ) {
                if( params != null && params.length > 0 && params[ 0 ] != null &&
                    params[ 0 ] instanceof Long ) {
                    // Javaシーケンスで既に発行されているものを対象とする.
                    seq = ( Long )params[ 0 ] ;
                }
            }
            // JavaシーケンスでIDが発行されなく、
            // DBがsequence命令をサポートしている場合.
            else if( seqFlag == true && insertFlag == true &&
                ( params != null && params.length > 0 &&
                params[ 0 ] instanceof Long ) == false ) {
                String s = kind.getSequenceId( model ) ;
                if( s != null ) {
                    seq = getSequenceId( record,s ) ;
                    if( seq != null ) {
                        // シーケンスIDが取得できた場合は、
                        // パラメータ0にセット.
                        //(insert文生成時に、カラムのはじめを[id]に必ず設定し、
                        //params[0] == nullにする必要がある).
                        params[ 0 ] = seq ;
                    }
                }
            }
        }
        // SQL文実行.
        int ret = 0 ;
        if( params == null || params.length <= 0 ) {
            ret = record.executeUpdate( sql ) ;
        }
        else {
            ret = record.executeUpdate( sql,params,size ) ;
        }
        // JavaシーケンスでIDが発行されなく、
        // SQL実行後にシーケンスIDを取得できる場合.
        if( outId != null && outId.length >= 1 &&
            DaoSessionFactory.getInstance().isJavaSequence() == false &&
            seqFlag == true && insertFlag == true ) {
            String s = kind.getInsertIdBySQL() ;
            if( s != null ) {
                seq = getInsertId( record,s ) ;
            }
        }
        // シーケンスが取得できた場合は、outIdに設定.
        if( outId != null ) {
            outId[ 0 ] = seq ;
        }
        return ret ;
    }
    
    /**
     * 情報数取得実行.
     * @param record SQL実行レコードを設定します.
     * @param model テーブル名を設定します.
     * @param meta このテーブルのメタデータを設定します.
     * @param sql 実行対象のSQL文を設定します.
     * @param params パラメータ群を設定します.
     * @param size 対象のパラメータ長を設定します.
     * @return int 実行件数が返されます.
     * @exception Exception 例外.
     */
    public static final int executionCount( Record record,String model,MetaColumn meta,
        String sql,Object[] params,int size )
        throws Exception {
        if( record == null ) {
            return 0 ;
        }
        if( sql == null || ( sql = sql.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "SQL文は不正です" ) ;
        }
        SupportKind kind = record.getSupportKind() ;
        if( kind == null ) {
            throw new IOException( "サポート外のアダプタです" ) ;
        }
        ResultSet rs = null ;
        try {
            // SQL文実行.
            if( params == null || params.length <= 0 ) {
                rs = record.executeQuery( sql ) ;
            }
            else {
                rs = record.executeQuery( sql,params,size ) ;
            }
            int count = 0 ;
            for( ;; ) {
                if( rs.next() == false ) {
                    break ;
                }
                count = rs.getInt( 1 ) ;
                break ;
            }
            rs.close() ;
            rs = null ;
            return count ;
        } finally {
            if( rs != null ) {
                try {
                    rs.close() ;
                } catch( Exception e ) {
                }
            }
            rs = null ;
        }
    }
    
    /**
     * Select実行.
     * @param record SQL実行レコードを設定します.
     * @param model テーブル名を設定します.
     * @param meta このテーブルのメタデータを設定します.
     * @param sql 実行対象のSQL文を設定します.
     * @param columnLen 取得対象のカラムサイズを設定します.
     * @param offset 取得オフセット値を設定します.
     * @param limit 取得リミット値を設定します.
     * @param params パラメータ群を設定します.
     * @param size 対象のパラメータ長を設定します.
     * @return List<Map<String,Object>> select文の実行結果が返されます.
     * @exception Exception 例外.
     */
    public static final List<Map<String,Object>> executionSelect(
        Record record,String model,MetaColumn meta,String sql,int columnLen,int offset,int limit,
        Object[] params,int size )
        throws Exception {
        if( record == null ) {
            return null ;
        }
        if( sql == null || ( sql = sql.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "SQL文は不正です" ) ;
        }
        SupportKind kind = record.getSupportKind() ;
        if( kind == null ) {
            throw new IOException( "サポート外のアダプタです" ) ;
        }
        ResultSet rs = null ;
        try {
            ArrayList<Map<String,Object>> ret = null ;
            // SQL文実行.
            if( params == null || params.length <= 0 ) {
                rs = record.executeQuery( sql ) ;
            }
            else {
                rs = record.executeQuery( sql,params,size ) ;
            }
            if( limit <= 0 ) {
                limit = Integer.MAX_VALUE ;
            }
            if( offset <= 0 ) {
                offset = 0 ;
            }
            ret = new ArrayList<Map<String,Object>>() ;
            pushResult( ret,model,rs,columnLen,offset,limit ) ;
            rs.close() ;
            rs = null ;
            return ret ;
        } finally {
            if( rs != null ) {
                try {
                    rs.close() ;
                } catch( Exception e ) {
                }
            }
            rs = null ;
        }
    }
    
    /**
     * １対１テーブル結合処理.
     * @param record SQL実行レコードを設定します.
     * @param model テーブル名を設定します.
     * @param meta このテーブルのメタデータを設定します.
     * @param sql 実行対象のSQL文を設定します.
     * @param offset 取得オフセット値を設定します.
     * @param limit 取得リミット値を設定します.
     * @param params パラメータ群を設定します.
     * @param size 対象のパラメータ長を設定します.
     * @param join テーブル結合条件を設定します.
     * @param joinNameList テーブル結合名群を設定します.
     * @return List<Map<String,Object>> select文の実行結果が返されます.
     * @exception Exception 例外.
     */
    public static final List<Map<String,Object>> executionJoin(
        Record record,String model,MetaColumn meta,String sql,int offset,int limit,
        Object[] params,int size,
        Map<String,String> join,List<String> joinNameList )
        throws Exception {
        if( record == null ) {
            return null ;
        }
        if( sql == null || ( sql = sql.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "SQL文は不正です" ) ;
        }
        SupportKind kind = record.getSupportKind() ;
        if( kind == null ) {
            throw new IOException( "サポート外のアダプタです" ) ;
        }
        ResultSet rs = null ;
        try {
            ArrayList<Map<String,Object>> ret = null ;
            // SQL文実行.
            if( params == null || params.length <= 0 ) {
                rs = record.executeQuery( sql ) ;
            }
            else {
                rs = record.executeQuery( sql,params,size ) ;
            }
            if( limit <= 0 ) {
                limit = Integer.MAX_VALUE ;
            }
            if( offset <= 0 ) {
                offset = 0 ;
            }
            ret = new ArrayList<Map<String,Object>>() ;
            pushResultJoin( ret,model,rs,offset,limit,join,joinNameList ) ;
            rs.close() ;
            rs = null ;
            return ret ;
        } finally {
            if( rs != null ) {
                try {
                    rs.close() ;
                } catch( Exception e ) {
                }
            }
            rs = null ;
        }
    }
    
    /**
     * １対他テーブル結合処理.
     * @param record SQL実行レコードを設定します.
     * @param model テーブル名を設定します.
     * @param meta このテーブルのメタデータを設定します.
     * @param sql 実行対象のSQL文を設定します.
     * @param limit 取得リミット値を設定します.
     * @param params パラメータ群を設定します.
     * @param size 対象のパラメータ長を設定します.
     * @param joinNameList テーブル結合名群を設定します.
     * @return List<Map<String,Object>> select文の実行結果が返されます.
     * @exception Exception 例外.
     */
    public static final List<Map<String,Object>> executionMany(
        Record record,String model,MetaColumn meta,String sql,int limit,
        Object[] params,int size,List<String> joinNameList )
        throws Exception {
        if( record == null ) {
            return null ;
        }
        if( sql == null || ( sql = sql.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "SQL文は不正です" ) ;
        }
        SupportKind kind = record.getSupportKind() ;
        if( kind == null ) {
            throw new IOException( "サポート外のアダプタです" ) ;
        }
        ResultSet rs = null ;
        try {
            ArrayList<Map<String,Object>> ret = null ;
            // SQL文実行.
            if( params == null || params.length <= 0 ) {
                rs = record.executeQuery( sql ) ;
            }
            else {
                rs = record.executeQuery( sql,params,size ) ;
            }
            if( limit <= 0 ) {
                limit = Integer.MAX_VALUE ;
            }
            ret = new ArrayList<Map<String,Object>>() ;
            pushResultMany( ret,model,rs,limit,joinNameList ) ;
            rs.close() ;
            rs = null ;
            return ret ;
        } finally {
            if( rs != null ) {
                try {
                    rs.close() ;
                } catch( Exception e ) {
                }
            }
            rs = null ;
        }
    }
    
    /**
     * 通常呼び出しSQL文の作成.
     */
    protected static final StringBuilder createSQL( MetaColumn meta,DaoSession session )
        throws Exception {
        StringBuilder joinBuf = new StringBuilder().append( "select " ) ;
        int len = meta.size() ;
        for( int i = 0 ; i < len ; i ++ ) {
            if( i != 0 ) {
                joinBuf.append( "," ) ;
            }
            joinBuf.append( "a." ).append( meta.getColumnName( i ) ) ;
        }
        joinBuf.append( " from " ).append( meta.getTable() ).append( " a " ) ;
        return joinBuf ;
    }
    
    /**
     * JoinSQL文(１対１結合)を生成.
     */
    protected static final StringBuilder createJoinSQL( ArrayList<String> out,MetaColumn meta,
        DaoSession session,Map<String,String> join )
        throws Exception {
        Iterator it = join.keySet().iterator() ;
        int cnt = 0 ;
        StringBuilder joinBuf = new StringBuilder().append( "select " ) ;
        int len = meta.size() ;
        // 取得カラム名一覧をセット.
        for( int i = 0 ; i < len ; i ++ ) {
            if( i != 0 ) {
                joinBuf.append( "," ) ;
            }
            joinBuf.append( "a." ).append( meta.getColumnName( i ) ) ;
        }
        // join対象のテーブルカラム一覧名をセット.
        while( it.hasNext() ) {
            String key = ( String )it.next() ;
            out.add( key ) ;
            String tblName = join.get( key ) ;
            MetaColumn m = session.getMeta( tblName ) ;
            if( m == null ) {
                throw new MaachangDaoException( "指定テーブル名[" + tblName + "]は存在しません" ) ;
            }
            len = m.size() ;
            // 取得カラム名一覧をセット.
            for( int i = 0 ; i < len ; i ++ ) {
                joinBuf.append( ",b" ).append( cnt ).append( "." ).append( m.getColumnName( i ) ) ;
            }
            cnt ++ ;
        }
        cnt = 0 ;
        joinBuf.append( " from " ).append( meta.getTable() ).append( " a " ) ;
        len = out.size() ;
        for( int i = 0 ; i < len ; i ++ ) {
            String key = out.get( i ) ;
            String value = DbUtil.convertJavaNameByDBName( join.get( key ) ) ;
            joinBuf.append( "left outer join " ).append( value ).append( " b" ).append( i ).
                append( " on a." ).append( DbUtil.convertJavaNameByDBName( key ) ).
                append( "=b" ).append( i ).append( ".id " ) ;
        }
        return joinBuf ;
    }
    
    /**
     * manySQL文(１対他結合)を生成.
     */
    protected static final StringBuilder createManySQL( ArrayList<String> out,MetaColumn meta,
        DaoSession session,Map<String,String> srcJoin )
        throws Exception {
        Iterator it = srcJoin.keySet().iterator() ;
        int cnt = 0 ;
        StringBuilder joinBuf = new StringBuilder().append( "select " ) ;
        int len = meta.size() ;
        // 取得カラム名一覧をセット.
        for( int i = 0 ; i < len ; i ++ ) {
            if( i != 0 ) {
                joinBuf.append( "," ) ;
            }
            joinBuf.append( "a." ).append( meta.getColumnName( i ) ) ;
        }
        // join対象のテーブルカラム一覧名をセット.
        while( it.hasNext() ) {
            String key = ( String )it.next() ;
            out.add( key ) ;
            MetaColumn m = session.getMeta( key ) ;
            if( m == null ) {
                throw new MaachangDaoException( "指定テーブル名[" + key + "]は存在しません" ) ;
            }
            len = m.size() ;
            // 取得カラム名一覧をセット.
            for( int i = 0 ; i < len ; i ++ ) {
                joinBuf.append( ",b" ).append( cnt ).append( "." ).append( m.getColumnName( i ) ) ;
            }
            cnt ++ ;
        }
        cnt = 0 ;
        joinBuf.append( " from " ).append( meta.getTable() ).append( " a " ) ;
        len = out.size() ;
        for( int i = 0 ; i < len ; i ++ ) {
            String key = out.get( i ) ;
            String tblName = DbUtil.convertJavaNameByDBName( key ) ;
            String value = DbUtil.convertJavaNameByDBName( srcJoin.get( key ) ) ;
            joinBuf.append( "left outer join " ).append( tblName ).append( " b" ).append( i ) ;
            joinBuf.append( " on a.id=b" ).append( i ).append( "." ).
            append( value ).append( " " ) ;
        }
        return joinBuf ;
    }
    
    /**
     * 判別処理内容に、テーブルシンボルを付加.
     */
    protected static final String addWhereByTableSimbol( String where,MetaColumn meta )
        throws Exception {
        if( where == null || where.length() <= 0 ) {
            return null ;
        }
        int len = meta.size() ;
        for( int i = 0 ; i < len ; i ++ ) {
            String key = meta.getColumnName( i ) ;
            int keyLen = key.length() ;
            int b = 0 ;
            for( ;; ) {
                int p = where.indexOf( key,b ) ;
                if( p <= -1 ) {
                    break ;
                }
                boolean execFlg = false ;
                if( p <= 0 || where.charAt( p-1 ) == ' ' ) {
                    int e = p + keyLen ;
                    if( where.length() >= e ) {
                        execFlg = true ;
                    }
                    else {
                        char c = where.charAt( e+1 ) ;
                        if( c == ' ' || c == '=' || c == '>' || c == '<' || c == '!' ) {
                            execFlg = true ;
                        }
                    }
                    if( execFlg ) {
                        where = new StringBuilder().append( where.substring( 0,p ) ).
                            append( "a." ).append( where.substring( p ) ).toString() ;
                        b = p + keyLen + 2 ;
                    }
                    else {
                        b = p + keyLen ;
                    }
                    continue ;
                }
                b = p + 1 ;
            }
        }
        return where ;
    }
    
    /**
     * シーケンス付与対象のパラメータが存在するかチェック.
     * <BR><BR>
     * シーケンス付与対象のパラメータが存在するかチェックします.
     * <BR>
     * @param meta 対象のメタデータを設定します.
     * @return boolean [true]の場合、シーケンス付与条件です.
     */
    public static final boolean isSequence( MetaColumn meta )
        throws Exception {
        int len = meta.size() ;
        for( int i = 0 ; i < len ; i ++ ) {
            if( SEQ_COLUMN.equals( meta.getColumnName( i ).toLowerCase() ) &&
                meta.getColumnType( i ) == Types.BIGINT ) {
                return true ;
            }
        }
        return false ;
    }
    
    /**
     * 楽観的ロック対象のパラメータが存在するかチェック.
     * <BR><BR>
     * 楽観的ロック対象のパラメータが存在するかチェックします.
     * <BR>
     * @param meta 対象のメタデータを設定します.
     * @return boolean [true]の場合、シーケンス付与条件です.
     */
    public static final boolean isOptimisticLock( MetaColumn meta )
        throws Exception {
        int len = meta.size() ;
        for( int i = 0 ; i < len ; i ++ ) {
            if( OPTIMISTIC_LOCK_COLUMN.equals( meta.getColumnName( i ).toLowerCase() ) &&
                meta.getColumnType( i ) == Types.BIGINT ) {
                return true ;
            }
        }
        return false ;
    }
    
    /**
     * 生成時間のパラメータが存在するかチェック.
     * <BR><BR>
     * 生成時間のパラメータが存在するかチェックします.
     * <BR>
     * @param meta 対象のメタデータを設定します.
     * @return boolean [true]の場合、生成時間パラメータが存在します.
     */
    public static final boolean isCreateTime( MetaColumn meta )
        throws Exception {
        int len = meta.size() ;
        for( int i = 0 ; i < len ; i ++ ) {
            if( CREATE_TIME.equals( meta.getColumnName( i ).toLowerCase() ) ) {
                return true ;
            }
        }
        return false ;
    }
    
    /**
     * 更新時間のパラメータが存在するかチェック.
     * <BR><BR>
     * 更新時間のパラメータが存在するかチェックします.
     * <BR>
     * @param meta 対象のメタデータを設定します.
     * @return boolean [true]の場合、更新時間パラメータが存在します.
     */
    public static final boolean isUpdateTime( MetaColumn meta )
        throws Exception {
        int len = meta.size() ;
        for( int i = 0 ; i < len ; i ++ ) {
            if( UPDATE_TIME.equals( meta.getColumnName( i ).toLowerCase() ) ) {
                return true ;
            }
        }
        return false ;
    }
    
    /**
     * シーケンスIDが利用できる場合は取得.
     */
    private static final Long getSequenceId( Record record,String sql )
        throws Exception {
        ResultSet result = null ;
        try {
            // シーケンスが利用できる場合.
            result = record.executeQuery( sql ) ;
            if( result != null ) {
                result.next() ;
                Long ret = result.getLong( 1 ) ;
                return ret ;
            }
            return null ;
        } catch( Exception e ) {
            throw e ;
        } finally {
            if( result != null ) {
                try {
                    result.close() ;
                } catch( Exception e ) {
                }
            }
        }
    }
    
    /**
     * Inset後にシーケンスIDを取得できる場合.
     */
    private static final Long getInsertId( Record record,String sql )
        throws Exception {
        ResultSet result = null ;
        try {
            // シーケンスが利用できる場合.
            result = record.executeQuery( sql ) ;
            if( result != null ) {
                result.next() ;
                Long ret = result.getLong( 1 ) ;
                return ret ;
            }
            return null ;
        } catch( Exception e ) {
            throw e ;
        } finally {
            if( result != null ) {
                try {
                    result.close() ;
                } catch( Exception e ) {
                }
            }
        }
    }
    
    /**
     * 実行結果を情報に設定.
     */
    private static final void pushResult( ArrayList<Map<String,Object>> out,String model,ResultSet result,int columnLen,int offset,int limit )
        throws Exception {
        ResultUtil.setPosition( result,offset ) ;
        MetaColumn meta = null ;
        for( int i = 0 ; i < limit ; i ++ ) {
            if( result.next() == false ) {
                break ;
            }
            if( meta == null ) {
                meta = MetaFactory.getInstance().getMetaColumn( true,model ) ;
                // 単一データ取得処理の場合.
                if( columnLen != meta.size() ) {
                    meta = new MetaColumn(model,result.getMetaData());
                }
            }
            Map<String,Object> o = new HashMap<String,Object>() ;
            convertResultByObject( o,meta,result,columnLen ) ;
            out.add( o ) ;
        }
    }
    
    /**
     * Join実行結果を情報に設定.
     */
    private static final void pushResultJoin( ArrayList<Map<String,Object>> out,String model,ResultSet result,int offset,int limit,
        Map<String,String> join,List<String> joinNameList )
        throws Exception {
        ResultUtil.setPosition( result,offset ) ;
        for( int i = 0 ; i < limit ; i ++ ) {
            if( result.next() == false ) {
                break ;
            }
            Map<String,Object> o = new HashMap<String,Object>() ;
            convertResultByObjectJoin( o,model,result,join,joinNameList ) ;
            out.add( o ) ;
        }
    }
    
    /**
     * Many実行結果を情報に設定.
     */
    private static final void pushResultMany( ArrayList<Map<String,Object>> out,String model,ResultSet result,int limit,
        List<String> joinNameList )
        throws Exception {
        Long targetId = null ;
        Map<String,Object> mainColumn = null ;
        MetaColumn meta = null ;
        while( true ) {
            if( result.next() == false ) {
                // 情報が存在する場合は、条件格納.
                if( mainColumn != null ) {
                    out.add( mainColumn ) ;
                }
                break ;
            }
            // メインMetaカラムを取得.
            meta = MetaFactory.getInstance().getMetaColumn( true,model ) ;
            // メインID条件が存在しない.
            if( targetId == null ) {
                // メイン条件を取得.
                mainColumn = new HashMap<String,Object>() ;
                int len = meta.size() ;
                for (int i = 0; i < len; i++) {
                    int type = meta.getColumnType( i ) ;
                    String name = meta.getColumnName( i );
                    Object value = getParamByNo( result,type,i+1 ) ;
                    String javaName = DbUtil.convertDBNameByJavaName(false, name);
                    putResult( mainColumn,type,javaName,value ) ;
                }
                // メインIDをセット.
                targetId = ( Long )mainColumn.get( "id" ) ;
            }
            // メインID条件が存在する場合.
            else {
                // 前回の続きであるか否かチェック.
                int len = meta.size() ;
                // 新しい条件の場合.
                if( "id".equals( meta.getColumnName( 0 ) ) &&
                    targetId.equals( getParamByNo( result,meta.getColumnType( 0 ),1 ) ) == false ) {
                    if( mainColumn != null ) {
                        out.add( mainColumn ) ;
                        // 最初の１件だけを取得する処理の場合.
                        if( limit == 1 ) {
                            break ;
                        }
                    }
                    // メイン条件を取得.
                    mainColumn = new HashMap<String,Object>() ;
                    for (int i = 0; i < len; i++) {
                        int type = meta.getColumnType( i ) ;
                        String name = meta.getColumnName( i );
                        Object value = getParamByNo( result,type,i+1 ) ;
                        String javaName = DbUtil.convertDBNameByJavaName(false, name);
                        putResult( mainColumn,type,javaName,value ) ;
                    }
                    targetId = ( Long )mainColumn.get( "id" ) ;
                }
            }
            int startPos = meta.size() ;
            int len = joinNameList.size() ;
            // 他結合情報を取得する.
            for( int i = 0 ; i < len ; i ++ ) {
                boolean noDataFlag = false ;
                String tblName = new StringBuilder().append( MANY_JOIN_NAME ).
                    append( joinNameList.get( i ) ).toString() ;
                Map<String,Object> joinElement = new HashMap<String,Object>() ;
                meta = MetaFactory.getInstance().getMetaColumn( true,joinNameList.get( i ) ) ;
                int lenJ = meta.size() ;
                for( int j = 0 ; j < lenJ ; j ++ ) {
                    int type = meta.getColumnType( j ) ;
                    String name = meta.getColumnName( j );
                    Object value = getParamByNo( result,type,j+startPos+1 ) ;
                    if( j == 0 && value == null ) {
                        noDataFlag = true ;
                        break ;
                    }
                    String javaName = DbUtil.convertDBNameByJavaName(false, name) ;
                    putResult( joinElement,type,javaName,value ) ;
                }
                startPos += lenJ ;
                if( noDataFlag == false ) {
                    List<Map<String,Object>> lst = ( List<Map<String,Object>> )mainColumn.get( tblName ) ;
                    if( lst == null ) {
                        lst = new ArrayList<Map<String,Object>>() ;
                        mainColumn.put( tblName,lst ) ;
                    }
                    lst.add( joinElement ) ;
                }
                joinElement = null ;
            }
        }
    }
    
    /**
     * 指定ResultSetの内容をオブジェクトに割り当てる.
     */
    public static final void convertResultByObject(Map<String,Object>out,MetaColumn meta,ResultSet result,int columnLen)
            throws Exception {
        if (out == null || meta == null || result == null) {
            return ;
        }
        for (int i = 0; i < columnLen; i++) {
            int type = meta.getColumnType( i ) ;
            String name = meta.getColumnName(i);
            Object value = getParamByNo( result,type,i+1 ) ;
            String javaName = DbUtil.convertDBNameByJavaName(false, name);
            putResult( out,type,javaName,value ) ;
        }
    }
    
    /**
     * １対１結合:指定ResultSetの内容をオブジェクトに割り当てる.
     */
    public static final void convertResultByObjectJoin(Map<String,Object>out,String model,ResultSet result,
        Map<String,String> join,List<String> joinNameList )
            throws Exception {
        if (out == null || model == null || result == null ||
            join == null || joinNameList == null ) {
            return ;
        }
        // Joinメイン処理.
        MetaColumn meta = MetaFactory.getInstance().getMetaColumn( true,model ) ;
        int len = meta.size();
        for (int i = 0; i < len; i++) {
            int type = meta.getColumnType( i ) ;
            String name = meta.getColumnName( i );
            Object value = getParamByNo( result,type,i+1 ) ;
            String javaName = DbUtil.convertDBNameByJavaName(false, name);
            putResult( out,type,javaName,value ) ;
        }
        // 結合対象の処理.
        int startPos = len ;
        len = joinNameList.size() ;
        Map<String,Object> joinElement = null ;
        for( int i = 0 ; i < len ; i ++ ) {
            joinElement = new HashMap<String,Object>() ;
            meta = MetaFactory.getInstance().getMetaColumn( true,join.get( joinNameList.get( i ) ) ) ;
            int lenJ = meta.size() ;
            for( int j = 0 ; j < lenJ ; j ++ ) {
                int type = meta.getColumnType( j ) ;
                String name = meta.getColumnName( j );
                Object value = getParamByNo( result,type,j+startPos+1 ) ;
                String javaName = DbUtil.convertDBNameByJavaName(false, name);
                putResult( joinElement,type,javaName,value ) ;
            }
            startPos += lenJ ;
            out.put( joinNameList.get( i ),joinElement ) ;
            joinElement = null ;
        }
    }
    
    /**
     * 取得データを戻り値に設定.
     */
    private static final void putResult( Map<String,Object> result,int type,String name,Object value )
        throws Exception {
        if (value != null) {
            // char(1)の条件は、Booleanと同様の扱いとする.
            if( type == Types.CHAR ) {
                String c = ( String )value ;
                if( c.length() == 1 ) {
                    if( c.equals( "0" ) ) {
                        value = new Boolean( false ) ;
                    }
                    else {
                        value = new Boolean( true ) ;
                    }
                }
            }
            result.put( name,value ) ;
        }
        else {
            result.put( name,null ) ;
        }
    }
    
    
    /**
     * 指定パラメータの要素を取得.
     */
    private static final Object getParamByNo( ResultSet result,int type,int no )
        throws Exception {
        if( result.getObject( no ) == null ) {
            return null ;
        }
        Object data = null ;
        switch( type ){
            case Types.BIT :
                data = new Boolean( result.getBoolean( no ) ) ;
                break ;
            case Types.BOOLEAN :
                data = new Boolean( result.getBoolean( no ) ) ;
                break ;
            case Types.TINYINT :
                data = new Byte( result.getByte( no ) ) ;
                if( data != null ) {
                    byte b = ( ( Byte )data ).byteValue() ;
                    if( b == 1 ) {
                        data =  new Boolean( true ) ;
                    }
                    data = new Boolean( false ) ;
                }
                break ;
            case Types.SMALLINT :
                data = new Integer( result.getInt( no ) ) ;
                break ;
            case Types.INTEGER :
                data = new Integer( result.getInt( no ) ) ;
                break ;
            case Types.BIGINT :
                data = new Long( result.getLong( no ) ) ;
                break ;
            case Types.FLOAT :
                data = new Float( result.getFloat( no ) ) ;
                break ;
            case Types.REAL :
                data = new Float( result.getFloat( no ) ) ;
                break ;
            case Types.DOUBLE :
                data = new Double( result.getDouble( no ) ) ;
                break ;
            case Types.NUMERIC :
                data = result.getBigDecimal( no ) ;
                break ;
            case Types.DECIMAL :
                data = result.getBigDecimal( no ) ;
                break ;
            case Types.CHAR :
                data = result.getString( no ) ;
                break ;
            case Types.VARCHAR :
                data = result.getString( no ) ;
                break ;
            case Types.LONGVARCHAR :
                data = result.getString( no ) ;
                break ;
            case Types.DATE :
                data = result.getDate( no ) ;
                break ;
            case Types.TIME :
                data = result.getTime( no ) ;
                break ;
            case Types.TIMESTAMP :
                data = result.getTimestamp( no ) ;
                break ;
            case Types.BINARY :
                data = result.getBytes( no ) ;
                break ;
            case Types.VARBINARY :
                data = result.getBytes( no ) ;
                break ;
            case Types.LONGVARBINARY :
                data = result.getBytes( no ) ;
                break ;
            case Types.BLOB :
                data = result.getBlob( no ) ;
                break ;
            case Types.STRUCT :// 未サポート.
            case Types.CLOB :// 未サポート.
            case Types.NCLOB :// 未サポート.
            case Types.REF :// 未サポート.
            case Types.DATALINK :
                data = result.getString( no ) ;
                break ;
            default :
                data = null ;
        }
        if( data == null ) {
            return null ;
        }
        // blob.
        if( data instanceof Blob ) {
            InputStream b = new BufferedInputStream(
                ( ( Blob )data ).getBinaryStream() ) ;
            ByteArrayOutputStream bo = new ByteArrayOutputStream() ;
            byte[] bin = new byte[ 4096 ] ;
            for( ;; ) {
                int len = b.read( bin ) ;
                if( len <= -1 ) {
                    break ;
                }
                if( len != 0 ) {
                    bo.write( bin,0,len ) ;
                }
            }
            b.close() ;
            b = null ;
            data = bo.toByteArray() ;
            bo.close() ;
            bo = null ;
        }
        return data ;
    }
    
}

