/*
 * Copyright 2005-2006 Sun Microsystems, Inc.  All rights reserved.
 * SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */

package org.maachang.jsr.script.javascript;

import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.Reader;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.script.Bindings;
import javax.script.ScriptContext;

import org.maachang.jsr.script.api.ExternalBindings;
import org.maachang.jsr.script.util.AtomicObject;

/**
 * Simple implementation of ScriptContext.
 *
 * @author Mike Grogan
 * @since 1.6
 */
public class RhinoSimpleScriptContext implements ScriptContext {
    private static final Writer default_writer = new PrintWriter(System.out , true) ;
    private static final Writer default_errorWriter = new PrintWriter(System.err, true) ;
    private static final Reader default_reader = new InputStreamReader(System.in) ;    
    private final AtomicObject<Writer> writer = new AtomicObject<Writer>() ;
    private final AtomicObject<Writer> errorWriter = new AtomicObject<Writer>() ;
    private final AtomicObject<Reader> reader = new AtomicObject<Reader>() ;

    
    private final AtomicObject<Bindings> engineScope = new AtomicObject<Bindings>() ;
    private final AtomicObject<Bindings> globalScope = new AtomicObject<Bindings>() ;


    public RhinoSimpleScriptContext() {
        engineScope.set( ExternalBindings.getInstance() ) ;
        globalScope.set( null ) ;
        reader.set( default_reader ) ;
        writer.set( default_writer ) ;
        errorWriter.set( default_errorWriter ) ;

    }

    /**
     * Sets a <code>Bindings</code> of attributes for the given scope.  If the value
     * of scope is <code>ENGINE_SCOPE</code> the given <code>Bindings</code> replaces the
     * <code>engineScope</code> field.  If the value
     * of scope is <code>GLOBAL_SCOPE</code> the given <code>Bindings</code> replaces the
     * <code>globalScope</code> field.
     *
     * @param bindings The <code>Bindings</code> of attributes to set.
     * @param scope The value of the scope in which the attributes are set.
     *
     * @throws IllegalArgumentException if scope is invalid.
     * @throws NullPointerException if the value of scope is <code>ENGINE_SCOPE</code> and
     * the specified <code>Bindings</code> is null.
     */
    public void setBindings(Bindings bindings, int scope) {
        throw new RuntimeException( "bindings直接指定は未サポート!!" ) ;
    }


    /**
     * Retrieves the value of the attribute with the given name in
     * the scope occurring earliest in the search order.  The order
     * is determined by the numeric value of the scope parameter (lowest
     * scope values first.)
     *
     * @param name The name of the the attribute to retrieve.
     * @return The value of the attribute in the lowest scope for
     * which an attribute with the given name is defined.  Returns
     * null if no attribute with the name exists in any scope.
     * @throws NullPointerException if the name is null.
     * @throws IllegalArgumentException if the name is empty.
     */
    public Object getAttribute(String name) {
        Bindings es = engineScope.get() ;
        Bindings gs = globalScope.get() ;
        if (es.containsKey(name)) {
            return getAttribute(name, ENGINE_SCOPE);
        } else if (gs != null && gs.containsKey(name)) {
            return getAttribute(name, GLOBAL_SCOPE);
        }

        return null;
    }

    /**
     * Gets the value of an attribute in a given scope.
     *
     * @param name The name of the attribute to retrieve.
     * @param scope The scope in which to retrieve the attribute.
     * @return The value of the attribute. Returns <code>null</code> is the name
     * does not exist in the given scope.
     *
     * @throws IllegalArgumentException
     *         if the name is empty or if the value of scope is invalid.
     * @throws NullPointerException if the name is null.
     */
    public Object getAttribute(String name, int scope) {

        switch (scope) {

            case ENGINE_SCOPE:
                return engineScope.get().get(name);

            case GLOBAL_SCOPE:
                if (globalScope.get() != null) {
                    return globalScope.get().get(name);
                }
                return null;

            default:
                throw new IllegalArgumentException("Illegal scope value.");
        }
    }

    /**
     * Remove an attribute in a given scope.
     *
     * @param name The name of the attribute to remove
     * @param scope The scope in which to remove the attribute
     *
     * @return The removed value.
     * @throws IllegalArgumentException
     *         if the name is empty or if the scope is invalid.
     * @throws NullPointerException if the name is null.
     */
    public Object removeAttribute(String name, int scope) {

        switch (scope) {

            case ENGINE_SCOPE:
                if (getBindings(ENGINE_SCOPE) != null) {
                    return getBindings(ENGINE_SCOPE).remove(name);
                }
                return null;

            case GLOBAL_SCOPE:
                if (getBindings(GLOBAL_SCOPE) != null) {
                    return getBindings(GLOBAL_SCOPE).remove(name);
                }
                return null;

            default:
                throw new IllegalArgumentException("Illegal scope value.");
        }
    }

    /**
     * Sets the value of an attribute in a given scope.
     *
     * @param name The name of the attribute to set
     * @param value The value of the attribute
     * @param scope The scope in which to set the attribute
     *
     * @throws IllegalArgumentException
     *         if the name is empty or if the scope is invalid.
     * @throws NullPointerException if the name is null.
     */
    public void setAttribute(String name, Object value, int scope) {

        switch (scope) {

            case ENGINE_SCOPE:
                engineScope.get().put(name, value);
                return;

            case GLOBAL_SCOPE:
                if (globalScope.get() != null) {
                    globalScope.get().put(name, value);
                }
                return;

            default:
                throw new IllegalArgumentException("Illegal scope value.");
        }
    }

    /** {@inheritDoc} */
    public Writer getWriter() {
        return writer.get();
    }

    /** {@inheritDoc} */
    public Reader getReader() {
        return reader.get();
    }

    /** {@inheritDoc} */
    public void setReader(Reader reader) {
        this.reader.set( reader );
    }

    /** {@inheritDoc} */
    public void setWriter(Writer writer) {
        this.writer.set( writer );
    }

    /** {@inheritDoc} */
    public Writer getErrorWriter() {
        return errorWriter.get();
    }

    /** {@inheritDoc} */
    public void setErrorWriter(Writer writer) {
        this.errorWriter.set( writer ) ;
    }

    /**
     * Get the lowest scope in which an attribute is defined.
     * @param name Name of the attribute
     * .
     * @return The lowest scope.  Returns -1 if no attribute with the given
     * name is defined in any scope.
     * @throws NullPointerException if name is null.
     * @throws IllegalArgumentException if name is empty.
     */
    public int getAttributesScope(String name) {
        Bindings es = engineScope.get() ;
        Bindings gs = globalScope.get() ;
        if (es.containsKey(name)) {
            return ENGINE_SCOPE;
        } else if (gs != null && gs.containsKey(name)) {
            return GLOBAL_SCOPE;
        } else {
            return -1;
        }
    }

    /**
     * Returns the value of the <code>engineScope</code> field if specified scope is
     * <code>ENGINE_SCOPE</code>.  Returns the value of the <code>globalScope</code> field if the specified scope is
     * <code>GLOBAL_SCOPE</code>.
     *
     * @param scope The specified scope
     * @return The value of either the  <code>engineScope</code> or <code>globalScope</code> field.
     * @throws IllegalArgumentException if the value of scope is invalid.
     */
    public Bindings getBindings(int scope) {
        if (scope == ENGINE_SCOPE) {
            return engineScope.get();
        } else if (scope == GLOBAL_SCOPE) {
            return globalScope.get();
        } else {
            throw new IllegalArgumentException("Illegal scope value.");
        }
    }

    /** {@inheritDoc} */
    public List<Integer> getScopes() {
        return scopes;
    }

    private static List<Integer> scopes;
    static {
        scopes = new ArrayList<Integer>(2);
        scopes.add(ENGINE_SCOPE);
        scopes.add(GLOBAL_SCOPE);
        scopes = Collections.unmodifiableList(scopes);
    }
}
