package org.maachang.jni.unsafe;

/**
 * メモリ直接操作処理.
 * 
 * @version 2010/06/03
 * @author  masahito suzuki
 * @since   Unsafe-1.0.0
 */
public final class DirectMemoryIO {
    private DirectMemoryIO() {}
    
    /**
     * malloc.
     * @param size 生成対象のメモリサイズを設定します.
     * @return long 生成された先頭アドレスが返されます.
     */
    public static final long malloc( int size ) {
        return Native.malloc( size ) ;
    }
    
    /**
     * realloc.
     * @param address 対象のメモリ先頭アドレスを設定します.
     * @param size 再生成対象のメモリサイズを設定します.
     * @return long 再生成された先頭アドレスが返されます.
     */
    public static final long realloc( long address,int size ) {
        return Native.realloc( address,size ) ;
    }
    
    /**
     * free.
     * @param address メモリ解放対象のアドレスを設定します.
     */
    public static final void free( long address ) {
        Native.free( address ) ;
    }
    
    /**
     * memset.
     * @param address 対象のメモリ先頭アドレスを設定します.
     * @param code 指定値を設定します.
     * @param size 指定値設定のサイズを設定します.
     */
    public static final void memset( long address,byte code,int size ) {
        Native.memset( address,code,size ) ;
    }
    
    /**
     * memcpy.
     * @param destAddr コピー先のメモリアドレスを設定します.
     * @param srcAddr コピー元のメモリアドレスを設定します.
     * @param size コピーサイズを設定します.
     */
    public static final void memcpy( long destAddr,long srcAddr,int size ) {
        Native.memcpy( destAddr,srcAddr,size ) ;
    }
    
    /** sun.misc.Unsafeオブジェクト利用 **/
    private static final boolean UnsafeMode = Unsafe.UNSAFE_MODE ;
    private static final sun.misc.Unsafe unsafe = Unsafe.unsafe ;
    
    /**
     * 1バイトの情報を取得.
     * @param address 対象のアドレスを設定します.
     * @param no 対象の項番を設定します.
     * @return byte バイト情報が返されます.
     */
    public static final byte get( long address,int no ) {
        if( UnsafeMode ) {
            return unsafe.getByte( address+no ) ;
        }
        else {
            return Native.getByte( address,no ) ;
        }
    }
    
    /**
     * 1バイトの情報を設定.
     * @param address 対象のアドレスを設定します.
     * @param no 対象の項番を設定します.
     * @param value 対象の１バイト情報を設定します.
     */
    public static final void put( long address,int no,byte value ) {
        if( UnsafeMode ) {
            unsafe.putByte( address+no,value ) ;
        }
        else {
            Native.putByte( address,no,value ) ;
        }
    }
}

