#***********************************************************************
#	JAPRO Template Engine
#     Ver. 3.0
#
#	ex.
#
#	$jte = new JTE() ;
#	$template->exexute( 'template/sample.html', $data ) ;
#
#***********************************************************************

use strict ;
use warnings ;


if ( __FILE__ =~ /\A(.*\/)([^\/]*)\z/s ) {
	require $1 . 'jte_dom.pl' ;
} else {
	require 'jte_dom.pl' ;
}


package JTE ;


sub new {
	my $class = shift ;
	
	bless {
		'templateAttribute'	=> 'jte',
		
		'encoding'			=> 'UTF-8',
		'docType'			=> '<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd">',
		'xhtml'				=> 1,
		
		'emptyTag'			=> [
			'meta',
			'base',
			'link',
			'basefont',
			'spacer',
			'br',
			'wbr',
			'hr',
			'img',
			'area',
			'param',
			'bgsound',
			'input'
		],
		
		'blankAttribute'	=> [
			'checked',
			'compact',
			'declare',
			'defer',
			'disabled',
			'ismap',
			'multiple',
			'nohref',
			'noresize',
			'noshade',
			'nowrap',
			'selected',
			'target'
		],
		
		'urlAttribute'		=> [
			'href',
			'src',
			'background',
			'cite',
			'action'
		],
		
		'data'				=> {
			'condition'			=> {},
			'value'				=> {},
			'record'			=> {},
			'function'			=> {}
		},
    	
		'dom'				=> new JTE_DOMDocument(),
		
		'log'				=> '',
		'log_file'			=> 'jte.log'
	},
	$class ;
}


sub templateAttribute:lvalue {
	$_[ 0 ]->{ 'templateAttribute' } ;
}


sub data {
	$_[ 0 ]->{ 'data' } ;
}


sub dom {
	$_[ 0 ]->{ 'dom' } ;
}


#=======================================================================
# set all

sub set {
	my $this = shift ;
	my $data = shift ;
	
    if ( defined( $data->{ 'condition' } ) ) {
    	$this->set_conditions( $data->{ 'condition' } )
    }
    
    if ( defined( $data->{ 'value' } ) ) {
    	$this->set_values( $data->{ 'value' } )
    }
    
    if ( defined( $data->{ 'record' } ) ) {
    	$this->set_records( $data->{ 'record' } )
    }
    
    if ( defined( $data->{ 'function' } ) ) {
    	$this->set_functions( $data->{ 'function' } )
    }
}


#=======================================================================
# set block

sub set_conditions {
	my $this = shift ;
	my $data = shift ;
	
	foreach my $name ( keys( %{ $data } ) ) {
		$this->set_condition( $name, $data->{ $name } ) ;
	}
}


sub set_values {
	my $this = shift ;
	my $data = shift ;
	
	foreach my $name ( keys( %{ $data } ) ) {
		$this->set_value( $name, $data->{ $name } ) ;
	}
}


sub set_records {
	my $this = shift ;
	my $data = shift ;
	
	foreach my $name ( keys( %{ $data } ) ) {
		$this->set_record( $name, $data->{ $name } ) ;
	}
}


sub set_functions {
	my $this = shift ;
	my $data = shift ;
	
	foreach my $name ( keys( %{ $data } ) ) {
		$this->set_function( $name, $data->{ $name } ) ;
	}
}


#=======================================================================
# set part

sub set_condition {
	my $this = shift ;
	my $name = shift ;
	my $value = shift ;
	
    $this->{ 'data' }->{ 'condition' }->{ $name } = $value ;
}


sub set_value {
	my $this = shift ;
	my $name = shift ;
	my $value = shift ;
	
    $this->{ 'data' }->{ 'value' }->{ $name } = $value ;
}


sub set_record {
	my $this = shift ;
	my $name = shift ;
	my $value = shift ;
	
    $this->{ 'data' }->{ 'record' }->{ $name } = $value ;
}


sub set_function {
	my $this = shift ;
	my $name = shift ;
	my $value = shift ;
	
    $this->{ 'data' }->{ 'function' }->{ $name } = $value ;
}


#=======================================================================
# load

sub load {
	my $this = shift ;
	my $path = shift ;
	
	my $html = $this->file_get_contents( $path ) ;
	$html = $this->delete_bom( $html ) ;
	$html =~ s/(\A\s+|\s+\z)//s ;	# trim
	
    if ( $html =~ /\A(<\?xml(\s+.*?)?\?>)\s*(.*)\z/si ) {
      $html = $2 ;
    }
    
    if ( $html =~ /\A(<\!DOCTYPE\s+.*?>)\s*(.*)\z/si ) {
      $this->{ 'doctype' } = $1 ;
      $html = $2 ;
	}
    
    unless ( $html =~ /\A<html(\s+.*?)?>/si ) {
		print "No html !<br />$html" ;
		return ;
    }
    
    $this->{ 'dom' }->{ 'encoding' } = $this->{ 'encoding' } ;
    
    $this->_parse( $this->dom, \$html ) ;
}


#=======================================================================
# _parse
  
sub _parse {
	my $this = shift ;
	my $parent = shift ;
	my $html_ref = shift ;
	my $open = shift || '' ;
	
	while ( $$html_ref ne '' ) {
		if ( $$html_ref =~ /\A(.*?)(<\!--.*?-->|<!\[CDATA\[.*?\]\]>|<.*?>)(.*)\z/s ) {
			my $text = $1 ;
			my $tag = $2 ;
			$$html_ref = $3 ;
			
			if ( $text ne '' ) {
				$parent->appendChild( $this->dom->createTextNode( $text ) ) ;
			}
			
			if ( $tag =~ /\A<\!--(.*?)-->\z/s ) {
				$parent->appendChild( $this->dom->createComment( $1 ) ) ;
				next ;
			}
			
			if ( $tag =~ /\A<!\[CDATA\[(.*?)\]\]>\z/s ) {
				$parent->appendChild( $this->dom->createCDataSection( $1 ) ) ;
				next ;
			}
			
			if ( $tag =~ /\A<\/\s*(.*?)\s*>\z/s ) {
				my $tagName = lc( $1 ) ;
				
				if ( ( $open eq '' ) || ( $open ne $tagName ) ) {
					print "Not opened or invalid &lt;${open}&gt; - &lt;/${tagName}&gt;<br />\n" ;
				}
				
				last ;	# 親タグへ帰る
			}
			
			$this->_parse_element( $parent, $tag, $html_ref ) ;
			
			next ;
		}
		
		$parent->appendChild( $this->dom->createTextNode( $$html_ref ) ) ;
		$$html_ref = '' ;
	}
}


sub _parse_element {
	my $this = shift ;
	my $parent = shift ;
	my $tag = shift ;
	my $html_ref = shift ;
	
	unless ( $tag =~ /\A<\s*(\S+)\s*(.*)>\z/s ) {
		# <理解不能>
		return ;
	}
	
	my $tagName = lc( $1 ) ;
	my $attributes = $2 ;
	my $closed = '' ;
	
	if ( $attributes =~ /\A(.*?)\s*( ?\/)\z/s ) {
		$attributes = $1 ;
		$closed = 1 ;
	}
	
	my $node = $this->dom->createElement( $tagName ) ;
	
	$parent->appendChild( $node ) ;
	
	$this->_parse_attributes( $node, $attributes ) ;
	
	if ( $closed ) {
		return ;
	}
	
	if ( $this->in_array( $tagName, $this->{ 'emptyTag' } ) ) {
		return ;
	}
	
	if ( $tagName eq 'script' ) {
		if ( $$html_ref =~ /\A(.*?)<\/\s*script\s*>(.*)\z/si ) {
			my $text = $1 ;
			$$html_ref = $2 ;
			
			unless ( $text eq '' ) {
				$node->appendChild( $this->dom->createTextNode( $text ) ) ;
			}
		} else {
			unless ( $$html_ref eq '' ) {
				$node->appendChild( $this->dom->createTextNode( $$html_ref ) ) ;
			}
			
			$$html_ref = '' ;
		}
		
		return ;
	}
	
	$this->_parse( $node, $html_ref, $tagName ) ;
}


sub _parse_attributes {
	my $this = shift ;
	my $node = shift ;
	my $attributes = shift ;
	
	while ( $attributes ne '' ) {
		if ( $attributes =~ /\A(\S+?)\s*=\s*("[^"]*"|\'[^\']*\'|\S+)\s*(.*?)\s*\z/s ) {
			my $name = lc( $1 ) ;
			my $value = $2 ;
			$attributes = $3 ;
			
			if ( $node->hasAttribute( $name ) ) {
				next ;
			}
			
			if ( $value =~ /\A"([^"]*)"\z/s ) {
				$value = $1 ;
			} else {
				if ( $value =~ /\A'([^']*)'\z/s ) {
					$value = $1 ;
					
					if ( $this->in_array( $name, $this->{ 'urlAttribute' } ) ) {
						$value =~ s/"/%22/g
					} else {
						$value =~ s/"/&quot;/g
					}
				}
			}
			
			$node->setAttribute( $name, $value ) ;
			
			next ;
		}
		
		if ( $attributes =~ /\A(\S+)\s*(.*?)\s*\z/s ) {
			my $name = lc( $1 ) ;
			$attributes = $2 ;
			
			if ( $node->hasAttribute( $name ) ) {
				next ;
			}
			
			$node->setAttribute( $name, $name ) ;
			
			next ;
		}
		
		last ;
	}
}


#=======================================================================
# replace

sub replace {
	my $this = shift ;
	
    $this->{ 'data' }->{ 'root' } = $this->{ 'data' } ;
    
    $this->{ 'data' }->{ 'parent' } = {
    	'condition'	=> {},
    	'value'		=> {},
    	'record'	=> {},
    	'function'	=> {}
    } ;
	
	my $records = {} ;
	
	$this->replace_element( $this->dom->documentElement, $this->{ 'data' }, $records ) ;
	
	foreach my $name ( keys( %{ $records } ) ) {
		$this->replace_record( $records->{ $name } ) ;
	}
}


#=======================================================================
# replace_element
  
sub replace_element {
	my $this = shift ;
	my $element = shift ;
	my $data = shift ;
	my $records = shift ;
	my $in_record = shift ;
	my $is_record = shift ;
	
	unless ( $element->hasAttribute( $this->{ 'templateAttribute' } ) ) {
		$this->replace_childs( $element, $data, $records, $in_record ) ;
		return ;
	}
	
	my $commands = $this->_get_template_commands( $element ) ;
	
	if ( ! $is_record ) {
		
		# fake
		
		if ( defined( $commands->{ 'fake' } ) ) {
			$this->replace_remove( $element ) ;
			return ;
		}
		
		# record -> pending to replace_child of parent
		
		if ( defined( $commands->{ 'record' } ) ) {
			my $name = $commands->{ 'record' } ;
			my ( $local, $localName ) = $this->_get_template_value_local( $data, $name ) ;
			
			if ( defined( $local->{ 'record' } ) ) {
				if ( defined( $local->{ 'record' }->{ $localName } ) ) {
					my $records_data = @{ $local->{ 'record' }->{ $localName } } ;
					
					if ( $records_data > 0 ) {
						unless ( defined( $records->{ $name } ) ) {
							$records->{ $name } = {
								'data'		=> $local->{ 'record' }->{ $localName },
								'elements'	=> []
							} ;
						}
						
						my $n = @{ $records->{ $name }->{ 'elements' } } ;
						
						if ( $n < $records_data ) {
							$records->{ $name }->{ 'data' }->[ $n ]->{ 'root' } = $this->data ;
							$records->{ $name }->{ 'data' }->[ $n ]->{ 'parent' } = $data ;
							
							push (
								@{ $records->{ $name }->{ 'elements' } },
								{
									'element' => $element,
									'records'	=> {}
								}
							) ;
							
							$this->replace_element(
								$records->{ $name }->{ 'elements' }->[ $n ]->{ 'element' },
								$records->{ $name }->{ 'data' }->[ $n ],
								$records->{ $name }->{ 'elements' }->[ $n ]->{ 'records' },
								1,
								1
							) ;
							
							return ;
						}
					}
				}
			}
			
			$this->replace_remove( $element ) ;
			return ;
			
			if ( defined( $local->{ 'record' }{ $localName } ) ) {
				unless ( defined( $records->{ $commands->{ 'record' } } ) ) {
					$records->{ $commands->{ 'record' } } = [] ;
				}
				
				push( @{ $records->{ $commands->{ 'record' } } }, $element ) ;
				
				return ;
			}
		}
		
		# if/else
		
		if ( defined( $commands->{ 'condition' } ) ) {
			my $remove = 0 ;
			
			for ( my $n = 0 ; $n <= $#{ $commands->{ 'condition' } } ; $n++ ) {
				my $condition = $commands->{ 'condition' }->[ $n ]->{ 'condition' } ;
				my $name = $commands->{ 'condition' }->[ $n ]->{ 'name' } ;
				my $check = $condition ;
				
				my ( $local, $localName ) = $this->_get_template_value_local( $data, $name ) ;
				
				if ( defined( $local->{ 'condition' }->{ $localName } ) ) {
					$check = $local->{ 'condition' }->{ $localName } ;
				} else {
					if ( defined( $local->{ 'record' }->{ $localName } ) ) {
						$check = ( $#{ $local->{ 'record' }->{ $localName } } != -1 ) ;
					} else {
						if ( defined( $local->{ 'value' }->{ $localName } ) ) {
							$check = ( $local->{ 'value' }->{ $localName } ne '' ) ;
						}
					}
				}
				
				if ( $check ) {
					unless ( $condition ) {
						$remove = 1 ;
						last ;
					}
				} else {
					if ( $condition ) {
						$remove = 1 ;
						last ;
					}
				}
			}
			
			if ( $remove ) {
				$this->replace_remove( $element ) ;
				return ;
			}
		}
		
		# parse_outer
		
		if ( defined( $commands->{ 'parse_outer' } ) ) {
			$this->replace_parse_outer( $element, $data, $commands->{ 'parse_outer' }, $records, $in_record ) ;
			return ;
		}
		
		# outer
		
		if ( defined( $commands->{ 'outer' } ) ) {
			if ( $in_record ) {
				# as parse_outer
				$this->replace_parse_outer( $element, $data, $commands->{ 'outer' }, $records, $in_record ) ;
			}
			
			return ;
		}
	}
	
	# attribute
	
	if ( defined( $commands->{ '@' } ) ) {
		for ( my $n = 0 ; $n <= $#{ $commands->{ '@' } } ; $n++ ) {
			my $attribute = $commands->{ '@' }->[ $n ]->{ 'attribute' } ;
			my $name = $commands->{ '@' }->[ $n ]->{ 'name' } ;
			
			my ( $local, $localName ) = $this->_get_template_value_local( $data, $name ) ;
			
			unless ( defined( $local->{ 'value' }->{ $localName } ) ) {
				next ;
			}
			
			if ( $local->{ 'value' }->{ $localName } eq '' ) {
				if ( $this->in_array( $attribute, $this->{ 'blankAttribute' } ) ) {
					if ( $element->hasAttribute( $attribute ) ) {
						$element->removeAttribute( $attribute ) ;
						next ;
					}
				}
			}
			
			$element->setAttribute( $attribute, $local->{ 'value' }->{ $localName } ) ;
		}
	}
	
	# other
	
	if ( defined( $commands->{ 'other' } ) ) {
		for ( my $n = 0 ; $n <= $#{ $commands->{ 'other' } } ; $n++ ) {
			my $command = $commands->{ 'other' }->[ $n ]->{ 'command' } ;
			my $name = $commands->{ 'other' }->[ $n ]->{ 'name' } ;
			
			if ( $command eq 'function' ) {
				my ( $local, $localName ) = $this->_get_template_value_local( $data, $name ) ;
				
				unless ( defined( $local->{ 'function' }->{ $localName } ) ) {
					return ;
				}
				
				my $method = $local->{ 'function' }->{ $localName } ;
				
				if ( ref( $method ) eq 'CODE' ) {
					&$method( $this, $element, $data ) ;
				}
			} else {
				my $method = "command_{$command}" ;
				
				if ( $this->can( $method ) ) {
					$this->$method( $element, $data, $command, $name ) ;	# TODO: check
				}
			}
		}
	}
	
	# text
	
	if ( defined( $commands->{ 'text' } ) ) {
		my $name = $commands->{ 'text' } ;
		
		my ( $local, $localName ) = $this->_get_template_value_local( $data, $name ) ;
		
		if ( defined( $local->{ 'value' }->{ $localName } ) ) {
			my $br = ( $this->{ 'xhtml' } ) ? '<br />' : '<br>' ;
			
			my $html = $this->htmlspecialchars( $local->{ 'value' }->{ $localName } ) ;
			$html =~ s/\r?\n/$br/sg ;
			
			while ( $element->childNodes->length > 0 ) {
				$element->removeChild( $element->lastChild ) ;
			}
			
			$this->_parse( $element, \$html ) ;
			$this->replace_childs( $element, $data, $records, $in_record ) ;
		}
		
		return ;
	}
	
	# textarea
	
	if ( defined( $commands->{ 'textarea' } ) ) {
		my $name = $commands->{ 'textarea' } ;
		
		( my $local, my $localName ) = $this->_get_template_value_local( $data, $name ) ;
		
		if ( defined( $local->{ 'value' }->{ $localName } ) ) {
			my $html = $this->htmlspecialchars( $local->{ 'value' }->{ $localName } ) ;
		
			while ( $element->childNodes->length > 0 ) {
				$element->removeChild( $element->firstChild ) ;
			}
			
			$this->_parse( $element, \$html ) ;
		}
		
		return ;
	}
	
	# inner
	
	if ( defined( $commands->{ 'inner' } ) ) {
		if ( $in_record ) {
			# as parse_inner
			$this->replace_parse_inner( $element, $data, $commands->{ 'inner' }, $records, $in_record ) ;
		} else {
			return ;
		}
	}
	
	# parse_inner
	
	if ( defined( $commands->{ 'parse_inner' } ) ) {
		$this->replace_parse_inner( $element, $data, $commands->{ 'parse_inner' }, $records, $in_record ) ;
	}
	
	# child
	
	$this->replace_childs( $element, $data, $records, $in_record ) ;
}


sub replace_childs {
	my $this = shift ;
	my $element = shift ;
	my $data = shift ;
	my $records = shift ;
	my $in_record = shift ;
	
	# removeChild もあるので一覧作成
	
	my @childs = () ;
	
	for ( my $n = 0 ; $n < $element->childNodes->length ; $n++ ) {
		if ( $element->childNodes->item( $n )->{ 'nodeType' } == $JTE_DOM::XML_ELEMENT_NODE ) {
			my $child = $element->childNodes->item( $n ) ;
			push( @childs, $child ) ;
		}
	}
	
	for ( my $n = 0 ; $n <= $#childs ; $n++ ) {
		$this->replace_element( $childs[ $n ], $data, $records, $in_record ) ;
	}
}


# replace_record

sub replace_record {
	my $this = shift ;
	my $record = shift ;
	
	my $records_elements = @{ $record->{ 'elements' } } ;
	my $records_data = @{ $record->{ 'data' } } ;
	
	# 必要な行を用意
	
	if ( $records_elements < $records_data ) {
		my $parent = $record->{ 'data' }->[ $records_elements - 1 ]->{ 'parent' } ;
		my $last = $record->{ 'elements' }->[ $records_elements - 1 ]->{ 'element' } ;
		
		# 追加位置
		
		my $anchor = $this->dom->createElement( 'div' ) ;
		
		if ( defined( $last->nextSibling ) ) {
			$last->parentNode->insertBefore( $anchor, $last->nextSibling ) ;
		} else {
			$last->parentNode->appendChild( $anchor ) ;
		}
		
		# 追加材料
		
		my @source = () ;
		
		if ( defined( $last->previousSibling ) ) {
			if ( $last->previousSibling->nodeType == $JTE_DOM::XML_TEXT_NODE ) {
				if ( $last->previousSibling->nodeValue =~ /\A\s*\z/s ) {
					push( @source, $last->previousSibling->cloneNode( 1 ) ) ;
				}
			}
		}
		
		for ( my $n = $records_elements ; $n < $records_data ; $n++ ) {
			for ( my $s = 0 ; $s <= $#{source} ; $s++ ) {
				$anchor->parentNode->insertBefore( $source[ $s ]->cloneNode( 1 ), $anchor ) ;
			}
			
			my $element = $last->cloneNode( 1 ) ;
			
			$anchor->parentNode->insertBefore( $element, $anchor ) ;
			
			$record->{ 'data' }->[ $n ]->{ 'root' } = $this->data ;
			$record->{ 'data' }->[ $n ]->{ 'parent' } = $parent ;
			
			push( @{ $record->{ 'elements' } }, { 'element' => $element, 'records' => {} } ) ;
			
			$this->replace_element(
				$record->{ 'elements' }->[ $n ]->{ 'element' },
				$record->{ 'data' }->[ $n ],
				$record->{ 'elements' }->[ $n ]->{ 'records' },
				1,
				1
			) ;
		}
		
		$last->parentNode->removeChild( $anchor ) ;
	}
	
	# 子レコード
	
	for ( my $n = 0 ; $n < $records_elements ; $n++ ) {
		foreach my $name ( %{ $record->{ 'elements' }->[ $n ]->{ 'records' } } ) {
			$this->replace_record( $record->{ 'elements' }->[ $n ]->{ 'records' }->{ $name } ) ;
		}
	}
}


# remove

sub replace_remove {
	my $this = shift ;
	my $element = shift ;
	
	if ( defined( $element->previousSibling ) ) {
		if ( $element->previousSibling->nodeType == $JTE_DOM::XML_TEXT_NODE ) {
			if ( $element->previousSibling->nodeValue =~ /\A\s*\z/s ) {
				$element->parentNode->removeChild( $element->previousSibling ) ;
			}
		}
	}
	
	$element->parentNode->removeChild( $element ) ;
}


# parse_outer

sub replace_parse_outer {
	my $this = shift ;
	my $element = shift ;
	my $data = shift ;
	my $name = shift ;
	my $records = shift ;
	my $in_record = shift ;
	
	my $html = $this->_get_template_contents( $data, $name ) ;
	
	if ( ! $html ) {
		return ;
	}
	
	$html = $this->delete_bom( $html ) ;
	chomp( $html ) ;
	
	my @childs = () ;
	
	my $div = $this->dom->createElement( 'div' ) ;
	
	$this->_parse( $div, \$html ) ;
	
	for ( my $n = 0 ; $n < $div->childNodes->length ; $n++ ) {
		my $child = $div->childNodes->item( $n )->cloneNode() ;
		
		$element->parentNode->insertBefore( $child, $element ) ;
		
		if ( $child->nodeType == $JTE_DOM::XML_ELEMENT_NODE ) {
			push( @childs, $child ) ;
		}
	}
	
	$element->parentNode->removeChild( $element ) ;	
	
	for ( my $n = 0 ; $n <= $#childs ; $n++ ) {
		$this->replace_element(
			$childs[ $n ],
			$data,
			$records,
			$in_record
		) ;
	}
}


# parse_inner

sub replace_parse_inner {
	my $this = shift ;
	my $element = shift ;
	my $data = shift ;
	my $name = shift ;
	my $records = shift ;
	my $in_record = shift ;
	
	my $html = $this->_get_template_contents( $data, $name ) ;
	
	if ( ! $html ) {
		return ;
	}
	
	$html = $this->delete_bom( $html ) ;
	chomp( $html ) ;
	$html = $this->_inner_keep_indent( $element, $html ) ;
	
	# 空にする
	
	while ( $element->childNodes->length > 0 ) {
		$element->removeChild( $element->firstChild ) ;
	}
	
	$this->_parse( $element, \$html ) ;
}


#=======================================================================
# save

sub save {
	my $this = shift ;
	
    my $html = $this->{ 'docType' } . "\n" ;
    
    $this->_tree( \$html, $this->dom->documentElement, $this->{ 'data' } ) ;
    
    return $html ;
}


#=======================================================================
# _tree

sub _tree {
	my $this = shift ;
	my $html_ref = shift ;
	my $node = shift ;
	my $data = shift ;
	my $in_record = shift ;
	
	my $emptyTagEnd = ( $this->{ 'xhtml' } ) ? ' />' : '>' ;
	
	if ( $node->nodeType == $JTE_DOM::XML_ELEMENT_NODE ) {
		my @tag = ( $node->tagName ) ;
		
		my $commands = $this->_get_template_commands( $node ) ;
		$node->removeAttribute( $this->{ 'templateAttribute' } ) ;
		
		unless ( $in_record ) {
			$in_record = ( defined( $commands->{ 'record' } ) ) ;
		}
		
		if ( ( ! $in_record ) and ( defined( $commands->{ 'outer' } ) ) ) {
			my $contents = $this->_get_template_contents( $data, $commands->{ 'outer' } ) ;
			$$html_ref .= $contents ;
			return ;
		}
		
		for ( my $n = 0 ; $n < $node->{ 'attributes' }->length ; $n++ ) {
			my $name = $node->attributes->item( $n )->name ;
			my $value = $node->attributes->item( $n )->value ;
			
			if ( $value eq '' ) {
				if ( $this->in_array( $name, $this->{ 'blankAttribute' } ) ) {
					next ;
				}
			}
			
			if ( $this->{ 'xhtml' } ) {
				push( @tag, $this->htmlspecialchars( $name ) . '="' . $value . '"' ) ;
			} else {
				if ( in_array( $name, $this->{ 'blankAttribute' } ) ) {
					push( @tag, $this->htmlspecialchars( $name ) ) ;
				} else {
					push( @tag, $this->htmlspecialchars( $name ) . '="' . $value . '"' ) ;
				}
			}
		}
		
		if ( $this->in_array( $node->tagName, $this->{ 'emptyTag' } ) ) {
			$$html_ref .= "<" . join( ' ', @tag ) . ( ( $this->{ 'xhtml' } ) ? ' />' : '>' ) ;
		} else {
			$$html_ref .= "<" . join( ' ', @tag ) . ">" ;
			
			if ( ( ! $in_record ) and ( defined( $commands->{ 'inner' } ) ) ) {
				my $text = $this->_get_template_contents( $data, $commands->{ 'inner' } ) ;
				$text = $this->delete_bom( $text ) ;
				chomp( $text ) ;
				$text = $this->_inner_keep_indent( $node, $text ) ;
				
				$$html_ref .= $text ;
			} else {
				for ( my $n = 0 ; $n < $node->childNodes->length ; $n++ ) {
					$this->_tree( $html_ref, $node->childNodes->item( $n ), $data, $in_record ) ;
				}
			}
			
			$$html_ref .= "</" . $node->tagName . ">" ;
		}
		
		return ;
	}
	
	if ( $node->nodeType == $JTE_DOM::XML_COMMENT_NODE ) {
		$$html_ref .= "<!--" . $node->nodeValue . "-->" ;
		return ;
	}
	
	if ( $node->nodeType == $JTE_DOM::XML_CDATA_SECTION_NODE ) {
		$$html_ref .= "<![CDATA[" . $node->nodeValue . "]]>" ;
		return ;
	}
	
	$$html_ref .= $node->nodeValue ;
}


#=======================================================================
# execute

sub execute {
	my $this = shift ;
	my $path = shift ;
	my $data = shift ;
	
    $this->load( $path ) ;
    $this->set( $data ) ;
    $this->replace() ;
    print $this->save() ;
}


#=====================================================================
# part of html

sub part_load {
	my $this = shift ;
	my $html = shift ;
	
	$html = $this->delete_bom( $html ) ;
	$html =~ s/\A\s*(.*?)\s*\z/$1/s ;	# trim
	
	$this->{ 'dom' }->{ 'encoding' } = $this->{ 'encoding' } ;
	
	my $part = "<part>$html</part>" ;
	
	$this->_parse( \$this->dom, \$part ) ;
}


sub part_replace {
	my $this = shift ;
	my $data = shift ;
	
	my $records = {} ;
	
	$this->replace_childs( $this->{ 'dom' }->{ 'documentElement' }, $data, $records ) ;
	
	foreach my $name ( keys( %{ $records } ) ) {
		$this->replace_record( $records->{ $name } ) ;
	}
}


sub part_save {
	my $this = shift ;
	my $data = shift ;
	my $xhtml = shift || 1 ;
	
	$this->{ 'xhtml' } = $xhtml ;
	
	my $html = "" ;
	
	for ( my $n = 0 ; $n < $this->dom->documentElement->childNodes->length ; $n++ ) {
		$this->_tree( \$html, $this->dom->documentElement->childNodes->item( $n ), $data ) ;
	}
	
	return $html ;
}


sub part_execute {
	my $this = shift ;
	my $html = shift ;
	my $data = shift ;
	my $xhtml = shift || 1 ;
	
	$this->part_load( $html ) ;
	$this->part_replace( $data ) ;
	return $this->part_save( $data, $xhtml ) ;
}


#=======================================================================
# common

sub log {
	my $this = shift ;
	my $text = shift ;
	
	unless ( $this->{ 'log' } ) {
		return ;
	}
	
	open( OUT, '>>' . $this->{ 'log_file' } ) ;
	print OUT $text . "\n" ;
	close( OUT ) ;
}


sub delete_bom {
	my $this = shift ;
	my $text = shift ;
	
    $text =~ s/\A\xef\xbb\xbf//s ;
	
	return $text ;
}


sub _get_template_commands {
	my $this = shift ;
	my $element = shift ;
	
	my @attributes = split( '\s*;\s*', $element->getAttribute( $this->{ 'templateAttribute' } ) ) ;
	
	my $commands = {} ;
	
	for ( my $a = 0 ; $a <= $#attributes ; $a++ ) {
		my $command_name = $attributes[ $a ] ;
		$command_name =~ s/\A\s*(.*?)\s*\z/$1/s ;
		
		if ( $command_name eq '' ) {
			next ;
		}
		
		my $command = $command_name ;
		my $name = '' ;
		
		if ( $command_name =~ /\A([^:]*)\s*:\s*(.*)\z/s ) {
			$command = $1 ;
			$name = $2 ;
		}
		
		if ( $command =~ /\A@(.+)\z/s ) {
			my $attribute = lc( $1 ) ;
			
			unless ( defined( $commands->{ '@' } ) ) {
				$commands->{ '@' } = [] ;
			}
			
			push( @{ $commands->{ '@' } }, { 'attribute' => $attribute, 'name' => $name } ) ;
			
			next ;
		}
		
		if ( $this->in_array( $command, [ 'fake', 'remove', 'dummy' ] ) ) {
			$commands->{ 'fake' } = 1 ;
			next ;
		}
		
		if ( $this->in_array( $command, [ 'record', 'parse_outer', 'parse_inner', 'outer', 'inner', 'text', 'textarea' ] ) ) {
			$commands->{ $command } = $name ;
			next ;
		}
		
		if ( $command eq 'pre' ) {
			$commands->{ 'textarea' } = $name ;
			next ;
		}
		
		if ( $this->in_array( $command, [ 'if', 'else', 'condition', 'condition_not' ] ) ) {
			my $condition = ( ( $command eq 'if' ) || (  $command eq 'condition'  ) ) ;
			
			unless ( defined( $commands->{ 'condition' } ) ) {
				$commands->{ 'condition' } = [] ;
			}
			
			push( @{ $commands->{ 'condition' } }, { 'condition' => $condition, 'name' => $name } ) ;
			
			next ;
		}
		
		unless ( defined( $commands->{ 'other' } ) ) {
			$commands->{ 'other' } = [] ;
		}
		
		push( @{ $commands->{ 'other' } }, { 'command' => $command, 'name' => $name } ) ;
	}
	
	return $commands ;
}


sub _get_template_contents {
	my $this = shift ;
	my $data = shift ;
	my $name = shift ;
	
	if ( $name =~ /\Aurl\(([^)]*)\)\z/s ) {
		my @parts = () ;
		my @urls = split( '(\s+|\s*,\s*)', $1 ) ;
		
		for ( my $n = 0 ; $n <= $#urls ; $n++ ) {
			my $url = $urls[ $n ] ;
			
			if ( $url eq '' ) {
				next ;
			}
			
			my $part = $this->file_get_contents( $url ) ;
			$part = $this->delete_bom( $part ) ;
			
			push( @parts, $part ) ;
		}
		
		return join( "\n", @parts ) ;
	}
	
	return $this->_get_template_value( $data, $name ) ;
}


sub _get_template_value {
	my $this = shift ;
	my $data = shift ;
	my $name = shift ;
	
	my ( $local, $localName ) = $this->_get_template_value_local( $data, $name ) ;
	
	unless ( defined( $local->{ 'value' }->{ $localName } ) ) {
		return '' ;
	}
	
	return $local->{ 'value' }->{ $localName } ;
}


sub _get_template_value_local {
	my $this = shift ;
	my $data = shift ;
	my $name = shift ;
	
	my $local = $data ;
	my $localName = $name ;
	
	if ( $name =~ /\A([^^\.]+?)\.(.+)\z/s ) {
		my $localSet = lc( $1 ) ;
		my $localSetName = $2 ;
		
		if ( $localSet eq 'root' ) {
			( $local, $localName ) = ( $data->{ 'root' }, $localSetName ) ;
		}
		
		if ( $localSet eq 'parent' ) {
			( $local, $localName ) = ( $data->{ 'parent' }, $localSetName ) ;
		}
	}
	
	return ( $local, $localName ) ;
}


sub _inner_keep_indent {
	my $this = shift ;
	my $element = shift ;
	my $html = shift ;
	
	if ( $element->childNodes->length == 0 ) {
		return $html ;
	}
	
	my $first = '' ;
	my $last = '' ;
	
	# 開始タグの後の改行までの空白・コメントを保持
	
	while ( $element->childNodes->length > 0 ) {
		if ( $element->firstChild->nodeType == $JTE_DOM::XML_TEXT_NODE ) {
			my $text = $element->firstChild->nodeValue ;
			
			# 改行がある場合(終了)
			
			if ( $text =~ /\A(.*?)(\r?\n)(.*)\z/s ) {
				my $space = $1 ;
				my $lf = $2 ;
				my $other = $3 ;
				
				if ( $space =~ /\A\s*\z/s ) {
					$first = "${first}${space}${lf}" ;
					$element->firstChild->nodeValue = "${lf}${other}" ;
				} else {
					$first = "${first}${lf}" ;
				}
				
				last ;
			}
			
			# 空白でない場合(終了)
			
			unless ( $text =~ /\A\s*\z/s ) {
				last ;
			}
			
			# 空白を保持
			
			$first = "${first}${text}" ;
			$element->removeChild( $element->firstChild ) ;
			
			next ;
		}
		
		# コメントを保持
		
		if ( $element->firstChild->nodeType == $JTE_DOM::XML_COMMENT_NODE ) {
			$first = "${first}<!--" . $element->firstChild->nodeValue . "-->" ;
			$element->removeChild( $element->firstChild ) ;
			
			next ;
		}
		
		last ;
	}
	
	# 閉じタグのインデントを保持
	
	while ( $element->{ 'childNodes' }->length > 0 ) {
		if ( $element->lastChild->{ 'nodeType' } == $JTE_DOM::XML_TEXT_NODE ) {
			my $text = $element->lastChild->nodeValue ;
			
			# 改行がある場合(終了)
			
			if ( $text =~ /\A(.*)(\r?\n)(.*?)\z/s ) {
				my $other = $1 ;
				my $lf = $2 ;
				my $space = $3 ;
				
				if ( $space =~ /\A\s*\z/s ) {
					$last = "${lf}${space}${last}" ;
					$element->lastChild->nodeValue = "${other}${lf}" ;
				} else {
					$last = "${lf}${last}" ;
				}
				
				last ;
			}
			
			# 空白でない場合(終了)
			
			unless ( $text =~ /\A\s*\z/s ) {
				last ;
			}
			
			# 空白を保持
			
			$last = "${text}${last}" ;
			$element->removeChild( $element->lastChild ) ;
			
			next ;
		}
		
		# コメントを保持
		
		if ( $element->lastChild->nodeType == $JTE_DOM::XML_COMMENT_NODE ) {
			$last = "<!--" . $element->lastChild->nodeValue . "-->${last}" ;
			$element->removeChild( $element->lastChild ) ;
			
			next ;
		}
		
		last ;
	}
	
	return "${first}${html}${last}" ;
}


#=======================================================================
# from PHP

sub htmlspecialchars {
	my $this = shift ;
	my $text = shift ;
	
	$text =~ s/&/&amp;/g ;
	$text =~ s/"/&quot;/g ;
	$text =~ s/</&lt;/g ;
	$text =~ s/>/&gt;/g ;
	
	return $text ;
}


sub in_array {
	my $this = shift ;
	my $value = shift ;
	my $array_ref = shift ;
	
	my $find = '' ;
	my $l = @{ $array_ref } ;
	
	for ( my $n = 0 ; $n < $l ; $n++ ) {
		if ( $array_ref->[ $n ] eq $value ) {
			$find = 1 ;
			last ;
		}
	}
	
	return $find ;
}

  
sub file_get_contents {
	my $this = shift ;
	my $path = shift ;
	
	my $contents = '' ;
	
	unless ( defined( open( IN, $path ) ) ) {
		return undef ;
	}
	
	while ( <IN> ) {
		$contents .= $_ ;
	}
	
	close( IN ) ;
	
	return $contents ;
}

  
1;