/*
 * Web Application Template
 * 
 * Copyright (C) 2006 Satoshi Nagashiba, All Rights Reserved.
 */
package org.jtp.dao;

import java.util.Calendar;
import java.util.Date;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Query;
import org.springframework.orm.hibernate3.support.HibernateDaoSupport;

/**
 * 基本データアクセスクラスです。 
 * 
 * @author <a href="mailto:sato-vista@jade.plala.or.jp">Satoshi Nagashiba</a>
 */
public abstract class BaseDao extends HibernateDaoSupport {

    /**
     * 検索用の条件文字列に変換します。
     * @param value 変換する文字列、または<tt>null</tt>
     * @return 前方一致の<tt>LIKE</tt>の条件
     */
    public String toCondition(String value) {
        if (StringUtils.isEmpty(value)) {
            return null;
        }
        return value;
    }

    /**
     * 検索用の条件文字列に変換します。
     * @param value 変換する文字列、または<tt>null</tt>
     * @return 前方一致の<tt>LIKE</tt>の条件
     */
    public int toCondition(Integer value) {
        return (value == null) ? -1 : value.intValue();
    }

    /**
     * 検索用の条件文字列に変換します。
     * @param value 変換する文字列、または<tt>null</tt>
     * @return 前方一致の<tt>LIKE</tt>の条件
     */
    public byte toCondition(Byte value) {
        return (value == null) ? -1 : value.byteValue();
    }

    /**
     * 前方一致の<tt>LIKE</tt>に文字列を変換します。
     * @param value 変換する文字列、または<tt>null</tt>
     * @return 前方一致の<tt>LIKE</tt>の条件
     */
    public String toForwardMatch(String value) {
        if (StringUtils.isEmpty(value)) {
            return null;
        }
        return value + "%";
    }

    /**
     * 後方一致の<tt>LIKE</tt>に文字列を変換します。
     * @param value 変換する文字列、または<tt>null</tt>
     * @return 後方一致の<tt>LIKE</tt>の条件
     */
    public String toBackwardMatch(String value) {
        if (StringUtils.isEmpty(value)) {
            return null;
        }
        return "%" + value;
    }

    /**
     * 部分一致の<tt>LIKE</tt>に文字列を変換します。
     * @param value 変換する文字列、または<tt>null</tt>
     * @return 部分一致の<tt>LIKE</tt>の条件
     */
    public String toPartMatch(String value) {
        if (StringUtils.isEmpty(value)) {
            return null;
        }
        return "%" + value + "%";
    }

    /**
     * <p>日付範囲検索の開始日として使用可能な当日の開始日時に変換します。</p>
     * <p>検索条件では<tt>xxxDate &gt;= :xxxDateFrom</tt>形式で指定します。</p>
     * @param value 変換する日付、または<tt>null</tt>
     * @return 日付範囲検索の開始日用の日付
     */
    public Date toDateFrom(Date value) {
        if (value == null) {
            return null;
        }
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(value);
        calendar.clear(Calendar.AM_PM);
        calendar.clear(Calendar.HOUR);
        calendar.clear(Calendar.HOUR_OF_DAY);
        calendar.clear(Calendar.MINUTE);
        calendar.clear(Calendar.SECOND);
        calendar.clear(Calendar.MILLISECOND);
        return calendar.getTime();
    }

    /**
     * <p>日付範囲検索の終了日として使用可能な翌日の開始日時に変換します。</p>
     * <p>検索条件では<tt>xxxDate &lt; :xxxDateTo</tt>形式で指定します。</p>
     * @param value 変換する日付、または<tt>null</tt>
     * @return 日付範囲検索の終了日用の日付
     */
    public Date toDateTo(Date value) {
        if (value == null) {
            return null;
        }
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(value);
        calendar.clear(Calendar.AM_PM);
        calendar.clear(Calendar.HOUR);
        calendar.clear(Calendar.HOUR_OF_DAY);
        calendar.clear(Calendar.MINUTE);
        calendar.clear(Calendar.SECOND);
        calendar.clear(Calendar.MILLISECOND);
        calendar.add(Calendar.DAY_OF_MONTH, 1);
        return calendar.getTime();
    }

    public int getCount(Query query) {
        List list = query.list();
        if (list != null) {
            return ((Integer) list.get(0)).intValue();
        }
        return -1;
    }
}
