/*
 * Java Template Project
 * 
 * Copyright (C) 2006 Satoshi Nagashiba, All Rights Reserved.
 */
package org.jtp.web.action;

import java.io.BufferedOutputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;


import org.apache.struts.action.ActionForm;
import org.apache.struts.action.ActionForward;
import org.apache.struts.action.ActionMapping;
import org.jtp.SampleException;
import org.jtp.common.Encode;

/**
 * ダウンロード機能を実装したアクションクラスです。
 *
 * @author <a href="mailto:sato-vista@jade.plala.or.jp">Satoshi Nagashiba</a>
 */
public abstract class BaseDownloadAction extends BaseAction implements Encode {

	/** クライアントに送り返されるレスポンスのコンテントタイプ - バイナリデータ */
	private static final String CONTENT_TYPE_STREAM = "application/octet-stream";
	/** レスポンスヘッダ名称 */
	private static final String HEADER_DISPOSITION = "Content-Disposition";
	/** レスポンスヘッダへ追加する値 */
	private static final String HEADER_VALUE = "attachment; filename=";

	
    /**
     * ファイルをダウンロードします。
     * @param mapping  このインスタンスを選択するために使用したActionMapping
     * @param form     存在するならば、このリクエストのためのActionForm Bean
     * @param request  処理しているHTTPリクエスト
     * @param response 処理しているHTTPレスポンス
     * @return 指定された非HTTPリクエストを処理して、 対応する非HTTPレスポンスを生成します
     * （またはレスポンスを生成する他のWebコンポーネントにリクエストをフォワードします）。
     * @throws SampleException システム例外
     */
	public ActionForward download(ActionMapping mapping, ActionForm form,
            HttpServletRequest request, HttpServletResponse response) throws Exception {
    	response.setContentType(CONTENT_TYPE_STREAM);
    	response.setHeader(HEADER_DISPOSITION, HEADER_VALUE  + encode(getDisplayName()));
        
		try {
	    	InputStream fis = getDownloadStream(form, request);
			BufferedOutputStream fos = new BufferedOutputStream(response.getOutputStream());
	        int i;
			while((i = fis.read()) != -1){
	        	fos.write(i);
	        }
			fos.flush();
			fos.close();
			fis.close();
        }
        catch (IOException e) {
        	throw new SampleException(e);
        }
        
        return null;
    }
    
    /**
     * 表示ファイル名を取得します。
     * @return 表示ファイル名
     */
    protected abstract String getDisplayName();

    /**
     * ダウンロードファイルの入力ストリームを取得します。
     * @param form 存在するならば、このリクエストのためのActionForm Bean
     * @param request 処理しているHTTPリクエスト
     * @return ダウンロードファイルの入力ストリーム
     * @throws FileNotFoundException 指定されたパス名で示されるファイルが開けなかったことを通知する例外
     * @throws UnsupportedEncodingException 文字のエンコーディングがサポートされないことを通知する例外
     */
    protected abstract InputStream getDownloadStream(ActionForm form,
    		HttpServletRequest request) throws FileNotFoundException, UnsupportedEncodingException;
    
    /**
     * 日本語の文字化け防止のエンコーディング
     * @param str 対象文字列
     * @return String 変換後の文字列
     * @throws SampleException システム例外
     */
    private String encode(String str) throws SampleException {
    	String result = null;
    	try {
    		if (str != null) {
    			result = java.net.URLEncoder.encode(str, UTF8);
    		}
        }
    	catch (UnsupportedEncodingException e) {
    		throw new SampleException(e);
    	}
    	return result;
    }
}
