/*
 * Java Template Project
 * 
 * Copyright (C) 2006 Satoshi Nagashiba, All Rights Reserved.
 */
package org.jtp.logic.interceptor;

import java.text.MessageFormat;

import org.aopalliance.intercept.MethodInterceptor;
import org.aopalliance.intercept.MethodInvocation;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jtp.common.util.StringUtils;
import org.springframework.util.ClassUtils;

/**
 * 実行メソッドの開始／終了をロギングするインターセプタークラスです。
 * 
 * @author <a href="mailto:sato-vista@jade.plala.or.jp">Satoshi Nagashiba</a>
 */
public class MethodTraceInterceptor implements MethodInterceptor {
    
    /** 開始ログメッセージ. */
    private String enterMessage = "Start - class[{0}] method[{1}],";
    /** 終了ログメッセージ. */
    private String exitMessage = "End - class[{0}] method[{1}],";
    
    /**
     * メソッドをインターセプトします。
     * @param invocation 実行メソッド
     */
	public Object invoke(MethodInvocation invocation) throws Throwable {
		Log logger = LogFactory.getLog(invocation.getThis().getClass());

    	long before = 0;
        long after = 0;
        Object[] param = null;

        param = initMessageFormatParameters(invocation);
        logger.info(MessageFormat.format(enterMessage, param));
        before = System.currentTimeMillis();
        try {
            return invocation.proceed();
        } finally {
            String emsg = MessageFormat.format(exitMessage, param);
            after = System.currentTimeMillis();
            emsg += " Time = " + (after - before) + "msec.,";
            logger.info(emsg);
        }
    }

    /**
     * インターセプト情報を取得する。<br>.
     * @param invocation メソッド呼び出し
     * @return プレースホルダー用メッセージ配列
     */
    private Object[] initMessageFormatParameters(MethodInvocation invocation) {
        String className = invocation.getMethod().getDeclaringClass().getName();
        String methodName = invocation.getMethod().getName();
        String argumentStr = getMethodArguments(invocation);
        String parameterTypeStr = getMethodParameterTypess(invocation);
        return new Object[]{className, methodName, argumentStr, parameterTypeStr};
    }    
    
    /**
     * メソッドの引数を取得します。<br>.
     * @param invocation メソッド呼び出し
     * @return メソッド引数文字列
     */
    private String getMethodArguments(MethodInvocation invocation) {
        return (invocation.getArguments() == null) ? "()" : "("
				+ StringUtils.toCommaDelimitedString(invocation.getArguments()) + ")";
    }

    /**
     * メソッドの引数タイプを取得します。<br>.
     * @param invocation メソッド呼び出し
     * @return メソッド引数タイプ文字列
     */
    private String getMethodParameterTypess(MethodInvocation invocation) {
        if (invocation.getMethod().getParameterTypes().length == 0) {
            return "()";
        }
        else {
            Class[] types = invocation.getMethod().getParameterTypes();
            String[] names = new String[types.length];
            for (int i = 0; i < types.length; i++) {
                Class type = types[i];
                names[i] = ClassUtils.getShortName(type);
            }
            return "(" + StringUtils.toCommaDelimitedString(names) + ")";
        }
    }
}
