/*
 * Java Template Project
 * 
 * Copyright (C) 2006 Satoshi Nagashiba, All Rights Reserved.
 */
package org.jtp.logic.mail;

import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.velocity.VelocityContext;
import org.jtp.SampleException;
import org.jtp.dao.mail.MailAttachTableDao;
import org.jtp.dao.mail.MailBccTableDao;
import org.jtp.dao.mail.MailCcTableDao;
import org.jtp.dao.mail.MailTableDao;
import org.jtp.entity.mail.MailAttachTable;
import org.jtp.entity.mail.MailBccTable;
import org.jtp.entity.mail.MailCcTable;
import org.jtp.entity.mail.MailTable;

import com.ozacc.mail.Mail;
import com.ozacc.mail.SendMail;
import com.ozacc.mail.VelocityMailBuilder;


/** 
 * メールテーブル情報のサービスクラスです。
 * 
 * @spring.bean id="mailTableService" 
 * @spring.property name="mailTableDao" ref="mailTableDao"
 * @spring.property name="mailCcTableDao" ref="mailCcTableDao"
 * @spring.property name="mailBccTableDao" ref="mailBccTableDao"
 * @spring.property name="mailAttachTableDao" ref="mailAttachTableDao"
 * @spring.property name="velocityMailBuilder" ref="velocityMailBuilder"
 * @spring.property name="sendMail" ref="sendMail"
 * @spring.property name="templatePath" value="/mail-template.xml"
 * 
 * @author <a href="mailto:sato-vista@jade.plala.or.jp">Satoshi Nagashiba</a>
 */
public class MailTableServiceImpl implements MailTableService {
	
    /** このクラスで発生した実行状況をログに記録するためのオブジェクトです。 */
    private Log _log = LogFactory.getLog(MailTableServiceImpl.class);

	/** メールテーブル情報DAO */
	private MailTableDao mailTableDao;

	/** CCメールテーブル情報DAO */
	private MailCcTableDao mailCcTableDao;

	/** BCCメールテーブル情報DAO */
	private MailBccTableDao mailBccTableDao;

	/** 添付メールテーブル情報DAO */
	private MailAttachTableDao mailAttachTableDao;

	/** Velocityと連携して動的にメールデータを生成するオブジェクト */
	private VelocityMailBuilder velocityMailBuilder;
	
	/** メール送信オブジェクト */
	private SendMail sendMail;
	
	/** メールテンプレートパス */
	private String templatePath;
	
    /**
     * Setter Injection
     */
	public void setMailTableDao(MailTableDao mailTableDao) {
		this.mailTableDao = mailTableDao;
	}

    /**
     * Setter Injection
     */
	public void setMailCcTableDao(MailCcTableDao mailCcTableDao) {
		this.mailCcTableDao = mailCcTableDao;
	}

    /**
     * Setter Injection
     */
	public void setMailBccTableDao(MailBccTableDao mailBccTableDao) {
		this.mailBccTableDao = mailBccTableDao;
	}

    /**
     * Setter Injection
     */
    public void setMailAttachTableDao(MailAttachTableDao mailAttachTableDao) {
		this.mailAttachTableDao = mailAttachTableDao;
	}

	/**
     * Setter Injection
     */
	public void setVelocityMailBuilder(VelocityMailBuilder velocityMailBuilder) {
		this.velocityMailBuilder = velocityMailBuilder;
	}

    /**
     * Setter Injection
     */
	public void setSendMail(SendMail sendMail) {
		this.sendMail = sendMail;
	}

    /**
     * Setter Injection
     */
	public void setTemplatePath(String templatePath) {
		this.templatePath = templatePath;
	}
    
	/**
	 * 検索します。
	 * @return エンティティのリスト
	 * @throws SampleException システム一般例外
	 */
    public List findHistory() throws SampleException {
    	return mailTableDao.findHistory();
    }
    
	/**
	 * 登録します。
	 * @param mailTable エンティティ
	 * @throws SampleException システム一般例外
	 */
    public void save(MailTable mailTable) throws SampleException {
    	mailTableDao.save(mailTable);
    	if (mailTable.getMailCcTables() != null) {
         	for (Iterator it = mailTable.getMailCcTables().iterator(); it.hasNext();) {
         		MailCcTable mailCcTable = (MailCcTable) it.next();
         		mailCcTable.getId().setId(mailTable.getId());
         	}
        	mailCcTableDao.save(mailTable.getMailCcTables());
    	}
    	if (mailTable.getMailBccTables() != null) {
         	for (Iterator it = mailTable.getMailBccTables().iterator(); it.hasNext();) {
         		MailBccTable mailBccTable = (MailBccTable) it.next();
         		mailBccTable.getId().setId(mailTable.getId());
         	}
        	mailBccTableDao.save(mailTable.getMailBccTables());
    	}
    	if (mailTable.getMailAttachTables() != null) {
         	for (Iterator it = mailTable.getMailAttachTables().iterator(); it.hasNext();) {
         		MailAttachTable mailAttachTable = (MailAttachTable) it.next();
         		mailAttachTable.getId().setId(mailTable.getId());
         	}
        	mailAttachTableDao.save(mailTable.getMailAttachTables());
    	}
    	mailSend(mailTable);
    }
    
    /**
     * メールを送信します。
     * @param mailTable エンティティ
     */
    private void mailSend(MailTable mailTable) {
        //メールの送信
        try {
            //メールの送信元／送信先を設定したOZACCメールオブジェクトを生成します
            VelocityContext context = new VelocityContext();
            context.put("mailTable", mailTable);
            Mail mail =  velocityMailBuilder.buildMail(templatePath, context);

            if (mail != null) {
                mail.setFrom(mailTable.getFrom());
                
                String to = mailTable.getTo();
                if (StringUtils.isNotEmpty(to)) {
                    mail.addTo(to);
                }

                Set cc = mailTable.getMailCcTables();
                if (cc != null) {
                    for (Iterator it = cc.iterator(); it.hasNext();) {
                    	MailCcTable entity = (MailCcTable) it.next();
                    	mail.addCc(entity.getCc());
                    }
                }
                
                Set bcc = mailTable.getMailBccTables();
                if (bcc != null) {
                    for (Iterator it = bcc.iterator(); it.hasNext();) {
                    	MailBccTable entity = (MailBccTable) it.next();
                    	mail.addCc(entity.getBcc());
                    }
                }
            
                Set attach = mailTable.getMailAttachTables();
                if (attach != null) {
                    for (Iterator it = attach.iterator(); it.hasNext();) {
                    	MailAttachTable entity = (MailAttachTable) it.next();
                    	mail.addFile(entity.getAttach(), entity.getAttachName());
                    }
                }

                sendMail.send(mail);
                _log.info("メール送信成功！！");
            }
            else {
            	_log.info("メール送信しない。");
            }
        }
        catch (Exception e) {
        	_log.error("メール送信失敗！！", e);
        }
    }
}
