/*
 * Java Template Project
 * 
 * Copyright (C) 2006 Satoshi Nagashiba, All Rights Reserved.
 */
package org.jtp.web.action;

import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.struts.action.ActionForm;
import org.apache.struts.action.ActionForward;
import org.apache.struts.action.ActionMapping;
import org.jtp.SampleException;
import org.jtp.common.Encode;
import org.jtp.web.WebConstants;

/**
 * ダウンロード機能を実装したアクションクラスです。
 *
 * @author <a href="mailto:sato-vista@jade.plala.or.jp">Satoshi Nagashiba</a>
 */
public abstract class BaseDownloadAction extends BaseAction implements Encode {

    /**
     * ファイルをダウンロードします。
     * @param mapping  このインスタンスを選択するために使用したActionMapping
     * @param form     存在するならば、このリクエストのためのActionForm Bean
     * @param request  処理しているHTTPリクエスト
     * @param response 処理しているHTTPレスポンス
     * @return 指定された非HTTPリクエストを処理して、 対応する非HTTPレスポンスを生成します
     * （またはレスポンスを生成する他のWebコンポーネントにリクエストをフォワードします）。
     * @throws SampleException システム例外
     */
	public ActionForward download(ActionMapping mapping, ActionForm form,
            HttpServletRequest request, HttpServletResponse response) throws Exception {
    	response.setContentType(getContentType(form, request));
    	response.setHeader(WebConstants.HEADER_DISPOSITION, WebConstants.HEADER_VALUE  + encode(getFileName(form, request)));

		try {
	    	InputStream fis = getDownloadStream(form, request, response);
			BufferedOutputStream fos = new BufferedOutputStream(response.getOutputStream());
	        int i;
			while((i = fis.read()) != -1){
	        	fos.write(i);
	        }
			fis.close();
			fos.flush();
			fos.close();
        }
        catch (IOException e) {
        	throw new SampleException(e);
        }
        
        return null;
    }
    
    /**
     * 日本語の文字化け防止のエンコーディング
     * @param str 対象文字列
     * @return String 変換後の文字列
     * @throws SampleException システム例外
     */
    protected String encode(String str) throws SampleException {
    	String result = null;
    	try {
    		if (str != null) {
    			result = URLEncoder.encode(str, UTF8);
    		}
        }
    	catch (UnsupportedEncodingException e) {
    		throw new SampleException(e);
    	}
    	return result;
    }

    /**
     * クライアントに送り返されるレスポンスのコンテントタイプを取得します。
     * @param form 存在するならば、このリクエストのためのActionForm Bean
     * @param request 処理しているHTTPリクエスト
     * @return クライアントに送り返されるレスポンスのコンテントタイプ
     */
    protected String getContentType(ActionForm form, HttpServletRequest request) {
    	return WebConstants.CONTENT_TYPE_STREAM;
    }
    
    /**
     * ダウンロードファイル名を取得します。
     * @param form 存在するならば、このリクエストのためのActionForm Bean
     * @param request 処理しているHTTPリクエスト
     * @return ダウンロードファイル名
     */
    protected abstract String getFileName(ActionForm form, HttpServletRequest request);
    
    /**
     * ダウンロードファイルの入力ストリームを取得します。
     * @param form 存在するならば、このリクエストのためのActionForm Bean
     * @param request 処理しているHTTPリクエスト
     * @param response 処理しているHTTPレスポンス
     * @return ダウンロードファイルの入力ストリーム
     * @throws IOException I/O例外
     */
    protected abstract InputStream getDownloadStream(ActionForm form,
    		HttpServletRequest request,	HttpServletResponse response) throws IOException;
}
