/* $Id: Int1.java,v 1.4 2004/06/20 15:23:40 fukasawa Exp $ */

/**
 *  @file    Int1.java
 *
 *  @author  Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 2001-2004 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

package com.jyugem.secs;

import java.*;

public final class Int1 extends Number implements Comparable {

    public static final int MIN_VALUE = -128;
    public static final int MAX_VALUE = 127;

    private int value;    // *** value ***

    // Constructor
    public Int1(Int1 value) {
        this.value = value.value;
    }

    public Int1(int value) {
        this.value = value;
    }

    public Int1(byte value) {
        this.value = (int)value;
    }

    public Int1(String s) throws NumberFormatException {
        this.value = parseInt1(s);
    }

    public static String toString(int x) {
        return Integer.toString(x);
    }

    public static int parseInt1(String s) throws NumberFormatException {
        return parseInt1(s, 10);
    }

    public static int parseInt1(String s, int radix)
                          throws NumberFormatException {
        int i = Integer.parseInt(s, radix);
        if (i < MIN_VALUE || i > MAX_VALUE)
            throw new NumberFormatException();
        return i;
    }

    public static Int1 valueOf(String s, int radix)
                           throws NumberFormatException {
        return new Int1(parseInt1(s, radix));
    }

    public static Int1 valueOf(String s) throws NumberFormatException {
        return valueOf(s, 10);
    }

    // Decodes a String into a Int1.  Accepts decimal, hexadecimal, and octal
    // numbers, in the following formats:
    //    1-9   decimal constant
    //    0x    hex constant
    //    #     hex constant
    //    0     octal constant
    public static Int1 decode(String numStr) throws NumberFormatException {
        int radix = 10;
        int index = 0;
        boolean negative = false;
        Int1 result;

        if (numStr.startsWith("-")) {
            negative = true;
            index++;
        }

        if (numStr.startsWith("0x", index) ||
            numStr.startsWith("0X", index)) {
            index += 2;
            radix = 16;
        } else if (numStr.startsWith("#", index)) {
            index++;
            radix = 16;
        } else if (numStr.startsWith("0", index) &&
                   numStr.length() > (index + 1)) {
            index++;
            radix = 8;
        }

        if (numStr.startsWith("-", index)) {
            throw new NumberFormatException("Negative sign in wrong position");
        }
        try {
            result = Int1.valueOf(numStr.substring(index), radix);
            result = negative ? new Int1(-result.intValue()) : result;
        } catch (NumberFormatException e) {
            String constant = "";
            if (negative) {
                constant = "-";
            }
            constant += numStr.substring(index);
            result = Int1.valueOf(constant, radix);
        }
        return result;
    }

    public byte byteValue() {
        return (byte)value;
    }

    public short shortValue() {
        return (short)value;
    }

    public int intValue() {
        return value;
    }

    public long longValue() {
        return (long)value;
    }

    public float floatValue() {
        return (float)value;
    }

    public double doubleValue() {
        return (double)value;
    }

    public String toString() {
        return Integer.toString(value, 10);
    }

    public int hashCode() {
        return value;
    }

    public boolean equals(Object obj) {
        if (obj instanceof Int1) {
            return value == ((Int1)obj).intValue();
        }
        return false;
    }

    public int compareTo(Int1 another) {
        return this.value - another.value;
    }

    public int compareTo(Object obj) {
        return compareTo((Int1)obj);
    }

}
