// $Id$

//=============================================================================
/**
 *  @file    HSCmdManager.cpp
 *
 *  @author  Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 2001-2004 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#define BEE_BUILD_DLL

#include "HSCmdManager.h"
#include "HSServer.h"
#include "HSInfoManager.h"
#include "jyugem/gem/JGTaskTrigger.h"
#include "BS2ACKMessage.h"
#include "BS2ErrorMessage.h"
#include "BS2ListItem.h"
#include "BS2DeclAtoms.h"


static HSCmdManager * _manager = NULL;
static char * _hcack_text[8] =
{
    "Acknowledge, command has been performed",
    "Command does not exist",
    "Cannot perform now",
    "At least one parameter is invalid",
    "Acknowledge, command will be performed with completion signaled later by an event",
    "Rejected, Already in Desired Condition",
    "No such object exists",
    "Unknown ack code (%d)",
};


//-----------------------------------------------------------------------------
// Constructor/Destructor
//-----------------------------------------------------------------------------
HSCmdManager::HSCmdManager() : JGRCmdManager(), m_sf(SFCODE(2,41))
{
    TRACE_FUNCTION(TRL_CONSTRUCT, "HSCmdManager::HSCmdManager");
}

//-----------------------------------------------------------------------------
HSCmdManager::~HSCmdManager()
{
    TRACE_FUNCTION(TRL_CONSTRUCT, "HSCmdManager::~HSCmdManager");

}

//-----------------------------------------------------------------------------
// Return own.
//-----------------------------------------------------------------------------
HSCmdManager * HSCmdManager::instance()
{
    TRACE_FUNCTION(TRL_LOW, "HSCmdManager::instance");
    if (_manager == NULL)
    {
        _manager = new HSCmdManager;
    }
    return _manager;
}

//-----------------------------------------------------------------------------
// Initialize
//-----------------------------------------------------------------------------
int HSCmdManager::init(void * parm)
{
    TRACE_FUNCTION(TRL_LOW, "HSCmdManager::init");

    return BEE_SUCCESS;
}

//-----------------------------------------------------------------------------
// Make parameter list in message
//-----------------------------------------------------------------------------
int HSCmdManager::makeBaseItems(const std::string& cmd, const std::string& port,
                                BS2Message ** msgptr, BS2ListItem ** paramsptr)
{
    TRACE_FUNCTION(TRL_LOW, "HSCmdManager::makeBaseItems");

    BS2Message * msg = BS2Message::factory(m_sf);
    BS2ListItem * rootlist = new BS2ListItem;
    msg->add(rootlist);
    BS2Item * item = BS2Item::factory(_TX("RCMD"), new BS2Ascii(cmd));
    rootlist->add(item);

    BS2ListItem * paramlist = new BS2ListItem;
    if (m_sf == SFCODE(2,49))
    {
        BS2Item * item = BS2Item::factory(_TX("OBJSPEC"), new BS2Ascii(port));
        rootlist->add(item);
    }
    rootlist->add(paramlist);
    BS2ListItem * plist;

    if (m_sf == SFCODE(2,41))
    {   // port id
        plist = new BS2ListItem;
        plist->add(BS2Item::factory(_TX("CPNAME"), new BS2Ascii(_TX("PortID"))));
        plist->add(BS2Item::factory(_TX("CPVAL"), new BS2Ascii(port)));
        paramlist->add(plist);
    }
    *msgptr = msg;
    *paramsptr = paramlist;
    return 0;
}

//-----------------------------------------------------------------------------
// Abort
//-----------------------------------------------------------------------------
int HSCmdManager::abort(int tid, const std::string& port, int level)
{
    TRACE_FUNCTION(TRL_LOW, "HSCmdManager::abort");

    BS2Message * msg;
    BS2ListItem * paramlist;
    int result = makeBaseItems(_TX("ABORT"), port, &msg, &paramlist);

    // level
    BS2ListItem * plist = new BS2ListItem;
    plist->add(BS2Item::factory(_TX("CPNAME"), new BS2Ascii(_TX("AbortLevel"))));
    plist->add(BS2Item::factory(_TX("CPVAL"), new BS2Int4(level)));
    paramlist->add(plist);

    result = this->send(msg, tid);

    return result;
}

//-----------------------------------------------------------------------------
// Pause
//-----------------------------------------------------------------------------
int HSCmdManager::pause(int tid, const std::string& port)
{
    TRACE_FUNCTION(TRL_LOW, "HSCmdManager::pause");

    BS2Message * msg;
    BS2ListItem * paramlist;
    int result = makeBaseItems(_TX("PAUSE"), port, &msg, &paramlist);
    result = this->send(msg, tid);
    return result;
}

//-----------------------------------------------------------------------------
// Resume
//-----------------------------------------------------------------------------
int HSCmdManager::resume(int tid, const std::string& port,
                         const std::vector<any_item>& params)
{
    TRACE_FUNCTION(TRL_LOW, "HSCmdManager::resume");

    BS2Message * msg;
    BS2ListItem * paramlist;
    int result = makeBaseItems(_TX("RESUME"), port, &msg, &paramlist);

    for (size_t i = 0; i < params.size(); i++)
    {
        BS2ListItem * plist = new BS2ListItem;
        plist->add(BS2Item::factory(_TX("CPNAME"), new BS2Ascii(params[i].m_pname)));
        plist->add(BS2Item::factory(_TX("CPVAL"),
                   BS2Atom::factory(params[i].m_value)));
        paramlist->add(plist);
    }
    result = this->send(msg, tid);
    return result;
}

//-----------------------------------------------------------------------------
// Start
//-----------------------------------------------------------------------------
int HSCmdManager::start(int tid, const std::string& port, const std::string& lotid,
                        int mbc_size, const std::vector<any_item>& params)
{
    TRACE_FUNCTION(TRL_LOW, "HSCmdManager::start");

    BS2Message * msg;
    BS2ListItem * paramlist;
    int result = makeBaseItems(_TX("START"), port, &msg, &paramlist);

    // lot id
    BS2ListItem * plist = new BS2ListItem;
    plist->add(BS2Item::factory(_TX("CPNAME"), new BS2Ascii(_TX("LotID"))));
    plist->add(BS2Item::factory(_TX("CPVAL"), new BS2Ascii(lotid)));
    paramlist->add(plist);

    // wafer count
    plist = new BS2ListItem;
    plist->add(BS2Item::factory(_TX("CPNAME"), new BS2Ascii(_TX("SlotCount"))));
    plist->add(BS2Item::factory(_TX("CPVAL"), new BS2Int4(mbc_size)));
    paramlist->add(plist);

    for (size_t i = 0; i < params.size(); i++)
    {
        plist = new BS2ListItem;
        plist->add(BS2Item::factory(_TX("CPNAME"), new BS2Ascii(params[i].m_pname)));
        plist->add(BS2Item::factory(_TX("CPVAL"),
                   BS2Atom::factory(params[i].m_value)));
        paramlist->add(plist);
    }
    result = this->send(msg, tid);
    return result;
}

//-----------------------------------------------------------------------------
// PP-SELECT
//-----------------------------------------------------------------------------
int HSCmdManager::pp_select(int tid, const std::string& port,
                            const std::string& lotid,
                            const std::string& ppid,
                            const std::vector<any_item>& params)
{
    TRACE_FUNCTION(TRL_LOW, "HSCmdManager::pp_select");

    BS2Message * msg;
    BS2ListItem * paramlist;
    int result = makeBaseItems(_TX("START"), port, &msg, &paramlist);

    // lotid
    BS2ListItem * plist = new BS2ListItem;
    plist->add(BS2Item::factory(_TX("CPNAME"), new BS2Ascii(_TX("LotID"))));
    plist->add(BS2Item::factory(_TX("CPVAL"), new BS2Ascii(lotid)));
    paramlist->add(plist);

    // ppid
    plist = new BS2ListItem;
    plist->add(BS2Item::factory(_TX("CPNAME"), new BS2Ascii(_TX("PPID"))));
    plist->add(BS2Item::factory(_TX("CPVAL"), new BS2Ascii(ppid)));
    paramlist->add(plist);

    for (size_t i = 0; i < params.size(); i++)
    {
        plist = new BS2ListItem;
        plist->add(BS2Item::factory(_TX("CPNAME"), new BS2Ascii(params[i].m_pname)));
        plist->add(BS2Item::factory(_TX("CPVAL"),
                   BS2Atom::factory(params[i].m_value)));
        paramlist->add(plist);
    }
    result = this->send(msg, tid);
    return result;
}


//-----------------------------------------------------------------------------
// Send Host Command
//-----------------------------------------------------------------------------
int HSCmdManager::request(const host_cmd& hcmd)
{
    TRACE_FUNCTION(TRL_LOW, "HSCmdManager::request");

    JGRemoteCmd * cmdinfo = this->find(hcmd.m_rcmd);
    if (cmdinfo == NULL)
    {
        TRACE_ERROR((_TX("Unknown command (%s)\n"), hcmd.m_rcmd.c_str()));
        return -1;
    }
    BS2Message * msg = makeCommand(cmdinfo, hcmd);
    if (msg == NULL)
    {
        return -1;
    }
    msg->dump();
    return this->send(msg, hcmd.m_tid);
}

//-----------------------------------------------------------------------------
// Send Host Command
//-----------------------------------------------------------------------------
BS2Message * HSCmdManager::makeCommand(JGRemoteCmd * cmdinfo,
                                       const host_cmd& hcmd)
{
    TRACE_FUNCTION(TRL_LOW, "HSCmdManager::makeCommand");

    JGRemoteCmd::Param * paraminfo;
    int sf = cmdinfo->m_sfcode;
    if (! (sf == SFCODE(2,49) || sf == SFCODE(2,41) || sf == SFCODE(2,21)))
    {
        TRACE_ERROR((_TX("Not support stream/function (0x%x)\n"), sf));
        return NULL;
    }

    BS2Message * msg = BS2Message::factory(sf);
    BS2Item * item;
    if (sf == SFCODE(2,21))
    {
        item = BS2Item::factory(_TX("RCMD"), BS2Atom::factory(cmdinfo->m_rcmd));
        msg->add(item);
        return msg;            // S2F21 is no parameter.
    }

    BS2ListItem * rootlist = new BS2ListItem;
    msg->add(rootlist);
    BS2ListItem * params = new BS2ListItem;
    BS2ListItem * plist;
    if (sf == SFCODE(2,49))
    {
        item = this->getDataID();  // DATAID
        rootlist->add(item);
        item = BS2Item::factory(_TX("OBJSPEC"), new BS2Ascii(hcmd.m_objspec));
        rootlist->add(item);
        item = BS2Item::factory(_TX("RCMD"), BS2Atom::factory(cmdinfo->m_rcmd));
        rootlist->add(item);
        rootlist->add(params);
    }
    else // if (sf == SFCODE(2,41))
    {
        item = BS2Item::factory(_TX("RCMD"), BS2Atom::factory(cmdinfo->m_rcmd));
        rootlist->add(item);
        rootlist->add(params);
        if (hcmd.m_objspec.size() > 0)
        {
            plist = new BS2ListItem;
            paraminfo = cmdinfo->find("OBJSPEC");
            item = BS2Item::factory(_TX("CPNAME"), new BS2Ascii(paraminfo->m_svcname));
            plist->add(item);
            item = BS2Item::factory(_TX("CPVAL"), new BS2Ascii(hcmd.m_objspec));
            plist->add(item);
            params->add(plist);
        }
    }

    for (size_t i = 0; i < hcmd.m_items.size(); i++)
    {
        paraminfo = cmdinfo->find(hcmd.m_items[i].m_pname);
        if (paraminfo == NULL)
        {
            TRACE_ERROR((_TX("Unknown parameter (%s)\n"),
                        hcmd.m_items[i].m_pname.c_str()));
            plist = new BS2ListItem;
            item = BS2Item::factory(_TX("CPNAME"), new BS2Ascii(hcmd.m_items[i].m_pname));
            plist->add(item);
            item = BS2Item::factory(_TX("CPVAL"), BS2Atom::factory(hcmd.m_items[i].m_value));
            plist->add(item);
        }
        else
        {
            plist = new BS2ListItem;
            item = BS2Item::factory(_TX("CPNAME"), new BS2Ascii(paraminfo->m_svcname));
            plist->add(item);
            item = BS2Item::factory(_TX("CPVAL"), BS2Atom::factory(hcmd.m_items[i].m_value));
            plist->add(item);
        }
        params->add(plist);
    }
    return msg;
}

//-----------------------------------------------------------------------------
// Host Command Acknowledge : S2F42
//-----------------------------------------------------------------------------
static char * _cpack_text[5] =
{
    "No Error",
    "Parameter Name (CPNAME) does not exist",
    "Illegal Value specified for CPVAL",
    "Illegal Format specified for CPVAL",
    "Unknown ack code (%d)",
};

class CPACKParser : public BS2Traverser
{
    friend class HSCmdManager;
public:
    string m_response;
    int    m_hcack;
    bool   m_first;

    CPACKParser() : m_response(_TX("")), m_hcack(0), m_first(true) {}
    virtual ~CPACKParser() {}

    virtual int parseItem(BS2Item * item)
    {
        TRACE_FUNCTION(TRL_LOW, "CEPACKParser::parseItem");
        BS2Atom * atom = item->atom();
        if (item->name() == _TX("CPNAME"))
        {
            if (! atom->isAscii())
            {
                TRACE_ERROR((_TX("Illegal CPNAME format(%o) \n"), atom->format()));
                return -1;
            }
            if (m_first)
            {
                m_first = false;
            }
            else
            {
                m_response += ", ";
            }
            string cpname = ((BS2Ascii *)atom)->value();
            m_response += cpname;
            m_response += ": ";
        }
        else if (item->name() == _TX("CPACK"))
        {
            JGvalue value;
            atom->get(value);
            int ack = value.getInt();
            if (ack > 3)
            {
                BCHAR buf[64];
                _stprintf(buf, _cpack_text[4], ack);
                m_response += buf;
            }
            else
            {
                m_response += _cpack_text[ack];
            }
        }
        else if (item->name() == _TX("HCACK"))
        {
            JGvalue value;
            atom->get(value);
            m_hcack = value.getInt();
        }
        return 0;
    }
};

//-----------------------------------------------------------------------------
int HSCmdManager::parseResponse42(BS2Message * msg, JGTrigger * notify)
{
    TRACE_FUNCTION(TRL_LOW, "HSCmdManager::parseResponse42");

    CPACKParser cpack;
    int result = msg->traverse(&cpack);
    if (result < 0)
    {
        TRACE_ERROR((_TX("Illegal remote command format \n")));
        return BEE_ERROR;
    }

    string errtext("");
    if (cpack.m_hcack == 0)
    {
        ;
    }
    else if (cpack.m_hcack > 6)
    {
        BCHAR buf[64];
        _stprintf(buf, _hcack_text[7], cpack.m_hcack);
        errtext = buf;
    }
    else
    {
        errtext = _hcack_text[cpack.m_hcack];
    }

    // Send ack to host

    return BEE_SUCCESS;
}

//-----------------------------------------------------------------------------
// Host Command Acknowledge : S2F50
//-----------------------------------------------------------------------------
static char * _cepack_text[6] =
{
    "No Error",
    "Parameter Name (CPNAME) does not exist",
    "Illegal Value specified for CEPVAL",
    "Illegal Format specified for CEPVAL",
    "Parameter name (CPNAME) not valid as used",
    "Unknown ack code (%d)",
};

class CEPACKParser : public BS2Traverser
{
    friend class HSCmdManager;
public:
    string m_response;
    int    m_hcack;
    bool   m_first;

    CEPACKParser() : m_response(_TX("")), m_hcack(0), m_first(true) {}
    virtual ~CEPACKParser() {}

    virtual int parseItem(BS2Item * item)
    {
        TRACE_FUNCTION(TRL_LOW, "CEPACKParser::parseItem");
        if (item->isList())
        {   // Ignore list item
            return 0;
        }

        BS2Atom * atom = item->atom();
        if (item->name() == _TX("CPNAME"))
        {
            if (! atom->isAscii())
            {
                TRACE_ERROR((_TX("Illegal CPNAME format(%o) \n"), atom->format()));
                return -1;
            }
            if (m_first)
            {
                m_first = false;
            }
            else
            {
                m_response += ", ";
            }
            string cpname = ((BS2Ascii *)atom)->value();
            m_response += cpname;
            m_response += ": ";
        }
        else if (item->name() == _TX("CEPVAL"))
        {
            JGvalue value;
            atom->get(value);
            int ack = value.getInt();
            if (ack > 4)
            {
                BCHAR buf[64];
                _stprintf(buf, _cepack_text[4], ack);
                m_response += buf;
            }
            else
            {
                m_response += _cepack_text[ack];
            }
        }
        else if (item->name() == _TX("HCACK"))
        {
            JGvalue value;
            atom->get(value);
            m_hcack = value.getInt();
        }
        return 0;
    }
};

//-----------------------------------------------------------------------------
int HSCmdManager::parseResponse50(BS2Message * msg, JGTrigger * notify)
{
    TRACE_FUNCTION(TRL_LOW, "HSCmdManager::parseResponse");

    CEPACKParser cepack;
    int result = msg->traverse(&cepack);
    if (result < 0)
    {
        TRACE_ERROR((_TX("Illegal remote command format \n")));
        return BEE_ERROR;
    }

    string errtext("");
    if (cepack.m_hcack == 0)
    {
        ;
    }
    else if (cepack.m_hcack > 6)
    {
        BCHAR buf[64];
        _stprintf(buf, _hcack_text[7], cepack.m_hcack);
        errtext = buf;
    }
    else
    {
        errtext = _hcack_text[cepack.m_hcack];
    }

    // Send ack to host

    return BEE_SUCCESS;
}

//-----------------------------------------------------------------------------
//
// Service of received message.
//
//-----------------------------------------------------------------------------
BS2Message * HSCmdManager::msg_svc(JGMessageTrigger * trigger, BS2Message * msg)
{
    BS2Message * replymsg = NULL;
    int result;

    if (msg->sf() == SFCODE(2,42))
    {
        result = parseResponse42(msg, trigger);
    }
    else if (msg->sf() == SFCODE(2,50))
    {
        result = parseResponse50(msg, trigger);
    }
    else
    {   // Unexpected message
        replymsg = this->unrecognized(msg);
    }

    return replymsg;
}


