// $Id$

//=============================================================================
/**
 *  @file    sqlite_dbio.h
 *
 *  @author  Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 2003-2004 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#ifndef SQLITE_DBIO_H
#define SQLITE_DBIO_H

#include <string>
using namespace std;
#include "jyugem/dbio/dbio.h"
#include "sqlite.h"

//
//
//
struct column_base
{
    column_base() { m_name = 0; }
    column_base(const char * name) { m_name = name; }
    virtual ~column_base() {}

    virtual int input(void * recp, const string& datap) = 0;

    const char * m_name;
};

//
// Column access
//
template <class Elm>
struct str_column : public column_base
{
    typedef void (Elm::* setfp)(const string&);

    str_column(char * name, setfp fptr) : column_base(name), m_setfp(fptr) {}
    virtual ~str_column() {}

    virtual int input(void * recp, const string& datap)
    {
        Elm * elm = static_cast<Elm *>(recp);
        (elm->*m_setfp)(datap);
        return 0;
    }
    setfp m_setfp;
};

template <class Elm>
struct bool_column : public column_base
{
    typedef void (Elm::* setfp)(bool);

    bool_column(char * name, setfp fptr) : column_base(name), m_setfp(fptr) {}
    virtual ~bool_column() {}

    virtual int input(void * recp, const string& datap)
    {
        Elm * elm = static_cast<Elm *>(recp);
        (elm->*m_setfp)((datap == _TX("Y")) ? true : false);
        return 0;
    }
    setfp m_setfp;
};


template <class Elm, typename T = int>
struct int_column : public column_base
{
    typedef void (Elm::* setfp)(T);
    typedef T    (*convfp)(const char *);

    int_column(char * name, setfp fptr, convfp cptr = 0)
        : column_base(name), m_setfp(fptr), m_convfp(cptr) {}
    virtual ~int_column() {}

    virtual int input(void * recp, const string& datap)
    {
        Elm * elm = static_cast<Elm *>(recp);
        T   intval;
        if (m_convfp == 0)
        {
            intval = (T)_tcstol(datap.c_str(), NULL, 10);
        }
        else
        {
            intval = m_convfp(datap.c_str());
        }
        (elm->*m_setfp)(intval);
        return 0;
    }
    setfp  m_setfp;
    convfp m_convfp;
};

template <class Elm, class IM = JGInfoManager>
struct id_column : public column_base
{
    typedef void (Elm::* setfp)(const JGid&);
    typedef int  (IM::* idfp)();

    id_column(char * name, setfp fptr, idfp idptr = 0)
        : column_base(name), m_setfp(fptr), m_idfp(idptr) {}
    virtual ~id_column() {}

    virtual int input(void * recp, const string& datap)
    {
        Elm * elm = static_cast<Elm *>(recp);
        IM * im = elm->info();
        JGid id;
        if (datap.size() > 0)
        {
            if (m_idfp != 0)
            {
                id.set(datap, (im->* m_idfp)());
            }
            else
            {
                id = datap;
            }
        }
        (elm->*m_setfp)(id);
        return 0;
    }
    setfp m_setfp;
    idfp  m_idfp;
};

//
// Equipment's information DataBase
//
class BEE_Export sqlite_dbio : public dbio
{
public:
    sqlite_dbio(const char * fname = NULL) : dbio(fname)
    {
        m_dbp = NULL;
    }
    virtual ~sqlite_dbio();

    virtual int  open(const string& dbfname);
    virtual int  drop(const string& tableName);
    virtual void close();

    virtual int  load(JGEquipment * equip);
    virtual int  save(JGEquipment * equip);
    virtual int  putcmds(const vector<string>& cmds);
    virtual int  sqlcmd(const string& sql);

    int  load(char * tableName, char * colNames, void * usrptr,
              sqlite_callback infptr);
    int  load(char * tableName, char * colNames, char * whereStr,
              void * usrptr, sqlite_callback infptr);

    static sqlite_dbio * instance();

//
protected:
    virtual int  begin_trans();         // Begin Transaction
    virtual int  commit();              // End Transaction
    virtual int  rollback();            // Abort Transaction

    int  create(const string& tableName, const string& fields);

// member variables
private:
    sqlite * m_dbp;
};


#endif  /* DBIO_H */
