//
// Sample program for Memory leak
//

#define _CRTDBG_MAP_ALLOC
#include <stdio.h>
#include <stdlib.h>
#include <crtdbg.h>
#include <string.h>
#include <time.h>

#include "b_memdbg.h"

//-----------------------------------------------------------------------------
// Macros for setting or clearing bits in the CRT debug flag
#ifdef _DEBUG
#define  SET_CRT_DEBUG_FIELD(a)   _CrtSetDbgFlag((a) | _CrtSetDbgFlag(_CRTDBG_REPORT_FLAG))
#define  CLEAR_CRT_DEBUG_FIELD(a) _CrtSetDbgFlag(~(a) & _CrtSetDbgFlag(_CRTDBG_REPORT_FLAG))
#else
#define  SET_CRT_DEBUG_FIELD(a)   ((void) 0)
#define  CLEAR_CRT_DEBUG_FIELD(a) ((void) 0)
#endif

static FILE * logFile;                // Used to log allocation information

/*
 * ALLOCATION HOOK FUNCTION:
 *   An allocation hook function can have many, many different uses.
 *   This one simply logs each allocation operation in a file.
 */
#define nNoMansLandSize 4

typedef struct _CrtMemBlockHeader
{
    struct _CrtMemBlockHeader * pBlockHeaderNext;
    struct _CrtMemBlockHeader * pBlockHeaderPrev;
    char *                      szFileName;
    int                         nLine;
    size_t                      nDataSize;
    int                         nBlockUse;
    long                        lRequest;
    unsigned char               gap[nNoMansLandSize];
    /* followed by:
     *   unsigned char          data[nDataSize];
     *   unsigned char          anotherGap[nNoMansLandSize];
     */
} _CrtMemBlockHeader;

#define pbData(pblock)  ((unsigned char *)((_CrtMemBlockHeader *)pblock + 1))
#define pHdr(pbData)    (((_CrtMemBlockHeader *)pbData)-1)

static bool _logging = false;
static int  _s28q = 0;
static int  _s24q = 0;

int __cdecl _AllocHook(int nAllocType, void * pvData, size_t nSize, int nBlockUse,
                       long lRequest, const unsigned char * szFileName, int nLine)
{
    char *operation[] = { "", "allocating", "re-allocating", "freeing" };
    char *blockType[] = { "Free", "Normal", "CRT", "Ignore", "Client" };
    _CrtMemBlockHeader * pHead;

    if (nBlockUse == _CRT_BLOCK) // Ignore internal C runtime library allocations
        return 7;

    _ASSERT((nAllocType > 0) && (nAllocType < 4));
    _ASSERT((nBlockUse >= 0) && (nBlockUse < 5));

    if (! _logging)
    {
        return 7;
    }

    if (nAllocType == 3)
    {
        pHead = pHdr(pvData);
        fprintf(logFile, "%ld: %s a %d-byte '%s' at 0x%X, %s line %d\n",
            pHead->lRequest, operation[nAllocType], pHead->nDataSize,
            blockType[pHead->nBlockUse], pvData, pHead->szFileName, pHead->nLine);
        if (pHead->nDataSize == 28 && nAllocType == 1)
        {
            _s28q++;
            if (_s28q == 2)
            {
                return 7;
            }
        }
    }
    else
    {
        fprintf(logFile, "%ld: %s a %d-byte '%s' at 0x%X, %s line %d\n",
            lRequest, operation[nAllocType], nSize, blockType[nBlockUse],
            pvData, szFileName, nLine);
        if (nSize == 28 && nAllocType == 1)
        {
            _s28q++;
            if (_s28q == 2)
            {
                return 7;
            }
        }
    }
    return 7;   // Let memory operation proceed
}

int memdbg_init(const char * filename)
{
    _CrtSetReportMode( _CRT_WARN, _CRTDBG_MODE_FILE );
    _CrtSetReportFile( _CRT_WARN, _CRTDBG_FILE_STDOUT );
    _CrtSetReportMode( _CRT_ERROR, _CRTDBG_MODE_FILE );
    _CrtSetReportFile( _CRT_ERROR, _CRTDBG_FILE_STDOUT );
    _CrtSetReportMode( _CRT_ASSERT, _CRTDBG_MODE_FILE );
    _CrtSetReportFile( _CRT_ASSERT, _CRTDBG_FILE_STDOUT );

    // SET_CRT_DEBUG_FIELD( _CRTDBG_LEAK_CHECK_DF | _CRTDBG_DELAY_FREE_MEM_DF );

    logFile = fopen(filename, "w");
    if (logFile == NULL)
    {
        printf(" Memory Logging File open error (%s).\n", filename);
        return -1;
    }

    struct tm * newtime;
    time_t aclock;
    time(&aclock);                  // Get time in seconds
    newtime = localtime(&aclock);   // Convert time to struct tm form
    fprintf(logFile,
        "Memory Allocation Log File for JYUGEM, run at %s",
        asctime(newtime));
    fputs("-----------------------------------------------------------------------------\n",
        logFile);

    // Install the hook functions
    _CrtSetAllocHook(_AllocHook);

    return 0;
}


/*
 * Memory Check Point
 */
static _CrtMemState checkPoint1;
static _CrtMemState checkPoint2;
static _CrtMemState checkPoint3;

void memdbg_begin(const char * text)
{
    fprintf(logFile, "------ memdbg_begin: %s ------\n", text);
    _CrtMemCheckpoint(&checkPoint1);
    _logging = true;
    return ;
}

void memdbg_end(const char * text)
{
    fprintf(logFile, "------ memdbg_end: %s ------\n", text);
    _CrtMemDumpAllObjectsSince(&checkPoint1);
    fflush(logFile);

    _logging = false;
    return ;
}

void memdbg_diff(const char * text)
{
    fprintf(logFile, "------ memdbg_diff: %s ------\n", text);
    _CrtMemCheckpoint(&checkPoint2);
    if (_CrtMemDifference(&checkPoint3, &checkPoint1, &checkPoint2))
    {
        _CrtMemDumpStatistics(&checkPoint3);
    }
    fflush(logFile);
    return ;
}

void memdbg_dump(const char * text)
{
    fprintf(logFile, "------ memdbg_dump: %s ------\n", text);
    _CrtMemDumpStatistics(&checkPoint1);
    return ;
}


