/* $Id: TestTarget.java,v 1.4 2004/06/20 15:23:40 fukasawa Exp $ */

/**
 *  @file    TestTarget.java
 *
 *  @author  Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 2001-2004 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package com.jyugem.jsunit.tfw;

import java.util.*;
import java.lang.*;
import java.io.*;
import javax.swing.*;
import javax.swing.tree.*;
import javax.swing.event.*;

import org.w3c.dom.Document;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Text;
import org.w3c.dom.Attr;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.SAXNotRecognizedException;
import org.xml.sax.SAXNotSupportedException;
import org.apache.xerces.parsers.DOMParser;
import junit.framework.*;

public class TestTarget extends Hashtable {

    // feature ids
    protected static final String NAMESPACES_FEATURE_ID = "http://xml.org/sax/features/namespaces";
    protected static final String VALIDATION_FEATURE_ID = "http://xml.org/sax/features/validation";
    protected static final String SCHEMA_VALIDATION_FEATURE_ID = "http://apache.org/xml/features/validation/schema";
    protected static final String CREATE_ENTITY_REF_NODES = "http://apache.org/xml/features/dom/create-entity-ref-nodes";
    protected static final String INCLUDE_IGNORABLE_WHITESPACE = "http://apache.org/xml/features/dom/include-ignorable-whitespace";
    // default settings
    protected static final boolean DEFAULT_NAMESPACES = true;
    protected static final boolean DEFAULT_NAMESPACE_PREFIXES = false;
    protected static final boolean DEFAULT_VALIDATION = false;
    protected static final boolean DEFAULT_SCHEMA_VALIDATION = true;
    protected static final boolean DEFAULT_CREATE_ENTITY_REF_NODES = false;
    protected static final boolean DEFAULT_INCLUDE_IGNORABLE_WHITESPACE = true;

    //
    // Data
    //
    DOMParser m_parser = new DOMParser();
    Document  m_xmldoc = null;
    TestSuite m_main;
    HashMap   m_testcases;
    String    m_testName = null;
    String    m_logName = null;
    SECSTestManager m_manager = null;  // jyusecs

    //
    // Public methods
    //
    // Constructor
    public TestTarget() {
    }

    public TestTarget(SECSTestManager mngr) {
        m_manager = mngr;
    }

    // Read message file that is formatted xml
    public int open(String filename) {
        boolean namespaces = DEFAULT_NAMESPACES;
        boolean validation = DEFAULT_VALIDATION;
        boolean schemaValidation = DEFAULT_SCHEMA_VALIDATION;
        boolean entityReference = DEFAULT_CREATE_ENTITY_REF_NODES;
        boolean includeIgnoreWhitespace = DEFAULT_INCLUDE_IGNORABLE_WHITESPACE;

        // set parser features
        try {
            m_parser.setFeature(NAMESPACES_FEATURE_ID, namespaces);
            m_parser.setFeature(VALIDATION_FEATURE_ID, validation);
            m_parser.setFeature(SCHEMA_VALIDATION_FEATURE_ID, schemaValidation);
            m_parser.setFeature(CREATE_ENTITY_REF_NODES, entityReference);
            m_parser.setFeature(INCLUDE_IGNORABLE_WHITESPACE, includeIgnoreWhitespace);
        }
        catch (SAXException e) {
            System.err.println("warning: Parser does not support feature");
        }

        //
        // parse file
        //
        try {
            m_parser.parse(filename);
            setDocument(m_parser.getDocument());
            build();
        }
        catch (SAXParseException e) {
            // ignore
        }
        catch (Exception e) {
            System.err.println("error: Parse error occurred - "+ e.getMessage());
            Exception se = e;
            if (e instanceof SAXException) {
                se = ((SAXException)e).getException();
            }
            if (se != null) {
                se.printStackTrace(System.err);
            } else {
                e.printStackTrace(System.err);
            }
        }
        return 0;
    }

    /** Sets the document. */
    public synchronized void setDocument(Document document) {
        m_xmldoc = document;        // save document
    }

    /** Returns the document. */
    public Document getDocument() {
        return m_xmldoc;
    }

    public TestSuite suite() { return m_main; }

    // Builds the test sequence
    private void build() throws SAXException {
        // is there anything to do?
        if (m_xmldoc == null) {
            return;
        }

        // iterate over children of this node
        NodeList nodes = m_xmldoc.getChildNodes();
        int len = (nodes != null) ? nodes.getLength() : 0;
        for (int i = 0; i < len; i++) {
            Node node = nodes.item(i);
            if (node.getNodeType() == Node.ELEMENT_NODE &&
                node.getNodeName().equals("secs:tests")) {
                parseTests(node);
            }
        }
    }

    // Builds the test. <secs:test>
    private void parseTests(Node node) throws SAXException {
        // Parse attribute
        NamedNodeMap attrs = node.getAttributes();
        int attrCount = (attrs != null) ? attrs.getLength() : 0;
        for (int i = 0; i < attrCount; i++) {
            Node attr = attrs.item(i);
            if (attr.getNodeName().equals("name")) {
                m_testName = attr.getNodeValue();
            } else if (attr.getNodeName().equals("logfile")) {
                m_logName = attr.getNodeValue();
            }
        }
        if (m_logName == null) {
            m_logName = new String(m_testName + ".log");
        }

        m_main = new TestSuite(m_testName);

        // Iterate over children of this node
        NodeList nodes = node.getChildNodes();
        int len = (nodes != null) ? nodes.getLength() : 0;
        for (int i = 0; i < len; i++) {
            Node childNode = nodes.item(i);
            if (childNode.getNodeType() == Node.ELEMENT_NODE) {
                if (childNode.getNodeName().equals("secs:test")) {
                    parseTestElementNode(childNode);
                } else if (childNode.getNodeName().equals("secs:sequence")) {
                    parseSequenceNode(childNode);
                }
            }
        }
    }

    // Builds the test. <secs:test>
    private void parseTestElementNode(Node node) throws SAXException {
        // Parse attribute
        String name = null;
        String description = null;
        NamedNodeMap attrs = node.getAttributes();
        int attrCount = (attrs != null) ? attrs.getLength() : 0;
        for (int i = 0; i < attrCount; i++) {
            Node attr = attrs.item(i);
            if (attr.getNodeName().equals("name")) {
                name = attr.getNodeValue();
            } else if (attr.getNodeName().equals("description")) {
                description = attr.getNodeValue();
            }
        }
        if (name == null) {
            throw new SAXException("<secs:test> Undefined test name");
        }
        SECSTestSuite testsuite = new SECSTestSuite(name, description);
        TestNode testnode = new TestNode(name, testsuite);
        put(name, testnode);

        // Iterate over children of this node
        NodeList nodes = node.getChildNodes();
        int len = (nodes != null) ? nodes.getLength() : 0;
        for (int i = 0; i < len; i++) {
            Node childNode = nodes.item(i);
            if (childNode.getNodeType() == Node.ELEMENT_NODE) {
                if (childNode.getNodeName().equals("secs:testsuite")) {
                    SECSTestSuite suite = parseTestSuites(childNode);
                    if (suite != null) {
                        String caseName = suite.getName();
                        TestNode caseNode = new TestNode(caseName, suite);
                        put(caseName, caseNode);
                        testsuite.addTest(suite);
                    }
                } else if (childNode.getNodeName().equals("secs:testcase")) {
                    SECSTestCase testcase = TestCaseParser.parse(childNode);
                    if (testcase != null) {
                        String caseName = testcase.getName();
                        TestNode caseNode = new TestNode(caseName, testcase);
                        put(caseName, caseNode);
                        testsuite.addTest(testcase);
                    }
                }
            }
        }
    }

    // Create test sequence.
    private void parseSequenceNode(Node node) throws SAXException {
        // iterate over children of this node
        NodeList nodes = node.getChildNodes();
        int len = (nodes != null) ? nodes.getLength() : 0;
        for (int i = 0; i < len; i++) {
            Node childNode = nodes.item(i);
            if (childNode.getNodeType() == Node.ELEMENT_NODE) {
                if (childNode.getNodeName().equals("secs:teststep")) {
                    String testName = null;
                    Node valueNode = childNode.getFirstChild();
                    if (valueNode != null) {
                        if (valueNode.getNodeType() == Node.TEXT_NODE) {
                            testName = valueNode.getNodeValue();
                        } else if (valueNode.getNodeType() == Node.ELEMENT_NODE) {
                            testName = valueNode.getNodeValue();
                        }
                    }

                    TestNode reftest = (TestNode)get(testName);
                    if (reftest == null) {
                        throw new SAXException("<secs:teststep> Not found reference. " +
                                               testName);
                    }
                    m_main.addTest((junit.framework.Test)reftest.get());
                }
            }
        }
        return ;
    }

    // Builds the test. <secs:testsuite>
    private SECSTestSuite parseTestSuites(Node node) throws SAXException {
        // Parse attribute
        String name = null;
        String description = null;
        NamedNodeMap attrs = node.getAttributes();
        int attrCount = (attrs != null) ? attrs.getLength() : 0;
        for (int i = 0; i < attrCount; i++) {
            Node attr = attrs.item(i);
            if (attr.getNodeName().equals("name")) {
                name = attr.getNodeValue();
            } else if (attr.getNodeName().equals("description")) {
                description = attr.getNodeValue();
            }
        }
        if (name == null) {
            throw new SAXException("<secs:testsuite> Undefined test name");
        }
        SECSTestSuite testsuite = new SECSTestSuite(name, description);
        TestNode testnode = new TestNode(name, testsuite);
        put(name, testnode);

        // Iterate over children of this node
        NodeList nodes = node.getChildNodes();
        int len = (nodes != null) ? nodes.getLength() : 0;
        for (int i = 0; i < len; i++) {
            Node childNode = nodes.item(i);
            if (childNode.getNodeType() == Node.ELEMENT_NODE) {
                if (childNode.getNodeName().equals("secs:testcase")) {
                    SECSTestCase testcase = TestCaseParser.parse(childNode);
                    if (testcase != null) {
                        String caseName = testcase.getName();
                        TestNode caseNode = new TestNode(caseName, testcase);
                        put(caseName, caseNode);
                        testsuite.addTest(testcase);
                    }
                }
            }
        }
        return testsuite;
    }

    //
    // Internal test node class
    //
    protected class TestNode {
        String m_name;
        String m_description;
        Object m_test;       // TestCase or TestSuite

        TestNode(String name, Object test) {
            m_name = name; m_test = test;
            if (test instanceof SECSTestCase) {
                SECSTestCase testcase = (SECSTestCase)test;
                m_description = testcase.getDescription();
            } else if (test instanceof SECSTestSuite) {
                SECSTestSuite testsuite = (SECSTestSuite)test;
                m_description = testsuite.getDescription();
            }
        }
        boolean isTestCase()  { return (m_test instanceof TestCase); }
        boolean isTestSuite() { return (m_test instanceof TestSuite); }
        Object get() { return m_test; }
        public String toString() { return (m_name + "F" + m_description); }
        void dump() {
            if (isTestSuite()) {
                TestSuite testsuite = (TestSuite)m_test;
                System.out.println("<secs:test name=\"" + testsuite.getName() + "\">");
                Enumeration enum = testsuite.tests();
                while (enum.hasMoreElements()) {
                    Object obj = enum.nextElement();
                    String testName;
                    if (obj instanceof SECSTestCase) {
                        SECSTestCase testcase = (SECSTestCase)obj;
                        System.out.println(testcase.toXmlString());
                    } else if (obj instanceof SECSTestSuite) {
                        SECSTestSuite suite = (SECSTestSuite)obj;
                        System.out.println(suite.toXmlString());
                    } else {
                        System.out.println("UNKNOWN: " + obj.getClass().getName());
                    }
                }
                System.out.println("</secs:test>\n");
            }
        }
    }

    public void dump() {
        System.out.println("*** TestCase ***");
        Enumeration enum = elements();
        while (enum.hasMoreElements()) {
            TestNode test = (TestNode)enum.nextElement();
            test.dump();
        }
        System.out.println("*** Test sequence ***");
        Enumeration seq = m_main.tests();
        while (seq.hasMoreElements()) {
            Object obj = seq.nextElement();
            if (obj instanceof TestCase) {
                System.out.println(((TestCase)obj).getName());
            } else {
                System.out.println(((TestSuite)obj).getName());
            }
        }
    }

    //
    // Test main
    //
    public static void main(String[] args) {
        SECSTestManager manager = SECSTestManager.getTestManager();
        manager.getTestTarget().dump();
    }
}

