/* $Id: XmlMessage.java,v 1.5 2004/06/27 08:01:56 fukasawa Exp $ */

/**
 *  @file    XmlMessage.java
 *
 *  @author  Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 2001-2004 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package com.jyugem.secs;

import java.util.*;
import java.lang.*;
import java.io.*;
import javax.swing.*;
import javax.swing.tree.*;
import javax.swing.event.*;

import org.w3c.dom.Document;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Text;
import org.w3c.dom.Attr;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.SAXNotRecognizedException;
import org.xml.sax.SAXNotSupportedException;
import org.apache.xerces.parsers.DOMParser;

public class XmlMessage extends TreeMap {

    // feature ids
    protected static final String NAMESPACES_FEATURE_ID = "http://xml.org/sax/features/namespaces";
    protected static final String VALIDATION_FEATURE_ID = "http://xml.org/sax/features/validation";
    protected static final String SCHEMA_VALIDATION_FEATURE_ID = "http://apache.org/xml/features/validation/schema";
    protected static final String CREATE_ENTITY_REF_NODES = "http://apache.org/xml/features/dom/create-entity-ref-nodes";

    // default settings
    protected static final boolean DEFAULT_NAMESPACES = true;
    protected static final boolean DEFAULT_NAMESPACE_PREFIXES = false;
    protected static final boolean DEFAULT_VALIDATION = false;
    protected static final boolean DEFAULT_SCHEMA_VALIDATION = true;
    protected static final boolean DEFAULT_CREATE_ENTITY_REF_NODES = false;

    //
    // Data
    //
    DOMParser parser = new DOMParser();
    Document      xmldoc;

    //
    // Public methods
    //
    // Constructor
    public XmlMessage() {
        // Sort permutation groups according to size
        super(new Comparator() {
            public int compare(Object o1, Object o2) {
                int sf1 = SECSMessage.toStreamNumber((String)o1);
                int sf2 = SECSMessage.toStreamNumber((String)o2);
                if (sf1 == sf2) {
                    return ((String)o1).compareTo((String)o2);
                }
                return (sf1 - sf2);
            }
        });
    }

    // Read message file that is formatted xml
    public int open(String filename) {
        boolean namespaces = DEFAULT_NAMESPACES;
        boolean validation = DEFAULT_VALIDATION;
        boolean schemaValidation = DEFAULT_SCHEMA_VALIDATION;
        boolean entityReference = DEFAULT_CREATE_ENTITY_REF_NODES;

        // set parser features
        try {
            parser.setFeature(NAMESPACES_FEATURE_ID, namespaces);
            parser.setFeature(VALIDATION_FEATURE_ID, validation);
            parser.setFeature(SCHEMA_VALIDATION_FEATURE_ID, schemaValidation);
            parser.setFeature(CREATE_ENTITY_REF_NODES, entityReference);
        } catch (SAXException e) {
            System.err.println("warning: Parser does not support feature");
        }

        //
        // parse file
        //
        try {
            parser.parse(filename);
            Document document = parser.getDocument();
            setDocument(document);
            buildTree();
        }
        catch (SAXParseException e) {
            // ignore
        }
        catch (Exception e) {
            System.err.println("error: Parse error occurred - "+ e.getMessage());
            Exception se = e;
            if (e instanceof SAXException) {
                se = ((SAXException)e).getException();
            }
            if (se != null) {
                se.printStackTrace(System.err);
            } else {
                e.printStackTrace(System.err);
            }
        }
        return 0;
    }

    /** Sets the document. */
    public synchronized void setDocument(Document document) {
        xmldoc = document;        // save document
    }

    /** Returns the document. */
    public Document getDocument() {
        return xmldoc;
    }

    // Builds the tree
    private void buildTree() {
        // is there anything to do?
        if (xmldoc == null) {
            return;
        }

        // iterate over children of this node
        NodeList nodes = xmldoc.getChildNodes();
        int len = (nodes != null) ? nodes.getLength() : 0;
        for (int i = 0; i < len; i++) {
            Node node = nodes.item(i);
            if (node.getNodeType() == Node.ELEMENT_NODE &&
                node.getNodeName().equals("secs:messages")) {
                insertMessageElementNode(node);
            }
        }
    }

    // Builds the message
    private void insertMessageElementNode(Node node) {

        // iterate over children of this node
        NodeList nodes = node.getChildNodes();
        int len = (nodes != null) ? nodes.getLength() : 0;
        for (int i = 0; i < len; i++) {
            Node msgnode = nodes.item(i);
            if (msgnode.getNodeType() == Node.ELEMENT_NODE &&
                msgnode.getNodeName().equals("secs:message")) {
                String sfname = null;
                NamedNodeMap attrs = msgnode.getAttributes();
                int attrCount = (attrs != null) ? attrs.getLength() : 0;
                for (int j = 0; j < attrCount; j++) {
                    Node attr = attrs.item(j);
                    if (attr.getNodeName().equals("sf")) {
                        if (sfname == null) {
                            sfname = attr.getNodeValue();
                        }
                    } else if (attr.getNodeName().equals("name")) {
                        sfname = attr.getNodeValue();
                    }
                }
                if (sfname == null) {
                    sfname = "------";
                }
                put(sfname, msgnode);
            }
        }
    }

    // Convert xml to SECS message.
    public static SECSMessage parseMessageElementNode(Node msgnode) throws
            SECSException {
        String sfname = null;
        String waitStr = "";
        String transStr = "";
        NamedNodeMap attrs = msgnode.getAttributes();
        int attrCount = (attrs != null) ? attrs.getLength() : 0;
        for (int i = 0; i < attrCount; i++) {
            Node attr = attrs.item(i);
            if (attr.getNodeName().equals("sf")) {    // Must use "sf" code
                sfname = attr.getNodeValue();
            } else if (attr.getNodeName().equals("wait")) {
                waitStr = attr.getNodeValue();
            } else if (attr.getNodeName().equals("transaction")) {
                transStr = attr.getNodeValue();
            }
        }

        if (sfname == null) {
            sfname = "------";
        }
        boolean wait = waitStr.toUpperCase().equals("TRUE") ? true : false;
        int transNum = 0;
        if (transStr.length() > 0) {
            transNum = Integer.parseInt(transStr);
        }
        SECSMessage msg = new SECSMessage(sfname, wait, transNum);

        // gather up attributes and children nodes
        try {
            NodeList children = msgnode.getChildNodes();
            int len = (children != null) ? children.getLength() : 0;
            for (int i = 0; i < len; i++) {
                Node node = children.item(i);
                if (node.getNodeType() == Node.ELEMENT_NODE) {
                    if (node.getNodeName().equals("secs:list")) {
                        SECSList secslist = new SECSList();
                        msg.set(secslist);
                        setListNode(node, secslist);
                    } else if (node.getNodeName().equals("secs:item")) {
                        SECSItem item = parseItemNode(node);
                        msg.set(item);
                    }
                }
            }
        } catch (SECSException e) {
            throw e;
        }
        return msg;
    }

    // Gets an item element node.
    private static SECSItem parseItemNode(Node inode) throws SECSException {
        String value = "";
        String iname = "";
        String typeStr = "";
        NamedNodeMap attrs = inode.getAttributes();
        int attrCount = (attrs != null) ? attrs.getLength() : 0;
        for (int i = 0; i < attrCount; i++) {
            Node attr = attrs.item(i);
            if (attr.getNodeName().equals("name")) {
                iname = attr.getNodeValue();
            } else if (attr.getNodeName().equals("type")) {
                typeStr = attr.getNodeValue();
            } else if (attr.getNodeName().equals("comment")) {
                String commentStr = attr.getNodeValue();   // ignore
            }
        }
        Node childNode = inode.getFirstChild();
        if (childNode != null) {
            if (childNode.getNodeType() == Node.TEXT_NODE) {
                value = childNode.getNodeValue();
            } else if (childNode.getNodeType() == Node.ELEMENT_NODE) {
                value = childNode.getNodeValue();
            } else if (childNode.getNodeType() == Node.CDATA_SECTION_NODE) {
                if (typeStr.equals("ascii") || typeStr.equals("jis") ||
                    typeStr.equals("utf-8")) {
                    value = childNode.getNodeValue();
                }
            }
        }

        // make item node
        try {
            SECSItem item = new SECSItem(iname, typeStr, value);
            return item;
        } catch (SECSException e) {
            throw e;
        }
    }

    private static SECSItem parseItemList(Node inode) throws SECSException {
        try {
            String iname = "";
            NamedNodeMap attrs = inode.getAttributes();
            int attrCount = (attrs != null) ? attrs.getLength() : 0;
            for (int i = 0; i < attrCount; i++) {
                Node attr = attrs.item(i);
                if (attr.getNodeName().equals("name")) {
                    iname = attr.getNodeValue();
                }
            }

            // make item list node
            ArrayList member = new ArrayList();
            NodeList children = inode.getChildNodes();
            int len = (children != null) ? children.getLength() : 0;
            for (int i = 0; i < len; i++) {
                Node node = children.item(i);
                if (node.getNodeType() == Node.ELEMENT_NODE) {
                    SECSItem item = parseItemNode(node);
                    member.add(item);
                }
            }
            SECSItem item = new SECSItem(iname, member);
            return item;
        } catch (SECSException e) {
            throw e;
        }
    }

    // Sets an item element node in list.
    private static void setListNode(Node lnode, SECSList slist) throws
            SECSException {
        try {
            // gather up attributes and children nodes
            NodeList children = lnode.getChildNodes();
            int len = (children != null) ? children.getLength() : 0;
            for (int i = 0; i < len; i++) {
                Node node = children.item(i);
                if (node.getNodeType() == Node.ELEMENT_NODE) {
                    if (node.getNodeName().equals("secs:list")) {
                        SECSList secslist = new SECSList();
                        slist.add(secslist);
                        setListNode(node, secslist);
                    } else if (node.getNodeName().equals("secs:item")) {
                        SECSItem item = parseItemNode(node);
                        slist.add(item);
                    } else if (node.getNodeName().equals("secs:itemlist")) {
                        SECSItem item = parseItemList(node);
                        slist.add(item);
                    }
                }
            }
        } catch (SECSException e) {
            throw e;
        }
    }

    // Make message by stream/function
    public SECSMessage makeMessage(String sfname) throws SECSException {
        try {
            Node node = (Node)get(sfname);
            if (node == null) {
                return null;
            }
            return parseMessageElementNode(node);
        } catch (SECSException e) {
            throw e;
        }
    }

    public void dump() {
        try {
            Iterator iter = values().iterator();
            while (iter.hasNext()) {
                Node root = (Node)iter.next();
                SECSMessage msg = parseMessageElementNode(root);  // XML to SECS
                msg.dump();
            }
        } catch (SECSException e) {
            System.err.println("error: Parse error occurred - "+ e.getMessage());
            e.printStackTrace(System.err);
        }
    }

    //
    // Test main
    //
    public static void main(String[] args) {
        XmlMessage resp = new XmlMessage();
        if (args.length > 0) {
            resp.open(args[0]);
        } else {
            resp.open("testResponse.xml");
        }

        resp.dump();

    }
}

