#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# This file is part of Pysilhouette.
#
# Copyright (c) 2009 HDE, Inc.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#

"""
@author: Kei Funagayama <kei@karesansui-project.info>
"""

import sys
import signal
import os
import traceback
import logging

from sqlalchemy.pool import SingletonThreadPool, QueuePool
from pysilhouette.log import reload_conf
from pysilhouette.prep import readconf, getopts, chkopts
from pysilhouette.db import Database
from pysilhouette.db.model import reload_mappers, JOBGROUP_STATUS
from pysilhouette.db.access import jobgroup_findbystatus, jobgroup_update
from pysilhouette.worker import SimpleWorker

from pysilhouette.util import kill_proc, write_pidfile, create_fifo

def sigterm_handler(signum, frame):
    logger = logging.getLogger('pysilhouette.performer.signal')
    logger.info('Stop the performerd with signal - pid=%s, signal=%s' % (os.getpid(), signum))

def performer(opts, cf):
    logger = logging.getLogger('pysilhouette.performer')

    # Initialization
    if os.access(cf["observer.mkfifo.path"], os.F_OK|os.R_OK|os.W_OK) is False:
        try:
            os.unlink(cf["observer.mkfifo.path"])
            logger.info('Deleted filo file. - file=%s' % cf["observer.mkfifo.path"])
        except:
            pass # Not anything
        
        create_fifo(cf["observer.mkfifo.path"],
                    cf["observer.mkfifo.user.name"],
                    cf["observer.mkfifo.group.name"],
                    cf["observer.mkfifo.perms"],
                    )

        logger.info('The fifo file was created. - file=%s' % cf["observer.mkfifo.path"])


    if opts.daemon is True:
        pid = os.getpid()
        if write_pidfile(opts.pidfile, pid):
            logger.info('The process file was created. - file=%s' % opts.pidfile)
        else:
            logger.error('Could not create process file. - file=%s' % opts.pidfile)
            return 1

    logger.info('performer : [started]')

    try:
        if cf['database.url'][:6].strip() == 'sqlite':
            db = Database(cf['database.url'],
                          encoding="utf-8",
                          convert_unicode=True,
                          #assert_unicode='warn', # DEBUG
                          #echo = opts.verbose,
                          #echo_pool = opts.verbose,
                          echo=True, # TODO
                          echo_pool=True # TODO
                          )
        else:
            if int(cf['database.pool.status']) == 1:
                db = Database(cf['database.url'],
                              encoding="utf-8",
                              convert_unicode=True,
                              #assert_unicode='warn', # DEBUG
                              poolclass=QueuePool,
                              pool_size=int(cf['database.pool.size']),
                              max_overflow=int(cf['database.pool.max.overflow']),
                              #echo = opts.verbose,
                              #echo_pool = opts.verbose,
                              echo=True, # TODO
                              echo_pool=True # TODO
                              )
            else:
                db = Database(cf['database.url'],
                              encoding="utf-8",
                              convert_unicode=True,
                              #assert_unicode='warn', # DEBUG
                              poolclass=SingletonThreadPool,
                              #echo = opts.verbose,
                              #echo_pool = opts.verbose,
                              echo=True, # TODO
                              echo_pool=True # TODO
                              )

        reload_mappers(db.get_metadata())

    except Exception, e:
        logger.error('Initializing a database error - %s' % str(e.args))
        t_logger = logging.getLogger('pysilhouette_traceback')
        t_logger.error(traceback.format_exc())
        return 1

    while True:
        fp = open(cf["observer.mkfifo.path"], 'r')
        try:
            code = fp.read()
        finally:
            fp.close()
                
        logger.info('Received code from the FIFO file. - code=%s' % code)
        session = db.get_session()
        m_jgs = jobgroup_findbystatus(session)
        session.close()
        
        logger.info('Queued the Job Group from the database. - Number of JobGroup=%d' % len(m_jgs))
        
        if code == cf["observer.mkfifo.start.code"]:
            if 0 < len(m_jgs):
                for m_jg in m_jgs:
                    try:
                        w = SimpleWorker(cf, db, m_jg.id)
                        w.run()
                    except Exception, e:
                        logger.info('Failed to perform the job group. Exceptions are not expected. - jobgroup_id=%d : %s'
                                     % (m_jg.id, str(e.args)))
                        print >>sys.stderr, traceback.format_exc()
                        t_logger = logging.getLogger('pysilhouette_traceback')
                        t_logger.error(traceback.format_exc())

                        try:
                            session = db.get_session()
                            jobgroup_update(session, m_jg, JOBGROUP_STATUS['APPERR'])
                            session.close()
                        except:
                            logger.error('Failed to change the status of the job group. - jobgroup_id=%d : %s'
                                         % (m_jg.id, str(e.args)))
                            t_logger = logging.getLogger('pysilhouette_traceback')
                            t_logger.error(traceback.format_exc())
                            
            else:
                logger.info('No Job Group.')
        elif code == cf["observer.mkfifo.stop.code"]:
            logger.info('Received stop code from the FIFO file. - code=%s' % code)
            break
        else:
            logger.info('Received illegal code from the FIFO file. - code=%s' % code)

def main():
    (opts, args) = getopts()
    if chkopts(opts) is True:
        return 1
    
    cf = readconf(opts.config)
    if cf is None:
        print >>sys.stderr, 'Failed to load the config file "%s". (%s)' % (opts.config, sys.argv[0])
        return 1

    # set env=PYSILHOUETTE_CONF
    os.environ['PYSILHOUETTE_CONF'] = opts.config
    
    if reload_conf(cf["env.sys.log.conf.path"]):
        logger = logging.getLogger('pysilhouette.performer')
    else:
        print >>sys.stderr, 'Failed to load the log file. (%s)' % sys.argv[0]
        return 1

    try:
        try:
            signal.signal(signal.SIGTERM, sigterm_handler)
            ret = performer(opts, cf) # start!!
            return ret
        except KeyboardInterrupt, k:
            logger.critical('Keyboard interrupt occurred. - %s' % str(k.args))
            print >>sys.stderr, 'Keyboard interrupt occurred. - %s' % str(k.args)
        except Exception, e:
            logger.critical('System error has occurred. - %s' % str(e.args))
            print >>sys.stderr, 'System error has occurred. - %s' % str(e.args)
            print >>sys.stderr, traceback.format_exc()
            t_logger = logging.getLogger('pysilhouette_traceback')
            t_logger.critical(traceback.format_exc())
            
    finally:
        if opts.daemon is True and os.path.isfile(opts.pidfile):
            os.unlink(opts.pidfile)
            logger.info('Process file has been deleted.. - pidfile=%s' % opts.pidfile)

if __name__ == '__main__':
    sys.exit(main())
