/*
Kasumi: Dictionary management tool for Anthy
Copyright (C) 2004 Takashi Nakamoto

This file is part of Kasumi

Kasumi is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Please send bug-reports, opinions and patches, if any, to me by E-Mail.
My E-mail address is following: bluedwarf@openoffice.org
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gtk/gtk.h>
#include "intl.h"
#include "topwindow.h"
#include "wordlist.h"
#include "anthy_dic.h"

GtkWidget *word_entry,*yomi_entry,*freq_spin;
GtkCombo *part_combo;
GtkWidget *noun_option_pane,*adv_option_pane;
GtkWidget *noun_na_check,*noun_sa_check,*noun_suru_check,*noun_gokan_check,
  *noun_kakujoshi_check,*adv_to_check,*adv_taru_check,*adv_suru_check,
  *adv_gokan_check;


static void changed_word_entry(GtkWidget *widget, gpointer data){
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;
  AnthyWord *selected_word;

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(wordview));
  model = gtk_tree_view_get_model(GTK_TREE_VIEW(wordview));
  
  if(gtk_tree_selection_get_selected(selection,&model,&iter)){
    selected_word = get_anthy_word_from_iter(model,&iter);
    free(selected_word->word);
    selected_word->word = gtk_editable_get_chars(GTK_EDITABLE(word_entry),
                                                 0,-1);
    gtk_list_store_set(wordlist, &iter,
                       COL_WORD, selected_word->word,
                       -1);
  }
}

static void changed_yomi_entry(GtkWidget *widget, gpointer data){
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;
  AnthyWord *selected_word;

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(wordview));
  model = gtk_tree_view_get_model(GTK_TREE_VIEW(wordview));
  
  if(gtk_tree_selection_get_selected(selection,&model,&iter)){
    selected_word = get_anthy_word_from_iter(model,&iter);
    free(selected_word->yomi);
    selected_word->yomi = gtk_editable_get_chars(GTK_EDITABLE(yomi_entry),
                                                 0,-1);
    gtk_list_store_set(wordlist, &iter,
                       COL_YOMI, selected_word->yomi,
                       -1);
  }
}

static void changed_freq_spin(GtkWidget *widget, gpointer data){
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;
  AnthyWord *selected_word;

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(wordview));
  model = gtk_tree_view_get_model(GTK_TREE_VIEW(wordview));
  
  if(gtk_tree_selection_get_selected(selection,&model,&iter)){
    selected_word = get_anthy_word_from_iter(model,&iter);
    selected_word->freq
      = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(freq_spin));
    gtk_list_store_set(wordlist, &iter,
                       COL_FREQ, (guint)(selected_word->freq),
                       -1);
  }
}

static void changed_part_combo(GtkWidget *widget, gpointer data){
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;
  AnthyWord *selected_word;

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(wordview));
  model = gtk_tree_view_get_model(GTK_TREE_VIEW(wordview));
  
  if(gtk_tree_selection_get_selected(selection,&model,&iter)){
    char *text;
    
    selected_word = get_anthy_word_from_iter(model,&iter);
    text = gtk_editable_get_chars(GTK_EDITABLE(part_combo->entry),0,-1);
    if(strcmp(text,UTF8_MEISHI) == 0){
      gtk_list_store_set(wordlist, &iter,
                         COL_PART, UTF8_MEISHI,
                         -1);
      selected_word->part = NOUN;
    }else if(strcmp(text,UTF8_FUKUSHI) == 0){
      gtk_list_store_set(wordlist, &iter,
                         COL_PART, UTF8_FUKUSHI,
                         -1);
      selected_word->part = ADVERB;
    }else if(strcmp(text,UTF8_JINNMEI) == 0){
      gtk_list_store_set(wordlist, &iter,
                         COL_PART, UTF8_JINNMEI,
                         -1);
      selected_word->part = PERSON;
    }else if(strcmp(text,UTF8_CHIMEI) == 0){
      gtk_list_store_set(wordlist, &iter,
                         COL_PART, UTF8_CHIMEI,
                         -1);
      selected_word->part = PLACE;
    }else if(strcmp(text,UTF8_KEIYOUSHI) == 0){
      gtk_list_store_set(wordlist, &iter,
                         COL_PART, UTF8_KEIYOUSHI,
                         -1);
      selected_word->part = ADJECTIVE;
    }
    
    free(text);
  }
}

static void save(GtkWidget *widget, gpointer data){
  save_anthy_dic();
}

static void add(GtkWidget *widget, gpointer data){
  AnthyWord newword,*new_p;
  GtkTreeSelection *selection;
  GtkTreePath *path;

  newword.word = "単語";
  newword.yomi = "読み";
  newword.freq = 1;
  newword.part = NOUN;
  newword.noun_na_connection = NO;
  newword.noun_sa_connection = NO;
  newword.noun_suru_connection = NO;
  newword.noun_gokan = NO;
  newword.noun_kakujoshi_connection = NO;
  newword.adv_to_connection = NO;
  newword.adv_taru_connection = NO;
  newword.adv_suru_connection = NO;
  newword.adv_gokan = NO;

  new_p = add_word(newword);

  path = gtk_tree_row_reference_get_path(new_p->row_reference);  
  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(wordview));
  gtk_tree_selection_select_path(selection,path);
  g_signal_emit_by_name(G_OBJECT(wordview),"cursor-changed");  
}

static void remove(GtkWidget *widget, gpointer data){
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreePath *path;
  GtkTreeIter iter;
  AnthyWord *selected_word;
  AnthyWord *next_word;

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(wordview));
  model = gtk_tree_view_get_model(GTK_TREE_VIEW(wordview));

  if(gtk_tree_selection_get_selected(selection,&model,&iter)){
    selected_word = get_anthy_word_from_iter(model,&iter);
    next_word = selected_word->next;
    remove_word(selected_word);

    if(next_word == NULL){
      next_word = last_anthy_word();
      if(next_word == NULL){
        gtk_entry_set_text(GTK_ENTRY(word_entry),"");
        gtk_entry_set_text(GTK_ENTRY(yomi_entry),"");
        gtk_spin_button_set_value(GTK_SPIN_BUTTON(freq_spin),1);
        gtk_entry_set_text(GTK_ENTRY(part_combo->entry),_("Noun"));
        gtk_widget_hide(noun_option_pane);
        gtk_widget_hide(adv_option_pane);
        return;
      }
    }

    path = gtk_tree_row_reference_get_path(next_word->row_reference);
    gtk_tree_selection_select_path(selection,path);
    g_signal_emit_by_name(G_OBJECT(wordview),"cursor-changed");
  }
}

static void toggled_check(GtkWidget *widget, gpointer data){
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;
  AnthyWord *selected_word;

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(wordview));
  model = gtk_tree_view_get_model(GTK_TREE_VIEW(wordview));

  if(gtk_tree_selection_get_selected(selection,&model,&iter)){
    selected_word = get_anthy_word_from_iter(model,&iter);
    
    if(widget==noun_na_check){
      selected_word->noun_na_connection
        = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(noun_na_check))
        ? YES : NO;
    }else if(widget==noun_sa_check){
      selected_word->noun_sa_connection
        = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(noun_sa_check))
        ? YES : NO;
    }else if(widget==noun_suru_check){
      selected_word->noun_suru_connection
        = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(noun_suru_check))
        ? YES : NO;
    }else if(widget==noun_gokan_check){
      selected_word->noun_gokan
        = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(noun_gokan_check))
        ? YES : NO;
    }else if(widget==noun_kakujoshi_check){
      selected_word->noun_kakujoshi_connection
        = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(noun_kakujoshi_check))
        ? YES : NO;
    }else if(widget==adv_to_check){
      selected_word->adv_to_connection
        = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(adv_to_check))
        ? YES : NO;
    }else if(widget==adv_taru_check){
      selected_word->adv_taru_connection
        = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(adv_taru_check))
        ? YES : NO;
    }else if(widget==adv_suru_check){
      selected_word->adv_suru_connection
        = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(adv_suru_check))
        ? YES : NO;
    }else if(widget==adv_gokan_check){
      selected_word->adv_gokan
        = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(adv_gokan_check))
        ? YES : NO;
    }
  }
}

static void wordlist_cursor_changed(GtkWidget *widget, gpointer data){
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;
  AnthyWord *selected_word;

  g_signal_handlers_disconnect_by_func(G_OBJECT(word_entry),
                                       G_CALLBACK(changed_word_entry), NULL);
  g_signal_handlers_disconnect_by_func(G_OBJECT(yomi_entry),
                                       G_CALLBACK(changed_yomi_entry), NULL);
  g_signal_handlers_disconnect_by_func(G_OBJECT(freq_spin),
                                       G_CALLBACK(changed_freq_spin), NULL);
  g_signal_handlers_disconnect_by_func(G_OBJECT(part_combo->entry),
                                       G_CALLBACK(changed_part_combo), NULL);
  g_signal_handlers_disconnect_by_func(G_OBJECT(noun_na_check),
                                       G_CALLBACK(toggled_check), NULL);
  g_signal_handlers_disconnect_by_func(G_OBJECT(noun_sa_check),
                                       G_CALLBACK(toggled_check), NULL);
  g_signal_handlers_disconnect_by_func(G_OBJECT(noun_suru_check),
                                       G_CALLBACK(toggled_check), NULL);
  g_signal_handlers_disconnect_by_func(G_OBJECT(noun_gokan_check),
                                       G_CALLBACK(toggled_check), NULL);
  g_signal_handlers_disconnect_by_func(G_OBJECT(noun_kakujoshi_check),
                                       G_CALLBACK(toggled_check), NULL);
  g_signal_handlers_disconnect_by_func(G_OBJECT(adv_to_check),
                                       G_CALLBACK(toggled_check), NULL);
  g_signal_handlers_disconnect_by_func(G_OBJECT(adv_taru_check),
                                       G_CALLBACK(toggled_check), NULL);
  g_signal_handlers_disconnect_by_func(G_OBJECT(adv_suru_check),
                                       G_CALLBACK(toggled_check), NULL);
  g_signal_handlers_disconnect_by_func(G_OBJECT(adv_gokan_check),
                                       G_CALLBACK(toggled_check), NULL);
  
  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(wordview));
  model = gtk_tree_view_get_model(GTK_TREE_VIEW(wordview));
  
  if(gtk_tree_selection_get_selected(selection,&model,&iter)){
    selected_word = get_anthy_word_from_iter(model,&iter);
    gtk_entry_set_text(GTK_ENTRY(word_entry),selected_word->word);
    gtk_entry_set_text(GTK_ENTRY(yomi_entry),selected_word->yomi);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(freq_spin),selected_word->freq);
    switch(selected_word->part){
    case NOUN:
      gtk_entry_set_text(GTK_ENTRY(part_combo->entry),_("Noun"));
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(noun_na_check),
                      selected_word->noun_na_connection == YES ? TRUE : FALSE);
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(noun_sa_check),
                      selected_word->noun_sa_connection == YES ? TRUE : FALSE);
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(noun_suru_check),
                    selected_word->noun_suru_connection == YES ? TRUE : FALSE);
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(noun_gokan_check),
                              selected_word->noun_gokan == YES ? TRUE : FALSE);
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(noun_kakujoshi_check),
               selected_word->noun_kakujoshi_connection == YES ? TRUE : FALSE);
      gtk_widget_show(noun_option_pane);
      gtk_widget_hide(adv_option_pane);
       break;
    case ADVERB:
      gtk_entry_set_text(GTK_ENTRY(part_combo->entry),_("Adverb"));
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(adv_to_check),
                      selected_word->adv_to_connection == YES ? TRUE : FALSE);
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(adv_taru_check),
                    selected_word->adv_taru_connection == YES ? TRUE : FALSE);
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(adv_suru_check),
                    selected_word->adv_suru_connection == YES ? TRUE : FALSE);
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(adv_gokan_check),
                      selected_word->adv_gokan == YES ? TRUE : FALSE);
      gtk_widget_hide(noun_option_pane);
      gtk_widget_show(adv_option_pane);
      break;
    case PERSON:
      gtk_entry_set_text(GTK_ENTRY(part_combo->entry),_("Person's name"));
      gtk_widget_hide(noun_option_pane);
      gtk_widget_hide(adv_option_pane);
      break;
    case PLACE:
      gtk_entry_set_text(GTK_ENTRY(part_combo->entry),_("Place-name"));
      gtk_widget_hide(noun_option_pane);
      gtk_widget_hide(adv_option_pane);
      break;
    case ADJECTIVE:
      gtk_entry_set_text(GTK_ENTRY(part_combo->entry),_("Adjective"));
      gtk_widget_hide(noun_option_pane);
      gtk_widget_hide(adv_option_pane);
      break;
    }
  }else{
    gtk_entry_set_text(GTK_ENTRY(word_entry),"");
    gtk_entry_set_text(GTK_ENTRY(yomi_entry),"");
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(freq_spin),1);
    gtk_entry_set_text(GTK_ENTRY(part_combo->entry),_("Noun"));
    gtk_widget_hide(noun_option_pane);
    gtk_widget_hide(adv_option_pane);
  }

  g_signal_connect(G_OBJECT(word_entry), "changed",
                   G_CALLBACK(changed_word_entry), NULL);
  g_signal_connect(G_OBJECT(yomi_entry), "changed",
                   G_CALLBACK(changed_yomi_entry), NULL);
  g_signal_connect(G_OBJECT(freq_spin), "value-changed",
                   G_CALLBACK(changed_freq_spin), NULL);
  g_signal_connect(G_OBJECT(part_combo->entry), "changed",
                   G_CALLBACK(changed_part_combo), NULL);
  g_signal_connect(G_OBJECT(noun_na_check), "toggled",
                   G_CALLBACK(toggled_check), NULL);
  g_signal_connect(G_OBJECT(noun_sa_check), "toggled",
                   G_CALLBACK(toggled_check), NULL);
  g_signal_connect(G_OBJECT(noun_suru_check), "toggled",
                   G_CALLBACK(toggled_check), NULL);
  g_signal_connect(G_OBJECT(noun_gokan_check), "toggled",
                   G_CALLBACK(toggled_check), NULL);
  g_signal_connect(G_OBJECT(noun_kakujoshi_check), "toggled",
                   G_CALLBACK(toggled_check), NULL);
  g_signal_connect(G_OBJECT(adv_to_check), "toggled",
                   G_CALLBACK(toggled_check), NULL);
  g_signal_connect(G_OBJECT(adv_taru_check), "toggled",
                   G_CALLBACK(toggled_check), NULL);
  g_signal_connect(G_OBJECT(adv_suru_check), "toggled",
                   G_CALLBACK(toggled_check), NULL);
  g_signal_connect(G_OBJECT(adv_gokan_check), "toggled",
                   G_CALLBACK(toggled_check), NULL);
}

static gboolean delete_event(GtkWidget *widget,
                             GdkEvent *event,
                             gpointer data)
{
  save(widget,data);
  return FALSE;
}

static void destroy(GtkWidget *widget, gpointer data){
  save(widget,data);
  gtk_main_quit();
}

static void quit(GtkWidget *widget, gpointer data){
  if(!delete_event(widget,NULL,data)){
    gtk_main_quit();
  }
}

int create_top_window()
{
  GtkWidget *window;
  GtkWidget *vbox,*hbox,*scroll,*entry_vbox,*label,*alignment,*hbutton_box,
    *button,*wordlist_view;
  GtkObject *adjustment;
  GList *items = NULL;

  /* set up window */
  window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(window), _("Kasumi"));
  g_signal_connect(G_OBJECT(window), "delete_event",
                   G_CALLBACK(delete_event), NULL);
  g_signal_connect(G_OBJECT(window), "destroy",
                   G_CALLBACK(destroy), NULL);

  /* creating top vbox */
  vbox = gtk_vbox_new(FALSE,8);
  gtk_container_set_border_width(GTK_CONTAINER(vbox),8);
  gtk_container_add(GTK_CONTAINER(window),vbox);

  /* creating hbox for wordlist and text entries */
  hbox = gtk_hbox_new(FALSE,8);
  gtk_box_pack_start(GTK_BOX(vbox),GTK_WIDGET(hbox),TRUE,TRUE,0);

  /* creating scrolled window for wordlist */
  scroll = gtk_scrolled_window_new(NULL,NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll),
                                 GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
  gtk_widget_set_size_request(GTK_WIDGET(scroll),300,300);
  gtk_box_pack_start(GTK_BOX(hbox),GTK_WIDGET(scroll),TRUE,TRUE,0);

  /* creating wordlist in scrolled window */
  wordlist_view = wordview;
  gtk_container_add(GTK_CONTAINER(scroll),GTK_WIDGET(wordlist_view));
  g_signal_connect(G_OBJECT(wordlist_view),"cursor-changed",
                   G_CALLBACK(wordlist_cursor_changed),NULL);

  /* creating vbox for text entries */
  entry_vbox = gtk_vbox_new(FALSE,0);
  gtk_box_pack_start(GTK_BOX(hbox),GTK_WIDGET(entry_vbox),FALSE,FALSE,0);

  /* creating text entries for "Word" */
  label = gtk_label_new(_("Word"));
  alignment = gtk_alignment_new(0,0.5,0,0);
  gtk_container_add(GTK_CONTAINER(alignment),GTK_WIDGET(label));
  gtk_box_pack_start(GTK_BOX(entry_vbox),GTK_WIDGET(alignment),FALSE,FALSE,0);

  word_entry = gtk_entry_new();
  gtk_box_pack_start(GTK_BOX(entry_vbox),GTK_WIDGET(word_entry),FALSE,FALSE,0);
  g_signal_connect(G_OBJECT(word_entry),"changed",
                   G_CALLBACK(changed_word_entry),NULL);

  /* creating text entries for "Yomi" */
  label = gtk_label_new(_("Yomi"));
  alignment = gtk_alignment_new(0,0.5,0,0);
  gtk_container_add(GTK_CONTAINER(alignment),GTK_WIDGET(label));
  gtk_box_pack_start(GTK_BOX(entry_vbox),GTK_WIDGET(alignment),FALSE,FALSE,0);

  yomi_entry = gtk_entry_new();
  gtk_box_pack_start(GTK_BOX(entry_vbox),GTK_WIDGET(yomi_entry),FALSE,FALSE,0);
  g_signal_connect(G_OBJECT(yomi_entry),"changed",
                   G_CALLBACK(changed_yomi_entry),NULL);

  /* creating spin button for "Frequency" */
  label = gtk_label_new(_("Frequency"));
  alignment = gtk_alignment_new(0,0.5,0,0);
  gtk_container_add(GTK_CONTAINER(alignment),GTK_WIDGET(label));
  gtk_box_pack_start(GTK_BOX(entry_vbox),GTK_WIDGET(alignment),FALSE,FALSE,0);

  adjustment = gtk_adjustment_new(1,1,32767,1,100,0);
  freq_spin = gtk_spin_button_new(GTK_ADJUSTMENT(adjustment),1.0,0);
  gtk_box_pack_start(GTK_BOX(entry_vbox),GTK_WIDGET(freq_spin),FALSE,FALSE,0);
  g_signal_connect(G_OBJECT(freq_spin),"value-changed",
                   G_CALLBACK(changed_freq_spin),NULL);

  /* creating spin button for "Part of Speech" */
  label = gtk_label_new(_("Part of Speech"));
  alignment = gtk_alignment_new(0,0.5,0,0);
  gtk_container_add(GTK_CONTAINER(alignment),GTK_WIDGET(label));
  gtk_box_pack_start(GTK_BOX(entry_vbox),GTK_WIDGET(alignment),FALSE,FALSE,0);

  part_combo = GTK_COMBO(gtk_combo_new());
  items = g_list_append(items,_("Noun"));
  items = g_list_append(items,_("Adverb"));
  items = g_list_append(items,_("Person's name"));
  items = g_list_append(items,_("Place-name"));
  items = g_list_append(items,_("Adjective"));
  gtk_combo_set_popdown_strings(GTK_COMBO(part_combo), items);
  gtk_entry_set_editable(GTK_ENTRY(part_combo->entry),FALSE);
  gtk_box_pack_start(GTK_BOX(entry_vbox),GTK_WIDGET(part_combo),FALSE,FALSE,0);
  g_signal_connect(G_OBJECT(part_combo->entry),"changed",
                   G_CALLBACK(changed_part_combo),NULL);

  /* creating noun option pane */
  noun_option_pane = gtk_vbox_new(FALSE,0);
  gtk_box_pack_start(GTK_BOX(entry_vbox),GTK_WIDGET(noun_option_pane),
                     FALSE,FALSE,0);

  noun_na_check = gtk_check_button_new_with_label(_("NA connection"));
  gtk_box_pack_start(GTK_BOX(noun_option_pane),GTK_WIDGET(noun_na_check),
                     FALSE,FALSE,0);
  g_signal_connect(G_OBJECT(noun_na_check),"toggled",
                   G_CALLBACK(toggled_check),NULL);

  noun_sa_check = gtk_check_button_new_with_label(_("SA connection"));
  gtk_box_pack_start(GTK_BOX(noun_option_pane),GTK_WIDGET(noun_sa_check),
                     FALSE,FALSE,0);
  g_signal_connect(G_OBJECT(noun_sa_check),"toggled",
                   G_CALLBACK(toggled_check),NULL);

  noun_suru_check = gtk_check_button_new_with_label(_("SURU connection"));
  gtk_box_pack_start(GTK_BOX(noun_option_pane),GTK_WIDGET(noun_suru_check),
                     FALSE,FALSE,0);
  g_signal_connect(G_OBJECT(noun_suru_check),"toggled",
                   G_CALLBACK(toggled_check),NULL);

  noun_gokan_check = gtk_check_button_new_with_label(_("Can be Bunnsetsu"));
  gtk_box_pack_start(GTK_BOX(noun_option_pane),GTK_WIDGET(noun_gokan_check),
                     FALSE,FALSE,0);
  g_signal_connect(G_OBJECT(noun_gokan_check),"toggled",
                   G_CALLBACK(toggled_check),NULL);

  noun_kakujoshi_check = gtk_check_button_new_with_label(_("KAKUJOSHI connection"));
  gtk_box_pack_start(GTK_BOX(noun_option_pane),
                     GTK_WIDGET(noun_kakujoshi_check),FALSE,FALSE,0);
  g_signal_connect(G_OBJECT(noun_kakujoshi_check),"toggled",
                   G_CALLBACK(toggled_check),NULL);

  /* creating adverb option pane */
  adv_option_pane = gtk_vbox_new(FALSE,0);
  gtk_box_pack_start(GTK_BOX(entry_vbox),GTK_WIDGET(adv_option_pane),
                     FALSE,FALSE,0);

  adv_to_check = gtk_check_button_new_with_label(_("TO connection"));
  gtk_box_pack_start(GTK_BOX(adv_option_pane),GTK_WIDGET(adv_to_check),
                     FALSE,FALSE,0);
  g_signal_connect(G_OBJECT(adv_to_check),"toggled",
                   G_CALLBACK(toggled_check),NULL);

  adv_taru_check = gtk_check_button_new_with_label(_("TARU connection"));
  gtk_box_pack_start(GTK_BOX(adv_option_pane),GTK_WIDGET(adv_taru_check),
                     FALSE,FALSE,0);
  g_signal_connect(G_OBJECT(adv_taru_check),"toggled",
                   G_CALLBACK(toggled_check),NULL);

  adv_suru_check = gtk_check_button_new_with_label(_("SURU connection"));
  gtk_box_pack_start(GTK_BOX(adv_option_pane),GTK_WIDGET(adv_suru_check),
                     FALSE,FALSE,0);
  g_signal_connect(G_OBJECT(adv_suru_check),"toggled",
                   G_CALLBACK(toggled_check),NULL);

  adv_gokan_check = gtk_check_button_new_with_label(_("Can be Bunnsetsu"));
  gtk_box_pack_start(GTK_BOX(adv_option_pane),GTK_WIDGET(adv_gokan_check),
                     FALSE,FALSE,0);
  g_signal_connect(G_OBJECT(adv_gokan_check),"toggled",
                   G_CALLBACK(toggled_check),NULL);

  /* creating box for buttons */
  hbutton_box = gtk_hbutton_box_new();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(hbutton_box),GTK_BUTTONBOX_SPREAD);
  gtk_box_pack_start(GTK_BOX(vbox),GTK_WIDGET(hbutton_box),FALSE,FALSE,0);

  /* creating buttons */
  button = gtk_button_new();
  gtk_button_set_label(GTK_BUTTON(button),_("Quit"));
  gtk_box_pack_start(GTK_BOX(hbutton_box),GTK_WIDGET(button),FALSE,FALSE,0);
  g_signal_connect(G_OBJECT(button),"clicked",
                   G_CALLBACK(quit),NULL);

  button = gtk_button_new();
  gtk_button_set_label(GTK_BUTTON(button),_("Save"));
  gtk_box_pack_start(GTK_BOX(hbutton_box),GTK_WIDGET(button),FALSE,FALSE,0);
  g_signal_connect(G_OBJECT(button),"clicked",
                   G_CALLBACK(save),NULL);

  button = gtk_button_new();
  gtk_button_set_label(GTK_BUTTON(button),_("Add"));
  gtk_box_pack_start(GTK_BOX(hbutton_box),GTK_WIDGET(button),FALSE,FALSE,0);
  g_signal_connect(G_OBJECT(button),"clicked",
                   G_CALLBACK(add),NULL);

  button = gtk_button_new();
  gtk_button_set_label(GTK_BUTTON(button),_("Remove"));
  gtk_box_pack_start(GTK_BOX(hbutton_box),GTK_WIDGET(button),FALSE,FALSE,0);
  g_signal_connect(G_OBJECT(button),"clicked",
                   G_CALLBACK(remove),NULL);
  
  gtk_widget_show_all(window);
  gtk_widget_hide(noun_option_pane);
  gtk_widget_hide(adv_option_pane);
  
  return 0;

}
