unit untBoard;

interface

uses
  untTopic, untHttp, SysUtils, untTool, Classes, Forms, StrUtils,
  untOption, untBBSSub, RegExpr;

type

  TTopicReceivedEvent = procedure(Sender : TObject; Topic : TTopic) of object;
  TBoard = class
  private
    FDownloading  : Boolean;
    FCategory     : string;
    FDisplayName  : string;
    FCurrentIndex : Integer;
    FOnTopicReceived: TTopicReceivedEvent;
    FIsDownloading: boolean;
    FBoardid: integer;
    procedure SetOnTopicReceived(const Value: TTopicReceivedEvent);
    procedure SetIsDownloading(const Value: boolean);
    procedure SetCategory(const Value: string);
    procedure SetDisplayName(const Value: string);
    procedure SetBoardid(const Value: integer);
  public
		property  Category    : string read FCategory write SetCategory;
		property  DisplayName : string read FDisplayName write SetDisplayName;
		property  Boardid     : integer read FBoardid write SetBoardid;
    property  OnTopicReceived : TTopicReceivedEvent read FOnTopicReceived write SetOnTopicReceived;
    property  IsDownloading : boolean read FIsDownloading write SetIsDownloading;
    constructor Create(BoardId : integer);
    procedure Download(); virtual; abstract;
    procedure RaiseTopicReceivedEvent(Topic : TTopic);
  end;

  TOnlineBoard = class(TBoard)
  private
    FHttpGet  : THttpGet;
    FWriteError: string;
    FBBSType: TBBSType;
    FServers  : TStringArray;
    FServer: string;
    FBoardName: string;
    procedure SetServer(const Value: string);
    function  GetBoardUrl()   : string;
    function  GetSubjectTxtUrl: string;
    function  GetBbsCgiUrl  : string;
    procedure SetWriteError(const Value: string);
    procedure SetBBSType(const Value: TBBSType);
    procedure HttpReceive(Sender: TObject; Complete: Boolean;
      ReceiveText: string);
    procedure HttpStateChange(Sender: TObject; HttpState: THttpState);
    procedure SetBoardName(const Value: string);
  public
    property  BBSType       : TBBSType read FBBSType write SetBBSType;
    property  BoardUrl      : string read GetBoardUrl;
    property  SubjectTxtUrl : string read GetSubjectTxtUrl;
    property  BbsCgiUrl     : string read GetBbsCgiUrl;
    property  WriteError    : string read FWriteError write SetWriteError;
    property  Server        : string read FServer write SetServer;
    property  BoardName     : string read FBoardName write SetBoardName;
    destructor  Destroy; override;
    procedure Download(); override;
  end;

  TOfflineBoard = class(TBoard)
  private
    FLoadedLog : Boolean;
    FIdxPath   : string;
    FTopicList : TList;
    procedure LoadLog();
  public
    constructor Create(BoardId : integer; idxpath : string);
    destructor  Destroy; override;
    procedure Download(); override;
    procedure AppendTopic(Topic : TTopic);
    procedure RemoveTopic(Topic : TTopic);
    procedure Save();
    procedure Patrol();
  end;

implementation

uses
  untGlobal;

{ TBoard }

//  vpeB 

procedure TBoard.SetBoardid(const Value: integer);
begin
  FBoardid := Value;
end;

procedure TBoard.SetCategory(const Value: string);
begin
  FCategory := Value;
end;

procedure TBoard.SetDisplayName(const Value: string);
begin
  FDisplayName := Value;
end;

procedure TBoard.SetOnTopicReceived(const Value: TTopicReceivedEvent);
begin
  FOnTopicReceived := Value;
end;

//  \bh 

{ --------------------------------------------------------
  ֐: Create
  pr  : RXgN^
    : BoardId
  @@@: DisplayName
  @@@; Category
  ߂l: Ȃ
  l  : Ȃ
  ------------------------------------------------------ }
constructor TBoard.Create(BoardId : integer);
begin
  FBoardId     := BoardId;
end;

{ --------------------------------------------------------
  ֐: RaiseTopicReceivedEvent
  pr  : TopicM|`Cxg𔭍s
    : Topic
  ߂l: Ȃ
  l  : Ȃ
  ------------------------------------------------------ }
procedure TBoard.RaiseTopicReceivedEvent(Topic: TTopic);
begin

  try
    if Assigned(FOnTopicReceived) then
      FOnTopicReceived(self, Topic);
  except
  end;

end;

{ TOnlineBoard }

//  vpeB 

procedure TOnlineBoard.SetServer(const Value: string);
begin
  FServer := Value;

  // BBSType𔻒f
  
  if Pos('jbbs', Value) > 0 then
    FBBSType := btJBBS
  else if Pos('machibbs.com', Value) > 0 then
    FBBSType := btJBBS
  else if Pos('shitaraba.com', Value) > 0 then
    FBBSType := btShitaraba
  else if Pos('www.mitinoku.jp', Value) > 0 then
    FBBSType := btMitinoku
  else
    FBBSType := bt2ch;

end;

procedure TOnlineBoard.SetBoardName(const Value: string);
begin
  FBoardName := Value;
end;

procedure TOnlineBoard.SetWriteError(const Value: string);
begin
  FWriteError := Value;
end;

procedure TBoard.SetIsDownloading(const Value: boolean);
begin
  FIsDownloading := Value;
end;

procedure TOnlineBoard.SetBBSType(const Value: TBBSType);
begin
  FBBSType := Value;
end;

function TOnlineBoard.GetBoardUrl: string;
begin
  result := 'http://' + FServer + '/' + FBoardName + '/';
end;

function TOnlineBoard.GetSubjectTxtUrl: string;
begin
  result := GetBoardUrl() + 'subject.txt';
end;

function TOnlineBoard.GetBbsCgiUrl: string;
begin
  result := 'http://' + FServer + '/test/bbs.cgi';
end;

{ --------------------------------------------------------
  ֐: Download
  pr  : gsbN̎MJn
    : Ȃ
  ߂l: Ȃ
  l  : Ȃ
  ------------------------------------------------------ }
procedure TOnlineBoard.Download;
begin

  if FDownloading then exit;
  FDownloading := true;

  // HTTPNX̗グ
  FCurrentIndex := 0;
  FHttpGet.Free;
  FHttpGet := THttpGet.Create;
  FHttpGet.ReceiveEventTiming := OnOneLine;
  FHttpGet.OnReceive          := HttpReceive;
  FHttpGet.OnStateChange      := HttpStateChange;
  if gReadProxyUse then FHttpGet.Proxy := gReadProxy;

  FHttpGet.Connect(SubjectTxtUrl);

end;

{ --------------------------------------------------------
  ֐: Destroy
  pr  : fXgN^
    : Ȃ
  ߂l: Ȃ
  l  : Ȃ
  ------------------------------------------------------ }
destructor TOnlineBoard.Destroy;
begin
  FHttpGet.Free;

  inherited;
end;

//  Cxg 

// f[^M
procedure TOnlineBoard.HttpReceive(Sender: TObject; Complete: Boolean;
  ReceiveText: string);
var
  P1, P2, P3    : Integer;
  TopicId       : string;
  TopicTitle    : string;
  TopicMsgCount : Integer;
  TopicPath     : string;
  Topic         : TTopic;
begin

  // TopicId
  P1 := Pos('.dat', ReceiveText);
  if P1 = 0 then exit;
  TopicId := Copy(ReceiveText, 1, P1 - 1);

  // TopicTitle
  P2 := LastDelimiter('(', ReceiveText);
  if P2 = 0 then exit;
  TopicTitle := Copy(ReceiveText, P1 + 6, P2 - P1 - 7);

  // TopicMsgCount
  P3 := LastDelimiter(')', ReceiveText);
  if P3 = 0 then exit;
  TopicMsgCount := StrToIntNeo(Copy(ReceiveText, P2 + 1, P3 - P2 - 1));

  // XbhNX
  Topic     := gBBSCore.GetTopic(self, TopicId);

  Inc(FCurrentIndex);
  Topic.Index := FCurrentIndex;
  Topic.Title        := TopicTitle;
  Topic.MessageCount := TopicMsgCount;

  // Cxgs
  RaiseTopicReceivedEvent(Topic);

end;

// 󋵕ω
procedure TOnlineBoard.HttpStateChange(Sender: TObject; HttpState: THttpState);
var
  contentsize : Integer;
begin

  if HttpState = hsDisconnected then
  begin

    FDownloading := false;

  end;

end;

{ TOfflineBoard }

{ --------------------------------------------------------
  ֐: Create
  pr  : RXgN^
    : idxpath
  @@@: BoardName
  ߂l: Ȃ
  l  : Ȃ
  ------------------------------------------------------ }
constructor TOfflineBoard.Create(BoardId : integer; idxpath : string);
begin
  inherited Create(BoardId);

  FIdxPath     := idxpath;

  FTopicList := TList.Create;
end;

{ --------------------------------------------------------
  ֐: Download
  pr  : gsbN̎MJn
    : Ȃ
  ߂l: Ȃ
  l  : Ȃ
  ------------------------------------------------------ }
procedure TOfflineBoard.Download;
var
  I      : Integer;
begin
  inherited;

  LoadLog();

  for I := 0 to FTopicList.Count - 1 do
    RaiseTopicReceivedEvent(FTopicList[i]);

end;

{ --------------------------------------------------------
  ֐: AppendTopic
  pr  : gsbN̒ǉ
    : Topic
  ߂l: Ȃ
  l  : Ȃ
  ------------------------------------------------------ }
procedure TOfflineBoard.AppendTopic(Topic: TTopic);
begin

  LoadLog();

  FTopicList.Insert(0, Topic);
  Topic.IsFavorite := true;
  Topic.SaveIdx;
  Save();

end;

{ --------------------------------------------------------
  ֐: RemoveTopic
  pr  : gsbN̍폜
    : Topic
  ߂l: Ȃ
  l  : Ȃ
  ------------------------------------------------------ }
procedure TOfflineBoard.RemoveTopic(Topic: TTopic);
begin

  LoadLog();

  FTopicList.Remove(Topic);
  Topic.IsFavorite := false;
  Topic.SaveIdx;
  Save();

end;

{ --------------------------------------------------------
  ֐: Destroy
  pr  : fXgN^
    : Ȃ
  ߂l: Ȃ
  l  : Ȃ
  ------------------------------------------------------ }
destructor TOfflineBoard.Destroy;
begin
  FTopicList.Free;

  inherited;
end;

{ --------------------------------------------------------
  ֐: Patrol
  pr  : 
    : Ȃ
  ߂l: Ȃ
  l  : Ȃ
  ------------------------------------------------------ }
procedure TOfflineBoard.Patrol;
var
  I : Integer;
  Topic : TTopic;
begin

  LoadLog();

  for I := 0 to FTopicList.Count - 1 do
  begin
    Application.ProcessMessages;
    Topic := TTopic(FTopicList[I]);

    //*: Ń^u̐V}[N͂
    //Ȃ΂ȂȂBۗB

    gBBSCore.DownloadTopic(Topic);
  end;

end;

//  vCx[g֐ 

{ --------------------------------------------------------
  ֐: LoadLog
  pr  : O̓ǂݍ
    : Ȃ
  ߂l: Ȃ
  l  : Ȃ
  ------------------------------------------------------ }
procedure TOfflineBoard.LoadLog;
var
  stlLog    : TStringList;
  Topic     : TTopic;
  I         : Integer;
  Regex     : TRegExpr;
  server    : string;
  boardname : string;
  topicid   : string;
  board     : TOnlineBoard;
begin

  Regex := TRegExpr.Create;
  Regex.Expression := '^(.+?)\\(.+?)\\(.+?)\.idx';

  if FLoadedLog = false then
  begin
    if FileExists(FIdxPath) then
    begin

      stlLog := TStringList.Create;
      stlLog.LoadFromFile(FIdxPath);

      // gsbN̓ǂݍ
      for I := 0 to stlLog.Count - 1 do
        if stlLog[I] <> '' then
          if Regex.Exec(stlLog[I]) then
          begin
            server    := Regex.Substitute('$1');
            boardname := Regex.Substitute('$2');
            topicid   := Regex.Substitute('$3');
            board := gBoardList.GetBoard(server, boardname); 

            Topic := gBBSCore.GetTopic(board, topicid);
            Topic.CountUp;     // ΉCountDownȂ
            FTopicList.Add(Topic);
          end;

      stlLog.Free;
    end;
    FLoadedLog := true;
  end;

  Regex.Free;

end;

{ --------------------------------------------------------
  ֐: Save
  pr  : ۑ
    : Ȃ
  ߂l: Ȃ
  l  : Ȃ
  ------------------------------------------------------ }
procedure TOfflineBoard.Save;
var
  I       : Integer;
  Topic   : TTopic;
  stlSave : TStringList;
  board   : TOnlineBoard;
begin

  LoadLog();

  stlSave := TStringList.Create;
  for I := 0 to FTopicList.Count - 1 do
  begin
    Topic := TTopic(FTopicList[I]);
    board := TOnlineBoard(Topic.Board);
    stlSave.Add(board.Server + '\' + board.BoardName + '\' + Topic.TopicId + '.idx');
  end;

  stlSave.SaveToFile(FIdxPath); 
  stlSave.Free;

end;


end.
