unit untHttp;

interface

uses
  Classes, SysUtils,
  IdThread, IdHttp, IdComponent, gzip,
  untStreamTool, SyncObjs;

type

  THttpErrpr = (heNoError, heBrokenGzip);

  TAsyncHttp = class(TIdThread)
  private
    FErrorCode     : THttpErrpr;
    FURL           : string;
    FHttp          : TIdHttp;
    FWriteEvent    : TMemoryStreamEx;
    FBufferReader  : TStreamReader;
    FReadPosition  : Integer;
    FOnReceived    : TNotifyEvent;
    FReceivedLines : TStringList;
    FStartRange    : Integer;
    FLastModified  : string;
    FOnStatus      : TIdStatusEvent;
    FStatus        : TIdStatus;
    FStatusText    : String;
    FOnComplete    : TNotifyEvent;
    FContentLength : Integer;
    FUserAgent     : string;
    FProxyPort     : Integer;
    FProxyHost     : string;
    FInited        : Boolean;
    FAddHeaders    : string;
    FResponseCode  : Integer;
    FBuffer        : TMemoryStream;
    FGzipStream    : TGzipDecompressStream;
    FUseGzip: Boolean;
    procedure HttpReceived(const Buff; Count : int64);
    procedure HttpStatus(ASender: TObject; const AStatus: TIdStatus; const AStatusText: string);
    procedure SetOnReceived(const Value: TNotifyEvent);
    procedure SetStartRange(const Value: Integer);
    procedure SetLastModified(const Value: string);
    procedure RaiseReceivedEvent;
    procedure RaiseStatusEvent;
    procedure SetOnStatus(const Value: TIdStatusEvent);
    function  GetReplayCode : Integer;
    procedure SetOnComplete(const Value: TNotifyEvent);
    procedure SetContentLength(const Value: Integer);
    procedure SetUserAgent(const Value: string);
    procedure SetProxyHost(const Value: string);
    procedure SetProxyPort(const Value: Integer);
    procedure SetUseGzip(const Value: Boolean);
    procedure SetErrorCode(const Value: THttpErrpr);
  public
    property    ErrorCode  : THttpErrpr read FErrorCode write SetErrorCode;
    property    OnReceived : TNotifyEvent read FOnReceived write SetOnReceived;
    property    OnStatus: TIdStatusEvent read FOnStatus write SetOnStatus;
    property    OnComplete : TNotifyEvent read FOnComplete write SetOnComplete;
    property    ReceivedLines : TStringList read FReceivedLines;
    property    StartRange : Integer read FStartRange write SetStartRange;
    property    LastModified : string read FLastModified write SetLastModified;
    property    UserAgent    : string read FUserAgent write SetUserAgent;
    property    ContentLength : Integer read FContentLength write SetContentLength;
    property    ResponseCode : Integer read GetReplayCode;
    property    ProxyHost : string read FProxyHost write SetProxyHost;
    property    ProxyPort : Integer read FProxyPort write SetProxyPort;
    property    UseGzip : Boolean read FUseGzip write SetUseGzip;
    constructor Create(ACreateSuspended: Boolean = True); override;
    destructor  Destroy; override;
    procedure   BeforeRun; override;
    procedure   Run; override;
    procedure   AfterRun; override;
    procedure   AddHeader(HeaderName, Value : string);
    procedure   Get(URL : string);
  end;

implementation

{ TAsyncHttp }

procedure TAsyncHttp.SetOnComplete(const Value: TNotifyEvent);
begin
  FOnComplete := Value;
end;

procedure TAsyncHttp.SetContentLength(const Value: Integer);
begin
  FContentLength := Value;
end;

procedure TAsyncHttp.AddHeader(HeaderName, Value: string);
begin
  FAddHeaders := FAddHeaders + HeaderName + ': ' + Value + #13#10;
end;

procedure TAsyncHttp.SetUserAgent(const Value: string);
begin
  FUserAgent := Value;
end;

procedure TAsyncHttp.SetProxyHost(const Value: string);
begin
  FProxyHost := Value;
end;

procedure TAsyncHttp.SetProxyPort(const Value: Integer);
begin
  FProxyPort := Value;
end;

procedure TAsyncHttp.SetOnReceived(const Value: TNotifyEvent);
begin
  FOnReceived := Value;
end;

constructor TAsyncHttp.Create(ACreateSuspended: Boolean = True);
begin
  inherited Create(ACreateSuspended);

end;

destructor TAsyncHttp.Destroy;
begin

  inherited;
end;

procedure TAsyncHttp.Get(URL: string);
begin

  FURL := URL;
  self.Start;

end;

procedure TAsyncHttp.BeforeRun;
begin
  inherited;

end;

procedure TAsyncHttp.Run;
var
  headers : TStringList;
  I       : Integer;
begin
  inherited;

  if FStartRange = 0 then
    FUseGzip := true;

  FErrorCode := heNoError;

  FReceivedLines := TStringList.Create;

  FGzipStream := nil;
  FWriteEvent := TMemoryStreamEx.Create;
  FWriteEvent.OnWrite := HttpReceived;
  FBuffer := TMemoryStream.Create;
  FBufferReader       := TStreamReader.Create(FBuffer);
  FReadPosition       := 0;

  {*debug Config.SetReadProxy(FHttp)
  FHttp.Request.UserAgent   := gUserAgent;
  FHttp.Request.RawHeaders.Add('X-2ch-UA: ' + APP_2chUA);
  }

  FHttp := TIdHttp.Create(nil);
  FHttp.Request.ContentRangeStart := FStartRange;
  FHttp.Request.Connection     := 'close';
  FHttp.Request.UserAgent      := FUserAgent;

  FHttp.OnStatus := HttpStatus;

  if FUseGzip then
    FHttp.Request.AcceptEncoding := 'gzip';

  if FLastModified <> '' then
    AddHeader('If-Modified-Since', FLastModified);

  headers := TStringList.Create;
  headers.Text := FAddHeaders;
  for I := 0 to headers.Count - 1 do
    FHttp.Request.CustomHeaders.Add(headers[I]);

  if FProxyHost <> '' then
  begin
    FHttp.ProxyParams.ProxyServer := FProxyHost;
    FHttp.ProxyParams.ProxyPort   := FProxyPort;
    FHttp.Request.Pragma := 'no-cache';
  end;

  try
    FHttp.Get(FURL, FWriteEvent);
  except on Exception do ;
  end;


  FHttp.Disconnect;

  FResponseCode  := FHttp.ResponseCode;
  FLastModified  := FHttp.Response.RawHeaders.Values['Last-Modified'];

  if FHttp.Response.ContentEncoding = 'gzip' then
    FContentLength := FBuffer.Size
  else
    FContentLength := FHttp.Response.ContentLength;

  if Assigned(FOnComplete) then FOnComplete(self);

  FBuffer.Free;
  FWriteEvent.Free;
  FBufferReader.Free;

  FReceivedLines.Free;
  FHttp.Free;

  FURL := '';
  FReadPosition := 0;
  FOnReceived   := nil;
  FStartRange   := 0;
  FLastModified := '';
  FOnStatus     := nil;
//  FStatus       := 0;
  FStatusText   := '';
  FOnComplete   := nil;
  FContentLength := 0;
  FUserAgent     :='';
  FProxyPort     := 0;
  FProxyHost     := '';
  FInited        := false;
  FAddHeaders    := '';
  //FReceivedLines.Clear;
  FGzipStream.Free;
  FUseGzip := false;
  FErrorCode := heNoError;

  Stop;

end;

procedure TAsyncHttp.AfterRun;
begin
  inherited;

end;

procedure TAsyncHttp.HttpReceived(const Buff; Count: int64);
var
  line : string;
begin

  FBuffer.Seek(0, soFromEnd);

  if FHttp.Response.ContentEncoding = 'gzip' then
  begin
    if FGzipStream = nil then
      FGzipStream := TGzipDecompressStream.Create(FBuffer);

    if FErrorCode = heNoError then
    begin
      try
        FGzipStream.Write(Buff, Count);
      except on Exception do
        begin
          FErrorCode := heBrokenGzip;
          exit;
        end;
      end;
    end;

  end else
  begin
    FBuffer.Write(Buff, Count);
  end;

  line := '';

  FBuffer.Seek(FReadPosition, soFromBeginning);
  while FBufferReader.ReadLine(line) do
  begin
    FReadPosition := FBuffer.Position;
    FReceivedLines.Add(line);
  end;

  Synchronize(RaiseReceivedEvent);

end;

procedure TAsyncHttp.SetStartRange(const Value: Integer);
begin
  FStartRange := Value;
end;

procedure TAsyncHttp.SetLastModified(const Value: string);
begin
  FLastModified := Value;
end;

procedure TAsyncHttp.RaiseReceivedEvent;
begin
  if Assigned(FOnReceived) then
    FOnReceived(self)
end;

procedure TAsyncHttp.SetOnStatus(const Value: TIdStatusEvent);
begin
  FOnStatus := Value;
end;

procedure TAsyncHttp.HttpStatus(ASender: TObject; const AStatus: TIdStatus;
  const AStatusText: string);
begin

  FStatus     := AStatus;
  FStatusText := AStatusText;

  Synchronize(RaiseStatusEvent);

end;

procedure TAsyncHttp.RaiseStatusEvent;
begin

  if Assigned(FOnStatus) then
    FOnStatus(self, FStatus, FStatusText)

end;

function TAsyncHttp.GetReplayCode: Integer;
begin

  result := FResponseCode;

end;

procedure TAsyncHttp.SetUseGzip(const Value: Boolean);
begin
  FUseGzip := Value;
end;

procedure TAsyncHttp.SetErrorCode(const Value: THttpErrpr);
begin
  FErrorCode := Value;
end;

end.
