/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Hiroyuki Ikezoe
 *  Copyright (C) 2003 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *  $Id: kz-bookmark.h,v 1.1.1.1 2005/03/27 12:58:27 poincare Exp $
 */

#ifndef __KZ_BOOKMARK_H__
#define __KZ_BOOKMARK_H__

#include <glib-object.h>

G_BEGIN_DECLS

#define KZ_TYPE_BOOKMARK		(kz_bookmark_get_type ())
#define KZ_BOOKMARK(obj)		(G_TYPE_CHECK_INSTANCE_CAST ((obj), KZ_TYPE_BOOKMARK, KzBookmark))
#define KZ_BOOKMARK_CLASS(klass)	(G_TYPE_CHECK_CLASS_CAST ((klass),  KZ_TYPE_BOOKMARK, KzBookmarkClass))
#define KZ_IS_BOOKMARK(obj)		(G_TYPE_CHECK_INSTANCE_TYPE ((obj), KZ_TYPE_BOOKMARK))
#define KZ_IS_BOOKMARK_CLASS(klass)	(G_TYPE_CHECK_CLASS_TYPE ((klass),  KZ_TYPE_BOOKMARK))
#define KZ_BOOKMARK_GET_CLASS(obj)	(G_TYPE_INSTANCE_GET_CLASS ((obj),  KZ_TYPE_BOOKMARK, KzBookmarkClass))

#define KZ_TYPE_BOOKMARK_TYPE		(kz_bookmark_type_get_type())

typedef struct _KzBookmark         KzBookmark;
typedef struct _KzBookmarkClass    KzBookmarkClass;

typedef enum {
	KZ_BOOKMARK_NORMAL,
	KZ_BOOKMARK_FOLDER,
	KZ_BOOKMARK_PURE_FOLDER,
	KZ_BOOKMARK_SEPARATOR,
	KZ_BOOKMARK_N_TYPES
} KzBookmarkType;

typedef enum {
	KZ_BOOKMARK_FOLDED_FLAG      = 1 << 0 
} KzBookmarkFlag;

#define KZ_BOOKMARK_SORT_LAST_MODIFIED "last-modified"

struct _KzBookmark
{
	GObject         parent;
	KzBookmarkType  type;
	KzBookmarkFlag  flags;
};

struct _KzBookmarkClass
{
	GObjectClass parent_class;

	/* -- signals -- */
	void (*insert_child)       (KzBookmark  *bookmark,
				    KzBookmark  *child,
				    KzBookmark  *sibling);
	void (*remove_child)       (KzBookmark  *bookmark,
				    KzBookmark  *child);
	void (*move_child)         (KzBookmark  *bookmark,
				    KzBookmark  *parent,
				    KzBookmark  *sibling);
	void (*children_reordered) (KzBookmark  *boomark);
};


/* GType */
GType		kz_bookmark_type_get_type	(void) G_GNUC_CONST;
GType		kz_bookmark_get_type		(void);

/* boomark item creating functions */
KzBookmark     *kz_bookmark_new			(void);
KzBookmark     *kz_bookmark_new_with_attrs	(const gchar *title,
						 const gchar *uri,
						 const gchar *description);
KzBookmark     *kz_bookmark_pure_folder_new	(void);
KzBookmark     *kz_bookmark_separator_new	(void);
/* set/get properties */
/*
 * "ID" is md5sum of URI of the parent bookmark file + ":" + ID
 * "title" means user defined title.
 * "document_title" is the original document title.
 */
const gchar    *kz_bookmark_get_id		(KzBookmark  *bookmark);
const gchar    *kz_bookmark_get_link		(KzBookmark  *bookmark);
const gchar    *kz_bookmark_get_title		(KzBookmark  *bookmark);
const gchar    *kz_bookmark_get_document_title	(KzBookmark  *bookmark);
const gchar    *kz_bookmark_get_description	(KzBookmark  *bookmark);
guint		kz_bookmark_get_last_modified	(KzBookmark  *bookmark);
guint		kz_bookmark_get_last_visited	(KzBookmark  *bookmark);
guint		kz_bookmark_get_added_time	(KzBookmark  *bookmark);
gboolean        kz_bookmark_is_editable         (KzBookmark  *bookmark);

void		kz_bookmark_set_id		(KzBookmark  *bookmark,
						 const gchar *id);
void		kz_bookmark_set_link		(KzBookmark  *bookmark,
						 const gchar *uri);
void		kz_bookmark_set_title		(KzBookmark  *bookmark,
						 const gchar *title);
void		kz_bookmark_set_document_title	(KzBookmark  *bookmark,
						 const gchar *doc_title);
void		kz_bookmark_set_description	(KzBookmark  *bookmark,
						 const gchar *description);
void		kz_bookmark_set_last_modified	(KzBookmark  *bookmark,
						 guint        time);
void		kz_bookmark_set_last_visited	(KzBookmark  *bookmark,
						 guint        time);
void		kz_bookmark_set_added_time	(KzBookmark  *bookmark,
						 guint        time);
/* separator interface */
gboolean	kz_bookmark_is_separator	(KzBookmark  *bookmark);

/* folder interface */
gboolean	kz_bookmark_is_folder		(KzBookmark  *bookmark);
gboolean	kz_bookmark_is_pure_folder	(KzBookmark  *bookmark);
gboolean	kz_bookmark_get_folded		(KzBookmark  *bookmark);
void		kz_bookmark_set_folded		(KzBookmark  *bookmark,
						 gboolean     folded);
void		kz_bookmark_insert_before	(KzBookmark  *bookmark,
						 KzBookmark  *child,
						 KzBookmark  *sibling);
void		kz_bookmark_append		(KzBookmark  *bookmark,
						 KzBookmark  *child);
void		kz_bookmark_prepend		(KzBookmark  *bookmark,
						 KzBookmark  *child);
void		kz_bookmark_remove		(KzBookmark  *bookmark,
						 KzBookmark  *child);
void		kz_bookmark_remove_all		(KzBookmark  *bookmark);
KzBookmark     *kz_bookmark_get_parent		(KzBookmark  *bookmark);
KzBookmark     *kz_bookmark_get_parent_file	(KzBookmark  *bookmark);
GList	       *kz_bookmark_get_children	(KzBookmark  *bookmark);
gboolean        kz_bookmark_has_children        (KzBookmark  *bookmark);
KzBookmark     *kz_bookmark_next		(KzBookmark  *bookmark);
KzBookmark     *kz_bookmark_prev		(KzBookmark  *bookmark);
void		kz_bookmark_sort		(KzBookmark  *bookmark,
						 const gchar *type);
guint		kz_bookmark_get_current		(KzBookmark  *bookmark);
void		kz_bookmark_set_current		(KzBookmark  *bookmark,
						 guint pos);
gboolean	kz_bookmark_get_lock		(KzBookmark  *bookmark);
void		kz_bookmark_set_lock		(KzBookmark  *bookmark,
						 gboolean lock);
gboolean	kz_bookmark_get_auto_refresh	(KzBookmark  *bookmark);
void		kz_bookmark_set_auto_refresh	(KzBookmark  *bookmark,
						 gboolean auto_refresh);

void		kz_bookmark_regist_sort_func	(const gchar *type,
						 GCompareFunc *func);

KzBookmark *    kz_bookmark_find_bookmark_from_uri (KzBookmark *bookmark,
						    const gchar *key_uri);
G_END_DECLS

#endif /* __KZ_BOOKMARK_H__ */
