/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003-2004 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *  $Id: kz-bookmarks-sidebar.c,v 1.1.1.1 2005/03/27 12:58:43 poincare Exp $
 */

#include "kz-bookmarks-sidebar.h"

#include <glib/gi18n.h>
#include "kazehakase.h"
#include "gobject-utils.h"
#include "kz-actions.h"
#include "kz-bookmarks-view.h"


enum {
	PROP_0,
	PROP_SIDEBAR
};


/* Object class methods */
static void     kz_bookmarks_sidebar_class_init   (KzBookmarksSidebarClass *klass);
static void     kz_bookmarks_sidebar_init         (KzBookmarksSidebar      *sidebar);
static void     kz_bookmarks_sidebar_dispose      (GObject      *obj);
static void     kz_bookmarks_sidebar_set_property (GObject      *object,
						   guint         prop_id,
						   const GValue *value,
						   GParamSpec   *pspec);
static void     kz_bookmarks_sidebar_get_property (GObject      *object,
						   guint         prop_id,
						   GValue       *value,
						   GParamSpec   *pspec);

static gboolean cb_button_press                   (GtkWidget          *widget,
						   GdkEventButton     *event,
						   KzBookmarksSidebar *sidebar);


static KzSidebarEntry kz_sidebar_bookmarks = 
{
	priority_hint: 0,
	label:         N_("Bookmarks"),
	icon:          NULL,
	create:        kz_bookmarks_sidebar_new,
};

static GtkVBoxClass *parent_class = NULL;


KzSidebarEntry *
kz_bookmarks_sidebar_get_entry (gint idx)
{
	if (idx == 0)
		return &kz_sidebar_bookmarks;
	else
		return NULL;
}


KZ_OBJECT_GET_TYPE(kz_bookmarks_sidebar, "KzBookmarksSidebar", KzBookmarksSidebar,
		   kz_bookmarks_sidebar_class_init, kz_bookmarks_sidebar_init,
		   GTK_TYPE_VBOX)


static void
kz_bookmarks_sidebar_class_init (KzBookmarksSidebarClass *klass)
{
	GObjectClass *gobject_class;

	parent_class = g_type_class_peek_parent(klass);
	gobject_class = (GObjectClass *) klass;

	gobject_class->dispose      = kz_bookmarks_sidebar_dispose;
	gobject_class->set_property = kz_bookmarks_sidebar_set_property;
	gobject_class->get_property = kz_bookmarks_sidebar_get_property;

	g_object_class_install_property
		(gobject_class,
		 PROP_SIDEBAR,
		 g_param_spec_object ("sidebar",
				      _("Sidebar"),
				      _("The parent sidebar"),
				      KZ_TYPE_SIDEBAR,
				      G_PARAM_READWRITE |
				      G_PARAM_CONSTRUCT_ONLY));
}


static void
kz_bookmarks_sidebar_init (KzBookmarksSidebar *sidebar)
{
	GtkWidget *scrwin, *view;

	/* create scrolled window and tree */
	scrwin = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrwin),
				       GTK_POLICY_AUTOMATIC,
				       GTK_POLICY_AUTOMATIC);
        gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrwin),
					    GTK_SHADOW_IN);
	gtk_box_pack_start(GTK_BOX(sidebar), scrwin, TRUE, TRUE, 0);
	gtk_widget_show(scrwin);

	view = kz_bookmarks_view_new();
	kz_bookmarks_view_set_root_folder(KZ_BOOKMARKS_VIEW(view),
					  KZ_BOOKMARK(kz_bookmarks),
					  TRUE, FALSE, FALSE, FALSE);
	gtk_container_add(GTK_CONTAINER(scrwin), view);
	g_signal_connect(view, "button-press-event",
			 G_CALLBACK(cb_button_press),
			 sidebar);
	gtk_widget_show(view);
}


static void
kz_bookmarks_sidebar_dispose (GObject *obj)
{
	KzBookmarksSidebar *sidebar = KZ_BOOKMARKS_SIDEBAR(obj);

	if (sidebar->sidebar)
		g_object_unref(sidebar->sidebar);
	sidebar->sidebar = NULL;

	if (G_OBJECT_CLASS (parent_class)->dispose)
		G_OBJECT_CLASS (parent_class)->dispose(obj);
}


static void
kz_bookmarks_sidebar_set_property (GObject         *object,
				   guint            prop_id,
				   const GValue    *value,
				   GParamSpec      *pspec)
{
	KzBookmarksSidebar *sidebar = KZ_BOOKMARKS_SIDEBAR(object);
  
	switch (prop_id)
	{
	case PROP_SIDEBAR:
		sidebar->sidebar = g_object_ref(g_value_get_object(value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}


static void
kz_bookmarks_sidebar_get_property (GObject         *object,
				   guint            prop_id,
				   GValue          *value,
				   GParamSpec      *pspec)
{
	KzBookmarksSidebar *sidebar = KZ_BOOKMARKS_SIDEBAR(object);

	switch (prop_id)
	{
	case PROP_SIDEBAR:
		g_value_set_object(value, sidebar->sidebar);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}


GtkWidget *
kz_bookmarks_sidebar_new (KzSidebar *sidebar)
{
	KzBookmarksSidebar *bsidebar;

	bsidebar = g_object_new(KZ_TYPE_BOOKMARKS_SIDEBAR,
			        "sidebar", sidebar,
			        NULL);

	return GTK_WIDGET(bsidebar);
}


static gboolean
cb_button_press (GtkWidget *widget, GdkEventButton *event,
		 KzBookmarksSidebar *sidebar)
{
	GtkWindow *kz;
	GtkTreeView *treeview = GTK_TREE_VIEW(widget);
	GtkTreeModel *model = gtk_tree_view_get_model(treeview);
	GtkTreeIter iter;
	GtkTreePath *treepath = NULL;
	gboolean success;
	KzBookmark *bookmark = NULL;
	gboolean retval = FALSE;

	g_return_val_if_fail(KZ_IS_BOOKMARKS_SIDEBAR(sidebar), FALSE);
	g_return_val_if_fail(KZ_IS_SIDEBAR(sidebar->sidebar), FALSE);

	kz = GTK_WINDOW(sidebar->sidebar->kz);

	success = gtk_tree_view_get_path_at_pos(GTK_TREE_VIEW(widget),
						event->x, event->y,
						&treepath, NULL, NULL, NULL);
	if (success)
	{
		const gchar *uri;

		gtk_tree_model_get_iter(model, &iter, treepath);
		bookmark = kz_bookmarks_view_get_bookmark(model, &iter);

		uri = kz_bookmark_get_link(bookmark);

		if (event->type == GDK_2BUTTON_PRESS)
		{
			if (kz_bookmark_is_folder(bookmark))
			{
				/* expand, collapse */
			}
			else if (kz_bookmark_is_separator(bookmark))
			{
				/* skip */
			}
			else if (uri)
			{
				kz_window_load_url(KZ_WINDOW(kz), uri);
			}
		}
		else if (event->button == 2)
		{
			if (kz_bookmark_is_folder(bookmark))
			{
				/* expand, collapse */
			}
			else if (kz_bookmark_is_separator(bookmark))
			{
				/* skip */
			}
			else if (uri)
			{
				kz_window_open_new_tab(KZ_WINDOW(kz), uri);
			}
		}
		else if (event->button == 3)
		{
			gtk_tree_view_set_cursor(GTK_TREE_VIEW(widget),
						 treepath, NULL, FALSE);
			kz_actions_popup_bookmark_menu_modal(KZ_WINDOW(kz),
							     bookmark,
							     event->button,
							     event->time);
			retval = TRUE;
 		}
	}

	if (treepath)
		gtk_tree_path_free(treepath);

	return retval;
}
