/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 *  Copyright (C) 2007 Hiroyuki Ikezoe  <poincare@ikezoe.net>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 *  Boston, MA  02111-1307  USA
 *
 */

#include "ie-bridge.h"

#include <gdk/gdkwin32.h>
#include <windows.h>
#include <exdisp.h>
#include <unknwn.h>
#include <mshtml.h>

#include "gtk-ie-embed.h"
#include "gtk-ie-embed-private.h"
#include "ie-browser-event-dispatcher.h"
#include "ie-document-event-dispatcher.h"

enum {
    PROP_0,
    PROP_GTK_WIDGET
};

typedef struct _IEBridgePriv IEBridgePriv;
struct _IEBridgePriv
{
    GtkWidget *widget;
    gboolean can_go_forward;
    gboolean can_go_back;

    IWebBrowser2 *web_browser;
    IEBrowserEventDispatcher *browser_event_dispatcher;
    IEDocumentEventDispatcher *document_event_dispatcher;
    IHTMLDocument2 *current_document;
    DWORD browser_event_cookie;
    DWORD document_event_cookie;
};

#define IE_BRIDGE_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), TYPE_IE_BRIDGE, IEBridgePriv))

G_DEFINE_TYPE (IEBridge, _ie_bridge, G_TYPE_OBJECT)

typedef HRESULT (*ATLAXGETCONTROL) (HWND, IUnknown **);
static HMODULE hModule = NULL;
static ATLAXGETCONTROL AtlAxGetControl = NULL;

/* virtual functions for GtkObject class */
static GObject *constructor        (GType                  type,
                                    guint                  n_props,
                                    GObjectConstructParam *props);
static void   dispose              (GObject               *object);
static void   set_property         (GObject               *object,
                                    guint                  prop_id,
                                    const GValue          *value,
                                    GParamSpec            *pspec);
static void   get_property         (GObject               *object,
                                    guint                  prop_id,
                                    GValue                *value,
                                    GParamSpec            *pspec);

static void
_ie_bridge_class_init (IEBridgeClass *klass)
{
    GObjectClass   *gobject_class = G_OBJECT_CLASS (klass);

    gobject_class->dispose      = dispose;
    gobject_class->constructor  = constructor;
    gobject_class->set_property = set_property;
    gobject_class->get_property = get_property;

    g_object_class_install_property (
        gobject_class,
        PROP_GTK_WIDGET,
        g_param_spec_object (
            "widget",
            "Parent Widget",
            "Parent GtkWidget",
            GTK_TYPE_WIDGET,
            (GParamFlags) (G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY)));

    g_type_class_add_private (gobject_class, sizeof (IEBridgePriv));
}

typedef BOOL (*ATLAXWININIT) (void);

static gboolean
atl_initialize (void)
{
    static gboolean atl_initialized = FALSE;

    if (!atl_initialized) {
        ATLAXWININIT AtlAxWinInit = NULL;
        hModule = LoadLibrary ("atl.dll");
        if (!hModule)
            return FALSE;
        CoInitialize (NULL);
        AtlAxWinInit = (ATLAXWININIT) GetProcAddress (hModule, "AtlAxWinInit");
        if (!AtlAxWinInit)
            return FALSE;
        AtlAxWinInit ();

        AtlAxGetControl = (ATLAXGETCONTROL) GetProcAddress (hModule, "AtlAxGetControl");

        atl_initialized = TRUE;
    }
    return TRUE;
}

static void
connect_browser_event_dispatcher (IEBridge *bridge)
{
    IConnectionPointContainer *iCPC;
    IConnectionPoint *iCP;
    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (bridge);

    priv->web_browser->QueryInterface (IID_IConnectionPointContainer, (void **) &iCPC);
    iCPC->FindConnectionPoint (DIID_DWebBrowserEvents2, &iCP);
    priv->browser_event_dispatcher = new IEBrowserEventDispatcher (bridge);
    iCP->Advise (priv->browser_event_dispatcher, &priv->browser_event_cookie);
    iCPC->Release ();
    iCP->Release ();
}

static void
disconnect_browser_event_dispatcher (IEBridge *bridge)
{
    IConnectionPointContainer *iCPC;
    IConnectionPoint *iCP;
    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (bridge);

    priv->web_browser->QueryInterface (IID_IConnectionPointContainer, (void **) &iCPC);
    iCPC->FindConnectionPoint (DIID_DWebBrowserEvents2, &iCP);
    iCP->Unadvise (priv->browser_event_cookie);
    iCPC->Release ();
    iCP->Release ();
}

static GObject *
constructor (GType type, guint n_props, GObjectConstructParam *props)
{
    GObject *object;
    GObjectClass *klass = G_OBJECT_CLASS (_ie_bridge_parent_class);
    IEBridgePriv *priv;
    HWND hwnd, hparent;
    HINSTANCE hInstance;

    object = klass->constructor (type, n_props, props);

    atl_initialize ();

    priv = IE_BRIDGE_GET_PRIVATE (object);
    hInstance = GetModuleHandle (NULL);
    hparent = (HWND) GDK_WINDOW_HWND (gtk_widget_get_parent_window (priv->widget));
    hwnd = CreateWindow ("AtlAxWin", "Shell.Explorer.2", WS_CHILD | WS_VISIBLE,
                         priv->widget->allocation.x, priv->widget->allocation.y,
                         priv->widget->allocation.width, priv->widget->allocation.height,
                         hparent, (HMENU)0, hInstance, NULL);
    if (hwnd) {
        IUnknown *iUnknown;

        priv->widget->window = gdk_window_foreign_new ((GdkNativeWindow) hwnd);
        ShowWindow (hwnd, SW_HIDE);

        if (AtlAxGetControl (hwnd, &iUnknown) == S_OK) {
            iUnknown->QueryInterface (IID_IWebBrowser2, (void **) &priv->web_browser);
            priv->web_browser->put_RegisterAsBrowser (VARIANT_TRUE);
            priv->web_browser->put_Silent (VARIANT_TRUE);
            priv->web_browser->put_AddressBar (VARIANT_FALSE);
            priv->web_browser->put_StatusBar (VARIANT_FALSE);
            priv->web_browser->put_ToolBar (VARIANT_FALSE);
            priv->web_browser->put_FullScreen (VARIANT_FALSE);
            priv->web_browser->put_TheaterMode (VARIANT_FALSE);
            iUnknown->Release ();
        }
        connect_browser_event_dispatcher (IE_BRIDGE (object));
    }

    return object;
}

static void
_ie_bridge_init (IEBridge *ie)
{
    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (ie);

    priv->widget = NULL;
    priv->web_browser = NULL;
    priv->current_document = NULL;
    priv->browser_event_dispatcher = NULL;
    priv->browser_event_cookie = 0;
    priv->document_event_cookie = 0;

    priv->document_event_dispatcher = new IEDocumentEventDispatcher (ie);
}

static void
dispose (GObject *object)
{
    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (object);

    if (priv->browser_event_dispatcher) {
        disconnect_browser_event_dispatcher (IE_BRIDGE (object));
        delete priv->browser_event_dispatcher;
        priv->browser_event_dispatcher = NULL;
    }

    if (priv->document_event_cookie) {
        _ie_bridge_disconnect_document_event_dispatcher (IE_BRIDGE (object));
    }

    if (priv->document_event_dispatcher) {
        delete priv->document_event_dispatcher;
        priv->document_event_dispatcher = NULL;
    }

    if (priv->web_browser) {
        priv->web_browser->Release ();
        priv->web_browser = NULL;
    }

    if (priv->widget) {
        g_object_unref (priv->widget);
        priv->widget = NULL;
    }

    if (G_OBJECT_CLASS (_ie_bridge_parent_class)->dispose)
        G_OBJECT_CLASS (_ie_bridge_parent_class)->dispose (object);
}

static void
set_property (GObject *object,
              guint prop_id,
              const GValue *value,
              GParamSpec *pspec)
{
    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (object);

    switch (prop_id) {
    case PROP_GTK_WIDGET:
    {
        GObject *obj = G_OBJECT (g_value_get_object (value));
        priv->widget = GTK_WIDGET (g_object_ref (obj));
        break;
    }
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}

static void
get_property (GObject *object,
              guint prop_id,
              GValue *value,
              GParamSpec *pspec)
{
    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (object);

    switch (prop_id) {
    case PROP_GTK_WIDGET:
        g_value_set_object (value, G_OBJECT (priv->widget));
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}

IEBridge *
_ie_bridge_new (GtkWidget *widget)
{
    return IE_BRIDGE (g_object_new (TYPE_IE_BRIDGE,
                                    "widget", widget,
                                    NULL));
}

void
_ie_bridge_set_visible (IEBridge *ie, gboolean visible)
{
    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (ie);

    if (priv->web_browser) {
        VARIANT_BOOL B = visible ? VARIANT_TRUE : VARIANT_FALSE;
        priv->web_browser->put_Visible (B);
    }
}

void
_ie_bridge_load_url (IEBridge *ie, const gchar *url)
{
    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (ie);
    gunichar2 *utf16_url;

    if (!priv->web_browser)
        return;

    utf16_url = g_utf8_to_utf16 (url, -1, NULL, NULL, NULL);
    if (utf16_url) {
        priv->web_browser->Navigate ((OLECHAR *) utf16_url, NULL, NULL, NULL, NULL);
        g_free (utf16_url);
    }
}


void
_ie_bridge_reload (IEBridge *ie, RefreshConstants flag)
{
    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (ie);

    if (priv->web_browser) {
        VARIANT level;
        level.vt = VT_I4;
        level.intVal = flag;
        priv->web_browser->Refresh2 (&level);
    }
}

void
_ie_bridge_stop (IEBridge *ie)
{
    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (ie);

    if (priv->web_browser) {
        priv->web_browser->Stop ();
    }
}

gboolean
_ie_bridge_is_loading (IEBridge *ie)
{
    VARIANT_BOOL vB;
    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (ie);

    if (!priv->web_browser)
        return FALSE;

    priv->web_browser->get_Busy (&vB);
    return vB == VARIANT_TRUE ? TRUE : FALSE;
}

void
_ie_bridge_go_back (IEBridge *ie)
{
    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (ie);

    if (priv->web_browser) {
        priv->web_browser->GoBack ();
    }
}

void
_ie_bridge_go_forward (IEBridge *ie)
{
    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (ie);

    if (priv->web_browser) {
        priv->web_browser->GoForward ();
    }
}

gchar *
_ie_bridge_get_location (IEBridge *ie)
{
    gchar *location = NULL;

    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (ie);

    if (priv->web_browser) {
        BSTR URL;
        priv->web_browser->get_LocationURL (&URL);
        if (URL) {
            location = g_utf16_to_utf8 ((gunichar2 *) URL, -1, NULL, NULL, NULL);
            SysFreeString (URL);
        }
    }
    return location;
}

gchar *
_ie_bridge_get_title (IEBridge *ie)
{
    gchar *title = NULL;

    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (ie);

    if (priv->web_browser) {
        BSTR Name;
        priv->web_browser->get_LocationName (&Name);
        if (Name) {
            title = g_utf16_to_utf8 ((gunichar2 *) Name, -1, NULL, NULL, NULL);
            SysFreeString (Name);
        }
    }
    return title;
}

gchar *
_ie_bridge_get_charset (IEBridge *ie)
{
    gchar *charset = NULL;

    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (ie);

    if (priv->web_browser) {
        BSTR Charset = NULL;
        priv->current_document->get_charset (&Charset);
        if (Charset) {
            charset = g_utf16_to_utf8 ((gunichar2 *) Charset, -1, NULL, NULL, NULL);
            SysFreeString (Charset);
        }
    }
    return charset;
}

void
_ie_bridge_set_charset (IEBridge *ie, const gchar *charset)
{
    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (ie);

    if (priv->web_browser) {
        BSTR Charset;
        Charset = (BSTR) g_utf8_to_utf16 (charset, -1, NULL, NULL, NULL);
        if (Charset) {
            priv->current_document->put_charset (Charset);
            SysFreeString (Charset);
        }
    }
}

gboolean
_ie_bridge_can_go_forward (IEBridge *ie)
{
    return IE_BRIDGE_GET_PRIVATE (ie)->can_go_forward;
}

gboolean
_ie_bridge_can_go_back (IEBridge *ie)
{
    return IE_BRIDGE_GET_PRIVATE (ie)->can_go_back;
}

static void
_ie_bridge_exec_default_command (IEBridge *ie, OLECMDID cmd_id)
{
    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (ie);

    if (priv->web_browser) {
        priv->web_browser->ExecWB (cmd_id, OLECMDEXECOPT_DODEFAULT,
                                   NULL, NULL);
    }
}

void
_ie_bridge_cut_clipboard (IEBridge *ie)
{
    _ie_bridge_exec_default_command (ie, OLECMDID_CUT);
}

void
_ie_bridge_copy_clipboard (IEBridge *ie)
{
    _ie_bridge_exec_default_command (ie, OLECMDID_COPY);
}

void
_ie_bridge_paste_clipboard (IEBridge *ie)
{
    _ie_bridge_exec_default_command (ie, OLECMDID_PASTE);
}

void
_ie_bridge_select_all (IEBridge *ie)
{
    _ie_bridge_exec_default_command (ie, OLECMDID_SELECTALL);
}

static gboolean
_ie_bridge_is_enable_command (IEBridge *ie, OLECMDID cmd_id)
{
    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (ie);
    OLECMDF cmdF;

    if (!priv->web_browser)
        return FALSE;

    priv->web_browser->QueryStatusWB (cmd_id, &cmdF);
    return (cmdF & OLECMDF_ENABLED);
}

gboolean
_ie_bridge_can_cut_clipboard (IEBridge *ie)
{
    return _ie_bridge_is_enable_command (ie, OLECMDID_CUT);
}

gboolean
_ie_bridge_can_copy_clipboard (IEBridge *ie)
{
    return _ie_bridge_is_enable_command (ie, OLECMDID_COPY);
}

gboolean
_ie_bridge_can_paste_clipboard (IEBridge *ie)
{
    return _ie_bridge_is_enable_command (ie, OLECMDID_PASTE);
}

void
_ie_bridge_set_font_size (IEBridge *ie, guint size)
{
    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (ie);

    if (priv->web_browser) {
        VARIANT zoom;
        zoom.vt = VT_I4;
        zoom.lVal = size;
        priv->web_browser->ExecWB (OLECMDID_ZOOM,
                                   OLECMDEXECOPT_DODEFAULT,
                                   &zoom, NULL);
    }
}

guint
_ie_bridge_get_font_size (IEBridge *ie)
{
    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (ie);

    if (priv->web_browser) {
        VARIANT zoom;
        zoom.vt = VT_I4;
        priv->web_browser->ExecWB (OLECMDID_ZOOM,
                                   OLECMDEXECOPT_DODEFAULT,
                                   NULL, &zoom);
        return zoom.lVal;
    }
}

void
_ie_bridge_page_setup (IEBridge *ie)
{
    _ie_bridge_exec_default_command (ie, OLECMDID_PAGESETUP);
}

void
_ie_bridge_print (IEBridge *ie)
{
    _ie_bridge_exec_default_command (ie, OLECMDID_PRINT);
}

void
_ie_bridge_print_preview (IEBridge *ie)
{
    _ie_bridge_exec_default_command (ie, OLECMDID_PRINTPREVIEW);
}

void
_ie_bridge_save (IEBridge *ie, const gchar *filename)
{
    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (ie);
    IDispatch *doc = NULL;
    IPersistStreamInit *stream = NULL;
    IStream *file_stream = NULL;

    if (!priv->web_browser)
        return;

    priv->web_browser->get_Document (&doc);
    if (!doc)
        return;
    doc->QueryInterface (IID_IPersistStreamInit, (void **) &stream);
    if (!stream)
        return;
    stream->InitNew ();
    stream->Save (file_stream, VARIANT_TRUE);
    stream->Release ();
}

gchar *
_ie_bridge_get_last_modified (IEBridge *ie)
{
    BSTR mTime = NULL;
    gchar *last_modified = NULL;
    IHTMLDocument2 *doc = NULL;

    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (ie);

    if (!priv->web_browser)
        return NULL;

    priv->web_browser->get_Document ((IDispatch **) &doc);
    if (!doc)
        return NULL;
    doc->get_lastModified (&mTime);
    if (mTime) {
        last_modified = g_utf16_to_utf8 ((gunichar2 *) mTime, -1, NULL, NULL, NULL);
        SysFreeString (mTime);
    }

    return last_modified;
}

static gboolean
get_selected_text_range (IHTMLDocument2 *document, IHTMLTxtRange **text_range)
{
    IHTMLSelectionObject *selection;
    gboolean ret = FALSE;

    if (!document)
        return FALSE;

    document->get_selection (&selection);
    if (selection) {
        IDispatch *range_dispatch;
        selection->createRange (&range_dispatch);

        if (range_dispatch) {
            range_dispatch->QueryInterface (IID_IHTMLTxtRange, (void **) text_range);
            if (text_range)
                ret = TRUE;
            range_dispatch->Release ();
        }
        selection->Release ();
    }

    return ret;
}

gchar *
_ie_bridge_get_selected_text (IEBridge *ie)
{
    gchar *text = NULL;
    IHTMLSelectionObject *selection;
    IHTMLTxtRange *text_range = NULL;
    gboolean ret;
    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (ie);

    if (!priv->web_browser)
        return NULL;
    if (!priv->current_document)
        return NULL;
    if (get_selected_text_range (priv->current_document, &text_range) &&
        text_range) {
        BSTR selected_text;
        text_range->get_text (&selected_text);
        if (selected_text) {
            text = g_utf16_to_utf8 ((gunichar2 *) selected_text, -1, NULL, NULL, NULL);
            SysFreeString (selected_text);
        }
        text_range->Release();
    }

    return text;
}

gboolean
_ie_bridge_find_string (IEBridge *ie,
                        const gchar *string,
                        gboolean forward,
                        gboolean wrap)
{
    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (ie);
    IHTMLDocument2 *doc = NULL;
    IHTMLTxtRange *text_range = NULL, *selected_range = NULL;
    IHTMLElement *element = NULL;
    IHTMLBodyElement *body = NULL;
    gboolean ret = FALSE;

    if (!priv->web_browser)
        return FALSE;

    priv->web_browser->get_Document ((IDispatch **) &doc);
    if (!doc)
        return FALSE;

    doc->get_body (&element);
    if (!element) {
        doc->Release ();
        return FALSE;
    }
    element->QueryInterface (IID_IHTMLBodyElement, (void **) &body);
    if (body) {
        body->createTextRange (&text_range);
    }

    if (get_selected_text_range (doc, &selected_range) && selected_range) {
        long ActualCount;
        gunichar2 *unit, *how;
        unit = g_utf8_to_utf16 ("character", -1, NULL, NULL, NULL);
        if (forward) {
            how = g_utf8_to_utf16 ("EndToEnd", -1, NULL, NULL, NULL);
            selected_range->moveStart ((OLECHAR *) unit, 1, &ActualCount);
        } else {
            how = g_utf8_to_utf16 ("StartToStart", -1, NULL, NULL, NULL);
            selected_range->moveEnd ((OLECHAR *) unit, -1, &ActualCount);
        }
        selected_range->setEndPoint ((OLECHAR *) how, text_range);
        text_range->Release ();
        text_range = selected_range;
        g_free (unit);
        g_free (how);
    }

    if (text_range) {
        gunichar2 *utf16_string;
        utf16_string = g_utf8_to_utf16 (string, -1, NULL, NULL, NULL);
        if (utf16_string) {
            VARIANT_BOOL vBool;
            gint direction = forward ? 1 : -1;
            text_range->findText ((OLECHAR *) utf16_string, direction, 0, &vBool);
            if (vBool == VARIANT_TRUE) {
                text_range->select ();
                ret = TRUE;
            } else if (selected_range && wrap) {
                text_range->Release ();
                body->createTextRange (&text_range);    
                text_range->findText ((OLECHAR *) utf16_string, direction, 0, &vBool);
                if (vBool == VARIANT_TRUE) {
                    text_range->select ();
                    ret = TRUE;
                }
            }
            g_free (utf16_string);
        }
        text_range->Release ();
    }

    if (body)
        body->Release ();
    element->Release ();

    return ret;
}

void
_ie_bridge_load_html_from_string (IEBridge *ie, const gchar *string)
{
    IStream *stream;
    IPersistStreamInit *persist_stream = NULL;
    IHTMLDocument2 *doc = NULL;
    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (ie);

    if (!priv->web_browser)
        return;

    priv->web_browser->get_Document ((IDispatch **) &doc);
    if (!doc)
        return;

    CreateStreamOnHGlobal ((HGLOBAL) string, TRUE, &stream);
    if (stream) {
        doc->QueryInterface (IID_IPersistStreamInit, (void **) persist_stream);
        if (persist_stream) {
            persist_stream->InitNew ();
            persist_stream->Load (stream);
            persist_stream->Release ();
        }
        stream->Release ();
    }

    doc->Release ();
}

void
_ie_bridge_title_changed (IEBridge *ie, gunichar2 *title)
{
    gchar *utf8_title;
    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (ie);
    utf8_title = g_utf16_to_utf8 ((gunichar2 *) title, -1, NULL, NULL, NULL);
    if (utf8_title) {
        g_signal_emit_by_name (G_OBJECT (priv->widget), "title", utf8_title, 0);
        g_free (utf8_title);
    }
}

void
_ie_bridge_location_changed (IEBridge *ie, gunichar2 *location)
{
    gchar *utf8_location;
    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (ie);
    utf8_location = g_utf16_to_utf8 ((gunichar2 *) location, -1, NULL, NULL, NULL);
    if (utf8_location) {
        g_signal_emit_by_name (G_OBJECT (priv->widget), "location", utf8_location, 0);
        g_free (utf8_location);
    }
}

void
_ie_bridge_status_text_change (IEBridge *ie, gunichar2 *text)
{
    gchar *utf8_text;
    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (ie);
    utf8_text = g_utf16_to_utf8 ((gunichar2 *) text, -1, NULL, NULL, NULL);
    if (utf8_text) {
        g_signal_emit_by_name (G_OBJECT (priv->widget), "text", utf8_text, 0);
        g_free (utf8_text);
    }
}

void
_ie_bridge_connect_document_event_dispatcher (IEBridge *ie)
{
    IConnectionPointContainer *iCPC;
    IConnectionPoint *iCP;
    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (ie);

    if (priv->current_document)
        priv->current_document->Release ();
    priv->web_browser->get_Document ((IDispatch **) &priv->current_document);
    priv->current_document->QueryInterface (IID_IConnectionPointContainer, (void **) &iCPC);
    iCPC->FindConnectionPoint (DIID_HTMLDocumentEvents, &iCP);
    iCP->Advise (priv->document_event_dispatcher, &priv->document_event_cookie);
    iCPC->Release ();
    iCP->Release ();
}

void
_ie_bridge_disconnect_document_event_dispatcher (IEBridge *ie)
{
    IConnectionPointContainer *iCPC;
    IConnectionPoint *iCP;
    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (ie);

    if (!priv->current_document)
        return;
    priv->current_document->QueryInterface (IID_IConnectionPointContainer, (void **) &iCPC);
    iCPC->FindConnectionPoint (DIID_HTMLDocumentEvents, &iCP);
    iCP->Unadvise (priv->document_event_cookie);
    iCPC->Release ();
    iCP->Release ();
    priv->current_document->Release ();
    priv->current_document = NULL;
    priv->document_event_dispatcher->Release ();
    priv->document_event_cookie = 0;
}

void
_ie_bridge_set_forward_state (IEBridge *ie, gboolean enable)
{
    IE_BRIDGE_GET_PRIVATE (ie)->can_go_forward = enable;
}

void
_ie_bridge_set_backward_state (IEBridge *ie, gboolean enable)
{
    IE_BRIDGE_GET_PRIVATE (ie)->can_go_back = enable;
}

gboolean
_ie_bridge_new_window (IEBridge *ie, gpointer *newDispatch)
{
    gboolean cancel = FALSE;
    IEBridge *newbridge = NULL;
    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (ie);

    cancel = _gtk_ie_embed_new_window (GTK_IE_EMBED (priv->widget), &newbridge);

    if (newbridge) {
        IEBridgePriv *newpriv = IE_BRIDGE_GET_PRIVATE (newbridge);
        IDispatch *dispatch = NULL;
        if (newpriv->web_browser->get_Application (&dispatch) == S_OK) {
            *newDispatch = dispatch;
        }
    }
    return cancel;
}

gboolean
_ie_bridge_close_window (IEBridge *ie, gboolean is_child)
{
    gboolean cancel = FALSE;
    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (ie);

    g_signal_emit_by_name (G_OBJECT (priv->widget), "close-window",
                           &cancel);
    return cancel;
}

void
_ie_bridge_net_start (IEBridge *ie)
{
    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (ie);

    g_signal_emit_by_name (G_OBJECT (priv->widget), "net-start", 0);
}

void
_ie_bridge_net_stop (IEBridge *ie)
{
    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (ie);

    g_signal_emit_by_name (G_OBJECT (priv->widget), "net-stop", 0);
}

void
_ie_bridge_progress_change (IEBridge *ie, glong current_progress, glong max_progress)
{
    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (ie);

    g_signal_emit_by_name (G_OBJECT (priv->widget), "progress",
                           current_progress, max_progress);

}

gpointer
_ie_bridge_get_browser_object (IEBridge *ie)
{
    return IE_BRIDGE_GET_PRIVATE (ie)->web_browser;
}

static void
_set_dom_event_target_attributes (GtkIEEmbedDOMEventTarget *target, IHTMLElement *target_element)
{
    GtkIEEmbedDOMEventTargetAttribute *attr = NULL;
#if defined(HAVE_IHTMLNODE) && defined (HAVE_IHTMLAttributeCollection)
    IHTMLDOMNode *node = NULL;
    IDispatch *attributes = NULL;
    IHTMLAttributeCollection *collection = NULL;
    target_element->QueryInterface (IID_IHTMLDOMNode, (IDispatch **) &node);
    node->get_attributes (&attributes);
    if (attributes) {
        long i, length;
        attributes->QueryInterface (IID_IHTMLAttributeCollection, (IHTMLAttributeCollection **) &collection);
        collection->get_length (&length);
        for (i = 0; i < length; i++) {
            IDispatch *disp = NULL;
            IHTMLDOMAttribute *dom_attr = NULL;
            BSTR nodeName
            VARIANT nodeValue;
            gchar *utf8_name, *utf8_value;
            VARIANT v;
            v.vt = VT_I4;
            v.intVal = i;
            collection->get_item (&v, (IDispatch **) disp);
            disp->QueryInterface (IID_IHTMLDOMAttribute, (IHTMLDOMAttribute **) &dom_attr);
            dom_attr->get_nodeName (&nodeName);
            dom_attr->get_nodeValue (&nodeValue);
            utf8_name = g_utf16_to_utf8 ((gunichar2 *) nodeName, -1, NULL, NULL, NULL);
            if (nodeValue.vt == VT_BSTR && nodeValue.bstrVal) {
                utf8_value = g_utf16_to_utf8 ((gunichar2 *) nodeValue.bstrVal, -1, NULL, NULL, NULL);
            }
            attr = g_new0 (GtkIEEmbedDOMEventTargetAttribute, 1);
            attr->name = utf8_name;
            attr->value = utf8_value;
            target_attribute_list = g_list_prepend (target_attribute_list, attr);
            disp->Release ();
            SysFreeString (nodeName);
        }
    }
#else
    /* workaround */
    VARIANT attr_value;
    gunichar2 *attr_name;

    attr_name = g_utf8_to_utf16 ("href", -1, NULL, NULL, NULL);
    target_element->getAttribute ((OLECHAR *) attr_name, 0, &attr_value);
    if (attr_value.vt == VT_BSTR && attr_value.bstrVal) {
        gchar *value = NULL;
        attr = g_new0 (GtkIEEmbedDOMEventTargetAttribute, 1);
        attr->name = g_strdup ("href");
        value = g_utf16_to_utf8 ((gunichar2 *) attr_value.bstrVal, -1, NULL, NULL, NULL);
        if (value)
            attr->value = value;
        target->attribute_list = g_list_prepend (target->attribute_list, attr);
    }
    g_free (attr_name);
#endif /* HAVE_IHTMLNODE */
}

static GtkIEEmbedDOMEventTarget *
_ie_bridge_create_dom_event_target (IHTMLElement *target_element)
{
    GtkIEEmbedDOMEventTarget *target;
    BSTR tag_name = NULL;

    target = g_new0 (GtkIEEmbedDOMEventTarget, 1);

    target_element->get_tagName (&tag_name);
    if (tag_name) {
        gchar *utf8_tag_name;
        utf8_tag_name = g_utf16_to_utf8 ((gunichar2 *) tag_name, -1, NULL, NULL, NULL);
        target->name = utf8_tag_name;
        SysFreeString (tag_name);
    }

    _set_dom_event_target_attributes (target, target_element);

    return target;
}

static GtkIEEmbedDOMMouseEvent *
_ie_bridge_create_dom_mouse_event (IEBridge *ie)
{
    gint x, y;
    GdkModifierType mask;
    GtkIEEmbedDOMMouseEvent *event;
    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (ie);

    gdk_window_get_pointer (priv->widget->window, &x, &y, &mask);

    event = g_new0 (GtkIEEmbedDOMMouseEvent, 1);

    event->x = x;
    event->y = y;
    event->shift_key = mask & GDK_SHIFT_MASK;
    event->control_key = mask & GDK_CONTROL_MASK;
    event->alt_key = mask & GDK_MOD1_MASK;
    event->meta_key = mask & GDK_META_MASK;
    if (mask & GDK_BUTTON1_MASK)
        event->button = 0;
    else if (mask & GDK_BUTTON2_MASK)
        event->button = 1;
    else if (mask & GDK_BUTTON3_MASK)
        event->button = 2;

    if (priv->current_document) {
        IHTMLElement *element = NULL;

        priv->current_document->elementFromPoint (x, y, &element);
        if (element) {
            event->target = _ie_bridge_create_dom_event_target (element);
            element->Release ();
        }
    }

    return event;
}

static void
_ie_bridge_dom_mouse_event_free (GtkIEEmbedDOMMouseEvent *event)
{
    if (!event)
        return;

    if (event->target)
        gtk_ie_embed_dom_event_target_free (event->target);

    g_free (event);
}

static gboolean
_ie_bridge_emit_mouse_event (IEBridge *ie, gchar *signal_name)
{
    GtkIEEmbedDOMMouseEvent *event;
    gboolean ret;
    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (ie);

    event = _ie_bridge_create_dom_mouse_event (ie);
    g_signal_emit_by_name (G_OBJECT (priv->widget), signal_name,
                           event, &ret);
    _ie_bridge_dom_mouse_event_free (event);

    return ret;
}

gboolean
_ie_bridge_mouse_down (IEBridge *ie)
{
    return _ie_bridge_emit_mouse_event (ie, "dom-mouse-down");
}

gboolean
_ie_bridge_mouse_move (IEBridge *ie)
{
    return _ie_bridge_emit_mouse_event (ie, "dom-mouse-move");
}

gboolean
_ie_bridge_mouse_up (IEBridge *ie)
{
    return _ie_bridge_emit_mouse_event (ie, "dom-mouse-up");
}

gboolean
_ie_bridge_mouse_click (IEBridge *ie)
{
    return _ie_bridge_emit_mouse_event (ie, "dom-mouse-click");
}

gboolean
_ie_bridge_is_mapped (IEBridge *ie)
{
    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (ie);
    return GTK_WIDGET_MAPPED (priv->widget);
}

gboolean
_ie_bridge_get_use_context_menu (IEBridge *ie)
{
    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (ie);
    return gtk_ie_embed_get_use_context_menu (GTK_IE_EMBED (priv->widget));
}

void
_ie_bridge_selection_changed (IEBridge *ie)
{
    IEBridgePriv *priv = IE_BRIDGE_GET_PRIVATE (ie);
    g_signal_emit_by_name (G_OBJECT (priv->widget), "selection-changed");
}

/*
vi:ts=4:nowrap:ai:expandtab:sw=4
*/
