/**************************************************************************
*   Copyright (C) 2003,2004 by Hideki Ikemoto , (c) 2004 by 421           *
*   ikemo@wakaba.jp                                                       *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
**************************************************************************/

#ifndef KITAIMGMANAGER_H
#define KITAIMGMANAGER_H

#include <kio/job.h>

#include <qobject.h>
#include <qdict.h>
#include <qmutex.h>
#include <qevent.h>

class QFile;
class QDataStream;
class QStringList;

struct IMGDAT
{
    int code;
    bool mosaic;
    int width;
    int height;
    unsigned int size;
    KURL datURL;
};


/*----------------------------------------------------------*/

namespace Kita
{
    class FileLoader;

    class ImgManager : public QObject
    {
        Q_OBJECT

        static ImgManager* instance;
        static QMutex m_mutex;
        QPtrList< FileLoader > m_loaderList;
        QDict< IMGDAT > m_imgDatDict;
        QWidget* m_mainwidget;

    public:
        ImgManager( QWidget* mainwidget );
        ~ImgManager();

        static void setup( QWidget* mainwidget );
        static void deleteInstance();
        static ImgManager* getInstance();
        static bool isImgFile( const KURL& url, bool use_mimetype );
        static bool isAnimationGIF( const KURL& url );
        static bool isBMP( const KURL& url );

        static bool load( const KURL& url, const KURL& datURL );
        static void stop( const KURL& url );
        static QString getPath( const KURL& url );
        static bool isLoadingNow( const KURL& url );
        static bool deleteCache( const KURL& url, QWidget* parent );
        static bool copyCache( const KURL& url, QWidget* parent );

        static bool mosaic( const KURL& url );
        static void setMosaic( const KURL& url, bool status );
        static int code( const KURL& url );
        static int width( const KURL& url );
        static int height( const KURL& url );
        static unsigned int size( const KURL& url );
        static QPixmap icon( const KURL& url );
        static KURL datURL( const KURL& url );

    private:
        bool isImgFilePrivate( const KURL& url, bool use_mimetype );
        bool loadPrivate( const KURL& url, const KURL& datURL );
        void stopPrivate( const KURL& url );
        bool isLoadingNowPrivate( const KURL& url );
        bool deleteCachePrivate( const KURL& url, QWidget* parent );
        bool copyCachePrivate( const KURL& url, QWidget* parent );
        void setMosaicPrivate( const KURL& url, bool status );

        FileLoader* getLoader( const KURL& url );
        void deleteLoader( FileLoader* loader );
        void createImgDat( const KURL& url, int code );
        IMGDAT* getImgDat( const KURL& url );
        void deleteImgDat( const KURL& url );
        void getSize( const KURL& url );
        bool cacheExists( const KURL& url );

    private slots:
        void slotData( Kita::FileLoader*, const QByteArray& );
        void slotResult( Kita::FileLoader* );

    protected:
        virtual void customEvent( QCustomEvent * e );

    signals:
        void receiveImgData( const KURL&, unsigned int, unsigned int );
        void finishImgLoad( const KURL& );
        void cacheDeleted( const KURL& );
    };

    /*---------------------------------------------------------*/

    class FileLoader : public QObject
    {
        Q_OBJECT

        /* target */
        KURL m_url;
        /* The URL of thread that refers url. */
        KURL m_datURL;
        /* save path */
        QString m_path;

        /* job */
        KIO::Job* m_currentJob;
        int m_code;
        QString m_header;

        /* file */
        QFile *m_file;
        QDataStream *m_ds;
        KIO::filesize_t m_size;
        KIO::filesize_t m_totalsize;

    public:
        FileLoader( const KURL& url, const KURL& datURL, const QString& path );
        ~FileLoader();

        void killJob();
        void stopJob();
        bool get();

        const bool isRunning() const;
        const int code() const;
        const KURL url() const;
        const QString path() const;
        const KIO::filesize_t size() const;
        const KIO::filesize_t totalsize() const;
        const KURL datURL() const;

    private:
        void closeFile();
        int responseCode();

    private slots:
        void slotData( KIO::Job* , const QByteArray& );
        void slotTotalSize( KIO::Job * , KIO::filesize_t );
        void slotRedirection( KIO::Job* , const KURL& );
        void slotResult( KIO::Job* );

    signals:
        void data( Kita::FileLoader*, const QByteArray& );
        void result( Kita::FileLoader* );
    };

}


/*---------------------------------------------------------*/

#define EVENT_EmitFinigh   ( QEvent::User + 200 )
#define EVENT_DeleteLoader ( QEvent::User + 201 )


class EmitFinishEvent : public QCustomEvent
{
    KURL m_url;

public:
    EmitFinishEvent( KURL url ) : QCustomEvent( EVENT_EmitFinigh ), m_url( url ) {}

    KURL url() const { return m_url; }
};


class DeleteLoaderEvent : public QCustomEvent
{
    Kita::FileLoader* m_loader;

public:
    DeleteLoaderEvent( Kita::FileLoader* loader ) : QCustomEvent( EVENT_DeleteLoader )
    , m_loader( loader ) {}

    Kita::FileLoader* getLoader() const { return m_loader; }
};



#endif
