/***************************************************************************
 *   Copyright (C) 2003 by Hideki Ikemoto                                  *
 *   ikemo@wakaba.jp                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/
#include "k2ch_articlefile.h"

#include <qfile.h>
#include <qregexp.h>

#include <kdebug.h>
#include <time.h>

K2chArticleFileItem::K2chArticleFileItem( QString& name, int seen, time_t accessTime )
  : m_name( name ),
  m_seen( seen ),
  m_accessTime( accessTime )
{
}

K2chArticleFileItem::~K2chArticleFileItem()
{
}

QString K2chArticleFileItem::toString()
{
  return QString("(\"%1\" :seen %2 :access-time (%3 %4 0))")
    .arg( m_name )
    .arg( m_seen )
    .arg( ( m_accessTime >> 16 ) & 0xffff )
    .arg( ( m_accessTime       ) & 0xffff );
}

K2chArticleFileItem& K2chArticleFile::item( int i )
{
  return m_list[ i ];
}

K2chArticleFile::K2chArticleFile( QString& filePath )
  : m_filePath( filePath )
{
  QFile file( m_filePath );

  if( ! file.open( IO_ReadOnly ) ) {
    kdError() << __func__ << ": can't open " << m_filePath << endl;
    return;
  }

  // article-summaryեȤASCIIʳʤϤʤΤcodecϻȤʤ
  parse( QString( file.readAll() ) );
}

K2chArticleFile::~K2chArticleFile()
{
}

void K2chArticleFile::parse( QString str )
{
  QRegExp regexp( "^\\s*\\((.*)\\)\\s*$" );

  if( regexp.search( str ) == -1 ) {
    kdWarning() << "K2chArticleFile::parse, no match!" << endl;
    return;
  }

  QString body = regexp.cap( 1 );
  kdDebug() << __func__ << "matched str:" << body << endl;

  QRegExp article_regexp( "\\("                  // left bracket
                          "\"([0-9]*)\""         // m_name
                          " :seen ([-0-9]+)"    // m_seen
                          " :access-time \\(([0-9]+) ([0-9]+) ([0-9]+)\\)" // m_accessTime
                          "\\)" );               // right bracket

  int pos = 0;
  while( ( pos = article_regexp.search( body, pos ) ) != -1 ) {
    QString name = article_regexp.cap( 1 );
    int seen = article_regexp.cap( 2 ).toInt();
    int time1 = article_regexp.cap( 3 ).toInt();
    int time2 = article_regexp.cap( 4 ).toInt();
    int time3 = article_regexp.cap( 5 ).toInt();
    kdDebug() << "matched str = '" << article_regexp.cap( 0 ) << "'" << endl;
    kdDebug() << "name = '" << name << "'" << endl;
    kdDebug() << "seen = " << seen << endl;
    kdDebug() << "accessTime = " << time1 << "," << time2 << "," << time3 << endl;
    time_t accessTime = ( time1 << 16 ) + time2;

    K2chArticleFileItem item( name, seen, accessTime );

    m_list.append( item );
    pos += article_regexp.matchedLength();
  }
}

int K2chArticleFile::count()
{
  return m_list.size();
}

QString K2chArticleFile::toString()
{
  QString ret = "(";

  QValueList<K2chArticleFileItem>::iterator it;
  for( it = m_list.begin(); it != m_list.end(); ++it )
  {
    ret += (*it).toString();
  }
  ret += ")";

  return ret;
}
