"""Graphical user interface"""

import Tkinter as Tk
import Pmw
import re
import sys
from config import Config
from util import *
from bookmark import Bookmark
import renderer
import bbs
import kittytk as ktk
import time
import Queue
from thread import start_new_thread
import observer
import threadpane
import guiecho
import guicommand
import font

CONFIG = Config()

class GUI(Tk.Tk):
    """Primary window

    This class also acts as the mediator class for all widgets.
    """
    def __init__(self, bbs):
        self.bbs = bbs
        Tk.Tk.__init__(self)
        self.title("KittyWalk")
        self.iconname("KittyWalk")
        self.geometry(CONFIG.get("Window","Geometry"))
        self.scroll_unit = int(CONFIG.get("Window","ScrollUnit"))
        self.make_widgets()
        self.bind_events()
        self.tk.call("tk","useinputmethods","1")

    def make_widgets(self):
        # Make containers.
        tri_pane_frame = ktk.TriPaneFrame(self)
        notebook = Pmw.NoteBook(tri_pane_frame.left, borderwidth=1, pagemargin=0)
        boardmenu = notebook.add(unicode("ꗗ","sjis"))
        bookmark = notebook.add(unicode("Cɓ","sjis"))
        bookmark_tab = notebook.tab(1)
        explorer = tri_pane_frame.left
        threads = tri_pane_frame.right_top
        messages = tri_pane_frame.right_bottom

        # Create widgets.
        self.menubar = MenuBar(self)
        self.statusbar = StatusBar(self)
        self.board_menu = BoardMenu(boardmenu, self.bbs)
        self.thread_menu = ThreadMenu(threads)
        self.thread_pane = threadpane.ThreadPane(messages)
        self.bookmark = BookmarkList(bookmark)

        self.thread_search_dialog = None

        # Pack.
        self.statusbar.pack(side=Tk.BOTTOM, fill=Tk.BOTH)
        tri_pane_frame.pack(side=Tk.TOP, expand=Tk.YES, fill=Tk.BOTH)
        notebook.pack(expand=Tk.YES, fill=Tk.BOTH)
        self.thread_pane.pack(expand=Tk.YES, fill=Tk.BOTH)

        # Configuration.
        self.config(menu=self.menubar)
        guiecho.set(self.statusbar)
        notebook.setnaturalsize()
        notebook.tab(0).config(font=CONFIG.get("Font","UI"))
        notebook.tab(1).config(font=CONFIG.get("Font","UI"))

        # GUI command setting.
        guicommand.store_widgets(window = self,
                                 threadmenu = self.thread_menu,
                                 bookmarkview = self.bookmark,
                                 messageview = self.thread_pane,
                                 boardmenu = self.board_menu)

        pass

    def bind_events(self):
        self.bind("<Configure>", self.on_resize)
        pass

    def on_resize(self, event):
        if event.widget == self:
            CONFIG.set("Window","Geometry",self.geometry())

    def popup_searchdialog(self):
        try:
            self.thread_search_dialog.focus_force()
            pass
        except:
            self.thread_search_dialog = ThreadSearchDialog(self,
                                                           self.thread_menu)
            pass
        # _CAOACEBhE̍ɔzuB
        size, x, y = self.geometry().split("+")
        self.thread_search_dialog.geometry("+%s+%s" % (x,y))
        pass

    def popup_config_dialog(self):
        ConfigDialog(self)
        pass

    pass


class ConfigDialog(Pmw.Dialog):
    _menu_font = CONFIG.get("Font","UI")
    _entry_font = CONFIG.get("Font","Entry")
    def __init__(self, parent):
        Pmw.Dialog.__init__(self, parent, buttons=("OK", "Cancel"),
                            command=self._on_pressed)
        self.title(unicode("ݒ","sjis"))
        self._make_widgets()
        self.transient(parent)

    def _make_widgets(self):
        interior = self.interior()
        # ǂݍݗpvNVB
        self._get_proxy = ktk.EntryField(
          interior,
          label=unicode("ǂݍݗpvNV","sjis"),
          entrywidth=32,
          labelfont=self._menu_font,
          entryfont=self._entry_font)
        self._get_proxy.set(CONFIG.get("Proxy","Get"))
        self._get_proxy.pack(expand=Tk.YES, fill=Tk.X)
        # ݗpvNVB
        self._post_proxy = ktk.EntryField(
          interior,
          label=unicode("ݗpvNV","sjis"),
          entrywidth=32,
          labelfont=self._menu_font,
          entryfont=self._entry_font)
        self._post_proxy.set(CONFIG.get("Proxy","Post"))
        self._post_proxy.pack(expand=Tk.YES, fill=Tk.X)
        # Behavior on bookmark list
        self._bookmarklist_command = Tk.BooleanVar()
        self._bookmarklist_command.set(CONFIG.get("Bookmark", "OpenNew"))
        Tk.Checkbutton(interior,
                       text=j("Cɓ_uNbNŐV^uJ"),
                       variable = self._bookmarklist_command).pack(anchor=Tk.W)
        # Behavior on thread menu
        self._threadmenu_command = Tk.BooleanVar()
        self._threadmenu_command.set(CONFIG.get("ThreadMenu", "OpenNew"))
        Tk.Checkbutton(interior,
                       text=j("Xbhꗗ_uNbNŐV^uJ"),
                       variable = self._threadmenu_command).pack(anchor=Tk.W)
        pass

    def _on_pressed(self, name):
        if name=="OK":
            CONFIG.set("Proxy","Get", self._get_proxy.get())
            CONFIG.set("Proxy","Post", self._post_proxy.get())
            opens = self._bookmarklist_command.get()
            CONFIG.set("Bookmark","OpenNew",opens)
            guicommand.set_if_bookmarklist_opens_new_page(opens)
            opens = self._threadmenu_command.get()
            CONFIG.set("ThreadMenu","OpenNew",opens)
            guicommand.set_if_threadmenu_opens_new_page(opens)
            pass
        self.destroy()
        pass

class ThreadSearchDialog(Tk.Toplevel):
    font = CONFIG.get("Font","UI")
    _direction_down = 1
    _direction_up = 2

    def __init__(self, parent, threadmenu):
        self.threadmenu = threadmenu
        
        Tk.Toplevel.__init__(self, parent)
        self.title(unicode("","sjis"))
        self.transient(parent)
        self.bind("<Escape>", lambda e:self.destroy())
        # Entry.
        self._entry = Tk.Entry(self, width=32, font=CONFIG.get("Font","Entry"))
        self._entry.bind("<Return>", self.search)
        self._entry.bind("<Escape>", lambda e: self.destroy())
        self._entry.pack(side=Tk.TOP)
        # B
        f = Tk.Frame(self)
        f.pack(side=Tk.TOP, expand=Tk.YES, fill=Tk.X)
        # WI{^ijB
        self._direction = Tk.IntVar()
        Tk.Radiobutton(f, text=unicode("","sjis"),
                       variable=self._direction,
                       value=self._direction_down,
                       font=CONFIG.get("Font","UI")).pack(side=Tk.LEFT)
        Tk.Radiobutton(f, text=unicode("","sjis"),
                       variable=self._direction,
                       value=self._direction_up,
                       font=CONFIG.get("Font","UI")).pack(side=Tk.LEFT)
        self._direction.set(self._direction_down)
        # {^B
        b = Tk.Button(f, text=unicode("","sjis"),
                      command=self.destroy,
                      font=CONFIG.get("Font","UI"))
        b.pack(side=Tk.RIGHT)
        self._button = Tk.Button(f, text=unicode("","sjis"),
                                 command = self.search,
                                 font=CONFIG.get("Font","UI"))
        self._button.pack(side=Tk.RIGHT)
        # Focus.
        self.focus_force()

    def search(self, event=None):
        """Search thread title"""
        if not self.threadmenu.board:
            return
        text = self.get().lower()
        listbox = self.threadmenu.component("listbox")
        # Determine currently selected index.
        selections = listbox.curselection()
        if len(selections)>0:
            selected_index = int(selections[0])
            current = selected_index
            pass
        else:
            current = 0
            pass
        # Determine index list to search through.
        threads = self.threadmenu.board.get_threads()
        max = len(threads)
        direction_is_down = self.direction_is_down()
        if direction_is_down:
            indices = range(current+1, max) + range(0, current)
            pass
        else:
            indices = range(current-1,0,-1) + range(max-1,current,-1)
            pass
        # Search query string from all subjects, and select first 
        # matched subject.
        for index in indices:
            title = threads[index].title.lower()
            if title.find(text) != -1:
                listbox.selection_clear(0, Tk.END)
                listbox.selection_set(index)
                listbox.see(index)
                break

    def focus_force(self):
        self._entry.focus_force()

    def get(self):
        return self._entry.get()

    def direction_is_down(self):
        if self._direction.get() == self._direction_down:
            return True
        else:
            return False

class BoardMenu(ktk.ScrolledListBox):
    """Board menu -- list of board names on the left pane"""

    _rmenu_font = font.ui

    def __init__(self, parent, bbs):
        self.bbs = bbs
        ktk.ScrolledListBox.__init__(self,
                                     parent,
                                     [(j("J"),self.open),
                                      (j("Cɓɒǉ"), self.bookmark)],
                                     dblclickcommand=self.open)
        listbox = self.component("listbox")
        listbox.config(font=CONFIG.get("Font","UI"),
                       bg="white",
                       relief=Tk.SUNKEN,
                       borderwidth=1)
        self.build()
        self.pack(expand=Tk.YES, fill=Tk.BOTH)
        # If BBS is empty, update.
        if len(self.bbs)==0:
            self.update()
        self.tk_focusFollowsMouse()

    def open(self):
        if self.selected_board():
            guicommand.open(self.selected_board())
            pass
        pass

    def build(self):
        """Display board names."""
        self.boards = []
        self.clear()
        for category in self.bbs:
            self.insert(Tk.END, category.name.encode("utf-8"))
            self.boards.append(None)
            for board in category:
                text = u"  " + board.name
                self.insert(Tk.END, text.encode("utf-8"))
                self.boards.append(board)

    def update(self):
        """Update BBS menu"""
        try:
            guiecho.write(unicode("ꗗ擾...","sjis"))
            self.bbs.update()
            self.bbs.build()
        except:
            guiecho.write(unicode("ꗗ擾ł܂ł","sjis"))
        else:
            guiecho.write(unicode("ꗗ쐬...","sjis"))
            self.build()
            guiecho.write(unicode("ꗗ̍XVI܂","sjis"))

    def selected_board(self):
        """ݑIĂ  (Board IuWFNg) ԂBI
        Ȃ None ԂB
        """
        selections = self.curselection()
        if len(selections)>0:
            index = int(selections[0])
            board = self.boards[index]
        else:
            board = None
        return board

    def bookmark(self):
        """Bookmark selected board"""
        if self.selected_board():
            guicommand.bookmark(self.selected_board())
            pass
        pass

    pass

class BookmarkList(ktk.ScrolledListBox):
    """Bookmark List - list of threads"""

    _rmenu_font = font.ui

    def __init__(self, parent):
        ktk.ScrolledListBox.__init__(self,
                                     parent,
                                     [(j("V^uŊJ"),
                                       self.open_new_page),
                                      (j("̃^uŊJ"),
                                       self.open_in_current_page),
                                      (j("폜"), self.remove_selection)],
                                      dblclickcommand=self.open_selected_item)
        # Initialize.
        self.parent = parent
        # GUI configurations.
        self._listbox.config(font=CONFIG.get("Font","UI"),
                             bg="white",
                             borderwidth=1,
                             selectmode=Tk.SINGLE)
        self.bookmark = Bookmark(CONFIG.get("Bookmark","File"))
        self.build()
        self.component("vertscrollbar").config(relief=Tk.SUNKEN, borderwidth=10)
        self.pack(expand=Tk.YES, fill=Tk.BOTH)
        self.set_if_opens_new(CONFIG.get("Bookmark", "OpenNew"))
        pass

    def open_new_page(self):
        if self.selected_item():
            guicommand.open_new_page(self.selected_item())
            pass
        pass

    def open_in_current_page(self):
        if self.selected_item():
            guicommand.open_in_current_page(self.selected_item())
            pass
        pass

    _open_command = open_in_current_page

    def open_selected_item(self):
        """Open selected board/thread"""
        self._open_command()
        pass

    def set_if_opens_new(self, yes):
        if yes:
            self._open_command = self.open_new_page
            pass
        else:
            self._open_command = self.open_in_current_page
            pass
        pass

    def build(self):
        """Display thread titles."""
        self.items = []
        self.clear()
        for item in self.bookmark:
            if isinstance(item, bbs.Thread):
                self.insert(Tk.END, item.title)
            elif isinstance(item, bbs.Board):
                self.insert(Tk.END, item.name)
            self.items.append(item)

    def selected_item(self):
        """Return the currently selected thread."""
        sel = self.curselection()
        if len(sel) > 0:
            index = int(sel[0])
            thread = self.items[index]
        else:
            thread = None
        return thread

    def selected_index(self):
        sel = self.curselection()
        if len(sel) > 0:
            index = int(sel[0])
        else:
            index = None
        return index

    def append(self, item):
        self.bookmark.append(item)
        self.update()
        pass

    def remove_selection(self):
        """Remove selected bookmark item"""
        if self.selected_index() != None:
            del self.bookmark[self.selected_index()]
            self.update()
            pass
        pass

    def update(self):
        self.bookmark.save()
        self.build()
        pass

    pass


class ThreadMenu(ktk.ScrolledListBox):
    _old_mark = unicode(" ","sjis")
    _new_mark = unicode(" ","sjis")
    _unread_mark = unicode("@ ","sjis")
    _rmenu_font = font.ui

    def __init__(self, parent):
        ktk.ScrolledListBox.__init__(self, parent,
                                     [(j("V^uŊJ"),
                                       self.open_new_page),
                                      (j("̃^uŊJ"),
                                       self.open_in_current_page),
                                      (j("XCɓɒǉ"),
                                       self.bookmark_selected_thread)],
                                     dblclickcommand=self._open_selection,
                                     labelpos = Tk.NW)
        self.config(relief=Tk.GROOVE, border=2)
        self._listbox.config(font = CONFIG.get("Font","UI"),
                             bg = "white",
                             borderwidth = 1,
                             selectmode=Tk.SINGLE)
        self.component("label").bind("<Double-1>", self.update)
        self.component("label").config(font=CONFIG.get("Font","UI"))
        self.pack(expand=Tk.YES, fill=Tk.BOTH)
        self.board = None
        self._make_labelmenu()
        self.set_if_opens_new(CONFIG.get("ThreadMenu", "OpenNew"))
        self.tk_focusFollowsMouse()
        pass

    def set_if_opens_new(self, yes):
        if yes:
            self._open_command = self.open_new_page
            pass
        else:
            self._open_command = self.open_in_current_page
            pass
        pass

    def _make_labelmenu(self):
        self._labelmenu = ktk.PopupMenu(self.component("label"),
                                        [(j("ēǂݍ"),
                                          self.update),
                                         (j("Cɓɒǉ"),
                                          self.bookmark_active_board)],
                                        font=font.ui)
        pass

    def bookmark_active_board(self):
        """Bookmark active board"""
        if self.board:
            guicommand.bookmark(self.board)
        pass

    def update(self, event=None):
        """Update current board"""
        self.build(self.board)
        pass

    def open_new_page(self):
        thread = self.selected_thread()
        if thread:
            guicommand.open_new_page(thread)
            self.restore(thread)
            pass
        pass
    
    def open_in_current_page(self):
        thread = self.selected_thread()
        if thread:
            guicommand.open_in_current_page(thread)
            self.restore(thread)
            pass
        pass
    
    _open_command = open_in_current_page

    def bookmark_selected_thread(self):
        """Bookmark selected thread."""
        thread = self.selected_thread()
        if thread != None:
            guicommand.bookmark(thread)
        pass

    def build(self, board=None):
        """Display thread names of the given board."""
        if not board:
            return False
        self.board = board
        self.label(board.name)
        # XVB
        try:
            guiecho.write(unicode("XꗗǍ ...","sjis"))
            board.update()
            guiecho.write(unicode("Xꗗ쐬 ...","sjis"))
            pass
        except:
            guiecho.write(unicode("XꗗXVł܂ł","sjis"))
            pass
        # For each thread, insert name into list box.
        self.clear()
        threads = board.get_threads()
        if len(threads) > 0:
            guiecho.write(unicode("Xꗗ쐬 ...","sjis"))
            self.threads = []
            for thread in threads:
                text = self._make_item(thread)
                self.insert(Tk.END, text.encode("utf-8"))
                self.threads.append(thread)
            guiecho.write(unicode("Xꗗ̍XV ...","sjis"))

    def _open_selection(self):
        self._open_command()
        pass

    def selected_thread(self):
        """Return the currently selected thread."""
        sel = self.curselection()
        if len(sel) > 0:
            index = int(sel[0])
            thread = self.threads[index]
        else:
            thread = None
        return thread

    def label(self, val=None):
        if val:
            self["label_text"] = val
        return self["label_text"]

    def scroll(self, delta):
        self.component("listbox").yview_scroll(delta, Tk.UNITS)

    def restore(self, thread):
        """Restore given thread slot"""
        if ((self.board != None) and
            (thread != None) and
            (thread in self.board.get_threads())):
            # Save currently selected items.
            old_selections = self.curselection()
            index = int(self.board.get_threads().index(thread))
            listbox = self.component("listbox")
            listbox.delete(index)
            listbox.insert(index, self._make_item(thread))
            # Re-select items.
            for i in old_selections:
                listbox.select_set(i)
                pass
            pass
        pass

    def _make_item(self, thread):
        # Determine mark.
        total = thread.remote_size
        read = thread.local_size
        if read == 0:
            mark = self._unread_mark
        elif read < total:
            mark = self._new_mark
        else:
            mark = self._old_mark

        text = u"%s %s    (%s/%s)" % (mark, thread.title, thread.local_size,
                                      thread.remote_size)
        text = renderer.extract_html_entities(text)
        return text


class StatusBar(Tk.Label):
    def __init__(self, parent=None, **kw):
        Tk.Label.__init__(self, parent)
        self.config(relief=Tk.SUNKEN, height=1, anchor=Tk.W,
                    font=CONFIG.get("Font","UI"),
                    borderwidth=1)
        pass

    def write(self, text):
        self.config(text=text.rstrip())
        self.update()
        pass

    pass


class MenuBar(Tk.Menu):
    def __init__(self, parent):
        Tk.Menu.__init__(self, parent)
        self.config(font=CONFIG.get("Font","UI"))
        self.build()

    def build(self):
        font = CONFIG.get("Font","UI")
        # t@CB
        filemenu = Tk.Menu(self)
        filemenu.add_command(label=unicode("I(X)","sjis"),
                             command=self.quit,
                             underline=3,
                             font=font)
        self.add_cascade(label=unicode("t@C(F)","sjis"),
                         menu=filemenu,
                         underline=5,
                         font=font)
        # ҏWB
        editmenu = Tk.Menu(self)
        editmenu.add_command(label=unicode("Rs[(C)","sjis"),
                             state=Tk.DISABLED,
                             underline=4,
                             font=font)
        editmenu.add_command(label=unicode("(F)","sjis"),
                             underline=3,
                             command= guicommand.search,
                             font=font)
        self.add_cascade(label=unicode("ҏW(E)","sjis"),
                         menu=editmenu,
                         underline=3,
                         font=font)
        # View
        viewmenu = Tk.Menu(self)
        viewmenu.add_command(label=j("ܕԂ̐ؑ(W)"),
                             command=guicommand.toggle_wordwrap,
                             underline=7,
                             font=font)
        self.add_cascade(label=unicode("\(V)","sjis"),
                         menu=viewmenu,
                         underline=3,
                         font=font)
        # Tools
        toolmenu = Tk.Menu(self)
        toolmenu.add_command(label=unicode("XCɓɒǉ(A)","sjis"),
                             command=guicommand.bookmark_active_thread,
                             underline=12,
                             font=font)
        toolmenu.add_command(label=unicode("Cɓ肩폜(R)","sjis"),
                             command=self.remove_bookmark,
                             underline=10,
                             font=font)
        toolmenu.add_separator()
        toolmenu.add_command(label=unicode("ݒ(C)","sjis"),
                             command=guicommand.configure,
                             underline=3,
                             font=font)
        toolmenu.add_command(label=unicode("ꗗXV(U)","sjis"),
                             command=guicommand.update_boardmenu,
                             underline=7,
                             font=font)
        self.add_cascade(label=unicode("c[(T)","sjis"),
                         menu=toolmenu,
                         underline=4,
                         font=font)

    def remove_bookmark(self): guicommand.remove_bookmark()

    pass
# EOF
