"""Inter-widget command.

This module provides the global access point to communicate GUI widget 
components.  Note that this module highly coupled to widgets in nature,  and therefore requires the appropriate initialization.
"""

import bbs
import Tkinter

WINDOW = "window"
THREADMENU = "threadmenu"
BOOKMARKVIEW = "bookmarkview"
MESSAGEVIEW = "messageview"
BOARDMENU = "boardmenu"

class Widget(dict):
    def is_ready(self):
        """Return whether each values is not None."""
        return not None in self.values()

    
WIDGET = Widget({WINDOW: None,
               THREADMENU: None,
               BOOKMARKVIEW: None,
               MESSAGEVIEW: None,
               BOARDMENU: None })

def store_widgets(**kw):
    global WIDGET
    for key in WIDGET.keys():
        key = key.lower()
        WIDGET[key] = kw[key]

def open(item):
    """Open item in appropriate view"""
    if isinstance(item, bbs.Thread):
        open_thread(item)
    elif isinstance(item, bbs.Board):
        open_board(item)

def open_board(board):
    """Open board in thread menu"""
    assert WIDGET.is_ready()
    if board:
        WIDGET[THREADMENU].build(board)

def open_thread(thread):
    """Open thread in message view"""
    assert WIDGET.is_ready()
    if thread:
        WIDGET[MESSAGEVIEW].open(thread)

def bookmark(item):
    assert WIDGET.is_ready()
    WIDGET[BOOKMARKVIEW].append(item)

def remove_bookmark():
    assert WIDGET.is_ready()
    WIDGET[BOOKMARKVIEW].remove_selection()

def search():
    assert WIDGET.is_ready()
    WIDGET[WINDOW].popup_searchdialog()

def configure():
    assert WIDGET.is_ready()
    WIDGET[WINDOW].popup_config_dialog()

def update_boardmenu():
    assert WIDGET.is_ready()
    WIDGET[BOARDMENU].update()

def bookmark_active_thread():
    assert WIDGET.is_ready()
    thread = WIDGET[MESSAGEVIEW].get_active_thread()
    if thread:
        WIDGET[BOOKMARKVIEW].append(thread)

def search_subject():
    assert WIDGET.is_ready()
    WIDGET[WINDOW].search_in_threadmenu()

def close_active_thread():
    assert WIDGET.is_ready()
    WIDGET[MESSAGEVIEW].close_active_thread()

def toggle_wordwrap():
    assert WIDGET.is_ready()
    WIDGET[MESSAGEVIEW].toggle_wordwrap()

def open_new_page(item):
    assert WIDGET.is_ready()
    if isinstance(item, bbs.Thread):
        WIDGET[MESSAGEVIEW].open_new_page(item)
    elif isinstance(item, bbs.Board):
        open_board(item)

def open_in_current_page(item):
    assert WIDGET.is_ready()
    if isinstance(item, bbs.Thread):
        WIDGET[MESSAGEVIEW].open_in_current_page(item)
    elif isinstance(item, bbs.Board):
        open_board(item)

def set_if_bookmarklist_opens_new_page(yes):
    assert WIDGET.is_ready()
    WIDGET[BOOKMARKVIEW].set_if_opens_new(yes)

def set_if_threadmenu_opens_new_page(yes):
    assert WIDGET.is_ready()
    WIDGET[THREADMENU].set_if_opens_new(yes)

def set_cursor(val):
    win = WIDGET[WINDOW]
    if isinstance(win, Tkinter.Tk):
        win.config(cursor=val)
        win.update()
