"""Thread frame widget, providing thread MVC classes and their container."""

import observer
import bbs
import Tkinter as Tk
import Pmw
import font
import renderer
import webbrowser
import re
import config
import postdialog
import sys
import codecs
import guiecho
import icon
from util import *
import kittytk as ktk
import guicommand
from thread import start_new_thread

webbrowser._tryorder.extend(["lynx %s", "w3m %s"])

class ThreadPane(Tk.Frame):
    def __init__(self, parent=None):
        Tk.Frame.__init__(self, parent)
        # Toolbar.
        f = Tk.Frame(self)
        f.pack(anchor=Tk.E)
        self.icon_images = []
        for name,command in [("cross",self.close_active_thread),
                             ("edit", self.post),
                             ("bookmark", guicommand.bookmark_active_thread),
                             ("reload",self.reload_active_thread)]:
            self.icon_images.append(icon.create(name))
            Tk.Button(f,
                      image=self.icon_images[-1],
                      command=command).pack(side=Tk.RIGHT)
            pass
        self.threadview_pages = ThreadViewPages(self)
        self.threadview_pages.pack(expand=Tk.YES, fill=Tk.BOTH)
        self.open = self.open_new_page
        # Place holder for dialog to post a message.
        self.postdialog = None
        pass

    def reload_active_thread(self):
        """Reload active thread"""
        self.threadview_pages.reload_active_thread()
        pass

    def close_active_thread(self):
        """Close active page."""
        self.threadview_pages.close_active_thread()
        pass

    def get_active_thread(self):
        return self.threadview_pages.get_active_thread()

    def post(self):
        thread = self.get_active_thread()
        if thread:
            self.postdialog = postdialog.PostDialog(self, thread)
            self.postdialog.open()
            pass
        pass

    def open_new_page(self, thread):
        self.threadview_pages.open_new_page(thread)
        pass
    
    def open_in_current_page(self, thread):
        self.threadview_pages.open_in_current_page(thread)
        pass

    def toggle_wordwrap(self):
        view = self.threadview_pages.get_active_view()
        if view:
            view.toggle_wordwrap()

    pass


class ThreadView(Tk.Frame):
    def __init__(self, thread, parent=None):
        Tk.Frame.__init__(self, parent)
        self.thread = thread
        self._requires_update = False
        # Make widgets.
        scrolled_text = Pmw.ScrolledText(self)
        scrolled_text.component("text").config(wrap=Tk.NONE, font=font.content)
        scrolled_text.pack(expand=Tk.YES, fill=Tk.BOTH)
        self.textbox = scrolled_text.component("text")
        self.make_balloon()
        self.make_rmenu()
        # Bind events.
        textbox = self.textbox
        textbox.bind("<Enter>", self.focus)
        textbox.bind("<Button-1>", self.start_selection)
        textbox.bind("<ButtonRelease-1>", self.stop_selection)
        textbox.bind("<Control-c>", self.copy_selection)
        textbox.bind("<Down>", self.scroll_line_down)
        textbox.bind("<Up>", self.scroll_line_up)
        textbox.bind("<space>", self.scroll_page_down)
        textbox.bind("<BackSpace>", self.scroll_page_up)
        textbox.tag_bind("cite", "<Enter>", self.on_cite)
        textbox.tag_bind("cite", "<Leave>", self.off_cite)
        textbox.tag_bind("link", "<Enter>", self.on_link)
        textbox.tag_bind("link", "<Leave>", self.off_link)
        textbox.tag_bind("link", "<Button-1>", self.open_link)
        # Configure tags.
        textbox.tag_config("cite", foreground="#0000cc")
        textbox.tag_config("body", lmargin1=32)
        textbox.tag_config("body", lmargin2=32)
        textbox.tag_config("link", foreground="#0000cc")
        # Initial actions.
        self._first_render()
        self._auto_update()
        pass

    def focus(self, event):
        self.textbox.config(state=Tk.NORMAL)
        self.textbox.focus()
        self.textbox.config(state=Tk.DISABLED)
        pass

    def render(self):
        """Render the thread content.

        The displaied position is maintained.
        """
        guicommand.set_cursor("watch")
        textbox = self.textbox
        if not textbox.get("0.0", Tk.END).isspace():
            self.save_seen_index()
            pass
        guiecho.write(j("`撆..."))
        textbox.configure(state=Tk.NORMAL)
        textbox.delete("0.0", Tk.END)
        renderer.AdvancedRenderer().render(self.thread, textbox)
        textbox.configure(state=Tk.DISABLED)
        guiecho.write(j(""))
        self.see_previously_seen_point()
        self._requires_update = None
        guicommand.set_cursor("")
        pass

    def make_rmenu(self):
        self.rmenu = ktk.PopupMenu(self.textbox,
                                   [(j("Rs["), self.copy_selection),
                                    (j("Python XNvgƂăRs["),
                                     self._copy_selection_as_python),
                                    (j("Cɓɒǉ"), self.bookmark),
                                    (j("ēǂݍ"),self.reload),
                                    (j(""), self.close) ],
                                   font.ui)
        pass

    def reload(self):
        self._download_and_render()
        pass

    def get_thread(self):
        return self.thread

    def set_thread(self, thread):
        self.thread = thread
        pass

    def start_selection(self, event):
        self.textbox.config(state=Tk.NORMAL)

    def stop_selection(self, event):
        self.textbox.config(state=Tk.DISABLED)

    def copy_selection(self, event=None):
        self.clipboard_clear()
        self.clipboard_append(self._get_selection())
        pass

    def toggle_wordwrap(self):
        if self.textbox["wrap"] == Tk.NONE: 
            self.textbox.config(wrap=Tk.WORD) 
            pass
        else: 
            self.textbox.config(wrap=Tk.NONE)
            pass
        pass

    def bookmark(self):
        guicommand.bookmark(self.thread)
        pass

    def close(self):
        guicommand.close_active_thread()
        pass

    def destroy(self):
        """Destroy this frame and its copmonents."""
        self.save_seen_index()
        pass

    def on_cite(self, event):
        textbox = self.textbox
        textbox.config(cursor="hand2")
        # Obtain cited message.  Return if not found.
        cited_index = self.get_target(Tk.CURRENT)
        ranges = textbox.tag_ranges(cited_index)[0:2]
        if len(ranges)>=2:
            start, end = ranges[0:2]
            message = textbox.get(start, end)
        else:
            return False
        # Popup.
        self.balloon.label["text"] = message.rstrip("\n") 
        x = event.x_root+8
        y = event.y_root-self.balloon.label.winfo_reqheight() 
        if y<0:
            y=0
        self.balloon.geometry("+%d+%d" % (x, y)) 
        self.balloon.deiconify() 
        self.balloon.lift() 

    def scroll_line_up(self, event):
        """Scroll up by a line."""
        self.textbox.yview_scroll(-1, Tk.UNITS)
        return "break" 

    def scroll_line_down(self, event):
        """Scroll down by a line."""
        self.textbox.yview_scroll(1, Tk.UNITS)
        return "break" 

    def scroll_page_up(self, event):
        """Scroll up by a page."""
        self.textbox.yview_scroll(-1, Tk.PAGES)
        return "break" 

    def scroll_page_down(self, event):
        """Scroll down by a line."""
        self.textbox.yview_scroll(1, Tk.PAGES)
        return "break" 

    def off_cite(self, event):
        self.textbox.config(cursor="")
        self.balloon.withdraw()

    def on_link(self, event):
        self.textbox.config(cursor="hand2")

    def off_link(self, event):
        self.textbox.config(cursor="")

    def open_link(self, event):
        url = self.get_target(Tk.CURRENT)
        if url:
            webbrowser.open(url)

    def make_balloon(self):
        self.balloon = Tk.Toplevel(self.textbox)
        self.balloon.withdraw() 
        self.balloon.overrideredirect(True) 
        self.balloon.label = Tk.Label(self.balloon, 
                                      justify=Tk.LEFT,
                                      bg="#ffffcc", 
                                      font=font.cite,
                                      borderwidth=1, 
                                      relief=Tk.RAISED) 
        self.balloon.label.pack()

    def get_target(self, index):
        """Return target contents if the given index has a tag 
        starting with "target=" """
        tags = self.textbox.tag_names(index)
        for tag in tags:
            match = re.match("target=(.+)", tag)
            if match:
                content = match.groups()[0]
                break
        else:
            content = None
        return content

    def save_seen_index(self):
        """Save currently being seen index in the textbox."""
        index = self.textbox.index(Tk.CURRENT)
        config.set("ThreadView", self.thread.identity(), index)
        pass

    def see_previously_seen_point(self):
        """Display previouly being seen point"""
        index = config.get("ThreadView", self.thread.identity())
        if index:
            self.textbox.see(index)
            pass
        pass

    def _get_selection(self):
        try:
            text = self.selection_get()
        except TclError:
            text = ""
        return text
        
    def _copy_selection_as_python(self):
        text = self._get_selection()
        text = text.replace(j("@"),"  ")
        self.clipboard_clear()
        self.clipboard_append(text)
        pass

    def _auto_update(self):
        if self._requires_update:
            self.render()
        self.after(self._update_interval, self._auto_update)
        pass
    
    _update_interval = 100
    
    def _download_and_render(self):
        if not self._requires_update:
            def procedure():
                guiecho.write(j("Ǎ..."))
                self.thread.download()
                guiecho.write(j(""))
                self._requires_update = True
                pass
            start_new_thread(procedure, ())
            pass
        pass
  
    def _first_render(self):
        if self.thread.size()==0:
            self._download_and_render()
            pass
        else:
            self.render()
            pass
        pass

    pass


class ThreadViewPages(Pmw.NoteBook):
    def __init__(self, parent=None):
        Pmw.NoteBook.__init__(self, parent)
        pass

    def __len__(self):
        return len(self.pagenames())

    def open_new_page(self, thread):
        if self.has(thread):
            self._reload(thread)
        else:
            self._append(thread)
            pass
        pass
    
    def open_in_current_page(self, thread):
        if len(self):
            self.tab(self.getcurselection()).config(font=font.ui,
                                                       anchor=Tk.W,
                                                       text=thread.title)
            self.get_active_view().set_thread(thread)
            self.get_active_view().reload()
            pass
        else:
            self.open_new_page(thread)
            pass
        pass
     
    def open(self, thread):
        if self.has(thread):
            self._reload(thread)
        else:
            self._append(thread)
            pass
        pass
    
    def has(self, thread):
        return thread in [self.page(name).view.get_thread() 
                          for name in self.pagenames()]
    
    def reload_active_thread(self):
        if self._get_active_page():
            self._get_active_page().view.reload()
            pass
        pass

    def close_active_thread(self):
        if self._get_active_page():
            self._get_active_page().view.destroy()
            self.delete(self.getcurselection())
            pass
        pass

    def get_active_thread(self):
        if self._get_active_page():
            return self._get_active_page().view.get_thread()
        else:
            return None
        pass

    def _get_active_page(self):
        name = self.getcurselection()
        if name:
            return self.page(name)
        else:
            return None
        pass

    def get_active_view(self):
        if self._get_active_page():
            return self._get_active_page().view
        else:
            return None
        
    def _append(self, thread):
        page = self.add(thread)
        self.tab(thread).config(font=font.ui,
                                     anchor=Tk.W,
                                     text=thread.title)
        view = ThreadView(thread, page)
        view.pack(expand=Tk.YES, fill=Tk.BOTH)
        self.selectpage(thread)
        self.setnaturalsize()
        page.view = view
        page.view.set_thread(thread)
        self._bind_tab_func(self.tab(thread))
        pass

    def _bind_tab_func(self, tab):
        tab.bind("<2>", lambda e: e.widget.invoke())
        tab.bind("<2>", lambda e: self.close_active_thread(), "+")
        tab.bind("<Double-1>", lambda e: self.reload_active_thread())
        tab._menu = ktk.PopupMenu(tab, 
                                  [(j("ēǂݍ"), 
                                    self.reload_active_thread),
                                   (j("Cɓɒǉ"), 
                                    guicommand.bookmark_active_thread),
                                   (j(""), 
                                    self.close_active_thread)],
                                  font.ui)
        pass
    
    def _reload(self, thread):
        self.selectpage(thread)
        self.reload_active_thread()
        pass

    def add(self, index):
        return Pmw.NoteBook.add(self, self._make_index(index))
    
    def tab(self, index):
        return Pmw.NoteBook.tab(self, self._make_index(index))
    
    def selectpage(self, index):
        return Pmw.NoteBook.selectpage(self, self._make_index(index))
    
    def _make_index(self, index):
        return str(index)

    pass


# EOF
