#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <netdb.h>
#include <signal.h>
#include <readline/readline.h>
#include <readline/history.h>

#include "port.h"
#include "host.h"
#include "common.h" 
#include "libWrapper.h"
#include "buffsize.h"
#include "sensorImageSize.h"
#include "slotsize.h"
#include "ipc.h"
#ifndef _SCHEMA_H
#define _SCHEMA_H
#include "schema.h"
#endif

#ifndef _SELECT_FROM_H
#define _SELECT_FROM_H
#include "selectFrom.h"
#endif

extern int connectCli(const int port, const char hostname[]);
extern int connectKraft(void);
extern void printResultForRelation(ANSWER answer);
extern void queryToArtemis(const int sockfd, const char query[]);
extern void sendExitAndCloseSockfd(const int sockfd);
extern void sendQuery(const char query[], const int sockfd);
extern void printResultOfOperation(const TYPE_OF_ANSWER typeOfAnswer);
extern void sendback(const int sockfd, const void *ptr, const int size, const int flag);
extern void recvback(const int sockfd, void *ptr, const int size, const int flag);
extern BOOLEAN isThisAppend(const char query[]);
extern TYPE_OF_ANSWER recvTypeOfAnswer(const int sockfd);

/***************************************************************
 *
 * Private Function
 *
 ***************************************************************/
static void
releaseSensorForClient(SENSOR *pHeadSensor)
{
  SENSOR *pSensor;

  while (pHeadSensor->next != NULL) {
    pSensor = pHeadSensor->next;
    pHeadSensor->next = pHeadSensor->next->next;
    free(pSensor);
  }
}

static void
releaseSensorSegmentForClient(SENSOR_SEGMENT *pHeadSensorSegment)
{
  SENSOR_SEGMENT *trash;

  while (pHeadSensorSegment->next != NULL) {
    trash = pHeadSensorSegment->next;
    pHeadSensorSegment->next = pHeadSensorSegment->next->next;
    releaseSensorForClient(&(trash->headSensor));
    free(trash);
  }
}

static void
releaseBaseTupleForClient(TUPLE *pHeadTuple, const ATTR attr[], const int numOfAttr)
{
  int attrid;
  int byteOffset;
  TUPLE *trash;
  SENSOR_SEGMENT *pSensorSegment;

  while (pHeadTuple->next != NULL) {
    trash = pHeadTuple->next;
    pHeadTuple->next = pHeadTuple->next->next;
    for (byteOffset = 0, attrid = 0; 
         attrid < numOfAttr; 
         byteOffset += attr[attrid].sizOfAttr, attrid++) {
      switch (attr[attrid].attrType) {
      case SENSOR_SEGMENT_DOUBLE:
        pSensorSegment = (SENSOR_SEGMENT *)(trash->obj + byteOffset);
        releaseSensorSegmentForClient(pSensorSegment);
        break;
      default: 
        break;
      }
    }
    free(trash->obj);
    free(trash);
  }
}

static void
releaseAnswerForClient(ANSWER answer)
{
  releaseBaseTupleForClient(&(answer.headTuple), answer.attr, answer.numOfAttr);
  free(answer.attr);
}

static void
writeLine(const int numOfAttr)
{
  int attrid, iter;

  printf("+");
  for (attrid = 0; attrid < numOfAttr; attrid++) {
    printf("-");
    for (iter = 0; iter < 8; iter++) 
      printf("-");
    printf("-+");
  }
  printf("\n");
}

static void
printAttribute(const ANSWER answer)
{
  int i;
  char buff[BUFSIZ];

  writeLine(answer.numOfAttr);
  printf("| ");

  bzero(buff, sizeof(buff));
  for (i = 0; i < answer.numOfAttr; i++) {
    if (answer.attr[i].f != NUF) {
      switch (answer.attr[i].f) {
      case MAX: 
        strcat(buff, " max(");
        break;
      case MIN: 
        strcat(buff, " min(");
        break;
      case CNT: 
        strcat(buff, " cnt(");
        break;
      case SUM: 
        strcat(buff, " sum(");
        break;
      case AVG: 
        strcat(buff, " avg(");
        break;
      case EUC: 
        strcat(buff, " euc(");
        break;
      case DTW: 
        strcat(buff, " dtw(");
        break;
      case LCS: 
        strcat(buff, " lcs(");
        break;
      case POW: 
        strcat(buff, " pow(");
        break;
      case FTR: 
        strcat(buff, " ftr(");
        break;
      case FTI: 
        strcat(buff, " fti(");
        break;
      case COR: 
        strcat(buff, " cor(");
        break;
      case NUF: 
        ERR;
        break;
      }
      strcat(buff, answer.attr[i].attrName);
      strcat(buff, ")");
      printf("%8s | ", buff);
    }
    else 
      printf("%8s | ", answer.attr[i].attrName);
  }
  printf("\n");
  writeLine(answer.numOfAttr);
}  

static void
printSensor(const SENSOR s)
{
  int i;
  SENSOR *p;

  for (p = s.next; p != NULL; p = p->next) {
    printf("[");
    for (i = 0; i < p->sdobj.szsa; i++) 
      printf("{%3.1f}", p->sdobj.v[i]);
    printf("]");
  }
}

static void
printSensorSegment(const SENSOR_SEGMENT head)
{
  SENSOR_SEGMENT *p;

  for (p = head.next; p != NULL; p = p->next) {
		if (p->f != NUF) {
      printf("[%3.1f]", p->v);
		}
		else {
			printf("=== ");
			printSensor(p->headSensor); 
			printf(" ===");
		}
  }
}

/***************************************************************
 *
 * Public Function
 *
 ***************************************************************/
extern int
connectKraftBySensor(void)
{
  int sockfd;
  sockfd = connectCli(DBMS_SENSOR_PORT, DBMS_HOST);
  return sockfd;
}

extern int
connectKraftByClient(void)
{
  int sockfd;
  sockfd = connectCli(DBMS_CLIENT_PORT, DBMS_HOST);
  return sockfd;
}

extern void 
sendQuery(const char query[], const int sockfd)
{
  if (send(sockfd, query, QUERY_BUFFSIZE, 0) == -1) ERR;
}

extern TYPE_OF_ANSWER
recvTypeOfAnswer(const int sockfd) 
{
  TYPE_OF_ANSWER typeOfAnswer;

  if (recv(sockfd, &typeOfAnswer, sizeof(TYPE_OF_ANSWER), 0) == -1) ERR;

  return typeOfAnswer;
}

extern void
sendExitAndCloseSockfd(const int sockfd)
{
  char query[QUERY_BUFFSIZE];

  bzero(query, QUERY_BUFFSIZE);
  strncpy(query, "exit", strlen("exit"));
  sendback(sockfd, query, QUERY_BUFFSIZE, 0);
  if (close(sockfd) == -1) ERR;
}

extern BOOLEAN
isThisAppend(const char query[])
{
  int qid;
  BOOLEAN thisIsAppend = FALSE;

  for (qid = 0; qid < (int)(QUERY_BUFFSIZE - strlen("append")); qid++) {
    if (strncmp(&query[qid], "append", strlen("append")) == 0) {
      thisIsAppend = TRUE;
      break;
    }
  }

  return thisIsAppend;
}

extern void
printResultForRelation(ANSWER answer)
{
  int attrid;
  size_t siz;
  char text[BUFFSIZE];
  TUPLE *p;
  SENSOR_SEGMENT hss; /* Head SS */

  printAttribute(answer);
  for (p = answer.headTuple.next; p != NULL; p = p->next) {
    printf("| ");
    for (siz = attrid = 0; attrid < answer.numOfAttr; siz += answer.attr[attrid].sizOfAttr, attrid++) {
      switch (answer.attr[attrid].attrType) {

      case INT:    
        printf("%8d | ", *(int *)(size_t)p->obj + siz);
        break;

      case DOUBLE: 
        printf("%8.1f | ", *(double *)(size_t)p->obj + siz);
        break;

      case TEXT:   
        bzero(text, BUFFSIZE);
        strncpy(text, p->obj + siz, BUFFSIZE);
        printf("%8s | ", text); 
        break;

      case SENSOR_SEGMENT_DOUBLE:  
        memcpy(&hss, p->obj + siz, sizeof(SENSOR_SEGMENT));
				printSensorSegment(hss); 
        break;

      default:
        ERR;
        break;
      }
    }
    printf("\n");
		fflush(stdout);
  }
  writeLine(answer.numOfAttr);
  releaseAnswerForClient(answer);
}  

static ANSWER
unMarshalize(char *pktp, const int szpkt)
{
  int o; /* Offset */
	int b; /* Byte Offset */
  METACOMM *mcp = NULL;
  ANSWER answer;
  TUPLE *tp = NULL;
  SENSOR *sp = NULL;
  SENSOR_SEGMENT *ssp = NULL;
  
  for (o = 0; o < szpkt; ) {
    mcp = (METACOMM *)(pktp + o);
    o += sizeof(METACOMM);

    switch (mcp->commType) {

    case TYPE_ANSWER:
      memcpy(&answer, (pktp + o), sizeof(ANSWER));
      tp = &(answer.headTuple);
      tp->next = NULL;
      o += sizeof(ANSWER);
      break;

    case TYPE_ATTR:
      if ((answer.attr = calloc(mcp->numOfObj, sizeof(ATTR))) == NULL) ERR;
      memcpy(answer.attr, (pktp + o), (sizeof(ATTR) * mcp->numOfObj));
      o += (sizeof(ATTR) * mcp->numOfObj);
      break;

    case TYPE_RELATION:
      if ((tp->next = calloc(1, sizeof(TUPLE))) == NULL) ERR;
      tp->next->prev = tp;
      tp = tp->next;
      if ((tp->obj = calloc(1, answer.byteOfTuple)) == NULL) ERR;
      memcpy(tp->obj, (pktp + o), answer.byteOfTuple);
      o += answer.byteOfTuple;
			ssp = NULL; /* init */
      break;

    case TYPE_SENSOR_SEGMENT:
			if (ssp == NULL) {
				ssp = (SENSOR_SEGMENT *)(pktp + o);
				b = ssp->offset;
				ssp = (SENSOR_SEGMENT *)(tp->obj + b);
			}
      if ((ssp->next = calloc(1, sizeof(SENSOR_SEGMENT))) == NULL) ERR;
			ssp = ssp->next;
      memcpy(ssp, pktp + o, sizeof(SENSOR_SEGMENT));
      sp = &ssp->headSensor;
      o += sizeof(SENSOR_SEGMENT);
      break;

    case TYPE_SENSOR:
      if ((sp->next = calloc(1, sizeof(SENSOR))) == NULL) ERR;
      sp = sp->next;
      memcpy(&sp->sdobj, pktp + o, sizeof(SDOBJ));
      o += sizeof(SDOBJ);      

      if ((sp->sdobj.v = calloc(sp->sdobj.szsa, sizeof(double))) == NULL) ERR;
      memcpy(sp->sdobj.v, pktp + o, sp->sdobj.szsa * sizeof(double));
      o += sp->sdobj.szsa * sizeof(double);

      break;
    }
  }

  return answer;
}

extern ANSWER
recvAnswer(const int sockfd)
{
  ANSWER answer;
  int szpkt;
	char *pktp;

  if (recv(sockfd, &szpkt, sizeof(int), MSG_WAITALL) == -1) ERR;
  if ((pktp = calloc(1, szpkt)) == NULL) ERR;
  if (recv(sockfd, pktp, szpkt, MSG_WAITALL) == -1) ERR;

  answer = unMarshalize(pktp, szpkt);
  free(pktp);

  return answer;
}

extern void
printResultForSchema(const int sockfd)
{
  char schemaInfo[BUFFSIZE];
  BOOLEAN fin;

  printf("--------------------\n");
  while (1) {
    recvback(sockfd, &fin, sizeof(BOOLEAN), 0);
    if (fin == TRUE) {
      break;
    }
    else {
      bzero(schemaInfo, BUFFSIZE);
      recvback(sockfd, schemaInfo, BUFFSIZE, 0);
      printf("%s\n", schemaInfo);
    }
  }
  printf("--------------------\n");
}

extern void
printResultOfOperation(const TYPE_OF_ANSWER typeOfAnswer)
{
  switch (typeOfAnswer) {

  case ANSWER_AGG4TUPLE_NOT_IMPLEMENTED:
    printf("AGG4TUPLE_NOT_IMPLEMENTED\n");
    break;

  case ANSWER_ATTR_NOT_EXIST:
    printf("ATTR DOES NOT EXIST\n");
    break;

  case ANSWER_APPEND:  
    //printf("APPEND\n");
    break; 

  case ANSWER_BEGIN: 
    printf("BEGIN\n"); 
    break;

  case ANSWER_COMMIT:  
    printf("COMMIT\n"); 
    break;

  case ANSWER_CREATE_INDEX: 
    printf("CREATE INDEX\n"); 
    break;

  case ANSWER_CREATE_MONITOR: 
    printf("CREATE MONITOR\n");  
    break;

  case ANSWER_CREATE_TABLE: 
    printf("CREATE TABLE\n");    
    break;

  case ANSWER_CREATE_WINDOW: 
    printf("CREATE WINDOW\n");    
    break;

  case ANSWER_DELETE:   
    printf("DELETE\n");
    break;

  case ANSWER_DROP_TABLE: 
    printf("DROP TABLE\n");                                 
    break;

  case ANSWER_DROP_INDEX: 
    printf("DROP INDEX\n");                                 
    break;

	case ANSWER_FUNCTION_WRONG:
    printf("ANSWER_FUNCTION_WRONG\n");
		break;

  case ANSWER_INSERT: 
    //printf("INSERT\n");                                     
    break;

  case ANSWER_INSERT_FAILED: 
    printf("INSERT_FAILED\n");
    break;

  case ANSWER_PARSE_ERROR: 
    printf("Sorry, parse error. Please try again\n");       
    break;

  case ANSWER_ROLLBACK_USER_REQUEST: 
    printf("ROLLBACK AS YOU WANT\n");                       
    break;

  case ANSWER_ROLLBACK_NO_BUFFERPOOL: 
    printf("ROLLBACK SINCE NO SPACE LEFT ON BUFFERPOOL\n"); 
    break;

  case ANSWER_RELATION: 
    ERR;
    break;

  case ANSWER_SHOW_SCHEMA:            
    ERR;
    break;

  case ANSWER_SHOW_MONITOR:           
    ERR;
    break;

  case ANSWER_TABLE_EXIST:
    printf("TABLE ALREADY EXISTS\n");
    break;

  case ANSWER_TABLE_NOT_EXIST:
    printf("TABLE DOES NOT EXIST\n");
    break;

  case ANSWER_TUPLE_EXIST:
    printf("TUPLE ALREADY EXISTS\n");
    break;

  case ANSWER_TUPLE_ID_MUST_BE_ZERO:
    printf("TUPLE ID MUST BE ZERO\n");
    break;

  case ANSWER_TUPLE_ID_WRONG:
    printf("TUPLE_ID_WRONG\n");
    break;

  case ANSWER_TUPLE_NOT_EXIST:
    printf("TUPLE DOES NOT EXIST\n");
    break;

  case ANSWER_FINISH_MONITOR_RUN: 
    ERR;
    break;

  case ANSWER_WRONG_ATTRIBUTE:
    printf("WRONG ATTRIBUTE\n");
    break;

  case ANSWER_WRONG_ORDER:
    printf("WRONG ORDER\n");
    break;

  case ANSWER_WRONG_SCHEMA:
    printf("WRONG SCHEMA\n");
    break;

  case ANSWER_WRONG_SENSOR_ID:
    printf("WRONG SENSOR ID\n");
    break;

  case ANSWER_WRONG_WHERE_CLAUSE:
    printf("WRONG WHERE CLAUSE\n");
    break;

  case ANSWER_UNACCEPTABLE: 
    printf("Invalid command\n");
    break;

  default:
    printf("Sorry, something is wrong: code(%d)\n", typeOfAnswer); 
    break;
  }
}

