#include "common.h"
#include "libWrapper.h"
#include "dataDir.h"
#include "schemaDir.h"
#ifndef _TRANS_INFO_H
#define _TRANS_INFO_H
#include "transInfo.h"
#endif
#ifndef _PAGE_H
#define _PAGE_H
#include "page.h"
#endif

#define SENSOR_DIR "sensor"
#define GARBAGE_FILE "garbage"
#define OBJECT_FILE "object"

/******************
 * 
 * Declaration
 *
 ******************/
extern void pin(DBPAGE *buffPageP, char mode);
extern void unpin(DBPAGE *buffPageP);
extern void unlockEntirePage(void);
extern void lockEntirePage(void);
extern BOOLEAN trypin(DBPAGE *pageP, char mode);

/******************
 * 
 * Global Variable 
 *
 ******************/
static DBPAGE *DBPageP[PAGENMBR];
static pthread_rwlock_t LockEntireDBPage;

/************
 * 
 * Function
 *
 ************/
static void
initPage(DBPAGE *pageP, SCHEMA *schemaP, const int attrid, const int tupleid, const char path[])
{
  /* 
   * Do NOT use "bzero(buffPageP, sizeof(DBPAGE))" !
   * Because it erases LOCK 
   */

  /* 
   * Zero Clear 
   */
  pageP->schemaP = NULL;
  bzero(pageP->data, PAGESIZE);
  pageP->access = 0;
  bzero(pageP->path, BUFFSIZE);
  bzero(&pageP->pi, sizeof(PAGEINFO));
  /* Never initialize rwlock here ! It is already initialized at the boot time */

  /* 
   * Set initial values 
   */
  pageP->schemaP = schemaP;
  pageP->access = 0;
  memcpy(pageP->path, path, BUFFSIZE);

  /* PAGEINFO */
  pageP->pi.tupleid = tupleid;
  pageP->pi.attrid = attrid;
  pageP->pi.numOfObj = 0;
  if (attrid == 0) { /* Relation */
    pageP->pi.sizOfObj = schemaP->sizOfObj;
  }
  else { /* Sensor */
    pageP->pi.sizOfObj = sizeof(SDOBJ) + sizeof(double) * schemaP->attr[attrid].szsa;
  }

  /* i don't care about offset here */
}

/*
 * First Fit Policy (Simplest implementation...)
 */
static DBPAGE *
selectVictimPage(void)
{
  int i;

  /*
   * Looking for the victim
   */
  while (1) {
    for (i = 0; i < PAGENMBR; i ++) {
      if (trypin(DBPageP[i], 'w') == TRUE) {
        assert(DBPageP[i]->schemaP != NULL);
        return DBPageP[i];
      }
    }
    pthread_yield();
  }
}

static void
writePageToDisk(DBPAGE *pageP, BOOLEAN doFsync)
{
  int fd;
  char *dir;

  memcpy(&pageP->data[PAGESIZE - sizeof(PAGEINFO)], &pageP->pi, sizeof(PAGEINFO));
  if ((fd = open(pageP->path, O_WRONLY|O_EXLOCK|O_CREAT, 0644)) == -1) {
    if (errno == ENOENT) {
      if ((dir = calloc(strlen(pageP->path), sizeof(char))) == NULL) ERR;
      strncpy(dir, pageP->path, strlen(pageP->path) - strlen("object"));
      if (mkdir(dir, 0755) == -1) ERR;
      free(dir);
      if ((fd = open(pageP->path, O_WRONLY|O_CREAT|O_EXLOCK, 0644)) == -1) ERR;
    }
    else {
      ERR;
    }
  }
  if (lseek(fd, pageP->pi.offset, SEEK_SET) == -1) ERR;
  if (write(fd, pageP->data, PAGESIZE) == -1) ERR;
  if (doFsync == TRUE)
    fsync(fd);
  if (close(fd) == -1) ERR;
}

static void
erasePointer(DBPAGE *pageP)
{
  int i;

  assert(pageP->schemaP != NULL);
  if (pageP->pi.attrid == 0) { /* Relation */
    if (pageP->schemaP->startP == pageP) 
      pageP->schemaP->startP = NULL;
    if (pageP->schemaP->endP == pageP) 
      pageP->schemaP->endP = NULL;
  }
  else { /* Sensor Attributes */
    assert(pageP->schemaP->attr[pageP->pi.attrid].p != NULL);
    for (i = 0; i < pageP->schemaP->numOfObj; i ++) {
      if (pageP->schemaP->attr[pageP->pi.attrid].p[i].startP == pageP) {
        pageP->schemaP->attr[pageP->pi.attrid].p[i].startP = NULL;
      }
      if (pageP->schemaP->attr[pageP->pi.attrid].p[i].endP == pageP) {
        pageP->schemaP->attr[pageP->pi.attrid].p[i].endP = NULL;
      }
    }
  }
}

static DBPAGE *
getVictimPage(void)
{
  DBPAGE *pageP;

  pageP = selectVictimPage();
  writePageToDisk(pageP, FALSE);
  erasePointer(pageP);
  assert(pageP != NULL);

  return pageP;
}

static DBPAGE *
getUnusedPage(const char path[], SCHEMA *schemaP, const int attrid, const int tupleid)
{
  int i;
  DBPAGE *pageP;

  /*
   * Find unlocked page 
   */
  for (i = 0; i < PAGENMBR; i ++) {
    if (trypin(DBPageP[i], 'w') == TRUE) {
      if (DBPageP[i]->schemaP == NULL) { /* This means CLEAN state */
        assert(DBPageP[i]->pi.numOfObj == 0);
        initPage(DBPageP[i], schemaP, attrid, tupleid, path);
        return DBPageP[i]; /* Return with lock */
      }
      else {
        unpin(DBPageP[i]);
      }
    }
  }

  /* Now I do not have any locks */
  assert(i == PAGENMBR);
  pageP = getVictimPage(); /* I will get the lock of page in "replacePage" */
  initPage(pageP, schemaP, attrid, tupleid, path);
  assert(pageP != NULL);

  return pageP; /* Return with lock */
}

static DBPAGE *
getEndPageInternal(const char path[], SCHEMA *schemaP, const int attrid, const int tupleid)
{
  int fd;
  char *dir;
  DBPAGE *pageP;

  pageP = getUnusedPage(path, schemaP, attrid, tupleid);

  /* Read data from disk */
  if ((fd = open(path, O_RDONLY|O_SHLOCK)) == -1) {
    if (errno == ENOENT) {
      if ((dir = calloc(strlen(path), sizeof(char))) == NULL) ERR;
      strncpy(dir, path, strlen(path) - strlen("object"));
      if (mkdir(dir, 0755) == -1) ERR;
      free(dir);
      if ((fd = open(path, O_RDONLY|O_CREAT|O_SHLOCK, 0644)) == -1) ERR;
    }
    else {
      ERR;
    }
  }

  /* Set the offset and pi */
  if ((pageP->pi.offset = lseek(fd, -1 * PAGESIZE, SEEK_END)) < 0) {
    if ((pageP->pi.offset = lseek(fd, 0, SEEK_SET)) == -1) ERR;
    if (close(fd) == -1) ERR;    
  }
  else {
    if (read(fd, pageP->data, PAGESIZE) == -1) ERR;
    if (close(fd) == -1) ERR;
    memcpy(&pageP->pi, &pageP->data[PAGESIZE - sizeof(PAGEINFO)], sizeof(PAGEINFO));
  }

  return pageP;
}

static void
setPath(SCHEMA *schemaP, const int attrid, const int tupleid, char path[])
{
	char *hdp;

	hdp = getenv("HOME");
  bzero(path, BUFFSIZE);
  if (attrid == 0) {
    sprintf(path, "%s/%s/%s/relation", hdp, DATA_DIR, schemaP->schemaName);
  }
  else {
    sprintf(path, "%s/%s/%s/sensor/%s/%d/object", hdp, DATA_DIR, schemaP->schemaName, schemaP->attr[attrid].attrName, tupleid);
  }
}

static DBPAGE *
getPageFromBuff(SCHEMA *schemaP, int aid, int tid, int offset)
{
  int i;

  for (i = 0; i < PAGENMBR; i ++) {
    if (trypin(DBPageP[i], 'w') == TRUE) {
      if (DBPageP[i]->schemaP == schemaP && DBPageP[i]->pi.attrid == aid && DBPageP[i]->pi.offset == offset) {
        if ((aid != 0 && DBPageP[i]->pi.tupleid == tid) || (aid == 0)) {
          assert(schemaP != NULL);
          assert(offset != -1);
          return DBPageP[i]; /* Return with lock */
        }
      }
      unpin(DBPageP[i]);
    }
  }
  return NULL;
}

static DBPAGE *
getPageFromDisk(SCHEMA *schemaP, int attrid, int tupleid, int offset)
{
  int fd;
  int n;  /* For return value of 'read' */
  char path[BUFFSIZE];
  char data[PAGESIZE];
  DBPAGE *pageP;
  
  /* 
   * Read data from disk 
   */
  setPath(schemaP, attrid, tupleid, path);
  if ((fd = open(path, O_RDONLY|O_SHLOCK)) == -1) {
    if (errno == ENOENT) {
      return NULL;
    }
    else {
      ERR;
    }
  }
  if (lseek(fd, offset, SEEK_SET) == -1) ERR;
  if ((n = read(fd, data, PAGESIZE)) <= 0) {
    if (n == 0) {
      if (close(fd) == -1) ERR;
      return NULL;
    }
    else {
      ERR;
    }
  }

  /* 
   * Paste file image to memory 
   */
  if (close(fd) == -1) ERR;
  pageP = getUnusedPage(path, schemaP, attrid, tupleid);
  memcpy(pageP->data, data, PAGESIZE);
  memcpy(&pageP->pi, &data[PAGESIZE - sizeof(PAGEINFO)], sizeof(PAGEINFO));
  pageP->pi.offset = offset;
  
  return pageP;
}

static DBPAGE *
getPage(SCHEMA *schemaP, int aid, int tid, int offset)
{
  DBPAGE *pageP;

  if ((pageP = getPageFromBuff(schemaP, aid, tid, offset)) != NULL) 
    return pageP;
  if ((pageP = getPageFromDisk(schemaP, aid, tid, offset)) != NULL) 
    return pageP;

  return NULL;
}

/********************************************************************
 *
 *                             PUBLIC
 *
 ********************************************************************/
extern void
pin(DBPAGE *pageP, char mode)
{
  assert(pageP != NULL);
  switch (mode) {
  case 'w':
    if (pthread_rwlock_wrlock(&pageP->rwlock)) ERR;
    break;
  case 'r':
    if (pthread_rwlock_rdlock(&pageP->rwlock)) ERR;
    break;
  default: 
    ERR;
    break;
  }
  pageP->access++;
}

extern BOOLEAN
trypin(DBPAGE *pageP, char mode)
{
  int en; /* Error Number */

  assert(pageP != NULL);
  switch (mode) {
  case 'w':
    if ((en = pthread_rwlock_trywrlock(&pageP->rwlock))) {
      if (en == EBUSY)
        return FALSE;
      else 
        ERR;
    }
    break;
  case 'r':
    if ((en = pthread_rwlock_tryrdlock(&pageP->rwlock))) {
      if (en == EBUSY)
        return FALSE;
      else 
        ERR;
    }
    break;
  default: 
    ERR;
    break;
  }
  pageP->access++;

  return TRUE;
}

extern void
unpin(DBPAGE *pageP)
{
  assert(pageP != NULL);
  if (pthread_rwlock_unlock(&pageP->rwlock)) ERR;
}

extern char *
getInsertPtr(SCHEMA *schemaP, const int tid)
{
  int i;
  int offset = 0;
  char *p; /* for data object */
  DBPAGE *pageP;

  for (pageP = getPage(schemaP, 0, tid, 0); /* 0 means TUPLE */
       pageP != NULL; offset += PAGESIZE,
       pageP = getPage(schemaP, 0, tid, offset)) {
    for (p = pageP->data, i = 0; 
         i < pageP->pi.numOfObj; i++) {
      D(*(int *)p);
      if (*(int *)p == tid)
        return p;
    }
  }

  return NULL;
}

extern DBPAGE *
getEndPage(SCHEMA *schemaP, int attrid, int tupleid)
{
  char path[BUFFSIZE];

  setPath(schemaP, attrid, tupleid, path);
  return getEndPageInternal(path, schemaP, attrid, tupleid);
}

extern DBPAGE *
getNewPage(SCHEMA *schemaP, int attrid, int tupleid)
{
  char path[BUFFSIZE];

  setPath(schemaP, attrid, tupleid, path);
  return getUnusedPage(path, schemaP, attrid, tupleid); /* Page has a lock */
}

extern int
getCurPageSize(DBPAGE *p)
{
  assert(p != NULL);
  return (p->pi.numOfObj * p->pi.sizOfObj);
}

extern void
initDBPages(void)
{
  int i;

  for (i = 0; i < PAGENMBR; i ++) {
    if ((DBPageP[i] = calloc(1, sizeof(DBPAGE))) == NULL) ERR;
    if (pthread_rwlock_init(&DBPageP[i]->rwlock, NULL)) ERR;
    /* Dirty Value, important */
    bzero(&DBPageP[i]->pi, sizeof(PAGEINFO));
  }
}

extern DBPAGE
getStartRelPage(SCHEMA *schemaP)
{
  DBPAGE *pageP, ret;

  lockEntirePage();
  if ((pageP = getPage(schemaP, 0, 0, 0)) != NULL) {
    memcpy(&ret, pageP, sizeof(DBPAGE));
    unpin(pageP);
  }
  else {
    ret.pi.offset = -1;
  }
  unlockEntirePage();
  
  return ret;
}

extern DBPAGE
getStartSenPage(SCHEMA *schemaP, int attrid, int tupleid)
{
  DBPAGE *pageP, ret;

  lockEntirePage();
  if ((pageP = getPage(schemaP, attrid, tupleid, 0)) != NULL) {
    memcpy(&ret, pageP, sizeof(DBPAGE));
    unpin(pageP);
  }
  else 
    ret.pi.offset = -1;

  unlockEntirePage();

  return ret;
}

extern DBPAGE
getNextPage(SCHEMA *schemaP, DBPAGE page)
{
  DBPAGE *pageP, ret;

  lockEntirePage();
  if ((pageP = getPage(schemaP, page.pi.attrid, page.pi.tupleid, (page.pi.offset + PAGESIZE))) != NULL) {
    memcpy(&ret, pageP, sizeof(DBPAGE));
    unpin(pageP);
  }
  else {
    ret.pi.offset = -1;
  }
  unlockEntirePage();

  return ret;
}

extern void
initPagePtr(SCHEMA *schemaP)
{
  int i;
  char *tmpP;

  for (i = 0; i < schemaP->numOfAttr; i ++) {
    switch (schemaP->attr[i].attrType) {
    case SENSOR_SEGMENT_DOUBLE:
      if (schemaP->attr[i].p == NULL) {
        if ((schemaP->attr[i].p = (PAGEPTR *)calloc((schemaP->numOfObj + 1), sizeof(PAGEPTR))) == NULL) ERR;
      }
      else {
        /* Set Cache */
        if ((tmpP = calloc(sizeof(PAGEPTR), schemaP->numOfObj)) == NULL) ERR;
        memcpy(tmpP, schemaP->attr[i].p, (sizeof(PAGEPTR) * (schemaP->numOfObj)));

        /* Reallocate */
        if ((schemaP->attr[i].p = (PAGEPTR *)realloc(schemaP->attr[i].p, (schemaP->numOfObj + 1) * sizeof(PAGEPTR))) == NULL) ERR;
        bzero(schemaP->attr[i].p, (sizeof(PAGEPTR) * (schemaP->numOfObj + 1)));

        /* Erase Cache */
        memcpy(schemaP->attr[i].p, tmpP, (sizeof(PAGEPTR) * (schemaP->numOfObj)));
        free(tmpP);
      }
      break;
    default: 
      break;
    }
  }
}

extern void
lockEntirePage(void)
{
  if (pthread_rwlock_wrlock(&LockEntireDBPage)) ERR;
}

extern void
unlockEntirePage(void)
{
  if (pthread_rwlock_unlock(&LockEntireDBPage)) ERR;
}

extern void
initLockOfDBPage(void)
{
  if (pthread_rwlock_init(&LockEntireDBPage, NULL)) ERR;
}

extern void
shutdownPages(void)
{
  int i;

  for (i = 0; i < PAGENMBR; i ++) {
    if (DBPageP[i]->schemaP != NULL) {
      writePageToDisk(DBPageP[i], TRUE);
    }
  }
}
