#include "common.h"
#include "libWrapper.h"

#ifdef PRINT_MATRIX
#define LOGFILE "LOGFILE"
#endif

#ifndef _SCHEMA_H
#define _SCHEMA_H
#include "schema.h"
#endif

#ifndef _SELECT_FROM_H
#define _SELECT_FROM_H
#include "selectFrom.h"
#endif

#ifndef _SIMSEQ_H
#define _SIMSEQ_H
#include "simseq.h"
#endif

#define PRINT_MATRIX

#ifdef PRINT_MATRIX
#define MATRIX_LOGFILE "/tmp/MATRIX_LOGFILE"
#endif

#ifndef _WINDOW_H
#define _WINDOW_H
#include "window.h"
#endif

/*******************************************************
 *
 * Typedef
 *
 *******************************************************/
typedef struct _PATH
{
  int sid; /* Coordinates X in matrix */
  int tid; /* Coordinates Y in matrix */
  double val;
} PATH;

/*******************************************************
 *
 * Global variable
 *
 *******************************************************/
static int MaxPathLength;

/*******************************************************
 *
 * Private Function
 *
 *******************************************************/
static void
setMatrix(double **matrix, const double s[], const double t[], const int windowSize)
{
  int sid;
  int tid;
  
  for (sid = 0; sid < windowSize; sid ++) {
    for (tid = 0; tid < windowSize; tid ++) {
      matrix[sid][tid] = abs(s[sid] - t[tid]);
    }
  }
}

#ifdef PRINT_MATRIX
static void
printMatrix(double **matrix, int windowSize)
{
  int sid;
  int tid;
  FILE *fp;

  if ((fp = fopen(MATRIX_LOGFILE, "a")) == NULL) ERR;
  fprintf(fp, "\n[Matrix]\n");
  for (sid = 0; sid < windowSize; sid ++) {
    fprintf(fp, " ");
    for (tid = 0; tid < windowSize; tid ++) {
      fprintf(fp, "%f ", matrix[sid][tid]);
    }
    fprintf(fp, "\n");
  }
  fclose(fp);
}
#endif

static PATH *
allocatePpath(void)
{
  PATH *ppath;
  
  if ((ppath = (PATH *)calloc(MaxPathLength, sizeof(PATH))) == NULL) ERR;

  return ppath;
}

static char 
getDirection(double **matrix, int pathid, PATH *ppath)
{
  int sval;
  int tval;
  int mval;
  char direction;

  sval = fabs((matrix[ppath[pathid-1].tid][ppath[pathid-1].sid+1]) 
              - (ppath[pathid-1].val));
  tval = fabs((matrix[ppath[pathid-1].tid+1][ppath[pathid-1].sid]) 
              - (ppath[pathid-1].val));
  mval = fabs((matrix[ppath[pathid-1].tid+1][ppath[pathid-1].sid+1]) 
              - (ppath[pathid-1].val));
  
  if ((mval <= sval) && (mval <= tval)) {
    direction = 'm';
  }
  else if (sval <= tval) {
    direction = 's';
  }
  else {
    direction = 't';
  }

  return direction;
}

static PATH *
createWarpingPath(double **matrix, int *pathlen, int windowSize)
{
  int pathid;
  char direction;
  PATH *ppath;

  ppath = allocatePpath();
  ppath[0].sid = 0;
  ppath[0].tid = 0;
  ppath[0].val = matrix[0][0];

  pathid = 0;
  while (1) {
    pathid ++;
    direction = getDirection(matrix, pathid, ppath);
    switch (direction) {
    case 's':
      ppath[pathid].sid = ppath[pathid-1].sid + 1;
      ppath[pathid].tid = ppath[pathid-1].tid;
      ppath[pathid].val = matrix[ppath[pathid].tid][ppath[pathid].sid];      
      break;
    case 't':
      ppath[pathid].sid = ppath[pathid-1].sid;
      ppath[pathid].tid = ppath[pathid-1].tid + 1;
      ppath[pathid].val = matrix[ppath[pathid].tid][ppath[pathid].sid];      
      break;
    case 'm':
      ppath[pathid].sid = ppath[pathid-1].sid + 1;
      ppath[pathid].tid = ppath[pathid-1].tid + 1;
      ppath[pathid].val = matrix[ppath[pathid].tid][ppath[pathid].sid];      
      break;
    default:
      printf("No such direction\n");
      exit(1);
      break;
    }

    /*
     * Check boundary condition
     */
    if (ppath[pathid].sid == (windowSize - 1)) {
      while (ppath[pathid].tid < windowSize - 1) {
        pathid ++;
        ppath[pathid].tid = ppath[pathid-1].tid + 1;
        ppath[pathid].sid = ppath[pathid-1].sid;
      }
      break;
    }
    else if (ppath[pathid].tid == (windowSize - 1)) {
      while (ppath[pathid].sid < windowSize - 1) {
        pathid ++;
        ppath[pathid].tid = ppath[pathid-1].tid;
        ppath[pathid].sid = ppath[pathid-1].sid + 1;
      }
      break;
    }
  }

  *pathlen = pathid+1;

  return ppath;
}

#ifdef PRINT_MATRIX
static void
printWarpingPath(double **matrix, PATH *ppath, int pathlen, int windowSize)
{
  int i;
  int j;
  int k;
  FILE *fp;

  if ((fp = fopen(MATRIX_LOGFILE, "a")) == NULL) ERR;
  fprintf(fp, "\n[Warping path]\n");
  for (i = 0; i < windowSize; i ++) {
    fprintf(fp, " ");
    for (j = 0; j < windowSize; j ++) {
      for (k = 0; k < pathlen; k ++) {
        if (ppath[k].tid == i && ppath[k].sid == j) {
          fprintf(fp, "%f ", matrix[i][j]);
          break;
        }
      }
      if (k == pathlen) {
        fprintf(fp, "  ");
      }
    }
    fprintf(fp, "\n");
  }
  fprintf(fp, "\n\n");
  fclose(fp);
}
#endif

static double
calcWarpingDistance(double **matrix, PATH *ppath, int pathlen, int windowSize)
{
  int i;
  int j;
  int k;
  double distance = 0.0;
  
  for (i = 0; i < windowSize; i ++) {
    for (j = 0; j < windowSize; j ++) {
      for (k = 0; k < pathlen; k ++) {
        if (ppath[k].tid == i && ppath[k].sid == j) {
	  distance += (matrix[i][j] * matrix[i][j]);
          break;
        }
      }
    }
  }
  distance = (sqrt(distance)/((double)pathlen));
  return distance;
}

static void
setMaxPathLength(int windowSize)
{
  MaxPathLength = windowSize * 2;
}

static double **
initMatrix(int windowSize)
{
  int i;
  double **matrix;

  if ((matrix = (double **)calloc(windowSize, sizeof(double *))) == NULL) ERR;
  for (i = 0; i < windowSize; i ++) {
    if ((matrix[i] = (double *)calloc(windowSize, sizeof(double))) == NULL) ERR;
  }

  return matrix;
}

#ifdef PRINT_MATRIX
static void
printDistance(double distance)
{
  FILE *fp;

  if ((fp = fopen(MATRIX_LOGFILE, "a")) == NULL) ERR;
  fprintf(fp, "\ndistance: %f\n", distance);
  fclose(fp);
}
#endif

static void
freeWarpingArea(double **matrix, int windowSize, PATH *ppath) 
{
  int i;

  for (i = 0; i < windowSize; i ++) {
    free(matrix[i]);
  }
  free(matrix);
  free(ppath);
}

static double
invokeCalcWarping(double s[], double t[], int windowSize)
{
  int pathlen;
  double distance;
  double **matrix;
  PATH *ppath;  

  setMaxPathLength(windowSize);
  matrix = initMatrix(windowSize);
  setMatrix(matrix, s, t, windowSize);

#ifdef PRINT_MATRIX
  printMatrix(matrix, windowSize);
#endif

  ppath = createWarpingPath(matrix, &pathlen, windowSize);
  printWarpingPath(matrix, ppath, pathlen, windowSize);
  distance = calcWarpingDistance(matrix, ppath, pathlen, windowSize);

#ifdef PRINT_MATRIX
  printDistance(distance);
#endif  

  freeWarpingArea(matrix, windowSize, ppath);

  return distance;
}

extern double
getWarpingDistance(WINDOW *lwP, WINDOW *rwP)
{
  double distance;
  return distance;
}
