#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <netdb.h>
#include <signal.h>
#include <readline/readline.h>
#include <readline/history.h>

#include "port.h"
#include "host.h"
#include "common.h" 
#include "libWrapper.h"
#include "buffsize.h"
#include "sensorImageSize.h"
#include "slotsize.h"
#include "extern.h"
#include "ipc.h"

#ifndef _SCHEMA_H
#define _SCHEMA_H
#include "schema.h"
#endif

#ifndef _SELECT_FROM_H
#define _SELECT_FROM_H
#include "selectFrom.h"
#endif

extern int connectKraft(void);
extern void printResultForRelation(ANSWER answer);
extern void queryToArtemis(const int sockfd, const char query[]);
extern void sendExitAndCloseSockfd(const int sockfd);
extern void sendQuery(const char query[], const int sockfd);
extern void printResultOfOperation(const TYPE_OF_ANSWER typeOfAnswer);
extern void sendback(const int sockfd, const void *ptr, const int size, const int flag);
extern void recvback(const int sockfd, void *ptr, const int size, const int flag);
extern BOOLEAN isThisAppend(const char query[]);
extern TYPE_OF_ANSWER recvTypeOfAnswer(const int sockfd);

/***************************************************************
 *
 * Private Function
 *
 ***************************************************************/
static void
releaseSensorForClient(SENSOR *pHeadSensor)
{
  SENSOR *pSensor;

  while (pHeadSensor->next != NULL) {
    pSensor = pHeadSensor->next;
    pHeadSensor->next = pHeadSensor->next->next;
    free(pSensor);
  }
}

static void
releaseSensorSegmentForClient(SENSOR_SEGMENT *pHeadSensorSegment)
{
  SENSOR_SEGMENT *trash;

  while (pHeadSensorSegment->next != NULL) {
    trash = pHeadSensorSegment->next;
    pHeadSensorSegment->next = pHeadSensorSegment->next->next;
    releaseSensorForClient(&(trash->headSensor));
    free(trash);
  }
}

static void
releaseBaseTupleForClient(TUPLE *pHeadTuple, const ATTR attr[], const int numOfAttr)
{
  int attrid;
  int byteOffset;
  TUPLE *trash;
  SENSOR_SEGMENT *pSensorSegment;

  while (pHeadTuple->next != NULL) {
    trash = pHeadTuple->next;
    pHeadTuple->next = pHeadTuple->next->next;
    for (byteOffset = 0, attrid = 0; 
         attrid < numOfAttr; 
         byteOffset += attr[attrid].sizOfAttr, attrid ++) {
      switch (attr[attrid].attrType) {
      case SENSOR_SEGMENT_DOUBLE:
        pSensorSegment = (SENSOR_SEGMENT *)(trash->obj + byteOffset);
        releaseSensorSegmentForClient(pSensorSegment);
        break;
      default: 
        break;
      }
    }
    free(trash->obj);
    free(trash);
  }
}

static void
releaseTupleList(TUPLE_LIST *pHeadTupleList)
{
  TUPLE_LIST *pTupleList;

  while (pHeadTupleList->next != NULL) {
    pTupleList = pHeadTupleList->next;
    pHeadTupleList->next = pHeadTupleList->next->next;
    free(pTupleList);
  }
}

static void
releaseAnswerForClient(ANSWER answer)
{
  releaseTupleList(&(answer.headAndorTupleList));
  releaseBaseTupleForClient(&(answer.headTuple), answer.attr, answer.numOfAttr);
  free(answer.attr);
}

static void
writeLine(const int numOfAttr)
{
  int attrid;
  int iter;

  printf("+");
  for (attrid = 0; attrid < numOfAttr; attrid ++) {
    printf("-");
    for (iter = 0; iter < 8; iter ++) {
      printf("-");
    }
    printf("-+");
  }
  printf("\n");
}

static void
printAttribute(const ANSWER answer)
{
  int attrid;

  writeLine(answer.numOfAttr);
  printf("| ");
  for (attrid = 0; attrid < answer.numOfAttr; attrid ++) {
    printf("%8s | ", answer.attr[attrid].attrName);
  }
  printf("\n");
  writeLine(answer.numOfAttr);
}  

static void
printSensor(const SENSOR headSensor, const ATTR_TYPE attrType)
{
  SENSOR *p;

  for (p = headSensor.next; p != NULL; p = p->next) {
    //printf("{");
    switch(attrType) {
    case SENSOR_SEGMENT_DOUBLE: 
      printf("{%3.1f}", p->sdobj.value);
      break;
    default: 
      ERR;
      break;
    }
    //printf("{%ld, %ld}", pSensor->sdobj.arrival.tv_sec, pSensor->sdobj.arrival.tv_usec);
    //printf("}");
  }
}

static void
printSensorSegment(const SENSOR_SEGMENT headSensorSegment, const ATTR_TYPE attrType)
{
  SENSOR_SEGMENT *p;

  for (p = headSensorSegment.next; p != NULL; p = p->next) {
    printf("=== ");
    switch(attrType) {
    case SENSOR_SEGMENT_DOUBLE: 
      printSensor(p->headSensor, SENSOR_SEGMENT_DOUBLE); 
      break;
    default: 
      break;
    }
    printf(" ===");
  }
  printf(" | ");
}

/***************************************************************
 *
 * Public Function
 *
 ***************************************************************/
extern int
connectKraftBySensor(void)
{
  int sockfd;

  sockfd = connectCli(DBMS_SENSOR_PORT, DBMS_HOST);
  return sockfd;
}

extern int
connectKraftByClient(void)
{
  int sockfd;

  sockfd = connectCli(DBMS_CLIENT_PORT, DBMS_HOST);
  return sockfd;
}

extern void 
sendQuery(const char query[], const int sockfd)
{
  if (send(sockfd, query, QUERY_BUFFSIZE, 0) == -1) ERR;
}

extern TYPE_OF_ANSWER
recvTypeOfAnswer(const int sockfd) 
{
  TYPE_OF_ANSWER typeOfAnswer;

  if (recv(sockfd, &typeOfAnswer, sizeof(TYPE_OF_ANSWER), 0) == -1) ERR;
  return typeOfAnswer;
}

extern void
sendExitAndCloseSockfd(const int sockfd)
{
  char query[QUERY_BUFFSIZE];

  bzero(query, QUERY_BUFFSIZE);
  strncpy(query, "exit", strlen("exit"));
  sendback(sockfd, query, QUERY_BUFFSIZE, 0);
  if (close(sockfd) == -1) ERR;
}

extern BOOLEAN
isThisAppend(const char query[])
{
  int qid;
  BOOLEAN thisIsAppend = FALSE;

  for (qid = 0; qid < (int)(QUERY_BUFFSIZE - strlen("append")); qid ++) {
    if (strncmp(&query[qid], "append", strlen("append")) == 0) {
      thisIsAppend = TRUE;
      break;
    }
  }

  return thisIsAppend;
}

extern void
printResultForRelation(ANSWER answer)
{
  int attrid;
  size_t siz;
  char textData[BUFFSIZE];
  /*
   * int intData;
   * double doubleData;
   */
  TUPLE *p;
  SENSOR_SEGMENT headSensorSegment;

  printAttribute(answer);
  for (p = answer.headTuple.next; p != NULL; p = p->next) {
    printf("| ");
    for (siz = attrid = 0; attrid < answer.numOfAttr; siz += answer.attr[attrid].sizOfAttr, attrid ++) {
      switch(answer.attr[attrid].attrType) {
      case INT:    
        /*
         * memcpy(&intData, pTuple->obj + siz, sizeof(int));
         * printf("%8d | ", intData);    
         */
        printf("%8d | ", *(int *)(size_t)p->obj + siz);
        break;
      case DOUBLE: 
        /*
         * memcpy(&doubleData, pTuple->obj + siz, sizeof(double));
         * printf("%8.1f | ", doubleData);    
         */
        printf("%8.1f | ", *(double *)(size_t)p->obj + siz);
        break;
      case TEXT:   
        bzero(textData, BUFFSIZE);
        strncpy(textData, p->obj + siz, BUFFSIZE);
        printf("%8s | ", textData); fflush(stdout);
        break;
      case SENSOR_SEGMENT_DOUBLE:  
        memcpy(&headSensorSegment, p->obj + siz, sizeof(SENSOR_SEGMENT));
        printSensorSegment(headSensorSegment, SENSOR_SEGMENT_DOUBLE); 
        break;
      default:
        ERR;
        break;
      }
    }
    printf("\n");
  }
  writeLine(answer.numOfAttr);
  releaseAnswerForClient(answer);
}  

extern ANSWER
unMarshalize(char *packetP, const int sizeOfPacket)
{
  int offset, byteOffset;
  METACOMM *metaCommP;
  ANSWER answer;
  TUPLE *tupleP;
  SENSOR *sensorP;
  SENSOR_SEGMENT *sensorSegmentP;
  
  offset = 0;
  while (offset < sizeOfPacket) {
    metaCommP = (METACOMM *)(packetP + offset);
    offset += sizeof(METACOMM);
    switch(metaCommP->commType) {
    case TYPE_ANSWER:
      memcpy(&answer, (packetP+offset), sizeof(ANSWER));
      tupleP = &(answer.headTuple);
      tupleP->next = NULL;
      offset += sizeof(ANSWER);
      break;
    case TYPE_ATTR:
      if ((answer.attr = calloc(metaCommP->numOfObj, sizeof(ATTR))) == NULL) ERR;
      memcpy(answer.attr, (packetP+offset), (sizeof(ATTR) * metaCommP->numOfObj));
      offset += (sizeof(ATTR) * metaCommP->numOfObj);
      break;
    case TYPE_RELATION:
      if ((tupleP->next = calloc(1, sizeof(TUPLE))) == NULL) ERR;
      tupleP->next->prev = tupleP;
      tupleP = tupleP->next;
      if ((tupleP->obj = calloc(1, answer.byteOfTuple)) == NULL) ERR;
      memcpy(tupleP->obj, (packetP+offset), answer.byteOfTuple);
      offset += answer.byteOfTuple;
      break;
    case TYPE_SENSOR_SEGMENT:
      sensorSegmentP = (SENSOR_SEGMENT*)(packetP+offset);
      byteOffset = sensorSegmentP->offset;
      sensorSegmentP = (SENSOR_SEGMENT*)(tupleP->obj + byteOffset);
      if ((sensorSegmentP->next = calloc(1, sizeof(SENSOR_SEGMENT))) == NULL) ERR;
      sensorSegmentP = sensorSegmentP->next;
      memcpy(sensorSegmentP, (packetP+offset), sizeof(SENSOR_SEGMENT));
      sensorP = &(sensorSegmentP->headSensor);
      offset += sizeof(SENSOR_SEGMENT);
      break;
    case TYPE_SENSOR:
      if ((sensorP->next = calloc(1, sizeof(SENSOR))) == NULL) ERR;
      sensorP = sensorP->next;
      memcpy(&sensorP->sdobj, packetP+offset, sizeof(SDOBJ));
      offset += sizeof(SDOBJ);      
      break;
    }
  }

  return answer;
}

extern ANSWER
recvAnswer(const int sockfd)
{
  ANSWER answer;
  int sizeOfPacket;
  char *packetP;

  if (recv(sockfd, &sizeOfPacket, sizeof(int), MSG_WAITALL) == -1) ERR;
  if ((packetP = calloc(1, sizeOfPacket)) == NULL) ERR;
  if (recv(sockfd, packetP, sizeOfPacket, MSG_WAITALL) == -1) ERR;
  answer = unMarshalize(packetP, sizeOfPacket);
  free(packetP);
  return answer;
}

extern void
printResultForSchema(const int sockfd)
{
  char schemaInfo[BUFFSIZE];
  BOOLEAN finish;

  printf("--------------------\n");
  while (1) {
    recvback(sockfd, &finish, sizeof(BOOLEAN), 0);
    if (finish == TRUE) {
      break;
    }
    else if (finish == FALSE) {
      bzero(schemaInfo, BUFFSIZE);
      recvback(sockfd, schemaInfo, BUFFSIZE, 0);
      printf("%s\n", schemaInfo);
    }
  }
  printf("--------------------\n");
}

extern void
printResultOfOperation(const TYPE_OF_ANSWER typeOfAnswer)
{
  switch(typeOfAnswer) {
  case ANSWER_APPEND:  
#ifndef EVAL
    printf("APPEND\n");
#endif
    break; 
  case ANSWER_BEGIN: 
    printf("BEGIN\n"); 
    break;
  case ANSWER_COMMIT:  
    printf("COMMIT\n"); 
    break;
  case ANSWER_CREATE_INDEX: 
    printf("CREATE INDEX\n"); 
    break;
  case ANSWER_CREATE_MONITOR: 
    printf("CREATE MONITOR\n");  
    break;
  case ANSWER_CREATE_TABLE: 
    printf("CREATE TABLE\n");    
    break;
  case ANSWER_DELETE:   
    printf("DELETE\n");
    break;
  case ANSWER_DROP_TABLE: 
    printf("DROP TABLE\n");                                 
    break;
  case ANSWER_DROP_INDEX: 
    printf("DROP INDEX\n");                                 
    break;
  case ANSWER_INSERT: 
    printf("INSERT\n");                                     
    break;
  case ANSWER_NOT_BEGIN: 
    printf("You must specify BEGIN at first\n");            
    break;
  case ANSWER_PARSE_ERROR: 
    printf("Sorry, parse error. Please try again\n");       
    break;
  case ANSWER_ROLLBACK_USER_REQUEST: 
    printf("ROLLBACK AS YOU WANT\n");                       
    break;
  case ANSWER_ROLLBACK_NO_BUFFERPOOL: 
    printf("ROLLBACK SINCE NO SPACE LEFT ON BUFFERPOOL\n"); 
    break;
  case ANSWER_RELATION: 
    N;
  case ANSWER_SHOW_SCHEMA:            
    N;
  case ANSWER_SHOW_MONITOR:           
    N;
  case ANSWER_FINISH_MONITOR_RUN: 
    N;
  case ANSWER_UNACCEPTABLE: 
    N;
  default:
    printf("I'm sorry, but I couldn't understand what you want...\n"); 
    break;
  }
}

extern void
getResultOfAdhocQuery(const int sockfd)
{
  TYPE_OF_ANSWER typeOfAnswer;
  ANSWER answer;

  typeOfAnswer = recvTypeOfAnswer(sockfd);
  switch(typeOfAnswer) {
  case ANSWER_RELATION:    
    answer = recvAnswer(sockfd);
    printResultForRelation(answer); 
    break;
  case ANSWER_SHOW_MONITOR: 
    printResultForSchema(sockfd);   
    break;
  case ANSWER_SHOW_SCHEMA:  
    printResultForSchema(sockfd);   
    break;
  default:
    printResultOfOperation(typeOfAnswer);            
    break;
  }
}
