﻿// This file is part of Notepad++ project
// Copyright (C)2003 Don HO <don.h@free.fr>
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// Note that the GPL places important restrictions on "derived works", yet
// it does not provide a detailed definition of that term.  To avoid
// misunderstandings, we consider an application to constitute a
// "derivative work" for the purpose of this license if it does any of the
// following:
// 1. Integrates source code from Notepad++.
// 2. Integrates/includes/aggregates Notepad++ into a proprietary executable
//    installer, such as those produced by InstallShield.
// 3. Links to a library or executes a program that does any of the above.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#define NO_TRY_CATCH	//[JOJO]
#define MOD_FILE_TOO_BIG_MSG	//[JOJO]

#include <deque>
#include <algorithm>
#include <time.h>
#include <sys/stat.h>
#include "Buffer.h"
#include "Scintilla.h"
#include "Parameters.h"
#include "Notepad_plus.h"
#include "ScintillaEditView.h"
#include "EncodingMapper.h"
#include "uchardet.h"
#include "LongRunningOperation.h"
#include "Notepad_plus_Window.h"	//+[JOJO] Notepad_plus_Window::gNppHWND

FileManager * FileManager::_pSelf = new FileManager();

static const int blockSize = 128 * 1024 + 4;
static const int CR = 0x0D;
static const int LF = 0x0A;

long Buffer::_recentTagCtr = 0;






namespace // anonymous
{

	static EolType getEOLFormatForm(const char* const data, size_t length, EolType defvalue = EolType::osdefault)
	{
		assert(length == 0 or data != nullptr && "invalid buffer for getEOLFormatForm()");

		for (size_t i = 0; i != length; ++i)
		{
			if (data[i] == CR)
			{
				if (i + 1 < length && data[i + 1] == LF)
					return EolType::windows;

				return EolType::macos;
			}

			if (data[i] == LF)
				return EolType::unix;
		}

		return defvalue; // fallback unknown
	}


} // anonymous namespace




Buffer::Buffer(FileManager * pManager, BufferID id, Document doc, DocFileStatus type, const TCHAR *fileName)
	// type must be either DOC_REGULAR or DOC_UNNAMED
	: _pManager(pManager)
	, _id(id)
	, _doc(doc)
	, _lang(L_TEXT)
{
	NppParameters* pNppParamInst = NppParameters::getInstance();
	const NewDocDefaultSettings& ndds = (pNppParamInst->getNppGUI()).getNewDocDefaultSettings();

	_eolFormat = ndds._format;
	_unicodeMode = ndds._unicodeMode;
	_encoding = ndds._codepage;
	if (_encoding != -1)
		_unicodeMode = uniCookie;
#ifdef INTERNAL_ENCODING_UTF8N
	assert(_unicodeMode != uni7Bit);
	assert(_unicodeMode != uni8Bit);
#endif

	_currentStatus = type;

	setFileName(fileName, ndds._lang);
	updateTimeStamp();
	checkFileState();
#ifdef MOD_USER_READONLY
	setUserReadOnly(getFileReadOnly());
#endif

	// reset after initialization
	_isDirty   = false;
	_canNotify = true;
	_needLexer = false; // new buffers do not need lexing, Scintilla takes care of that
}


void Buffer::doNotify(int mask)
{
	if (_canNotify)
	{
		assert(_pManager != nullptr);
		_pManager->beNotifiedOfBufferChange(this, mask);
	}
}


void Buffer::setDirty(bool dirty)
{
	_isDirty = dirty;
	doNotify(BufferChangeDirty);
}


void Buffer::setEncoding(int encoding)
{
	_encoding = encoding;
	doNotify(BufferChangeUnicode | BufferChangeDirty);
}


void Buffer::setUnicodeMode(UniMode mode)
{
#ifdef INTERNAL_ENCODING_UTF8N
	assert(_unicodeMode != uni7Bit);
	assert(_unicodeMode != uni8Bit);
	assert(mode != uni7Bit);
	assert(mode != uni8Bit);
#endif
	_unicodeMode = mode;
	doNotify(BufferChangeUnicode | BufferChangeDirty);
}


void Buffer::setLangType(LangType lang, const TCHAR* userLangName)
{
	if (lang == _lang && lang != L_USER)
		return;

	_lang = lang;
	if (_lang == L_USER)
		_userLangExt = userLangName;

	_needLexer = true;	//change of lang means lexern needs updating
	doNotify(BufferChangeLanguage|BufferChangeLexing);
}


void Buffer::updateTimeStamp()
{
	struct _stat buf;
	time_t timeStamp = (generic_stat(_fullPathName.c_str(), &buf)==0)?buf.st_mtime:0;

	if (timeStamp != _timeStamp)
	{
		_timeStamp = timeStamp;
		doNotify(BufferChangeTimestamp);
	}
}


// Set full path file name in buffer object,
// and determinate its language by its extension.
// If the ext is not in the list, the defaultLang passed as argument will be set.
void Buffer::setFileName(const TCHAR *fn, LangType defaultLang)
{
	NppParameters *pNppParamInst = NppParameters::getInstance();
	if (_fullPathName == fn)
	{
		updateTimeStamp();
		doNotify(BufferChangeTimestamp);
		return;
	}

	_fullPathName = fn;
	_fileName = PathFindFileName(_fullPathName.c_str());

	// for _lang
	LangType newLang = defaultLang;
	TCHAR *ext = PathFindExtension(_fullPathName.c_str());
	if (*ext == '.') // extension found
	{
		ext += 1;

		// Define User Lang firstly
		const TCHAR* langName = pNppParamInst->getUserDefinedLangNameFromExt(ext, _fileName);
		if (langName)
		{
			newLang = L_USER;
			_userLangExt = langName;
		}
		else // if it's not user lang, then check if it's supported lang
		{
			_userLangExt.clear();
			newLang = pNppParamInst->getLangFromExt(ext);
		}
	}

	if (newLang == defaultLang || newLang == L_TEXT)	//language can probably be refined
	{
		if ((!generic_stricmp(_fileName, TEXT("makefile"))) || (!generic_stricmp(_fileName, TEXT("GNUmakefile"))))
			newLang = L_MAKEFILE;
		else if (!generic_stricmp(_fileName, TEXT("CmakeLists.txt")))
			newLang = L_CMAKE;
		else if ((!generic_stricmp(_fileName, TEXT("SConstruct"))) || (!generic_stricmp(_fileName, TEXT("SConscript"))) || (!generic_stricmp(_fileName, TEXT("wscript"))))
			newLang = L_PYTHON;
		else if (!generic_stricmp(_fileName, TEXT("Rakefile")))
			newLang = L_RUBY;
	}

	updateTimeStamp();
	if (newLang != _lang || _lang == L_USER)
	{
		_lang = newLang;
		doNotify(BufferChangeFilename | BufferChangeLanguage | BufferChangeTimestamp);
		return;
	}

	doNotify(BufferChangeFilename | BufferChangeTimestamp);
}


bool Buffer::checkFileState() //eturns true if the status has been changed (it can change into DOC_REGULAR too). false otherwise
{
 	if (_currentStatus == DOC_UNNAMED)	//unsaved document cannot change by environment
		return false;

	struct _stat buf;
	bool isWow64Off = false;
	NppParameters *pNppParam = NppParameters::getInstance();

	if (not PathFileExists(_fullPathName.c_str()))
	{
		pNppParam->safeWow64EnableWow64FsRedirection(FALSE);
		isWow64Off = true;
	}

	bool isOK = false;
	if (_currentStatus != DOC_DELETED && not PathFileExists(_fullPathName.c_str()))	//document has been deleted
	{
		_currentStatus = DOC_DELETED;
		_isFileReadOnly = false;
		_isDirty = true;	//dirty sicne no match with filesystem
		_timeStamp = 0;
		doNotify(BufferChangeStatus | BufferChangeReadonly | BufferChangeTimestamp);
		isOK = true;
	}
	else if (_currentStatus == DOC_DELETED && PathFileExists(_fullPathName.c_str()))
	{	//document has returned from its grave
		if (not generic_stat(_fullPathName.c_str(), &buf))
		{
			_isFileReadOnly = (bool)(!(buf.st_mode & _S_IWRITE));

			_currentStatus = DOC_MODIFIED;
			_timeStamp = buf.st_mtime;
			doNotify(BufferChangeStatus | BufferChangeReadonly | BufferChangeTimestamp);
			isOK = true;
		}
	}
	else if (not generic_stat(_fullPathName.c_str(), &buf))
	{
		int mask = 0;	//status always 'changes', even if from modified to modified
		bool isFileReadOnly = (bool)(not(buf.st_mode & _S_IWRITE));
		if (isFileReadOnly != _isFileReadOnly)
		{
			_isFileReadOnly = isFileReadOnly;
			mask |= BufferChangeReadonly;
		}
		if (_timeStamp != buf.st_mtime)
		{
			_timeStamp = buf.st_mtime;
			mask |= BufferChangeTimestamp;
			_currentStatus = DOC_MODIFIED;
			mask |= BufferChangeStatus;	//status always 'changes', even if from modified to modified
		}

		if (mask != 0)
		{
			doNotify(mask);
			isOK = true;
		}
		isOK = false;
	}

	if (isWow64Off)
	{
		pNppParam->safeWow64EnableWow64FsRedirection(TRUE);
	}
	return isOK;
}

void Buffer::reload()
{
	struct _stat buf;
	if (PathFileExists(_fullPathName.c_str()) && not generic_stat(_fullPathName.c_str(), &buf))
	{
		_timeStamp = buf.st_mtime;
		_currentStatus = DOC_NEEDRELOAD;
		doNotify(BufferChangeTimestamp | BufferChangeStatus);
	}
}

int Buffer::getFileLength() const
{
	if (_currentStatus == DOC_UNNAMED)
		return -1;

	struct _stat buf;

	if (PathFileExists(_fullPathName.c_str()))
	{
		if (!generic_stat(_fullPathName.c_str(), &buf))
			return buf.st_size;
	}
	return -1;
}


generic_string Buffer::getFileTime(fileTimeType ftt) const
{
	if (_currentStatus == DOC_UNNAMED)
		return generic_string();

	struct _stat buf;

	if (PathFileExists(_fullPathName.c_str()))
	{
		if (!generic_stat(_fullPathName.c_str(), &buf))
		{
			time_t rawtime = (ftt == ft_created ? buf.st_ctime : (ftt == ft_modified ? buf.st_mtime : buf.st_atime));
			tm *timeinfo = localtime(&rawtime);
			const int temBufLen = 64;
			TCHAR tmpbuf[temBufLen];

			generic_strftime(tmpbuf, temBufLen, TEXT("%Y-%m-%d %H:%M:%S"), timeinfo);
			return tmpbuf;
		}
	}

	return generic_string();
}


void Buffer::setPosition(const Position & pos, ScintillaEditView * identifier)
{
	int index = indexOfReference(identifier);
	if (index == -1)
		return;
	_positions[index] = pos;
}


Position& Buffer::getPosition(ScintillaEditView* identifier)
{
	int index = indexOfReference(identifier);
	return _positions.at(index);
}


void Buffer::setHeaderLineState(const std::vector<size_t> & folds, ScintillaEditView * identifier)
{
	int index = indexOfReference(identifier);
	if (index == -1)
		return;

	//deep copy
	std::vector<size_t> & local = _foldStates[index];
	local.clear();
	size_t size = folds.size();
	for(size_t i = 0; i < size; ++i)
		local.push_back(folds[i]);
}


const std::vector<size_t> & Buffer::getHeaderLineState(const ScintillaEditView * identifier) const
{
	int index = indexOfReference(identifier);
	return _foldStates.at(index);
}


Lang * Buffer::getCurrentLang() const
{
	NppParameters *pNppParam = NppParameters::getInstance();
	int i = 0;
	Lang *l = pNppParam->getLangFromIndex(i);
	++i;
	while (l)
	{
		if (l->_langID == _lang)
			return l;

		l = pNppParam->getLangFromIndex(i);
		++i;
	}
	return nullptr;
}


int Buffer::indexOfReference(const ScintillaEditView * identifier) const
{
	size_t size = _referees.size();
	for (size_t i = 0; i < size; ++i)
	{
		if (_referees[i] == identifier)
			return static_cast<int>(i);
	}
	return -1;	//not found
}


int Buffer::addReference(ScintillaEditView * identifier)
{
	if (indexOfReference(identifier) != -1)
		return _references;

	_referees.push_back(identifier);
	_positions.push_back(Position());
	_foldStates.push_back(std::vector<size_t>());
	++_references;
	return _references;
}


int Buffer::removeReference(ScintillaEditView * identifier)
{
	int indexToPop = indexOfReference(identifier);
	if (indexToPop == -1)
		return _references;

	_referees.erase(_referees.begin() + indexToPop);
	_positions.erase(_positions.begin() + indexToPop);
	_foldStates.erase(_foldStates.begin() + indexToPop);
	_references--;
	return _references;
}


void Buffer::setHideLineChanged(bool isHide, int location)
{
	//First run through all docs without removing markers
	for(int i = 0; i < _references; ++i)
		_referees.at(i)->notifyMarkers(this, isHide, location, false); // (i == _references-1));

	if (!isHide) // no deleting if hiding lines
	{
		//Then all docs to remove markers.
		for(int i = 0; i < _references; ++i)
			_referees.at(i)->notifyMarkers(this, isHide, location, true);
	}
}


void Buffer::setDeferredReload() // triggers a reload on the next Document access
{
	_isDirty = false;	//when reloading, just set to false, since it sohuld be marked as clean
	_needReloading = true;
	doNotify(BufferChangeDirty);
}


#ifdef MOD_EACH_VIEW_TABSIZE
void Buffer::setLastXChosen()
{
	for (int index = _positions.size(); --index >= 0;)
		_positions.at(index).lastXChosenD = -1.0;	// execute SCI_CHOOSECARETX
}
#endif


/*
pair<size_t, bool> Buffer::getLineUndoState(size_t currentLine) const
{
	for (size_t i = 0 ; i < _linesUndoState.size() ; i++)
	{
		if (_linesUndoState[i].first == currentLine)
			return _linesUndoState[i].second;
	}
	return pair<size_t, bool>(0, false);
}

void Buffer::setLineUndoState(size_t currentLine, size_t undoLevel, bool isSaved)
{
	bool found = false;
	for (size_t i = 0 ; i < _linesUndoState.size() ; i++)
	{
		if (_linesUndoState[i].first == currentLine)
		{
			_linesUndoState[i].second.first = undoLevel;
			_linesUndoState[i].second.second = isSaved;
		}
	}
	if (!found)
	{
		_linesUndoState.push_back(pair<size_t, pair<size_t, bool> >(currentLine, pair<size_t, bool>(undoLevel, false)));
	}
}
*/

//filemanager

FileManager::~FileManager()
{
	for (std::vector<Buffer *>::iterator it = _buffers.begin(), end = _buffers.end(); it != end; ++it)
	{
		delete *it;
	}
}

void FileManager::init(Notepad_plus * pNotepadPlus, ScintillaEditView * pscratchTilla)
{
	_pNotepadPlus = pNotepadPlus;
	_pscratchTilla = pscratchTilla;
	_pscratchTilla->execute(SCI_SETUNDOCOLLECTION, false);	//dont store any undo information
	_scratchDocDefault = (Document)_pscratchTilla->execute(SCI_GETDOCPOINTER);
	_pscratchTilla->execute(SCI_ADDREFDOCUMENT, 0, _scratchDocDefault);
}

void FileManager::checkFilesystemChanges()
{
	for (int i = int(_nrBufs) - 1; i >= 0 ; i--)
    {
        if (i >= int(_nrBufs))
        {
            if (_nrBufs == 0)
                return;

            i = int(_nrBufs) - 1;
        }
        _buffers[i]->checkFileState();	//something has changed. Triggers update automatically
	}
}


int FileManager::getBufferIndexByID(BufferID id)
{
	for(size_t i = 0; i < _nrBufs; ++i)
	{
		if (_buffers[i]->_id == id)
			return static_cast<int>(i);
	}
	return -1;
}

Buffer* FileManager::getBufferByIndex(size_t index)
{
	if (index >= _buffers.size())
		return nullptr;
	return _buffers.at(index);
}


void FileManager::beNotifiedOfBufferChange(Buffer* theBuf, int mask)
{
	_pNotepadPlus->notifyBufferChanged(theBuf, mask);
}


void FileManager::addBufferReference(BufferID buffer, ScintillaEditView * identifier)
{
	Buffer* buf = getBufferByID(buffer);
	buf->addReference(identifier);
}


void FileManager::closeBuffer(BufferID id, ScintillaEditView * identifier)
{
	int index = getBufferIndexByID(id);
	Buffer* buf = getBufferByIndex(index);

	int refs = buf->removeReference(identifier);

	if (!refs) // buffer can be deallocated
	{
		_pscratchTilla->execute(SCI_RELEASEDOCUMENT, 0, buf->_doc);	//release for FileManager, Document is now gone
		_buffers.erase(_buffers.begin() + index);
		delete buf;
		_nrBufs--;
	}
}


// backupFileName is sentinel of backup mode: if it's not NULL, then we use it (load it). Otherwise we use filename
BufferID FileManager::loadFile(const TCHAR * filename, Document doc, int encoding, const TCHAR *backupFileName, time_t fileNameTimestamp)
{
#ifdef SNAPSHOT_ALWAYS_UTF8
	const bool isSnapshotFile = backupFileName != NULL && PathFileExists(backupFileName);
	const bool autoDetectEncoding = encoding == -1;
	if (isSnapshotFile) assert(!autoDetectEncoding);
#endif
#ifdef MOD_SESSION_CODEPAGE
	Utf8_16_Read UnicodeConvertor;	//declare here so we can get information after loading is done
	if (encoding != -1) {/*TODO: if 無しでもいいのでは？*/
		const int en = SessionToBufEncode(encoding);
		const UniMode um = SessionToUniMode(encoding);
		encoding = en;
		UnicodeConvertor.setEncoding(um);
	}
#endif
	bool ownDoc = false;
	if (doc == NULL)
	{
		doc = (Document)_pscratchTilla->execute(SCI_CREATEDOCUMENT);
		ownDoc = true;
	}

	TCHAR fullpath[MAX_PATH];
	::GetFullPathName(filename, MAX_PATH, fullpath, NULL);
	if (_tcschr(fullpath, '~'))
	{
		::GetLongPathName(fullpath, fullpath, MAX_PATH);
	}

#ifdef SNAPSHOT_ALWAYS_UTF8
#else
	bool isSnapshotFile = backupFileName != NULL && PathFileExists(backupFileName);
#endif
	if (isSnapshotFile && !PathFileExists(fullpath)) // if backup mode and fullpath doesn't exist, we guess is UNTITLED
	{
		lstrcpy(fullpath, filename); // we restore fullpath with filename, in our case is "new  #"
	}

#ifdef MOD_SESSION_CODEPAGE
#else
	Utf8_16_Read UnicodeConvertor;	//declare here so we can get information after loading is done
#endif

	char data[blockSize + 8]; // +8 for incomplete multibyte char
	EolType bkformat = EolType::unknown;
	LangType detectedLang = L_TEXT;
#ifdef SNAPSHOT_ALWAYS_UTF8
	bool res;
	if (isSnapshotFile) {
		assert(!autoDetectEncoding);
		// Snapshot file format is always UTF-8 with BOM.
		// Do not auto detect encoding. But need call Utf8_16_Read::determineUniMode().
		Utf8_16_Read u; u.setEncoding(uniUTF8); // dummy. u.determineUniMode(u8"\uFEFF", _countof(u8"\uFEFF") - 1);
		int e = -1; // encoding dummy
		LangType l = L_TEXT; // dummy
		res = loadFileData(doc, backupFileName, data, &u, l, e, bkformat, false);
	} else {
		res = loadFileData(doc, fullpath, data, &UnicodeConvertor, detectedLang, encoding, bkformat, autoDetectEncoding);
	}
#else
	bool res = loadFileData(doc, backupFileName ? backupFileName : fullpath, data, &UnicodeConvertor, detectedLang, encoding, bkformat);
#endif
	if (res)
	{
		Buffer* newBuf = new Buffer(this, _nextBufferID, doc, DOC_REGULAR, fullpath);
		BufferID id = static_cast<BufferID>(newBuf);
		newBuf->_id = id;

		if (backupFileName != NULL)
		{
			newBuf->_backupFileName = backupFileName;
			if (!PathFileExists(fullpath))
				newBuf->_currentStatus = DOC_UNNAMED;
		}

		if (fileNameTimestamp != 0)
			newBuf->_timeStamp = fileNameTimestamp;

		_buffers.push_back(newBuf);
		++_nrBufs;
		Buffer* buf = _buffers.at(_nrBufs - 1);

#ifdef ORZ // encoding
#else
		// restore the encoding (ANSI based) while opening the existing file
		NppParameters *pNppParamInst = NppParameters::getInstance();
		const NewDocDefaultSettings & ndds = (pNppParamInst->getNppGUI()).getNewDocDefaultSettings();
		buf->setUnicodeMode(ndds._unicodeMode);
		buf->setEncoding(-1);
#endif

		// if no file extension, and the language has been detected,  we use the detected value
		if ((buf->getLangType() == L_TEXT) && (detectedLang != L_TEXT))
			buf->setLangType(detectedLang);

		if (encoding == -1)
		{
#ifdef ORZ // encoding
			buf->setEncoding(encoding);
#endif
			UniMode um = UnicodeConvertor.getEncoding();
#ifdef INTERNAL_ENCODING_UTF8N
			assert(um != uni7Bit);
			assert(um != uni8Bit);
#endif
#ifdef FIX_CODEPAGE_AUTODETECT
#else
			if (um == uni7Bit)
				um = (ndds._openAnsiAsUtf8) ? uniCookie : uni8Bit;

#endif
			buf->setUnicodeMode(um);
		}
		else // encoding != -1
		{
            // Test if encoding is set to UTF8 w/o BOM (usually for utf8 indicator of xml or html)
            buf->setEncoding((encoding == SC_CP_UTF8)?-1:encoding);
            buf->setUnicodeMode(uniCookie);
		}
#ifdef FIX_L_ASCII_ENCODING
		if (!autoDetectEncoding)
			buf->canEncodeAutoChange = false;
#endif

		buf->setEolFormat(bkformat);

		//determine buffer properties
		++_nextBufferID;
		return id;
	}
	else //failed loading, release document
	{
		if (ownDoc)
			_pscratchTilla->execute(SCI_RELEASEDOCUMENT, 0, doc);	//Failure, so release document
		return BUFFER_INVALID;
	}
}


bool FileManager::reloadBuffer(BufferID id)
{
	Buffer* buf = getBufferByID(id);
#ifdef MOD_CHANGE_ENCODING
	return reloadBuffer(buf, buf->getUnicodeMode(), buf->getEncoding());
#else
	Document doc = buf->getDocument();
	Utf8_16_Read UnicodeConvertor;
	buf->_canNotify = false;	//disable notify during file load, we dont want dirty to be triggered
	int encoding = buf->getEncoding();
	char data[blockSize + 8]; // +8 for incomplete multibyte char
	EolType bkformat;
	LangType lang = buf->getLangType();


	buf->setLoadedDirty(false);	// Since the buffer will be reloaded from the disk, and it will be clean (not dirty), we can set _isLoadedDirty false safetly.
								// Set _isLoadedDirty false before calling "_pscratchTilla->execute(SCI_CLEARALL);" in loadFileData() to avoid setDirty in SCN_SAVEPOINTREACHED / SCN_SAVEPOINTLEFT

	bool res = loadFileData(doc, buf->getFullPathName(), data, &UnicodeConvertor, lang, encoding, bkformat);
	buf->_canNotify = true;

	if (res)
	{
		if (encoding == -1)
		{
			buf->setUnicodeMode(UnicodeConvertor.getEncoding());
		}
		else
		{
			buf->setEncoding(encoding);
			buf->setUnicodeMode(uniCookie);
		}
	}
	return res;
#endif
}

#ifdef MOD_CHANGE_ENCODING
bool FileManager::reloadBuffer(Buffer* buf, const UniMode unicodeMode, int encoding)
{
	const bool isDirty = buf->_isDirty;
	Utf8_16_Read UnicodeConvertor;
	UnicodeConvertor.setEncoding(unicodeMode);
	char data[blockSize + 8]; // +8 for incomplete multibyte char
	EolType bkformat;// = buf->getEolFormat();
	LangType lang = buf->getLangType();

	buf->setLoadedDirty(false);	// Since the buffer will be reloaded from the disk, and it will be clean (not dirty), we can set _isLoadedDirty false safetly.
								// Set _isLoadedDirty false before calling "_pscratchTilla->execute(SCI_CLEARALL);" in loadFileData() to avoid setDirty in SCN_SAVEPOINTREACHED / SCN_SAVEPOINTLEFT

	buf->_canNotify = false;	//disable notify during file load, we dont want dirty to be triggered
	bool res = loadFileData(buf->getDocument(), buf->getFullPathName(), data, &UnicodeConvertor, lang, encoding, bkformat, false);
	buf->_canNotify = true;
#ifdef FIX_DETECT_EOL
	buf->setEolFormat(bkformat);
#endif
	if (isDirty != buf->_isDirty)
		buf->doNotify(BufferChangeDirty);	// trigger dirty. draw titlebar "*" and tabbar icon.
	return res;
}
#endif

bool FileManager::reloadBufferDeferred(BufferID id)
{
	Buffer* buf = getBufferByID(id);
	buf->setDeferredReload();
	return true;
}

bool FileManager::deleteFile(BufferID id)
{
	Buffer* buf = getBufferByID(id);
	generic_string fileNamePath = buf->getFullPathName();

	// Make sure to form a string with double '\0' terminator.
	fileNamePath.append(1, '\0');

	if (!PathFileExists(fileNamePath.c_str()))
		return false;
	//return ::DeleteFile(fileNamePath) != 0;

	SHFILEOPSTRUCT fileOpStruct = {0};
	fileOpStruct.hwnd = NULL;
	fileOpStruct.pFrom = fileNamePath.c_str();
	fileOpStruct.pTo = NULL;
	fileOpStruct.wFunc = FO_DELETE;
	fileOpStruct.fFlags = FOF_ALLOWUNDO;
	fileOpStruct.fAnyOperationsAborted = false;
	fileOpStruct.hNameMappings         = NULL;
	fileOpStruct.lpszProgressTitle     = NULL;

	return SHFileOperation(&fileOpStruct) == 0;
}


bool FileManager::moveFile(BufferID id, const TCHAR * newFileName)
{
	Buffer* buf = getBufferByID(id);
	const TCHAR *fileNamePath = buf->getFullPathName();
	if (::MoveFileEx(fileNamePath, newFileName, MOVEFILE_REPLACE_EXISTING) == 0)
		return false;

	buf->setFileName(newFileName);
	return true;
}


/*
Specs and Algorithm of session snapshot & periodic backup system:
Notepad++ quits without asking for saving unsaved file.
It restores all the unsaved files and document as the states they left.

For existing file (c:\tmp\foo.h)
	- Open
	In the next session, Notepad++
	1. load backup\FILENAME@CREATION_TIMESTAMP (backup\foo.h@198776) if exist, otherwise load FILENAME (c:\tmp\foo.h).
	2. if backup\FILENAME@CREATION_TIMESTAMP (backup\foo.h@198776) is loaded, set it dirty (red).
	3. if backup\FILENAME@CREATION_TIMESTAMP (backup\foo.h@198776) is loaded, last modif timestamp of FILENAME (c:\tmp\foo.h), compare with tracked timestamp (in session.xml).
	4. in the case of unequal result, tell user the FILENAME (c:\tmp\foo.h) was modified. ask user if he want to reload FILENAME(c:\tmp\foo.h)

	- Editing
	when a file starts being modified, a file will be created with name: FILENAME@CREATION_TIMESTAMP (backup\foo.h@198776)
	the Buffer object will associate with this FILENAME@CREATION_TIMESTAMP file (backup\foo.h@198776).
	1. sync: (each 3-5 second) backup file will be saved, if buffer is dirty, and modification is present (a bool on modified notificatin).
	2. sync: each save file, or close file, the backup file will be deleted (if buffer is not dirty).
	3. before switch off to another tab (or close files on exit), check 1 & 2 (sync with backup).

	- Close
	In the current session, Notepad++
	1. track FILENAME@CREATION_TIMESTAMP (backup\foo.h@198776) if exist (in session.xml).
	2. track last modified timestamp of FILENAME (c:\tmp\foo.h) if FILENAME@CREATION_TIMESTAMP (backup\foo.h@198776) was tracked  (in session.xml).

For untitled document (new  4)
	- Open
	In the next session, Notepad++
	1. open file UNTITLED_NAME@CREATION_TIMESTAMP (backup\new  4@198776)
	2. set label as UNTITLED_NAME (new  4) and disk icon as red.

	- Editing
	when a untitled document starts being modified, a backup file will be created with name: UNTITLED_NAME@CREATION_TIMESTAMP (backup\new  4@198776)
	the Buffer object will associate with this UNTITLED_NAME@CREATION_TIMESTAMP file (backup\new  4@198776).
	1. sync: (each 3-5 second) backup file will be saved, if buffer is dirty, and modification is present (a bool on modified notificatin).
	2. sync: if untitled document is saved, or closed, the backup file will be deleted.
	3. before switch off to another tab (or close documents on exit), check 1 & 2 (sync with backup).

	- CLOSE
	In the current session, Notepad++
	1. track UNTITLED_NAME@CREATION_TIMESTAMP (backup\new  4@198776) in session.xml.
*/
bool FileManager::backupCurrentBuffer()
{
	LongRunningOperation op;

	Buffer* buffer = _pNotepadPlus->getCurrentBuffer();
	bool result = false;
	bool hasModifForSession = false;

	if (buffer->isDirty())
	{
		if (buffer->isModified()) // buffer dirty and modified, write the backup file
		{
			// Synchronization
			// This method is called from 2 differents place, so synchronization is important
			HANDLE writeEvent = ::OpenEvent(EVENT_ALL_ACCESS, TRUE, TEXT("nppWrittingEvent"));
			if (not writeEvent)
			{
				// no thread yet, create a event with non-signaled, to block all threads
				writeEvent = ::CreateEvent(NULL, TRUE, FALSE, TEXT("nppWrittingEvent"));
				if (not writeEvent)
				{
					printStr(TEXT("CreateEvent problem in backupCurrentBuffer()!"));
					return false;
				}
			}
			else
			{
				if (::WaitForSingleObject(writeEvent, INFINITE) != WAIT_OBJECT_0)
				{
					printStr(TEXT("WaitForSingleObject problem in backupCurrentBuffer()!"));
					return false;
				}

				// unlocled here, set to non-signaled state, to block all threads
				if (not ::ResetEvent(writeEvent))
				{
					printStr(TEXT("ResetEvent problem in backupCurrentBuffer()!"));
					return false;
				}
			}

#ifdef SNAPSHOT_ALWAYS_UTF8
#else
			UniMode mode = buffer->getUnicodeMode();
			if (mode == uniCookie)
				mode = uni8Bit;	//set the mode to ANSI to prevent converter from adding BOM and performing conversions, Scintilla's data can be copied directly

			Utf8_16_Write UnicodeConvertor;
			UnicodeConvertor.setEncoding(mode);
			int encoding = buffer->getEncoding();
#endif

			generic_string backupFilePath = buffer->getBackupFileName();
			if (backupFilePath.empty())
			{
				// Create file
				backupFilePath = NppParameters::getInstance()->getUserPath();
				backupFilePath += TEXT("\\backup\\");

				// if "backup" folder doesn't exist, create it.
				if (!PathFileExists(backupFilePath.c_str()))
				{
					::CreateDirectory(backupFilePath.c_str(), NULL);
				}

				backupFilePath += buffer->getFileName();

				const int temBufLen = 32;
				TCHAR tmpbuf[temBufLen];
				time_t ltime = time(0);
				struct tm* today = localtime(&ltime);
				generic_strftime(tmpbuf, temBufLen, TEXT("%Y-%m-%d_%H%M%S"), today);

				backupFilePath += TEXT("@");
				backupFilePath += tmpbuf;

				// Set created file name in buffer
				buffer->setBackupFileName(backupFilePath);

				// Session changes, save it
				hasModifForSession = true;
			}

			TCHAR fullpath[MAX_PATH];
			::GetFullPathName(backupFilePath.c_str(), MAX_PATH, fullpath, NULL);
			if (_tcschr(fullpath, '~'))
			{
				::GetLongPathName(fullpath, fullpath, MAX_PATH);
			}

			// Make sure the backup file is not read only
			DWORD dwFileAttribs = ::GetFileAttributes(fullpath);
			if (dwFileAttribs & FILE_ATTRIBUTE_READONLY) // if file is read only, remove read only attribute
			{
#ifdef ORZ // FILE_ATTRIBUTE_READONLY
				dwFileAttribs &= ~FILE_ATTRIBUTE_READONLY;
#else
				dwFileAttribs ^= FILE_ATTRIBUTE_READONLY;
#endif
				::SetFileAttributes(fullpath, dwFileAttribs);
			}

#ifdef SNAPSHOT_ALWAYS_UTF8
			FILE *fp = _wfopen(fullpath, L"wb");
#else
			FILE *fp = UnicodeConvertor.fopen(fullpath, TEXT("wb"));
#endif
			if (fp)
			{
				int lengthDoc = _pNotepadPlus->_pEditView->getCurrentDocLen();
				char* buf = (char*)_pNotepadPlus->_pEditView->execute(SCI_GETCHARACTERPOINTER);	//to get characters directly from Scintilla buffer
#ifdef SNAPSHOT_ALWAYS_UTF8
				if (lengthDoc) fwrite(u8"\uFEFF", 1, _countof(u8"\uFEFF") - 1, fp); // UTF-8 BOM
				size_t items_written = fwrite(buf, 1, lengthDoc, fp);
				fclose(fp);
				if (items_written == (size_t) lengthDoc) // backup file has been saved
#else
				size_t items_written = 0;
				if (encoding == -1) //no special encoding; can be handled directly by Utf8_16_Write
				{
					items_written = UnicodeConvertor.fwrite(buf, lengthDoc);
					if (lengthDoc == 0)
						items_written = 1;
				}
				else
				{
					WcharMbcsConvertor *wmc = WcharMbcsConvertor::getInstance();
					int grabSize;
					for (int i = 0; i < lengthDoc; i += grabSize)
					{
						grabSize = lengthDoc - i;
						if (grabSize > blockSize)
							grabSize = blockSize;

						int newDataLen = 0;
						int incompleteMultibyteChar = 0;
						const char *newData = wmc->encode(SC_CP_UTF8, encoding, buf+i, grabSize, &newDataLen, &incompleteMultibyteChar);
						grabSize -= incompleteMultibyteChar;
						items_written = UnicodeConvertor.fwrite(newData, newDataLen);
					}
					if (lengthDoc == 0)
						items_written = 1;
				}
				UnicodeConvertor.fclose();

				// Note that fwrite() doesn't return the number of bytes written, but rather the number of ITEMS.
				if(items_written == 1) // backup file has been saved
#endif
				{
					buffer->setModifiedStatus(false);
					result = true;	//all done
				}
			}
			// set to signaled state
			if (::SetEvent(writeEvent) == NULL)
			{
				printStr(TEXT("oups!"));
			}
			// printStr(TEXT("Event released!"));
			::CloseHandle(writeEvent);
		}
		else // buffer dirty but unmodified
		{
			result = true;
		}
	}
	else // buffer not dirty, sync: delete the backup file
	{
		generic_string backupFilePath = buffer->getBackupFileName();
		if (not backupFilePath.empty())
		{
			// delete backup file
#ifdef ORZ // redundant
			buffer->setBackupFileName(generic_string());
			result = (::DeleteFile(backupFilePath.c_str()) != 0);
#else
			generic_string file2Delete = buffer->getBackupFileName();
			buffer->setBackupFileName(generic_string());
			result = (::DeleteFile(file2Delete.c_str()) != 0);
#endif

			// Session changes, save it
			hasModifForSession = true;
		}
		//printStr(TEXT("backup deleted in backupCurrentBuffer"));
		result = true; // no backup file to delete
	}
	//printStr(TEXT("backup sync"));

	if (result && hasModifForSession)
	{
		//printStr(buffer->getBackupFileName().c_str());
		_pNotepadPlus->saveCurrentSession();
	}
	return result;
}

class EventReset final
{
public:
	explicit EventReset(HANDLE h)
	{
		_h = h;
	}

	~EventReset()
	{
		::SetEvent(_h);
		::CloseHandle(_h);
	}

private:
	HANDLE _h;
};

bool FileManager::deleteCurrentBufferBackup()
{
	HANDLE writeEvent = ::OpenEvent(EVENT_ALL_ACCESS, TRUE, TEXT("nppWrittingEvent"));
	if (!writeEvent)
	{
		// no thread yet, create a event with non-signaled, to block all threads
		writeEvent = ::CreateEvent(NULL, TRUE, FALSE, TEXT("nppWrittingEvent"));
	}
	else
	{
		if (::WaitForSingleObject(writeEvent, INFINITE) != WAIT_OBJECT_0)
		{
			// problem!!!
			printStr(TEXT("WaitForSingleObject problem in deleteCurrentBufferBackup()!"));
			return false;
		}

		// unlocled here, set to non-signaled state, to block all threads
		::ResetEvent(writeEvent);
	}

	EventReset reset(writeEvent); // Will reset event in destructor.

	Buffer* buffer = _pNotepadPlus->getCurrentBuffer();
	bool result = true;
	generic_string backupFilePath = buffer->getBackupFileName();
	if (not backupFilePath.empty())
	{
		// delete backup file
		buffer->setBackupFileName(generic_string());
		result = (::DeleteFile(backupFilePath.c_str()) != 0);
	}

	// set to signaled state via destructor EventReset.
	return result;
}


#ifdef CHECK_ENCODE_BEFORE_SAVE
int FileManager::checkEncodeBuffer(BufferID id, const WCHAR *filename)
{
	const ScintillaEditView *pView = _pNotepadPlus->_pEditView;
	const Buffer *buffer = getBufferByID(id);
	const int encoding = buffer->getEncoding();
	if (encoding == -1) {
		switch (buffer->getUnicodeMode()) {
		case uniUTF8: case uniCookie:
			return CHECK_ENCODE_NOP;
		case uni16BE: case uni16LE: case uni16BE_NoBOM: case uni16LE_NoBOM:
			break;
		default:
			assert(true); return CHECK_ENCODE_NOP;	// ???
		}
	} else {
		assert(buffer->getUnicodeMode() == uniCookie);
	}

	const int lineCount = pView->execute(SCI_GETLINECOUNT);
	const unsigned char *buf = (unsigned char*) pView->execute(SCI_GETCHARACTERPOINTER);	//to get characters directly from Scintilla buffer
	WcharMbcsConvertor *wmc = WcharMbcsConvertor::getInstance();
	for (int lineNumber = 0; lineNumber < lineCount; ++lineNumber) {
		const int startPos   = pView->execute(SCI_POSITIONFROMLINE, lineNumber);
		const int lineLength = pView->execute(SCI_LINELENGTH, lineNumber);
		const unsigned char *linebuf = buf + startPos;

		const unsigned char *decodeData;
		int decodeLen = 0;
		if (encoding == -1) {
		 //	switch (buffer->getUnicodeMode()) {
		 //	case uni16BE: case uni16LE: case uni16BE_NoBOM: case uni16LE_NoBOM:
				decodeData = (unsigned char*) wmc->decode(65001/*CP_UTF8*/, (char*) linebuf, lineLength, &decodeLen);	// 65001 --> 1200(WCHAR) --> 65001
		 //		break;
		 //	default:
		 //		assert(true);	// ???
		 //	}
		} else {
			int encodeLen = 0;
			const char *encodeData =      wmc->encode(encoding, (char*) linebuf, lineLength, &encodeLen);	// 65001 --> WCHAR -> ANSI
			decodeData = (unsigned char*) wmc->decode(encoding, encodeData     , encodeLen , &decodeLen);	// ANSI --> WCHAR --> 65001
		}

		int xxOffset = -1;
		for (int i = 0; i < lineLength; ++i) {
			if (i >= decodeLen || linebuf[i] != decodeData[i]/* && decodeData[i] == '?'*/) {
				xxOffset = i;	// ?
				break;
			}
		}
		if (xxOffset >= 0) {
			const int columnNumber = pView->execute(SCI_GETCOLUMN, startPos + xxOffset);
//TODO:		int position = pView->execute(SCI_POSITIONBEFORE, startPos + xxOffset + 1);
//TODO:		int xxBytes  = pView->execute(*** SCI_CHAR_BYTECOUNT, position);				// int UTF8Classify(const unsigned char *us, int len)	-- scintilla/src/UniConversion.cxx(221)
//TODO:		uint32_t u   = pView->execute(*** SCI_UNICODE, position);						// int SCI_METHOD Document::GetCharacterAndWidth(int position, int *pWidth)	-- scintilla/src/Document.cxx(832)

			int xxBytes; {{ int i = xxOffset;
			while (i > 0 && UTF8IsTrailByte(buf[startPos + i])) --i;
			xxBytes = UTF8BytesOfLead[buf[startPos + i]];
			if (xxOffset < i + xxBytes) xxOffset = i; }}

			int position = startPos + xxOffset;
			xxBytes = UTF8BytesOfLead[buf[position]];
			if (xxOffset + xxBytes > lineLength) xxBytes = 1;

			std::wstring msg = filename;
			swsprintf(msg, L"\r\r行:%d  桁:%d  ", lineNumber + 1, columnNumber + 1);

			WCHAR wideCharStr[3]; wideCharStr[0] = wideCharStr[1] = wideCharStr[2] = L'\0';
			int wwLength = 0;
			MultiByteToWideChar(65001/*CP_UTF8*/, 0, (LPCSTR)(buf + position), xxBytes, wideCharStr, 2);
			uint32_t u = wideCharStr[0], v = wideCharStr[1];
			if (u == 0xFFFD) {
				swsprintf(msg, L"文字 0x%02X", buf[position]);
			}
			else if (0xD800 <= u && u < 0xDC00 && 0xDC00 <= v && v < 0xE000) {
				u = (u - 0xD800) * 0x0400 + (v - 0xDC00) + 0x10000;
				wwLength = 2;
				swsprintf(msg, L"文字[%s] U+%04X", wideCharStr, u);
			}
			else {
				wwLength = 1;
				swsprintf(msg, L"文字[%s] U+%04X", wideCharStr, u);
			}

			unsigned char ansiCharStr[10];
			WCHAR verifyWStr[3]; verifyWStr[0] = verifyWStr[1] = verifyWStr[2] = L'\0';
			int aaLength = WideCharToMultiByteN(encoding, wideCharStr, wwLength, (char*) ansiCharStr, _countof(ansiCharStr));
			int vvLength = MultiByteToWideCharN(encoding, (LPCSTR) ansiCharStr, aaLength, verifyWStr, 2);
			uint32_t j = verifyWStr[0], k = verifyWStr[1];
			if (0xD800 <= j && j < 0xDC00 && 0xDC00 <= k && k < 0xE000)
				j = (j - 0xD800) * 0x0400 + (k - 0xDC00) + 0x10000;
			if (vvLength > 0) {
				swsprintf(msg, L"\r    ⇒ 代替文字[%s] U+%04X (", verifyWStr, j);
				for (int i = 0; i < aaLength; ++i) swsprintf(msg, i == 0 ? L"%02X" : L" %02X", ansiCharStr[i]);
				msg += L')';
			}

			std::wstring ANSI;
			if (encoding == -1) {
				ANSI = L"UTF-16";
			} else {
				EncodingMapper *em = EncodingMapper::getInstance();
				int cmdID = em->getIndexFromEncoding(encoding);
				if (cmdID != -1) getNameStrFromCmd(IDM_FORMAT_ENCODE + cmdID, ANSI);
			}
			swsprintf(msg, L"\r\r このファイルは、%s テキストファイルとして保存すると失われてしまう Unicode 形式の文字を含んでいます。Unicode の情報を保存するには、下の [キャンセル] をクリックし、[文字コード] から [UTF-8] を選択してください。続行しますか?\r\rいいえ:該当文字にジャンプ", !ANSI.empty() ? ANSI.c_str() : L"ANSI");
		//	swsprintf(msg, L"\r\r このファイルは、%s テキストファイルとして保存すると失われてしまう Unicode 形式の文字を含んでいます。Unicode の情報を保存するには、下の [キャンセル] をクリックし、[文字コード] から [Unicode] を選択してください。続行しますか?\r\rいいえ:該当文字にジャンプ", !ANSI.empty() ? ANSI.c_str() : L"ANSI");
			switch (::MessageBoxW(Notepad_plus_Window::gNppHWND,
					msg.c_str(),
					L"Notepad++",
					MB_YESNOCANCEL | MB_ICONEXCLAMATION | MB_APPLMODAL | MB_SETFOREGROUND | MB_DEFBUTTON3)) {
			case IDYES:
				return CHECK_ENCODE_PASS;
			case IDNO:
				pView->execute(SCI_GOTOPOS, position);
				return CHECK_ENCODE_CANCEL;
			case IDCANCEL:
				return CHECK_ENCODE_CANCEL;
			default:
				assert(true);
			}
		}
	}

	return CHECK_ENCODE_NOP;
}
#endif

bool FileManager::saveBuffer(BufferID id, const TCHAR * filename, bool isCopy, generic_string * error_msg)
{
	HANDLE writeEvent = ::OpenEvent(EVENT_ALL_ACCESS, TRUE, TEXT("nppWrittingEvent"));
	if (!writeEvent)
	{
		// no thread yet, create a event with non-signaled, to block all threads
		writeEvent = ::CreateEvent(NULL, TRUE, FALSE, TEXT("nppWrittingEvent"));
	}
	else
	{		//printStr(TEXT("Locked. I wait."));
		if (::WaitForSingleObject(writeEvent, INFINITE) != WAIT_OBJECT_0)
		{
			// problem!!!
			printStr(TEXT("WaitForSingleObject problem in saveBuffer()!"));
			return false;
		}

		// unlocled here, set to non-signaled state, to block all threads
		::ResetEvent(writeEvent);
	}

	EventReset reset(writeEvent); // Will reset event in destructor.
	Buffer* buffer = getBufferByID(id);
	bool isHidden = false;
	bool isSys = false;
	DWORD attrib = 0;

	TCHAR fullpath[MAX_PATH];
	::GetFullPathName(filename, MAX_PATH, fullpath, NULL);
	if (_tcschr(fullpath, '~'))
	{
		::GetLongPathName(fullpath, fullpath, MAX_PATH);
	}

	if (PathFileExists(fullpath))
	{
		attrib = ::GetFileAttributes(fullpath);

		if (attrib != INVALID_FILE_ATTRIBUTES)
		{
			isHidden = (attrib & FILE_ATTRIBUTE_HIDDEN) != 0;
			if (isHidden)
				::SetFileAttributes(filename, attrib & ~FILE_ATTRIBUTE_HIDDEN);

			isSys = (attrib & FILE_ATTRIBUTE_SYSTEM) != 0;
			if (isSys)
				::SetFileAttributes(filename, attrib & ~FILE_ATTRIBUTE_SYSTEM);
		}
	}

	UniMode mode = buffer->getUnicodeMode();
#ifdef INTERNAL_ENCODING_UTF8N
	assert(mode != uni7Bit);
	assert(mode != uni8Bit);
#else
	if (mode == uniCookie)
		mode = uni8Bit;	//set the mode to ANSI to prevent converter from adding BOM and performing conversions, Scintilla's data can be copied directly
#endif

	Utf8_16_Write UnicodeConvertor;
	UnicodeConvertor.setEncoding(mode);

	int encoding = buffer->getEncoding();

#ifdef FIX_SYMBOLIC_LINK
	if (UnicodeConvertor.fopen(fullpath))
#else
	FILE *fp = UnicodeConvertor.fopen(fullpath, TEXT("wb"));
	if (fp)
#endif
	{
		_pscratchTilla->execute(SCI_SETDOCPOINTER, 0, buffer->_doc);	//generate new document

		int lengthDoc = _pscratchTilla->getCurrentDocLen();
		char* buf = (char*)_pscratchTilla->execute(SCI_GETCHARACTERPOINTER);	//to get characters directly from Scintilla buffer
		size_t items_written = 0;
		if (encoding == -1) //no special encoding; can be handled directly by Utf8_16_Write
		{
			items_written = UnicodeConvertor.fwrite(buf, lengthDoc);
			if (lengthDoc == 0)
				items_written = 1;
		}
		else
		{
#ifdef MLANG_DLL
			WcharMbcsConvertor *wmc = WcharMbcsConvertor::getInstance();
			DWORD mlangMode = 0;
			int grabSize;
			for (int i = 0; i < lengthDoc; i += grabSize)
			{
				const bool fEOF = i + blockSize >= lengthDoc;
				grabSize = lengthDoc - i;
				if (grabSize > blockSize)
					grabSize = blockSize;
				const int bbSize = grabSize;
				int newDataLen = 0;
				const char *newData = wmc->encodeStream(&mlangMode, encoding, buf + i, &grabSize, &newDataLen);
				int incompleteMultibyteChar = bbSize - grabSize;
				if (newDataLen != 0)
					items_written = UnicodeConvertor.fwrite(newData, newDataLen);
				if (fEOF) {	// [EOF] ?
					if (incompleteMultibyteChar != 0) {
						items_written += UnicodeConvertor.fwrite(buf + i + grabSize, incompleteMultibyteChar);
						grabSize += incompleteMultibyteChar;
						incompleteMultibyteChar = 0;
					}
				}
			}
			if (lengthDoc == 0)
				items_written = 1;
			wmc->free();
#else // MLANG_DLL
			WcharMbcsConvertor *wmc = WcharMbcsConvertor::getInstance();
			int grabSize;
			for (int i = 0; i < lengthDoc; i += grabSize)
			{
				grabSize = lengthDoc - i;
				if (grabSize > blockSize)
					grabSize = blockSize;

				int newDataLen = 0;
				int incompleteMultibyteChar = 0;
				const char *newData = wmc->encode(SC_CP_UTF8, encoding, buf+i, grabSize, &newDataLen, &incompleteMultibyteChar);
				grabSize -= incompleteMultibyteChar;
				items_written = UnicodeConvertor.fwrite(newData, newDataLen);
			}
			if (lengthDoc == 0)
				items_written = 1;
#endif // MLANG_DLL
		}
		
		// check the language du fichier
		LangType language = detectLanguageFromTextBegining((unsigned char *)buf, lengthDoc);

		UnicodeConvertor.fclose();

		// Error, we didn't write the entire document to disk.
		// Note that fwrite() doesn't return the number of bytes written, but rather the number of ITEMS.
		if(items_written != 1)
		{
			if(error_msg != NULL)
				*error_msg = TEXT("Failed to save file.\nNot enough space on disk to save file?");

			// set to signaled state via destructor EventReset.
			return false;
		}

		if (isHidden)
			::SetFileAttributes(fullpath, attrib | FILE_ATTRIBUTE_HIDDEN);

		if (isSys)
			::SetFileAttributes(fullpath, attrib | FILE_ATTRIBUTE_SYSTEM);

		if (isCopy)
		{
			_pscratchTilla->execute(SCI_SETDOCPOINTER, 0, _scratchDocDefault);

			/* for saveAs it's not necessary since this action is for the "current" directory, so we let manage in SAVEPOINTREACHED event
			generic_string backupFilePath = buffer->getBackupFileName();
			if (not backupFilePath.empty())
			{
				// delete backup file
				generic_string file2Delete = buffer->getBackupFileName();
				buffer->setBackupFileName(generic_string());
				::DeleteFile(file2Delete.c_str());
			}
			*/

			// set to signaled state via destructor EventReset.
			return true;	//all done
		}

		buffer->setFileName(fullpath, language);
		buffer->setDirty(false);
		buffer->setStatus(DOC_REGULAR);
		buffer->checkFileState();
		_pscratchTilla->execute(SCI_SETSAVEPOINT);
		_pscratchTilla->execute(SCI_SETDOCPOINTER, 0, _scratchDocDefault);

		generic_string backupFilePath = buffer->getBackupFileName();
		if (not backupFilePath.empty())
		{
			// delete backup file
			buffer->setBackupFileName(generic_string());
			::DeleteFile(backupFilePath.c_str());
		}

		// set to signaled state via destructor EventReset.
		return true;
	}
	// set to signaled state via destructor EventReset.
	return false;
}

size_t FileManager::nextUntitledNewNumber() const
{
	std::vector<size_t> usedNumbers;
	for(size_t i = 0; i < _buffers.size(); i++)
	{
		Buffer *buf = _buffers.at(i);
#ifdef MOD_NEWDOC_DIRECTORY
		if (buf->isUntitled() && wcsstartsWith(buf->_fileName, UNTITLED_STR))
#else
		if (buf->isUntitled())
#endif
		{
			// if untitled document is invisible, then don't put its number into array (so its number is available to be used)
			if ((buf->_referees[0])->isVisible())
			{
				TCHAR *numberStr = buf->_fileName + lstrlen(UNTITLED_STR);
				int usedNumber = generic_atoi(numberStr);
				usedNumbers.push_back(usedNumber);
			}
		}
	}

	size_t newNumber = 1;
	bool numberAvailable = true;
	bool found = false;
	do
	{
		for(size_t j = 0; j < usedNumbers.size(); j++)
		{
			numberAvailable = true;
			found = false;
			if (usedNumbers[j] == newNumber)
			{
				numberAvailable = false;
				found = true;
				break;
			}
		}
		if (!numberAvailable)
			newNumber++;

		if (!found)
			break;

	} while (!numberAvailable);

	return newNumber;
}

BufferID FileManager::newEmptyDocument()
{
#ifdef MOD_NEWDOC_DIRECTORY
	std::wstring newTitle;
	swsprintf(newTitle, L"%s\\%s%d", NppParameters::getInstance()->getWorkingDir(), UNTITLED_STR, nextUntitledNewNumber());
#else
	generic_string newTitle = UNTITLED_STR;
	TCHAR nb[10];
	wsprintf(nb, TEXT("%d"), nextUntitledNewNumber());
	newTitle += nb;
#endif

	Document doc = (Document)_pscratchTilla->execute(SCI_CREATEDOCUMENT);	//this already sets a reference for filemanager
	Buffer* newBuf = new Buffer(this, _nextBufferID, doc, DOC_UNNAMED, newTitle.c_str());
	BufferID id = static_cast<BufferID>(newBuf);
	newBuf->_id = id;
	_buffers.push_back(newBuf);
	++_nrBufs;
	++_nextBufferID;
	return id;
}

BufferID FileManager::bufferFromDocument(Document doc, bool dontIncrease, bool dontRef)
{
#ifdef MOD_NEWDOC_DIRECTORY
	std::wstring newTitle;
	swsprintf(newTitle, L"%s\\%s%d", NppParameters::getInstance()->getWorkingDir(), UNTITLED_STR, nextUntitledNewNumber());
#else
	generic_string newTitle = UNTITLED_STR;
	TCHAR nb[10];
	wsprintf(nb, TEXT("%d"), nextUntitledNewNumber());
	newTitle += nb;
#endif

	if (!dontRef)
		_pscratchTilla->execute(SCI_ADDREFDOCUMENT, 0, doc);	//set reference for FileManager
	Buffer* newBuf = new Buffer(this, _nextBufferID, doc, DOC_UNNAMED, newTitle.c_str());
	BufferID id = static_cast<BufferID>(newBuf);
	newBuf->_id = id;
	_buffers.push_back(newBuf);
	++_nrBufs;

	if (!dontIncrease)
		++_nextBufferID;
	return id;
}

int FileManager::detectCodepage(char* buf, size_t len)
{
	uchardet_t ud = uchardet_new();
	uchardet_handle_data(ud, buf, len);
	uchardet_data_end(ud);
	const char* cs = uchardet_get_charset(ud);
	int codepage = EncodingMapper::getInstance()->getEncodingFromString(cs);
	uchardet_delete(ud);
	return codepage;
}

//[///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
LangType FileManager::detectLanguageFromTextBegining(const unsigned char *data, size_t dataLen)
{
	struct FirstLineLanguages
	{
		std::string pattern;
		LangType lang;
	};

#ifdef ORZ // BOM
	if (1 < dataLen)
		if (data[0] == 0xFE && data[1] == 0xFF) // UTF16 BE BOM
			return L_TEXT;
		else if (data[0] == 0xFF && data[1] == 0xFE) // UTF16 LE BOM
			return L_TEXT;	// L_TEXT or L_RC or L_INI ...

	// Eliminate BOM if present
	size_t i = 0;
	if (2 < dataLen)
		if (data[0] == 0xEF && data[1] == 0xBB && data[2] == 0xBF) // UTF8 BOM
			i += 3;
#else
	// Is the buffer at least the size of a BOM?
	if (dataLen <= 3)
		return L_TEXT;

	// Eliminate BOM if present
	size_t i = 0;
	if ((data[0] == 0xEF && data[1] == 0xBB && data[2] == 0xBF) || // UTF8 BOM
		(data[0] == 0xFE && data[1] == 0xFF && data[2] == 0x00) || // UTF16 BE BOM
		(data[0] == 0xFF && data[1] == 0xFE && data[2] == 0x00))   // UTF16 LE BOM
		i += 3;
#endif

	// Skip any space-like char
	for (; i < dataLen; ++i)
	{
		if (data[i] != ' ' && data[i] != '\t' && data[i] != '\n' && data[i] != '\r')
			break;
	}

	// Create the buffer to need to test
	const size_t longestLength = 40; // shebangs can be large
	std::string buf2Test = std::string((const char *)data + i, longestLength);

	// Is there a \r or \n in the buffer? If so, truncate it
	auto cr = buf2Test.find("\r");
	auto nl = buf2Test.find("\n");
	auto crnl = min(cr, nl);
	if (crnl != std::string::npos && crnl < longestLength)
		buf2Test = std::string((const char *)data + i, crnl);

	// First test for a Unix-like Shebang
	// See https://en.wikipedia.org/wiki/Shebang_%28Unix%29 for more details about Shebang
	std::string shebang = "#!";

	size_t foundPos = buf2Test.find(shebang);
	if (foundPos == 0)
	{
		// Make a list of the most commonly used languages
		const size_t NB_SHEBANG_LANGUAGES = 6;
		FirstLineLanguages ShebangLangs[NB_SHEBANG_LANGUAGES] = {
			{ "sh",		L_BASH },
			{ "python", L_PYTHON },
			{ "perl",	L_PERL },
			{ "php",	L_PHP },
			{ "ruby",	L_RUBY },
			{ "node",	L_JAVASCRIPT }
		};

		// Go through the list of languages
		for (i = 0; i < NB_SHEBANG_LANGUAGES; ++i)
		{
			if (buf2Test.find(ShebangLangs[i].pattern) != std::string::npos)
			{
				return ShebangLangs[i].lang;
			}
		}

		// Unrecognized shebang (there is always room for improvement ;-)
		return L_TEXT;
	}

	// Are there any other patterns we know off?
	const size_t NB_FIRST_LINE_LANGUAGES = 5;
	FirstLineLanguages languages[NB_FIRST_LINE_LANGUAGES] = {
		{ "<?xml",			L_XML },
		{ "<?php",			L_PHP },
		{ "<html",			L_HTML },
		{ "<!DOCTYPE html",	L_HTML },
		{ "<?",				L_PHP } // MUST be after "<?php" and "<?xml" to get the result as accurate as possible
	};

	for (i = 0; i < NB_FIRST_LINE_LANGUAGES; ++i)
	{
		foundPos = buf2Test.find(languages[i].pattern);
		if (foundPos == 0)
		{
			return languages[i].lang;
		}
	}

	// Unrecognized first line, we assume it is a text file for now
	return L_TEXT;
}

#ifdef MOD_FILE_TOO_BIG_MSG
static void fileTooBigMsg(const WCHAR *filename, const unsigned __int64 fileSize/*, const char *function, const int line*/)
{
	static const int countof_msg = MAX_PATH + 80;
	WCHAR msg[countof_msg]; /* WCHAR *msg = new WCHAR[countof_msg]; */
	swprintf_s(msg, countof_msg, L"%s\n\n" L"\t(%I64u bytes)\n\n" L"is too big to be opened by Notepad++" /*L"\n\n%S(%d)"*/, filename, fileSize/*, function, line*/);
	::MessageBoxW(Notepad_plus_Window::gNppHWND, msg, L"File open problem", MB_OK | MB_ICONWARNING/*MB_ICONERROR*/ | MB_APPLMODAL /*| MB_SETFOREGROUND*/ | MB_TOPMOST);
	/* delete [] msg; */
}
#endif

#ifdef MOD_CHANGE_ENCODING
bool FileManager::loadFileData(Document doc, const WCHAR * filename, char* data, Utf8_16_Read * unicodeConvertor, LangType & language, int & encoding, EolType & eolFormat, const bool detectEncoding/* = false*//*TODO:=> , const bool isSnapshotFile = false */)
#else
bool FileManager::loadFileData(Document doc, const TCHAR * filename, char* data, Utf8_16_Read * unicodeConvertor, LangType & language, int & encoding, EolType & eolFormat)
#endif
{
#ifdef FIX_CODEPAGE_AUTODETECT
	const bool _determineBOM = true; //TODO: NppParameters::getInstance()->getNppGUI().determineBOM
#endif
	assert(encoding >= -1);
	FILE *fp = generic_fopen(filename, TEXT("rb"));
	if (not fp)
		return false;

	//Get file size
	_fseeki64 (fp , 0 , SEEK_END);
	unsigned __int64 fileSize =_ftelli64(fp);
	rewind(fp);
	// size/6 is the normal room Scintilla keeps for editing, but here we limit it to 1MiB when loading (maybe we want to load big files without editing them too much)
	unsigned __int64 bufferSizeRequested = fileSize + min(1<<20,fileSize/6);
	// As a 32bit application, we cannot allocate 2 buffer of more than INT_MAX size (it takes the whole address space)
	if (bufferSizeRequested > INT_MAX)
	{
#ifdef MOD_FILE_TOO_BIG_MSG
		fileTooBigMsg(filename, fileSize/*, __FUNCTION__, __LINE__*/);
#else
		::MessageBox(NULL, TEXT("File is too big to be opened by Notepad++"), TEXT("File size problem"), MB_OK|MB_APPLMODAL);
#endif
		/*
		_nativeLangSpeaker.messageBox("NbFileToOpenImportantWarning",
										_pPublicInterface->getHSelf(),
										TEXT("File is too big to be opened by Notepad++"),
										TEXT("File open problem"),
										MB_OK|MB_APPLMODAL);
		*/
		fclose(fp);
		return false;
	}

	//Setup scratchtilla for new filedata
	_pscratchTilla->execute(SCI_SETSTATUS, SC_STATUS_OK); // reset error status
	_pscratchTilla->execute(SCI_SETDOCPOINTER, 0, doc);
	bool ro = _pscratchTilla->execute(SCI_GETREADONLY) != 0;
	if (ro)
	{
		_pscratchTilla->execute(SCI_SETREADONLY, false);
	}
	_pscratchTilla->execute(SCI_CLEARALL);


	if (language < L_EXTERNAL)
	{
		_pscratchTilla->execute(SCI_SETLEXER, ScintillaEditView::langNames[language].lexerID);
	}
	else
	{
		int id = language - L_EXTERNAL;
		TCHAR * name = NppParameters::getInstance()->getELCFromIndex(id)._name;
		WcharMbcsConvertor *wmc = WcharMbcsConvertor::getInstance();
		const char *pName = wmc->wchar2char(name, CP_ACP);
		_pscratchTilla->execute(SCI_SETLEXERLANGUAGE, 0, reinterpret_cast<LPARAM>(pName));
	}

	if (encoding != -1)
		_pscratchTilla->execute(SCI_SETCODEPAGE, SC_CP_UTF8);

	bool success = true;
#ifdef FIX_DETECT_EOL
#else
	EolType format = EolType::unknown;
#endif
#ifdef NO_TRY_CATCH
	/*TRY:*/ int exception_line = 0;
#else
	__try
#endif
	{
		// First allocate enough memory for the whole file (this will reduce memory copy during loading)
		_pscratchTilla->execute(SCI_ALLOCATE, WPARAM(bufferSizeRequested));
		if (_pscratchTilla->execute(SCI_GETSTATUS) != SC_STATUS_OK)
#ifdef NO_TRY_CATCH
			{ exception_line = __LINE__; goto SCI_GETSTATUS_EXCEPTION; } // throw
#else
			throw;
#endif

#ifdef ORZ
#else
		size_t lenFile = 0;
		size_t lenConvert = 0;	//just in case conversion results in 0, but file not empty
#endif
		bool isFirstTime = true;
		int incompleteMultibyteChar = 0;

#ifdef MLANG_DLL
		DWORD mlangMode = 0;
#endif

#ifdef ORZ
		for (;;)
#else
		do
#endif
		{
#ifdef ORZ
			const size_t
#endif
			lenFile = fread(data+incompleteMultibyteChar, 1, blockSize-incompleteMultibyteChar, fp) + incompleteMultibyteChar;
#ifdef MLANG_DLL
			incompleteMultibyteChar = 0;
#endif
#ifdef MOD_CHANGE_ENCODING
#else
			if (lenFile == 0) break;
#endif

            if (isFirstTime)
            {
#ifdef FIX_CODEPAGE_AUTODETECT // --------------------------------------------------------------------------------------
				if (detectEncoding) {
					unicodeConvertor->determineUniMode((const unsigned char *)data, lenFile);
					const UniMode um = unicodeConvertor->getEncoding();
					if (um == uni7Bit || um == uni8Bit) {
						unicodeConvertor->setEncoding(uniCookie);
						if (encoding == -1) {
							if (NppParameters::getInstance()->getNppGUI()._detectEncoding) {
								encoding = detectCodepage(data, lenFile);	// Shift-JIS, JIS, EUC-JP, Big5 ...
					// TODO:	if (encoding == -1) {
					// TODO:		// -*- coding: XXXX -*-
					// TODO:		// vim:fileencoding=XXXX
					// TODO:		// "coding[=:]\\s*[\"\']?([-\\w.]+)"
					// TODO:		encoding = detectXXXXXXXX(data, lenFile);	// encoding="utf-8"
					// TODO:	}
								if (encoding == -1) {
									if (um == uni7Bit && NppParameters::getInstance()->getNppGUI().getNewDocDefaultSettings()._openAnsiAsUtf8)
										/* encoding = -1 */; // UTF-8N
									else
										encoding = NppParameters::getInstance()->getNppGUI().getNewDocDefaultSettings().openAnsiCodepage; // ANSI
								}
							}
						}
					}
					else if (um == uniCookie) {
						/* encoding = encoding */; // UTF-8N
					}
					else { // if uniUTF8, uni16BE, uni16LE, uni16BE_NoBOM, uni16LE_NoBOM
						encoding = -1; // UTF-8, UTF-16**
					}
				} // detectEncoding
				else if ((_determineBOM || unicodeConvertor->getEncoding() == uni16BE || unicodeConvertor->getEncoding() == uni16LE || unicodeConvertor->getEncoding() == uniUTF8)
				  && unicodeConvertor->determineBOM((const unsigned char *)data, lenFile) != uni8Bit) {
					// uniUTF8  UTF-8 with BOM
					// uni16BE  UTF-16 big-endian with BOM
					// uni16LE  UTF-16 little-endian with BOM
					unicodeConvertor->determineUniMode((const unsigned char *)data, lenFile);
					encoding = -1;
				}
				// else not detect. sessuin.xml
#else // FIX_CODEPAGE_AUTODETECT ---------------------------------------------------------------------------------------
				// check if file contain any BOM
                if (Utf8_16_Read::determineEncoding((unsigned char *)data, lenFile) != uni8Bit)
                {
                    // if file contains any BOM, then encoding will be erased,
                    // and the document will be interpreted as UTF
                    encoding = -1;
				}
				else if (encoding == -1)
				{
					if (NppParameters::getInstance()->getNppGUI()._detectEncoding)
						encoding = detectCodepage(data, lenFile);
                }
#endif // FIX_CODEPAGE_AUTODETECT --------------------------------------------------------------------------------------

#ifdef FIX_DETECT_LANGUAGE
#else
				if (language == L_TEXT)
				{
					// check the language du fichier
					language = detectLanguageFromTextBegining((unsigned char *)data, lenFile);
				}
#endif

                isFirstTime = false;
            }
#ifdef MOD_CHANGE_ENCODING
			if (lenFile == 0) break;
#endif

			if (encoding != -1)
			{
				if (encoding == SC_CP_UTF8)
				{
					// Pass through UTF-8 (this does not check validity of characters, thus inserting a multi-byte character in two halfs is working)
					_pscratchTilla->execute(SCI_APPENDTEXT, lenFile, reinterpret_cast<LPARAM>(data));
				}
				else
				{
					WcharMbcsConvertor* wmc = WcharMbcsConvertor::getInstance();
					int newDataLen = 0;
#ifdef MLANG_DLL
					int processedMultibyteChar = lenFile;
					const char *newData = wmc->decodeStream(&mlangMode, encoding, data, &processedMultibyteChar, &newDataLen);
					incompleteMultibyteChar = lenFile - processedMultibyteChar;
					if (newDataLen != 0)
						_pscratchTilla->execute(SCI_APPENDTEXT, newDataLen, (LPARAM) newData);
					if (incompleteMultibyteChar != 0 && feof(fp)) {
						_pscratchTilla->execute(SCI_APPENDTEXT, incompleteMultibyteChar, (LPARAM)(data + processedMultibyteChar));
						incompleteMultibyteChar = 0;
					}
#else // MLANG_DLL
					const char *newData = wmc->encode(encoding, SC_CP_UTF8, data, static_cast<int32_t>(lenFile), &newDataLen, &incompleteMultibyteChar);
					_pscratchTilla->execute(SCI_APPENDTEXT, newDataLen, reinterpret_cast<LPARAM>(newData));
#endif // MLANG_DLL
				}
#ifdef FIX_DETECT_EOL
#else

				if (format == EolType::unknown)
					format = getEOLFormatForm(data, lenFile, EolType::unknown);
#endif
			}
			else
			{
#ifdef ORZ
				size_t lenConvert;	//just in case conversion results in 0, but file not empty
#endif
				lenConvert = unicodeConvertor->convert(data, lenFile);
#ifdef UTF16_SURROGATE_PAIR
				incompleteMultibyteChar = unicodeConvertor->getIncompleteBytes();
				int processedMultibyteChar = lenFile - incompleteMultibyteChar;
#endif
#ifdef UTF16_SURROGATE_PAIR
				if (lenConvert != 0)
					_pscratchTilla->execute(SCI_APPENDTEXT, lenConvert, (LPARAM)(unicodeConvertor->getNewBuf()));
				if (incompleteMultibyteChar != 0 && feof(fp)) {
					_pscratchTilla->execute(SCI_APPENDTEXT, incompleteMultibyteChar, (LPARAM)(data + processedMultibyteChar));
					incompleteMultibyteChar = 0;
				}
#else
				_pscratchTilla->execute(SCI_APPENDTEXT, lenConvert, reinterpret_cast<LPARAM>(unicodeConvertor->getNewBuf()));
#endif
#ifdef FIX_DETECT_EOL
#else
				if (format == EolType::unknown)
					format = getEOLFormatForm(unicodeConvertor->getNewBuf(), unicodeConvertor->getNewSize(), EolType::unknown);
#endif
			}

			if (_pscratchTilla->execute(SCI_GETSTATUS) != SC_STATUS_OK)
#ifdef NO_TRY_CATCH
				{ exception_line = __LINE__; goto SCI_GETSTATUS_EXCEPTION; } // throw
#else
				throw;
#endif

			if (incompleteMultibyteChar != 0)
			{
				// copy bytes to next buffer
#ifdef ORZ
				memcpy(data, data + lenFile - incompleteMultibyteChar, incompleteMultibyteChar);
#else
				memcpy(data, data + blockSize - incompleteMultibyteChar, incompleteMultibyteChar);
#endif
			}

			assert(lenFile != 0);
		}
#ifdef ORZ
#else
		while (lenFile > 0);
#endif
#ifdef NO_TRY_CATCH
		goto FINALLY;
#endif
	}
#ifdef NO_TRY_CATCH
	SCI_GETSTATUS_EXCEPTION: // catch SCI_GETSTATUS_EXCEPTION
#else
	__except(EXCEPTION_EXECUTE_HANDLER) //TODO: should filter correctly for other exceptions; the old filter(GetExceptionCode(), GetExceptionInformation()) was only catching access violations
#endif
	{
#ifdef MOD_FILE_TOO_BIG_MSG
		fileTooBigMsg(filename, fileSize/*, __FUNCTION__, exception_line*/);
#else
		::MessageBox(NULL, TEXT("File is too big to be opened by Notepad++"), TEXT("File open problem"), MB_OK|MB_APPLMODAL);
#endif
		success = false;
#ifdef NO_TRY_CATCH
		goto FINALLY;
#endif
	}

#ifdef NO_TRY_CATCH
	FINALLY: {}
#endif

#ifdef MLANG_DLL
	WcharMbcsConvertor::getInstance()->free();
#endif
	fclose(fp);

	// broadcast the format
#ifdef FIX_DETECT_EOL
	eolFormat = getEOLFormatForm((const char *const)_pscratchTilla->execute(SCI_GETCHARACTERPOINTER),
		_pscratchTilla->execute(SCI_GETTEXTLENGTH),
		NppParameters::getInstance()->getNppGUI().getNewDocDefaultSettings()._format);
#else
	if (format == EolType::unknown)
	{
		NppParameters *pNppParamInst = NppParameters::getInstance();
		const NewDocDefaultSettings & ndds = (pNppParamInst->getNppGUI()).getNewDocDefaultSettings(); // for ndds._format
		eolFormat = ndds._format;
	}
	else
	{
		eolFormat = format;
	}
#endif
#ifdef FIX_DETECT_LANGUAGE
	if (detectEncoding)
		if (language == L_TEXT)
			// check the language du fichier
			language = detectLanguageFromTextBegining(_pscratchTilla->execute(SCI_GETCHARACTERPOINTER), _pscratchTilla->execute(SCI_GETTEXTLENGTH));
#endif

	_pscratchTilla->execute(SCI_EMPTYUNDOBUFFER);
	_pscratchTilla->execute(SCI_SETSAVEPOINT);

	if (ro)
		_pscratchTilla->execute(SCI_SETREADONLY, true);

	_pscratchTilla->execute(SCI_SETDOCPOINTER, 0, _scratchDocDefault);

	return success;
}


BufferID FileManager::getBufferFromName(const TCHAR* name)
{
	TCHAR fullpath[MAX_PATH];
	::GetFullPathName(name, MAX_PATH, fullpath, NULL);
	if (_tcschr(fullpath, '~'))
	{
		::GetLongPathName(fullpath, fullpath, MAX_PATH);
	}
#ifdef FIX_SYMBOLIC_LINK
	HANDLE hFile = ::CreateFileW(fullpath, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
	BY_HANDLE_FILE_INFORMATION fileInformation = {};
	GetFileInformationByHandle(hFile, &fileInformation);
	::CloseHandle(hFile);
#endif

	for(size_t i = 0; i < _buffers.size(); i++)
	{
		if (!lstrcmpi(name, _buffers.at(i)->getFullPathName()))
			return _buffers.at(i)->getID();
#ifdef FIX_SYMBOLIC_LINK
		hFile = ::CreateFileW(_buffers.at(i)->getFullPathName(), GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
		if (hFile != INVALID_HANDLE_VALUE) {
			BY_HANDLE_FILE_INFORMATION fi;
			auto res = GetFileInformationByHandle(hFile, &fi);
			::CloseHandle(hFile);
			if (res != 0)
				if (fileInformation.dwVolumeSerialNumber == fi.dwVolumeSerialNumber
				 && fileInformation.nFileIndexHigh       == fi.nFileIndexHigh
				 && fileInformation.nFileIndexLow        == fi.nFileIndexLow)
					return _buffers.at(i)->getID();
		}
#endif
	}
	return BUFFER_INVALID;
}


BufferID FileManager::getBufferFromDocument(Document doc)
{
	for (size_t i = 0; i < _nrBufs; ++i)
	{
		if (_buffers[i]->_doc == doc)
			return _buffers[i]->_id;
	}
	return BUFFER_INVALID;
}


bool FileManager::createEmptyFile(const TCHAR * path)
{
	FILE * file = generic_fopen(path, TEXT("wb"));
	if (!file)
		return false;
	fclose(file);
	return true;
}


int FileManager::getFileNameFromBuffer(BufferID id, TCHAR * fn2copy)
{
	if (getBufferIndexByID(id) == -1)
		return -1;

	Buffer* buf = getBufferByID(id);
	if (fn2copy)
		lstrcpy(fn2copy, buf->getFullPathName());
	return lstrlen(buf->getFullPathName());
}


int FileManager::docLength(Buffer* buffer) const
{
	_pscratchTilla->execute(SCI_SETDOCPOINTER, 0, buffer->_doc);
	int docLen = _pscratchTilla->getCurrentDocLen();
	_pscratchTilla->execute(SCI_SETDOCPOINTER, 0, _scratchDocDefault);
	return docLen;
}