﻿// Utf8_16.cxx
// Copyright (C) 2002 Scott Kirkwood
//
// Permission to use, copy, modify, distribute and sell this code
// and its documentation for any purpose is hereby granted without fee,
// provided that the above copyright notice appear in all copies or
// any derived copies.  Scott Kirkwood makes no representations
// about the suitability of this software for any purpose.
// It is provided "as is" without express or implied warranty.
////////////////////////////////////////////////////////////////////////////////
// 
// Modificated 2006 Jens Lorenz
// 
// - Clean up the sources
// - Removing UCS-Bug in Utf8_Iter
// - Add convert function in Utf8_16_Write
////////////////////////////////////////////////////////////////////////////////

#include "Utf8_16.h"


const Utf8_16::utf8 Utf8_16::k_Boms[][3] = {
	{0x00, 0x00, 0x00},  // 0.uni8Bit	Unknown
	{0xEF, 0xBB, 0xBF},  // 1.uniUTF8	UTF-8
	{0xFE, 0xFF, 0x00},  // 2.uni16BE	UTF-16 Big endian
	{0xFF, 0xFE, 0x00},  // 3.uni16LE	UTF-16 Little endian
 //	{0x00, 0x00, 0x00},  // 4.uniCookie		UTF-8N
 //	{0x00, 0x00, 0x00},  // 5.uni7Bit		ASCII
 //	{0x00, 0x00, 0x00},  // 6.uni16BE_NoBOM	UTF-16BE
 //	{0x00, 0x00, 0x00},  // 7.uni16LE_NoBOM	UTF-16LE
#if 0000 // TODO:
 //	{0x00, 0x00, 0xFE, 0xFF},  // 8.uni32BE		UTF-32 Big endian
 //	{0xFF, 0xFE, 0x00, 0x00},  // 9.uni32LE		UTF-32 Little endian
 //	{0x00, 0x00, 0x00},  // 10.uni32BE_NoBOM	UTF-32BE
 //	{0x00, 0x00, 0x00},  // 11.uni32LE_NoBOM	UTF-32LE
 //	{0x2B, 0x2F, 0x76, (0x38,0x39,0x2B,0x2F)},  // 12.uniUTF7	UTF-7
 //	{0x00, 0x00, 0x00},  // 13.uni7_NoBOM		UTF7
 //	{0xEF, 0xBB, 0xBF},  // 14.uniCESU8			CESU-8
 //	{0x00, 0x00, 0x00},  // 15.uniCESU8_NoBOM	CESU-8
#endif
};

static const int /*Utf8_16::*/k_nSkip[] = {
	0,  // 0.uni8Bit			Unknown
	3,  // 1.uniUTF8			UTF-8
	2,  // 2.uni16BE			UTF-16 Big endian
	2,  // 3.uni16LE			UTF-16 Little endian
 //	0,  // 4.uniCookie			UTF-8N
 //	0,  // 5.uni7Bit			ASCII
 //	0,  // 6.uni16BE_NoBOM		UTF-16BE
 //	0,  // 7.uni16LE_NoBOM		UTF-16LE
#if 0000 // TODO:
 //	4,  // 8.uni32BE			UTF-32 Big endian
 //	4,  // 9.uni32LE			UTF-32 Little endian
 //	0,  // 10.uni32BE_NoBOM		UTF-32BE
 //	0,  // 11.uni32LE_NoBOM		UTF-32LE
 //	3,  // 12.uniUTF7			UTF-7
 //	0,  // 13.uni7_NoBOM		UTF7
 //	3,  // 14.uniCESU8			CESU-8
 //	0,  // 15.uniCESU8_NoBOM	CESU-8
#endif
};

// ==================================================================

Utf8_16_Read::Utf8_16_Read() {
	m_eEncoding		= uni8Bit;
	m_nAllocatedBufSize = 0;
	m_nNewBufSize   = 0;
	m_pNewBuf		= NULL;
#ifdef FIX_CODEPAGE_AUTODETECT
	nSkip = m_nSkip = 0;
#else
	m_bFirstRead	= true;
#endif
}

Utf8_16_Read::~Utf8_16_Read()
{
	if ((m_eEncoding == uni16BE) || (m_eEncoding == uni16LE) || (m_eEncoding == uni16BE_NoBOM) || (m_eEncoding == uni16LE_NoBOM))
    {
		delete [] m_pNewBuf;
		m_pNewBuf = NULL;
	}
}

// Returned value :
// 0 : utf8
// 1 : 7bits
// 2 : 8bits
u78 Utf8_16_Read::utf8_7bits_8bits()
{
	int rv = 1;
	int ASCII7only = 1;
	utf8 *sx	= (utf8 *)m_pBuf;
	utf8 *endx	= sx + m_nLen;

	while (sx<endx)
	{
		if (!*sx)
		{											// For detection, we'll say that NUL means not UTF8
			ASCII7only = 0;
			rv = 0;
			break;
		} 
		else if (*sx < 0x80)
		{			// 0nnnnnnn If the byte's first hex code begins with 0-7, it is an ASCII character.
			++sx;
		} 
		else if (*sx < (0x80 + 0x40)) 
		{											  // 10nnnnnn 8 through B cannot be first hex codes
			ASCII7only=0;
			rv=0;
			break;
		} 
		else if (*sx < (0x80 + 0x40 + 0x20))
		{					  // 110xxxvv 10nnnnnn  If it begins with C or D, it is an 11 bit character
			ASCII7only=0;
			if (sx+2 > endx) 
				break;
			if ((sx[0] & 0xC0) != 0xC0 || (sx[1] & 0xC0) != 0x80) {
				rv=0; break;
			}
			sx+=2;
		} 
		else if (*sx < (0x80 + 0x40 + 0x20 + 0x10))
		{								// 1110qqqq 10xxxxvv 10nnnnnn If it begins with E, it is 16 bit
			ASCII7only=0;
			if (sx+3 > endx) 
				break;
			if ((sx[0] & 0xE0) != 0xE0 || (sx[1] & 0xC0) != 0x80 || (sx[2] & 0xC0) != 0x80) {
				rv=0; break;
			}
			sx+=3;
		} 
		else 
		{													  // more than 16 bits are not allowed here
			ASCII7only=0;
			rv=0;
			break;
		}
	}
	if (ASCII7only) 
		return ascii7bits;
	if (rv)
		return utf8NoBOM;
	return ascii8bits;
}

size_t Utf8_16_Read::convert(char* buf, size_t len)
{
	// bugfix by Jens Lorenz
#ifdef FIX_CODEPAGE_AUTODETECT
#else
	static	size_t nSkip = 0;
#endif

	m_pBuf = (ubyte*)buf;
	m_nLen = len;
	m_nNewBufSize = 0;
#ifdef UTF16_SURROGATE_PAIR
	m_nIncompleteBytes = 0;
#endif

#ifdef FIX_CODEPAGE_AUTODETECT
#else
	if (m_bFirstRead == true)
    {
		determineEncoding();
		nSkip = m_nSkip;
		m_bFirstRead = false;
	}
#endif

#ifdef INTERNAL_ENCODING_UTF8N
	assert(m_eEncoding != uni7Bit);
	assert(m_eEncoding != uni8Bit);
#endif
    switch (m_eEncoding)
    {
		case uni7Bit:
        case uni8Bit:
        case uniCookie: {
            // Do nothing, pass through
			m_nAllocatedBufSize = 0;
            m_pNewBuf = m_pBuf;
			m_nNewBufSize = len;
            break;
        }
        case uniUTF8: {
            // Pass through after BOM
			m_nAllocatedBufSize = 0;
            m_pNewBuf = m_pBuf + nSkip;
			m_nNewBufSize = len - nSkip;
            break;
        }    
        case uni16BE_NoBOM:
        case uni16LE_NoBOM:
        case uni16BE:
        case uni16LE: {
            size_t newSize = len + len / 2 + 1;	//TODO: #if UTF16_SURROGATE_PAIR size_t newSize = len * 2;
            
			if (m_nAllocatedBufSize != newSize)
            {
				if (m_pNewBuf)
					delete [] m_pNewBuf;
                m_pNewBuf  = NULL;
                m_pNewBuf  = new ubyte[newSize];
				m_nAllocatedBufSize = newSize;
            }
            
            ubyte* pCur = m_pNewBuf;
            
            m_Iter16.set(m_pBuf + nSkip, len - nSkip, m_eEncoding);

            for (; m_Iter16; ++m_Iter16)
            {
                *pCur++ = m_Iter16.get();
            }
			m_nNewBufSize = pCur - m_pNewBuf;
#ifdef UTF16_SURROGATE_PAIR
			m_nIncompleteBytes = m_Iter16.m_nIncompleteBytes;
#endif
            break;
        }
        default:
            break;
    }

	// necessary for second calls and more
	nSkip = 0;

	return m_nNewBufSize;
}


#ifdef FIX_CODEPAGE_AUTODETECT
void Utf8_16_Read::determineUniMode(const unsigned char* buf, size_t bufLen)
#else
void Utf8_16_Read::determineEncoding()
#endif
{
#ifdef FIX_CODEPAGE_AUTODETECT
	m_pBuf = (ubyte*)buf;
	m_nLen = bufLen;
	m_nNewBufSize = 0;
#endif

	INT uniTest = IS_TEXT_UNICODE_STATISTICS;
	m_eEncoding = uni8Bit;
	m_nSkip = 0;

    // detect UTF-16 big-endian with BOM
	if (m_nLen > 1 && m_pBuf[0] == k_Boms[uni16BE][0] && m_pBuf[1] == k_Boms[uni16BE][1])
	{
		m_eEncoding = uni16BE;
		m_nSkip = k_nSkip[m_eEncoding]; // 2
	}
    // detect UTF-16 little-endian with BOM
	else if (m_nLen > 1 && m_pBuf[0] == k_Boms[uni16LE][0] && m_pBuf[1] == k_Boms[uni16LE][1])
	{
		m_eEncoding = uni16LE;
		m_nSkip = k_nSkip[m_eEncoding]; // 2
	}
    // detect UTF-8 with BOM
	else if (m_nLen > 2 && m_pBuf[0] == k_Boms[uniUTF8][0] && 
		m_pBuf[1] == k_Boms[uniUTF8][1] && m_pBuf[2] == k_Boms[uniUTF8][2])
	{
		m_eEncoding = uniUTF8;
		m_nSkip = k_nSkip[m_eEncoding]; // 3
	}
	// try to detect UTF-16 little-endian without BOM
	else if (m_nLen > 1 && m_nLen % 2 == 0 && m_pBuf[0] != NULL && m_pBuf[1] == NULL && IsTextUnicode(m_pBuf, static_cast<int32_t>(m_nLen), &uniTest))
	{
		m_eEncoding = uni16LE_NoBOM;
		m_nSkip = 0;
	}
	/* UTF-16 big-endian without BOM detection is taken away scince this detection is very week
    // try to detect UTF-16 big-endian without BOM
    else if (m_nLen > 1 && m_pBuf[0] == NULL && m_pBuf[1] != NULL)
	{
		m_eEncoding = uni16BE_NoBOM;
		m_nSkip = 0;
	}
	*/
	else
	{
		u78 detectedEncoding = utf8_7bits_8bits();
		if (detectedEncoding == utf8NoBOM)
			m_eEncoding = uniCookie;
		else if (detectedEncoding == ascii7bits)
			m_eEncoding = uni7Bit;
		else //(detectedEncoding == ascii8bits)
			m_eEncoding = uni8Bit;
		m_nSkip = 0;
	}
#ifdef FIX_CODEPAGE_AUTODETECT
	nSkip = m_nSkip;
#endif
}

#ifdef FIX_CODEPAGE_AUTODETECT
UniMode Utf8_16_Read::determineBOM(const unsigned char* buf, size_t bufLen)
#else
UniMode Utf8_16_Read::determineEncoding(const unsigned char *buf, size_t bufLen)
#endif
{
    // detect UTF-16 big-endian with BOM
	if (bufLen > 1 && buf[0] == k_Boms[uni16BE][0] && buf[1] == k_Boms[uni16BE][1])
	{
		return uni16BE;
	}
    
    // detect UTF-16 little-endian with BOM
    if (bufLen > 1 && buf[0] == k_Boms[uni16LE][0] && buf[1] == k_Boms[uni16LE][1])
	{
		return uni16LE;
	}
    
    // detect UTF-8 with BOM
	if (bufLen > 2 && buf[0] == k_Boms[uniUTF8][0] && 
		buf[1] == k_Boms[uniUTF8][1] && buf[2] == k_Boms[uniUTF8][2])
	{
		return uniUTF8;
	}

    return uni8Bit;
}


// ==================================================================

Utf8_16_Write::Utf8_16_Write()
{
	m_eEncoding = uni8Bit;
	m_pFile = NULL;
	m_pNewBuf = NULL;
	m_bFirstWrite = true;
	m_nBufSize = 0;
}

Utf8_16_Write::~Utf8_16_Write()
{
	fclose();
}

#ifdef FIX_SYMBOLIC_LINK
// @Override ::fwrite()
static DWORD fwrite(LPCVOID lpBuffer, DWORD size, DWORD count, HANDLE hFile)
{
	DWORD nNumberOfBytesToWrite = size * count;
	DWORD numberOfBytesWritten = 0;
	::WriteFile(hFile, lpBuffer, nNumberOfBytesToWrite, &numberOfBytesWritten, NULL);
	return numberOfBytesWritten / size;
}

bool Utf8_16_Write::fopen(const WCHAR *name)
{
	HANDLE hFile = ::CreateFileW(name, GENERIC_WRITE, FILE_SHARE_DELETE | FILE_SHARE_READ | FILE_SHARE_WRITE, NULL, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);
	if (hFile == INVALID_HANDLE_VALUE)
		m_pFile = NULL;
	else
		m_pFile = hFile;

	m_bFirstWrite = true;

	return m_pFile != NULL;
}
#else // FIX_SYMBOLIC_LINK
FILE * Utf8_16_Write::fopen(const TCHAR *_name, const TCHAR *_type)
{
	m_pFile = ::generic_fopen(_name, _type);

	m_bFirstWrite = true;

	return m_pFile;
}
#endif // FIX_SYMBOLIC_LINK

size_t Utf8_16_Write::fwrite(const void* p, size_t _size)
{
    // no file open
	if (!m_pFile)
    {
		return 0;
	}

    size_t  ret = 0;
    
	if (m_bFirstWrite)
    {
        switch (m_eEncoding)
        {
            case uniUTF8: {
                ::fwrite(k_Boms[m_eEncoding], k_nSkip[m_eEncoding], 1, m_pFile);
///             ::fwrite(k_Boms[m_eEncoding], 3, 1, m_pFile);
                break;
            }    
            case uni16BE:
            case uni16LE:
                ::fwrite(k_Boms[m_eEncoding], k_nSkip[m_eEncoding], 1, m_pFile);
///             ::fwrite(k_Boms[m_eEncoding], 2, 1, m_pFile);
                break;
            default:
                // nothing to do
                break;
        }
		m_bFirstWrite = false;
    }
    
#ifdef INTERNAL_ENCODING_UTF8N
	assert(m_eEncoding != uni7Bit);
	assert(m_eEncoding != uni8Bit);
#endif
    switch (m_eEncoding)
    {
		case uni7Bit:
        case uni8Bit:
        case uniCookie:
        case uniUTF8: {
            // Normal write
            ret = ::fwrite(p, _size, 1, m_pFile);
            break;
        }
        case uni16BE_NoBOM:
        case uni16LE_NoBOM:
        case uni16BE:
        case uni16LE: {
			static const int bufSize = 64*1024;
			utf16 buf[bufSize];
            
            Utf8_Iter iter8;
            iter8.set(static_cast<const ubyte*>(p), _size, m_eEncoding);
            
            int bufIndex = 0;
            while (iter8) {
                if (iter8.canGet()) {
                    buf[bufIndex++] = iter8.get();
                }
				++iter8;
				if(bufIndex == bufSize || !iter8) {
					if(!::fwrite(buf, bufIndex*sizeof(utf16), 1, m_pFile)) return 0;
					bufIndex = 0;
				}
            }
            ret = 1;
            break;
        }    
        default:
            break;
    }
    
    return ret;
}


size_t Utf8_16_Write::convert(char* p, size_t _size)
{
	if (m_pNewBuf)
    {
		delete [] m_pNewBuf;
	}

#ifdef INTERNAL_ENCODING_UTF8N
	assert(m_eEncoding != uni7Bit);
	assert(m_eEncoding != uni8Bit);
#endif
    switch (m_eEncoding)
    {
		case uni7Bit:
        case uni8Bit:
        case uniCookie: {
            // Normal write
            m_nBufSize = _size;
            m_pNewBuf = (ubyte*)new ubyte[m_nBufSize];
            memcpy(m_pNewBuf, p, _size);
            break;
        }
        case uniUTF8: {
            m_nBufSize = _size + 3;
            m_pNewBuf = (ubyte*)new ubyte[m_nBufSize];
            memcpy(m_pNewBuf, k_Boms[m_eEncoding], 3);
            memcpy(&m_pNewBuf[3], p, _size);
            break;
        }
        case uni16BE_NoBOM:
        case uni16LE_NoBOM:
        case uni16BE:
        case uni16LE:
		{
			utf16* pCur = NULL;
            
            if (m_eEncoding == uni16BE || m_eEncoding == uni16LE) {
                // Write the BOM
				m_pNewBuf = (ubyte*)new ubyte[sizeof(utf16) * (_size + 1)];
                memcpy(m_pNewBuf, k_Boms[m_eEncoding], 2);
	            pCur = (utf16*)&m_pNewBuf[2];
            } else {
				m_pNewBuf = (ubyte*)new ubyte[sizeof(utf16) * _size];
	            pCur = (utf16*)m_pNewBuf;
			}

            Utf8_Iter iter8;
            iter8.set(reinterpret_cast<const ubyte*>(p), _size, m_eEncoding);
            
            for (; iter8; ++iter8) {
                if (iter8.canGet()) {
                    *pCur++ = iter8.get();
                }
            }
            m_nBufSize = (const char*)pCur - (const char*)m_pNewBuf;
			break;
        }
        default:
            break;
    }
    
	return m_nBufSize;
}


void Utf8_16_Write::setEncoding(UniMode eType)
{
	m_eEncoding = eType;
}


void Utf8_16_Write::fclose()
{
	if (m_pNewBuf)
		delete [] m_pNewBuf;

	if (m_pFile)
#ifdef FIX_SYMBOLIC_LINK
		::CloseHandle(m_pFile);
#else
		::fclose(m_pFile);
#endif
}


//=================================================================
Utf8_Iter::Utf8_Iter()
{
	reset();
}

void Utf8_Iter::reset()
{
	m_pBuf = NULL;
	m_pRead = NULL;
	m_pEnd = NULL;
#ifdef UTF16_SURROGATE_PAIR
	m_index = m_len = 0;
#else
	m_eState = eStart;
#endif
	m_nCur = 0;
	m_eEncoding = uni8Bit;
}

void Utf8_Iter::set(const ubyte* pBuf, size_t nLen, UniMode eEncoding)
{
	m_pBuf      = pBuf;
	m_pRead     = pBuf;
	m_pEnd      = pBuf + nLen;
	m_eEncoding = eEncoding;
	m_isBigEndian = m_eEncoding == uni16LE || m_eEncoding == uni16LE_NoBOM;
	operator++();
	// Note: m_eState, m_nCur not set
}

// Go to the next byte.
void Utf8_Iter::operator++()
{
#ifdef UTF16_SURROGATE_PAIR
	if (m_index >= m_len) {
		/*
		 * UTF-8 ---decode---> U+XXXXXX (UTF-32) ---encode---> UTF-16
		 */
		uint32_t u = *m_pRead++;

		ubyte *di = (ubyte *)m_utf16; // (uint16_t*) --> (unsigned char*)
		if (m_isBigEndian) {
			*di++ = (ubyte) u; *di++ = 0;
		} else {
			*di++ = 0; *di++ = (ubyte) u;
		}
		m_nCur = m_utf16[0];
		if (m_nCur < 0x80) return;	// ASCII

		const int trailLen = UTF8BytesOfLead[u] - 1;
		if (trailLen == 0 || m_pRead + trailLen > m_pEnd) goto ILLEGAL_BYTE;
		u &= 0x3F >> trailLen;
		for (int i = 0; i < trailLen; ++i) {
			const ubyte v = m_pRead[i];
			if (!UTF8IsTrailByte(v)) goto ILLEGAL_BYTE;
			u = u << 6 | v & 0b00111111;
		}
		/* redundant utf8 sequence */ {
			const int t = u <=      0x7FF ? 2 - 1
			            : u <=     0xFFFF ? 3 - 1
 #if 1
			            : u <=   0x10FFFF ? 4 - 1	// maxUnicode
 #else
			            : u <=   0x1FFFFF ? 4 - 1
			            : u <=  0x3FFFFFF ? 5 - 1
			            : u <= 0x7FFFFFFF ? 6 - 1
 #endif
			            :                     - 1;
			if (t != trailLen) goto ILLEGAL_BYTE;
		}
		m_pRead += trailLen;
		di = (ubyte *)m_utf16; // (uint16_t*) --> (unsigned char*)
		if (u >= 0x010000) {
			const uint32_t a = u - 0x010000;
			const uint16_t hi = (uint16_t)(a / 0x0400 + 0xD800);
			const uint16_t lo = (uint16_t)(a % 0x0400 + 0xDC00);
			if (m_isBigEndian) {
				*di++ = (ubyte) hi; *di++ = (ubyte)(hi >> 8);
				*di++ = (ubyte) lo; *di++ = (ubyte)(lo >> 8);
			} else {
				*di++ = (ubyte)(hi >> 8); *di++ = (ubyte) hi;
				*di++ = (ubyte)(lo >> 8); *di++ = (ubyte) lo;
			}
			m_len = 2;
		} else {
			if (m_isBigEndian) {
				*di++ = (ubyte) u; *di++ = (ubyte)(u >> 8);
			} else {
				*di++ = (ubyte)(u >> 8); *di++ = (ubyte) u;
			}
			m_len = 1;
		}
		m_index = 0;
	}
	m_nCur = m_utf16[m_index++];
	return;
ILLEGAL_BYTE:
	return;
#else // !UTF16_SURROGATE_PAIR
	switch (m_eState)
    {
        case eStart:
            if (*m_pRead < 0x80) {
                m_nCur = *m_pRead;
                toStart();
            } else if (*m_pRead < 0xE0) {
                m_nCur = static_cast<utf16>((0x1F & *m_pRead) << 6);
                m_eState = e2Bytes_Byte2;
            } else {
                m_nCur = static_cast<utf16>((0xF & *m_pRead) << 12);
                m_eState = e3Bytes_Byte2;
            }
            break;
        case e2Bytes_Byte2:
        case e3Bytes_Byte3:
            m_nCur |= static_cast<utf8>(0x3F & *m_pRead);
            toStart();
            break;
        case e3Bytes_Byte2:
            m_nCur |= static_cast<utf16>((0x3F & *m_pRead) << 6);
            m_eState = e3Bytes_Byte3;
            break;
	}
	++m_pRead;
#endif // UTF16_SURROGATE_PAIR
}

#ifdef UTF16_SURROGATE_PAIR
#else
void Utf8_Iter::toStart()
{
	m_eState = eStart;
	if (m_eEncoding == uni16BE || m_eEncoding == uni16BE_NoBOM)
    {
		swap();
	}
}

void Utf8_Iter::swap()
{
	utf8* p = reinterpret_cast<utf8*>(&m_nCur);
	utf8 swapbyte = *p;
	*p = *(p + 1);
	*(p + 1) = swapbyte;
}
#endif

//==================================================
Utf16_Iter::Utf16_Iter()
{
	reset();
}

void Utf16_Iter::reset()
{
	m_pBuf = NULL;
	m_pRead = NULL;
	m_pEnd = NULL;
#ifdef UTF16_SURROGATE_PAIR
	m_index = m_len = 0;
#else
	m_eState = eStart;
#endif
	m_nCur = 0;
#ifdef UTF16_SURROGATE_PAIR
#else
	m_nCur16 = 0;
#endif
	m_eEncoding = uni8Bit;
}

void Utf16_Iter::set(const ubyte* pBuf, size_t nLen, UniMode eEncoding)
{
	m_pBuf = pBuf;
	m_pRead = pBuf;
	m_pEnd = pBuf + nLen;
	m_eEncoding = eEncoding;
	if (m_eEncoding == uni16BE || m_eEncoding == uni16BE_NoBOM) { H = 0; L = 1; }
	else                                                        { L = 0; H = 1; }
#ifdef UTF16_SURROGATE_PAIR
	m_nIncompleteBytes = 0;
#else
	m_eState = eStart;
#endif
	operator++();
	// Note: m_eState, m_nCur, m_nCur16 not reinitalized.
}

// Goes to the next byte.
// Not the next symbol which you might expect.
// This way we can continue from a partial buffer that doesn't align
void Utf16_Iter::operator++()
{
#ifdef UTF16_SURROGATE_PAIR
	if (m_index >= m_len) {
		/*
		 * UTF-16 ---decode---> U+XXXXXX (UTF-32) ---encode---> UTF-8
		 */
		uint32_t u;
		u = m_pRead[H] << 8 | m_pRead[L];
		m_pRead += 2;
		if (u < 0x80) {
			m_nCur = (ubyte) u;
			return;
		}
		if (0xD800 <= u && u < 0xDC00) {
			if (m_pRead + 2 <= m_pEnd) {
				const uint32_t v = m_pRead[H] << 8 | m_pRead[L];
				if (0xDC00 <= v && v < 0xE000) {
					u = (u - 0xD800) * 0x0400 + (v - 0xDC00) + 0x10000;
					m_pRead += 2;
				}
			}
			else {
				m_pRead -= 2;
				m_nIncompleteBytes = m_pRead - m_pBuf; // 2 bytes
				m_pRead = m_pEnd + 1; // operator bool
				return;
			}
		}
 //		if (0xD800 <= u && u < 0xE000) {
 //			u = 0xFFFD;
 //		}
		assert(u <= maxUnicode);
		m_len = u <=      0x7FF ? 2
		      : u <=     0xFFFF ? 3
 #if 1
		      :                   4;	// u <= 0x10FFFF maxUnicode
 #else
		      : u <=   0x1FFFFF ? 4
		      : u <=  0x3FFFFFF ? 5
		      : u <= 0x7FFFFFFF ? 6
		      : /*             */ 6;
 #endif
		for (int i = m_len; --i > 0; ) {
			m_utf8[i] = u & 0b00111111 | 0b10000000;
			u >>= 6;
		}
		m_utf8[0] = (ubyte)(0xFFFFFF00 >> m_len | u);
		m_index = 0;
	}
	m_nCur = m_utf8[m_index++];
#else // !UTF16_SURROGATE_PAIR
	switch (m_eState)
    {
        case eStart:
            if (m_eEncoding == uni16LE || m_eEncoding == uni16LE_NoBOM) 
            {
                m_nCur16 = *m_pRead++;
                m_nCur16 |= static_cast<utf16>(*m_pRead << 8);
            }
            else //(m_eEncoding == uni16BE || m_eEncoding == uni16BE_NoBOM)
            {
                m_nCur16 = static_cast<utf16>(*m_pRead++ << 8);
                m_nCur16 |= *m_pRead;
            }
            ++m_pRead;
            
            if (m_nCur16 < 0x80) {
                m_nCur = static_cast<ubyte>(m_nCur16 & 0xFF);
                m_eState = eStart;
            } else if (m_nCur16 < 0x800) {
                m_nCur = static_cast<ubyte>(0xC0 | m_nCur16 >> 6);
                m_eState = e2Bytes2;
            } else {
                m_nCur = static_cast<ubyte>(0xE0 | m_nCur16 >> 12);
                m_eState = e3Bytes2;
            }
            break;
        case e2Bytes2:
        case e3Bytes3:
            m_nCur = static_cast<ubyte>(0x80 | m_nCur16 & 0x3F);
            m_eState = eStart;
            break;
        case e3Bytes2:
            m_nCur = static_cast<ubyte>(0x80 | ((m_nCur16 >> 6) & 0x3F));
            m_eState = e3Bytes3;
            break;
    }
#endif // UTF16_SURROGATE_PAIR
}


