/* Copyright (C) 2021 Momi-g

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program. If not, see <http://www.gnu.org/licenses/>.
*/

/*SH_doc
title=lbc section=3 repnl=\040

@name lbc
@_brief helper macro/api to embed luajit code into C
@_syno
#include "lbc.h"
typedef struct struct lbc_tag {
	lua_t* L;
	char* emsg;
	int rc;
	char glname[64];
	int lpflg;
	int used;
} lbc_t;
	
typedef struct lbc_rttag {
	int inner_flg; 
	int  tp;
	int i;
	double d;
	const char* s;
	int sz;
} lbc_rt;
	
//runner
lbc_t* lbc_new(void);
void lbc_free(lbc_t* obj);
int lbc_reqbcM(lbc_t* obj, ...);
int lbc_runbcM(lbc_t* obj, int ac, char** av, FILE_IDENT );
int lbc_runstr(lbc_t* obj, int ac, char** av, const char* str);
int lbc_prunbcM(lbc_t* obj, int ac, char** av, FILE_IDENT );
int lbc_prunstr(lbc_t* obj, int ac, char** av, const char* str);

//result-reader	
lbc_rt lbc_tbval(lbc_t* obj, ...);
int lbc_tbvalsv(int nidx, lbc_t* obj, ...);
int lbc_delsv(lbc_t* obj, int idx);
int lbc_cleansv(lbc_t* obj);
int lbc_directf(int nidx, lbc_t* obj, ...);
int lbc_pdirectf(int nidx, lbc_t* obj, ...);
#define LBC_C2P	//see NOTES. for directf/pdirectf 
#define LBC_P2C
#define LBC_P2S

//result-support
lbc_pairs(k, v, obj, ...){...}	//macro
lbc_ipairs(i, v, obj, ...){...}
int lbc_keysv(int nidx, lbc_t* obj);
int lbc_valsv(int nidx, lbc_t* obj);
	
//lhh, use luatb as hash instead of c99 hsearch()
typedef struct luahash_tag{
	lua_t* L;
	char* emsg;
	int rc;
	char glname[64];
	int lpflg;
	int cnt;
} lhh_t;
typedef lbc_rt lhh_rt;

lhh_t* lhh_new(void);
void lhh_free(lhh_t* obj);
int lhh_set(lhh_t* obj, key, val, ...);
int lhh_unset(lhh_t* obj, key);
lhh_rt lhh_get(lhh_t* obj, key);

// use `double` args instead of `int`
int lhh_setd(lhh_t* obj, key, val, ...);
int lhh_unsetd(lhh_t* obj, key);
lhh_rt lhh_getd(lhh_t* obj, key);

@tl;dr
	@(code)@
	#include "lbc.h"
	int main(int argc, char** argv) {
	  lbc_t* obj = lbc_new();
	  int rc = lbc_runstr(obj, argc, argv, "print('from_lua');return 123");
	  lbc_rt val = lbc_tbval(obj, 0, 1);
	  printf("%d\n", val.i);	//>> 123
	  lbc_free(obj);
	  return 0;
	}
	//~$ cc src.c liblbc.a -lm -ldl
	@()
@_desc
	lbc assists embed luajit code to C. basic API is:
	@(pre)
	- lbc_new() makes luajit vm
	- lbc_reqbcM() imports dependent lj bytecode file as require()
	- lbc_runbcM(), lbc_runstr() execute byte/str code
	- lbc_tbval() gets rtn/tb val with C type, int, char* etc
	- lbc_free() closes luajit vm and release memory
	@()--
	additional API is:
	@(pre)
	- lbc_directf() executes luajit code directly as lbc_runstr()
		only available after runbcM()/runstr()
	- prunbcM(), prunstr(), pdirectf() catches assert() and rtn emsg
	- lbc_tbvalsv(), delsv(), cleansv() treats rtn/tb val
	- lbc_pairs(), ipairs(), keysv(), valsv() handle luatb from C.
		mimic the syntax of lua.
	@()--
	'lhh' is hash api using luatb. lhh is independet from lbc.
	lhh can uses lbc_pairs() / lbc_ipairs() macro. see samples.
	@(pre)
	- lhh_new() creates hashtable. setup luavm and luatb internally.
	- lhh_set() sets key/val to hash.
	- lhh_unset() removes key/val from hash.
	- lhh_get() gets val from hash.
	- lhh_free() closes hash(luavm) and releases memory.
	@()--
	setd(),unsetd(),getd() uses double type instead of integer.

@_eg
		@(code)@
	--mod.lua
	local M={}
	function M.myf() print("from_mod"); return 10; end
	return M

	--run.lua
	local m = require("mod")
	m.myf()
	return -3, "gw"
	-- ~$ luajit run.lua	#>> (test_run) from_mod
	-- ~$ luajit -b mod.lua mod.h	#>> prepare bytecode mod.h
	-- ~$ luajit -b run.lua run.h	#>> run.h
	
	//--src.c 
	#include "lbc.h"
	int main(int argc, char** argv) {
	  lbc_t* obj = lbc_new();
	  #include "mod.h"
	  lbc_reqbcM(obj, mod);	//macro: mod >> "mod.h" strlen(mod_XX);
	  #include "run.h"
	  int n = lbc_runbcM(obj, argc, argv, run);	//n=len{-3, "gw"}=2
	    // set argv to _G.arg, works as ~$ luajit run.lua ag1 ag2 ...
	  lbc_rt val = lbc_tbval(obj, 0, 2);	// val=obj[0][2]
	  printf("%s\n", val.s);	// val.tp=='s'tring, val.s== "gw"
	  lbc_free(obj);
	  return 0;
	}
	//~$ gcc src.c -static liblbc.a -lm -ldl
	//(or ~$ gcc src.c -lluajit5.1 -lm -ldl liblbc.a)
	//..liblbc.a holds whole libluajit5.1.a files.
	
	//--------src2.c,	detail sample code
	#include "lbc.h"
	int main(int ac, char** av) {
	  lbc_t* obj = lbc_new();
	  #include "mod1.h"
	  #include "mod2.h"
	  #include "mod3.h"
	  #include "mod4.h"
	  lbc_reqbcM(obj, mod1, mod2, mod3);	// support at least 9 args
	  lbc_reqbcM(obj, mod4);	// allow req many times
	  int n = lbc_runstr(obj, ac, av, "return require('mod2').myf()");
	//call only once lbc_runbcM() or lbc_runstr() 
		
	  lbc_rt val=lbc_tbval(obj,0,1);	//obj[0]={r1,r2}, obj[0][1]=r1 etc
	//luartns are saved to [0], return 1,2,{10,20} >> obj[0]={1,2,{10,20}} 
	//- int val.tp='s': (n)um(s)tr(t)b(b)ool(N)il(f)c(T)hread(u)data(l)udata
	//- tp='s' >> sz=bytesz, tp='t' >> sz==#tb (tb[1]-tb[n], cnt only seq)
	//- tp: val.i/val.d/val.s/val.sz == int/dbl/const char* /int 
	//- test yourself if needs tp check: val.d-val.i==0 (...maybe int)
	//- true/false,nil is converted to int val.i 1/0
	//- val cant hold non-C data, table/func etc. val.tp check only.
	
	// allows int/str key, tbval(obj,0,"k",&p,n)==obj[0]["k"]["str"][3]
	//- use as str if ag starts with "/&. char* p="str", char** p=&str
	//- strkey cant holds '\0'. not "/& args are treated as int
	//- dbl cant use as key, lbc_tbval(obj, 3.01, "key1") == obj[3]["key1"]
	
		int idx=lbc_tbvalsv(3,obj,0,2); // obj[3]=obj[0][2]={1,k="a\0z"} etc
	// copy from idx[0][2] to dst idx[3]. dstidx allows only intnum.
	// lbc_runXX(), directf() uses obj[0] as rtnbuff and overwrite.
	
	  idx = lbc_tbvalsv(-1, obj, 0, 1);	//dstidx<0: use autoidx. rtn= -1
	  idx = lbc_tbvalsv(-9, obj, 0,"a");	//rtn=-2,obj[-2]=obj[0]["a"]=nil
	  idx = lbc_tbvalsv(-1, obj, 0, 2);	//rtn= -2 (upper obj[-2]=nil)
	  lbc_delsv(obj, -2);	// obj[-2]=nil
	  lbc_cleansv(obj);	// del all autoidx, obj[idx<0]
	  idx = lbc_tbvalsv(-7,obj,0); //obj[-1]=obj[0],rtn= -1
	// you must manage saveidx yourself if use idx>0
	// you may use autoidx for local val or tmpbuff pool etc.
	
	//------support api usage
	//mimic the lua forloop, lbc_pairs/ipairs/keysv/valsv (macro/funcs)
	//(k,v,obj,0,"key") >> obj[0]["key"]	...you cant use dbl as keyidx
	  lbc_pairs(k, v, obj, 0){
	    printf("%c, %c\n", k.tp, v.tp);		// obj[0] must be tb
	    printf("%d, %d\n", k.i, v.i);		//k,v holds lbc_rt data
	  }
	  lbc_ipairs(num, val, obj, -3){
	    printf("%d,%c\n",num,val.tp);	//obj[-3], ipair key is int
	    int idx;
	    idx=lbc_keysv(obj, -8); //obj[autoidx]=keydata, no needs ag 'num'
	    idx=lbc_valsv(obj, 2 ); //obj[2]=obj[-3][keydata]
	    if(num>3){break;}
	  }
	  lbc_rt ck=lbc_tbval(obj, 2); //ck.i==123 etc. saved at pairs/ipairs
	
	//---
	//you can use lbc_directf() after lbc_runXX()
	//almost the same as lbc_runstr()
	//1: no argc, argv
	//2: share the luajit global env. share the _G value 
	//3: lua_pushfstring() fmt, restricted printf() fmt. see lua manual
	//4: rtn is saved to obj[0]
	  n = lbc_directf(obj, "print('hw'); return 10,%d", 20); //>> hw, n=2
	  val = lbc_tbval(obj, 0, 2);		//>> retuns to obj[0], val.i==20
	
	//you can access/edit obj[0][2] to use obj->glname as follows
	  lbc_directf(obj,"print(%s[0][2]);return 1,'gw'",obj->glname); //20
	  val = lbc_tbval(obj, 0, 2);	// val.s == "gw", overwrote obj[0]
	
	//prunbcM(), prunstr(), pdirectf() works as pcall()
	//catches inner error, assert() and 'return emsg' , returns 1 str if err.
	  n= lbc_pdirectf(obj,"assert(nil,'abc')"); // err: n= -1, rtnsz=1
	  if(n<0){ val = lbc_tbval(obj, 0, 1); }	// val.s == "abc", errmsg.
	// rc 'n' is the same as runbcM(), count of return args if noerr
	
	  lbc_free(obj);
		
	//----
	//--luahash, lhh is independent from lbc, lhh->L != lbc->L
	//keystr cant holds '\0' but valstr can if set ag4.
	  lhh_t* rec = lhh_new();
	  int sum = lhh_set(rec,1,10); // rec[1]=10, sum=1(cnt holds data)
	  lhh_set(rec, "abc", -3);	// rec["abc"] = -3
	  lhh_set(rec, 11, "xy") ;	// rec[11] = "xy", sum=2
	  lhh_set(rec,"a","a\0b",3);	// rec["a"]=a(\0)b, sz=3 (ag4)
	  const char* ky = "abc";
	  lhh_set(rec, &ky, -4);	// rec["abc"] = -4, sum=4, overwrite
	
	//args numtype is int in dfl. XXd treats args as double type.
	//set the correct type/value or cast (int) (double) if use numarg 
	  lhh_setd(rec, 2.2, "a");	//rec[2.2]="a", setd() uses k/v dbl
	  lhh_setd(rec, (double)2, 2.7); // rec[2]=2.7, treated by the luarule
	  lhh_unsetd(rec, 2.2);	//rec[2.2]=nil, unset()/unsetd()
	  lhh_unset(rec, "nosetkey");	//noerr
		
	//get
	  lhh_rt v = lhh_get(rec, "abc");	// v.tp='n', v.i= -4, v.tp/i/d/s/sz
	  v = lhh_getd(rec, (double)11);	// v.tp='s', v.s="xy", v.sz=2
	  v = lhh_get(rec, "a");	// v.tp='s', v.sz=3
	//lhh can borrow lbc pairloop
	  lbc_pairs(k, v, rec){
	    printf("%s/%d %s/%d\n", k.s?k.s:"", k.i, v.s?v.s:"", v.i);
	    lhh_unset(rec, 1);	// ok, del in loop
	    lhh_set(rec, 11, "xxyz");	// ok, replace in loop
	    //lhh_set(rec, 99, 1);
	    //NG, add newval. use other buff, lhh_set(rec2, 99, 1) etc
	    //lhh_get(rec, 1);
	    //NG, get() breaks stack sequence. use k.i, v.i etc
	  }
	  lhh_free(rec);
	  return 0;
	}
	//~$ gcc src2.c -lluajit5.1-a -lm -ldl liblbc.a
	//if you use valgrind memcheck, use -static to avoid miscount.
	//~$ gcc src2.c -static liblbc.a -lm -ldl
		@()

@_opt	-
@exit_staus	-
@return_value
	@(code)@
	int lbc_reqbcM() : suc/fail == 0/not0
	int lbc_delsv()  : same 
	int lbc_cleansv(): same

	int lbc_runbcM() : cnt return args. eg) (lua)return 0,"a",{1,2} >> 3
	int lbc_runstr() : same
	int lbc_directf(): same
	int lbc_prunbcM(): same if no err. return -1 and set one emsg if err.
	int lbc_prunstr(): same
	int lbc_pdirectf(): same

	int lbc_tbvalsv(): savedist index
	int lbc_keysv()  : same	
	int lbc_valsv()  : same

	int lhh_set()  : count of all holding keys
	int lhh_setd() : same
	int lhh_unset(): same
	int lhh_unsetd(): same
		@()
@errors output emsg and exit(1)
@notes
- sloppy benchmark:
	@(code)@
	--luajit vs lbc
	lj : for i=1,num do local local v=i+1 end
	C  : while(num){ lbc_tbvalsv(2, obj, 0, 1); }
	//lj : real 154.00ms    : loop: 100*1000*1000: (1)
	//lbc: real 2943.116 ms : loop:   1*1000*1000: 20x100 == (2000)

	--c99 hash vs lhh
	  FAST: c_arr,lj(1) > c99hash(2) >> lhh(10) >>> lbc(200) :SLOW

	- c99 posix hsearch()
	  real	46.685 ms	: c99setloop
	  real	44.315 ms	: c99getloop
	  
	- c arr[], arr[10]=100 etc
	  real	21.068 ms	: arr, setloop
	  real	16.977 ms	: arr, getloop
	  
	- luajit, table/arr tb[123]="abc"	etc
	  set/get 22ms
	  
	- lhh() lhh_set(obj,"key",10) etc
	  real 197.392 ms : lhh, setloop
	  real 192.476 ms : lhh, getloop
	  
	- lbc_directf(), "local tb[123]=456; return" etc
	  real 3936.665 ms: .directf
		@()--
	..luajit vm is very fast but lua_stack() handling cause bottlenecks.
	maybe you can bypass it with luajit ffi() cdata, direct pointer access.
	(https://luajit.org/ext_ffi_tutorial.html, Cast pointer to address)
	@(code)--
	normal usage                  use ffi() cdata
-----+          +-----          -----+          +-----
     | push/pop |               lj() +----------+ C_func()
lj() +----------+ C_func()           +----------+ 
     +----------+                    |          | 
     |          |            cdata <--------------> arr[]
-----+          +-----          -----+          +-----
	@()--
	--- bypass code: lj >>> C
		@(code)@
	const char* cmd = 
	  " ffi = require('ffi')"	//head "(space)", lua is free format  
	  " local tb = {}; for i=1,1000 do tb[#tb+1]=i end"
	  " local box= ffi.new('int[?]', #tb, tb )"
	  " local adrs= " LBC_C2P("box")
	//-- macro 'LBC_C2P()' works as below, cdata to ptrnum
	//" local adrs=tonumber(ffi.cast('intptr_t',ffi.cast('void *',box)))"
 
	  " return adrs, box"		//holds datasrc 'box' to protect from gc.
	;
	lbc_directf(obj, "%s", cmd);
	lbc_rt res = lbc_tbval(obj, 0, 1);	//adrs
	int* arr = (int*)(intptr_t)res.d;	// adrs(number) >> arr(ptr)
	printf("%d %d\n", arr[0], arr[1]);		//>>1,2
	@()--
	--- bypass code: lj <<< C
		@(code)@
	#include <inttypes.h>
	const char* s = "a\0bc";
	const char* fmt = " ffi = require('ffi')"
	  " local str = " LBC_P2S
	
	//-- macro 'LBC_P2S' works as printf fmt. needs ptr + sz
	// " local cdata = ffi.new('uintptr_t', %" PRIuPTR ")"
	// " cdata = ffi.cast('const char*', cdata)"
	// " local str = ffi.string(cdata, %d)"
	
	//-- macro 'LBC_P2C' works as below. needs ptr only
	// " local cdata =" LBC_P2C
	// " cdata = ffi.cast(void*, ffi.new('uintptr_t', %" PRIuPTR "))"

	" return str..'xyz'"	//gc deletes cdata. res is copied to 'str'
	;
	lbc_directf(obj, fmt, s, 4);	// ptr,binsz == "a\0bc", 4
	res = lbc_tbval(obj, 0, 1);	
	printf("%c %s %s, %d\n", res.tp,res.s,res.s+2,res.sz);
	  // 's', a(\0bcxyz), bcxyz, 7
	lbc_free(obj);

// num<->ptr cast allows only throught intptr_t/uintptr_t in luajit.
// printf("%p\n", p) / printf("%" PRIuPTR "\n", p) work allmost the same
// the latter is more portable (c99 defined)
// ffi.string(cdata, len) can get str/bin[len] data from C

--- concept
  - use luajit code as __asm__ in C
  - i want to embed normal luajit src.lua directly to C src
  - no glue code, glue file
  - get luacode result without complex stack handling
  - output C code as 1 file without dependent files (a.out, libX.so etc) 
  - easy api, low learning cost
  - portable, posix
	@()

@conforming_to POSIX.1-2001+
@COPYRIGHT Copyright 2021 momi-g, GPLv3+
@_ver 2021-08-28 v1.1.1
@_see
	@(code)
https://www.lua.org/
https://luajit.org/
https://stackoverflow.com/questions/2632300
https://stackoverflow.com/questions/44479282
	@()
//SH_docE*/
/* tool macros */
#ifndef ERRact
#include <stdio.h>
#include <errno.h>
#include <unistd.h>
#if  (200112L <= _POSIX_C_SOURCE +0)  /* nealy 200112L, _POSIX_C_SOURCE	c99*/
#define ERRact(xpr, msg, act)	if(xpr){ fprintf(stderr, \
	        "ERR: %s %d %s() pid:%d %s msg:%s sys:%s\n",__FILE__,__LINE__, __func__ \
	        ,getpid(), "hit(" #xpr ")", msg, strerror(errno) );act ;}
#else
#define ERRact(xpr, msg, act)	if(xpr){ fprintf(stderr, \
	        "ERR: %s %d %s() pid:%d %s msg:%s sys:%s\n",__FILE__,__LINE__, "func:c99+" \
	        ,0 , "hit(" #xpr ")", msg, strerror(errno) );act ;}
#endif
#define STOP(xpr, msg)	ERRact(xpr, msg, fputs("STOP\n",stderr);exit(1))
#endif
#define loop(a)		for(int lpcnt=1;lpcnt<=a;lpcnt++)
/*tool end*/

#include <stdlib.h>
#include <string.h>

#include "luaconf.h"
#include "lua.h"
#include "lauxlib.h"
#include "lualib.h"

#define LBC_SOURCECODE
#include "lbc.h"

lbc_t* lbc_new(void) {
	lua_State *L = luaL_newstate();
	luaL_openlibs(L);
	lbc_t* res = (lbc_t*)calloc(1, sizeof(lbc_t));
	res->L = L;
	return res;
}
void lbc_free(lbc_t* obj) {
	lua_close(obj->L);
	free(obj->emsg);
	*obj=(lbc_t) {0};
	free(obj);
}



//bcどもをmod登録。bcはtbでrtnしていることを前提にしてる
int lbc_reqbc_impl(lbc_t* obj, ...) {
	// const char* bc, int bcsz, const char* name, ...
	lua_t* L = obj->L;
	lua_getglobal(L, "package");
	lua_getfield(L, -1, "preload");	//preloadはtbでnameなtbが有るとmodとして登録される
	//preloadはsearch関数返し。bcは先頭当たりにsearch関数を持ってるのでそれでブート
	va_list vp;
	va_start(vp, obj);
	//bc >> tb
	while(1){
		const char* bc = va_arg(vp, char*);
		int bcsz = va_arg(vp, int);
		const char* name = va_arg(vp, char*);
		if(bc==NULL){break;}
		lua_pushstring(L, name);
		int rc = luaL_loadbuffer(L, bc, bcsz, name);
		; ERRact(rc, "fatal: load bc failed", printf("name: %s\n", name); exit(1));
		lua_settable(L, -3);
	}
	va_end(vp);
	lua_settop(L, -3);
	return 0;
}

int lbc_runbc_impl(int pflg, lbc_t* obj, int ac, char **av, const void* bcs, int bcsz, const char* name) {
	//bcはstrもいれた。自動判別なのでうまくいくはず
	; STOP(obj->used, "obj already ran");
	const char* bc=bcs;
	lua_t* L = obj->L;
	obj->used=1;
	int rc = luaL_loadbuffer(L, bc, bcsz, name);
	; ERRact(rc, "fatal: load bc failed", printf("name: %s\n", name); exit(1));
	// http://milkpot.sakura.ne.jp/lua/lua51_manual_ja.html#6
	lua_settop(L, 0);
	lua_newtable(L);
	for(int i=0; i<ac; i++) {
		lua_pushstring(L, av[i]);
		lua_rawseti(L, -2, i);
	}
	lua_setglobal(L, "arg");
	//
	rc = luaL_loadbuffer(L, bc, bcsz, name);
	; ERRact(rc, "fatal: load bc failed", printf("name: %s\n", name); exit(1));
	lua_getglobal(L, "arg");
	int narg;
	for(narg=1; !lua_isnil(L,-1); narg++) {lua_rawgeti(L, 2, narg); }
	lua_pop(L, 1);
	lua_remove(L, 2);
	rc = lua_pcall(L, narg-2, LUA_MULTRET, 0);	//0はerrcbがあればfcidx

	//rc!=0なら1にemsgが詰まってる
	; STOP(pflg==0 && rc, lua_tolstring(L, 1, NULL) );
	if(rc){ pflg= -1;}
	
	//rtnはtbに詰めて返す
	int rtnsz = lua_gettop(L);	// a|b|c, 3
	lua_newtable(L);	//	a|b|c| tb
	lua_insert(L, 1);	//	tb |a|b|c
	for(int i=rtnsz; i>0; i--) {
		lua_pushnumber(L, i);	//	tb |a|b|c|3
		lua_insert(L, -2);	//tb |a|b|3|c
		lua_rawset(L, 1);	// tb |a|b  tb[3]=c
	}
	if(pflg<0){ rtnsz=1;}	//pcallはrc== -1でemsgが一つだけ
	
	//globalのarrを探す
	int i=0;
	for(i=0;; i++) {
		sprintf(obj->glname, "%s%d", LBC_GLNAME, i);
		lua_getglobal(L, obj->glname);	// tb | gl
		if(lua_isnil(L, -1)) { break;}
		lua_pop(L,1);	// tb
	}
	lua_pop(L, 1);
	//nameを見つけたのでtbをセット
	lua_newtable(L);	//	res|tb
	lua_pushnumber(L, 0);	//	res|tb|0
	lua_insert(L, 1);
	lua_insert(L, 1);//	tb|0|res
	lua_rawset(L, 1);	// tb[0]=restb, tbr
	lua_setglobal(L, obj->glname);
	return rtnsz;	//rtn nargsを返す errは-1でobjにemsgが入る
	//Lとglname[0]={1,2,3} とtbszのrtnszが出来た
	//ハンドリングは別関数で試す。とりあえずここまでテストしてmcroも準備する
}

static const char* lbc_tbval_ag2tb(lua_t* L, va_list vp);
lbc_rt lbc_tbval_impl(lbc_t* obj, const char* pidx, int idx, ...) {
	; ERRact(obj->lpflg, "thisfunc cant use in pairloop blk", exit(1) );
	lua_t* L = obj->L;
//
	va_list vp;
	va_start(vp, idx);
	const char* rs;
	rs = lbc_tbval_ag2tb(L, vp);
	va_end(vp);
	rs = lua_pushfstring(L, "return %s[%d]%s", obj->glname, idx, rs);
//
	int rc = luaL_dostring(L, rs);
	; ERRact(rc, "catch err, bad code?", fputs(rs, stderr); exit(1));
	lua_replace(L, 1);	//強制的にtop==1 rtnでstk保持必須だからしょうがないね
	lua_settop(L, 1);
//
	lbc_rt res= {0};
	//val
	if(lua_type(L, -1) == LUA_TNUMBER) {
		res.tp='n';
		res.s=NULL;
		res.i = lua_tointeger(L, -1);
		res.d = lua_tonumber(L, -1);
	} else if(lua_type(L, -1) == LUA_TSTRING) {
		res.tp='s';
		res.i=0;
		res.d=0;
		size_t sz=0;
		res.s = lua_tolstring(L, -1, &sz);
		res.sz = (int)sz;
	} else if(lua_type(L, -1) == LUA_TTABLE) {
		res.tp='t';
		res.i=0;
		res.d=0;
		res.s=NULL;
		res.sz = lua_objlen(L, -1);
	} else if(lua_type(L, -1) == LUA_TNIL) { res.tp='N'; }
	else if(lua_type(L, -1) == LUA_TBOOLEAN) {
		res.tp='b';
		const char* p = lua_tostring(L, -1);
		if(p[0]=='t') {res.i=1;}
	} else if(lua_type(L, -1) == LUA_TFUNCTION) { res.tp='f'; }
	else if(lua_type(L, -1) == LUA_TUSERDATA) { res.tp='u'; }
	else if(lua_type(L, -1) == LUA_TTHREAD) { res.tp='T'; }
	else if(lua_type(L, -1) == LUA_TLIGHTUSERDATA) { res.tp='l'; }
	else { STOP(1, "unreachable block"); }
	return res;
}

static int lbc_searchnidx(lbc_t* obj);
int lbc_tbvalsv_impl(int nidx, lbc_t* obj, const char* pidx, int idx, ...) {
//	; ERRact(nidx==0, "[0] is blocked idx", exit(1));
//	; ERRact(obj->lpflg, "thisfunc cant use in pairloop blk", exit(1));
	if(nidx<0){ nidx = lbc_searchnidx(obj); }
	lua_t* L = obj->L;
//
	va_list vp;
	va_start(vp, idx);
	const char* rs;
	rs = lbc_tbval_ag2tb(L, vp);
	va_end(vp);
//
	rs = lua_pushfstring(L, "%s[%d]=%s[%d]%s", obj->glname, nidx, obj->glname, idx, rs);
	int rc = luaL_dostring(L, rs);
	lua_settop(L, -3);	//ag2とcmdの二つ
	; STOP(rc, "fatal err");
	return nidx;
}

// return系の共通コード ""とか&とか normalとsvで2x2で4倍になるので固めてる
// ("a",1,"z")	>> ["a"][1]["z"]
static const char* lbc_tbval_ag2tb(lua_t* L, va_list vp) {
	const char* rs="";
	lua_pushstring(L, "");	//dmy
	while(1) {
		char* p = va_arg(vp, char*);
		; if(p==NULL) {break;}	//sentinel
		if(p[0]=='"' || p[0]=='&') {
			char* s;
			if(p[0]=='"') { s = va_arg(vp, char*); }
			if(p[0]=='&') { s = *va_arg(vp, char**); }	//&系は二重なので細工
			//key == str "anc" or &s == ["abc"] ["ptr_s"]
			; ERRact(strchr(s, '"'), "keystr holds double-quote",fputs(p, stderr); exit(1));
			rs = lua_pushfstring(L, "%s[\"%s\"]", rs, s);
		} else {
			//key==num
			int inum = va_arg(vp, int);
			rs = lua_pushfstring(L, "%s[%d]", rs, inum);
		}
		lua_remove(L, -2);
	}
	return rs;
}
static int lbc_searchnidx(lbc_t* obj) {
	lua_t* L = obj->L;
	const char* fmt =
	"local i=-1;while 1 do local v = %s[i];if v==nil then return i end;i=i-1;end";
	const char* p = lua_pushfstring(L, fmt, obj->glname);
	int rc = luaL_dostring(L, p);	//(luaL_loadstring(L, str) || lua_pcall(L, 0, LUA_MULTRET, 0))
	; STOP(rc, "fatal err");
	int idx = lua_tonumber(L, -1);
	lua_pop(L, 2);
	return idx;
}

int lbc_delsv(lbc_t* obj,int idx) {
//	; ERRact(idx==0, "[0] is blocked idx", exit(1));
	lua_t* L = obj->L;
	const char*	fmt = "%s[%d] = nil";
	const char* cs = lua_pushfstring(L, fmt,  obj->glname, idx);
	int rc = luaL_dostring(L, cs);
	lua_pop(L, 1);
	; STOP(rc, "fatal err");
	return 0;
}
int lbc_cleansv(lbc_t* obj) {
	lua_t* L = obj->L;
	const char*	fmt = "for k,v in pairs(%s) do "
	                  "if type(k)=='number' and k<0 then %s[k]=nil end; end";
	const char* cs = lua_pushfstring(L, fmt,  obj->glname, obj->glname);
	int rc = luaL_dostring(L, cs);
	; ERRact(rc, "fatal err", fprintf(stderr, "%s\n",lua_tostring(L,-1)); exit(1));
	lua_pop(L, 1);
	return 0;
}
int lbc_directf_impl(int pflg, lbc_t* obj) {
//	; ERRact(nidx==0, "svidx 0 is booked", printf("save:%d\n", nidx); exit(1));
//	if(nidx<0) { nidx= lbc_searchnidx(obj);}
	lua_t* L = obj->L;
	int lv = lua_gettop(L);	//strがいる
	const char* p = lua_tostring(L, -1);
	int rc = luaL_dostring(L, p);	//(luaL_loadstring(L, str) || lua_pcall(L, 0, LUA_MULTRET, 0))
	; STOP(rc&&pflg==0, lua_tostring(L,-1) );
	if(rc){ pflg= -1; }
	//rtnはtbに詰めて返す
	int rtnsz = lua_gettop(L) - lv;	// cmd|a|b|c
//
	lua_newtable(L);	//	cmd|a|b|c| tb
	lua_replace(L, lv);	//	tb |a|b|c	..overwrite
	for(int i=rtnsz; 0<i; i--) {
		lua_pushnumber(L, i);	//	tb |a|b|c|3
		lua_insert(L, -2);	//tb |a|b|3|c
		lua_rawset(L, lv);	// tb |a|b  tb[3]=c
	}
	if(pflg<0){ rtnsz= -1; }
	lua_getglobal(L, obj->glname);	//	tb|gl
	lua_insert(L, -2);//	gl|tb
	lua_rawseti(L, -2, 0);	// gl[0]=tb
	lua_pop(L, 1);
	return rtnsz;
}


int lbc_keyvalsv_impl(int nidx, lbc_t* obj, int tgt) {
	lua_t* L = obj->L;
//	; ERRact(nidx==0, "use [1] or upper", printf("save:%d\n", nidx); exit(1));
	; STOP(obj->lpflg==0, "use only lbc_pair() loop");
	if(nidx<0) { nidx= lbc_searchnidx(obj);}
	if(tgt=='k') {tgt= -4;}
	if(tgt=='v') {tgt= -3;}
	//	..|k|v 初期スタック状態 forの中のみ
	lua_getglobal(L, obj->glname);
	lua_pushnumber(L, nidx);	//	k|v|name|3
	lua_pushvalue(L, tgt);	//...|k|v|name|3|kcopy の順になってるはず コピー追加
	lua_rawset(L, -3);		//...|k|v|name[3]
	lua_pop(L, 1);		//	..|k|v
	return nidx;
}

int lbc_pairexit(lbc_t* obj) {
	lua_settop(obj->L, -4);	// tb|k|v なので-4が本来の位置
	obj->lpflg=0;
	return 0;
}
int lbc_pairinit(int flg, lbc_t* obj, ...) {
	if(flg==0) {return 0;}
	obj->lpflg=1;
	lua_t* L = obj->L;
	lua_settop(obj->L, 0);
//
	va_list vp;
	va_start(vp, obj);
	const char* rs;
	rs = lbc_tbval_ag2tb(L, vp);
	va_end(vp);
	rs = lua_pushfstring(L, "return %s%s", obj->glname, rs);
//
	int rc = luaL_dostring(L, rs);
	; ERRact(rc, "catch err, bad code?", fputs(rs, stderr); exit(1));
	; ERRact(lua_type(L, -1)!=LUA_TTABLE, "idx doesnt have tb", printf("idx:%s\n", rs) );
	lua_insert(L, -3);	//agとcmdとrtnの三つでrtnを残す
	lua_pop(L, 2);
	//nilstartでnext準備で2つ乗っけておく
	lua_pushnil(obj->L);
	lua_pushnil(obj->L);
	return 1;
}
int lbc_isnextval(int mode, lbc_t* obj, void* k, lbc_rt* v) {
	lua_t* L=obj->L;
	lua_pop(L,1);	//最初に2こ積んでおけばいい
	lbc_rt* key = k;
	if(mode>0) {
		// arr系は直接問い合わせ popが邪魔だけど合わせる vが残るからそれでちょうどいい
		// breakは二つで起きるので常に2つ乗っける状態にする
		int* num = k;
		*num=*num+1;	//ここで追加なので0start
//		lua_pushnil(L);	//ダミー追加 stackのlenを合わせる
		lua_rawgeti(L, -2, *num );
		if(lua_type(L, -1) == LUA_TNIL) { v->inner_flg=0; goto lb_RTN; }
	} else {
		//key系はiter問い合わせ 空は終了 で2つ埋めて合わせる
		*key = (lbc_rt){0};
		if(lua_next(L, -2)==0) {
			//break
			 v->inner_flg=0;
			lua_pushnil(obj->L);
			lua_pushnil(obj->L);
			goto lb_RTN;
		}
		//key
		if(lua_type(L, -2) == LUA_TNUMBER) {
			key->tp='n';
			key->s=NULL;
			key->i = lua_tointeger(L, -2);
			key->d = lua_tonumber(L, -2);
		} else if(lua_type(L, -2) == LUA_TSTRING) {
			key->tp='s';
			key->i=0;
			key->d=0;
			size_t sz=0;
			key->s = lua_tolstring(L, -2, &sz);
			key->sz = (int)sz;
		} else if(lua_type(L, -2) == LUA_TNIL) { key->tp='N'; }
		else if(lua_type(L, -2) == LUA_TTABLE) { key->tp='t'; }
		else if(lua_type(L, -2) == LUA_TFUNCTION) { key->tp='f'; }
		else if(lua_type(L, -2) == LUA_TUSERDATA) { key->tp='u'; }
		else if(lua_type(L, -2) == LUA_TTHREAD) { key->tp='T'; }
		else if(lua_type(L, -2) == LUA_TLIGHTUSERDATA) { key->tp='l'; }
	}
	lbc_rt* val = v;
	*val = (lbc_rt){1};	//rcが入ってる
	//val
	if(lua_type(L, -1) == LUA_TNUMBER) {
		val->tp='n';
		val->s=NULL;
		int i = lua_tointeger(L, -1);
		double d = lua_tonumber(L, -1);
		val->i=i;
		val->d=d;
		if(i!=d) { val->d=d; val->i=0; }
	} else if(lua_type(L, -1) == LUA_TSTRING) {
		val->tp='s';
		val->i=0;
		val->d=0;
		size_t sz=0;
		val->s = lua_tolstring(L, -1, &sz);
		val->sz = (int)sz;
	} else if(lua_type(L, -1) == LUA_TTABLE) {
		val->tp='t';
		val->i=0;
		val->d=0;
		val->s=NULL;
		val->sz = 0;
	} else if(lua_type(L, -1) == LUA_TNIL) {
		val->tp='N';
		val->i=0;
		val->d=0;
		val->s=NULL;
		val->sz = 0;
	} else if(lua_type(L, -1) == LUA_TBOOLEAN) {
		val->tp='b';
		const char* p = lua_tostring(L, -1);
		if(p[0]=='t') {val->i=1;}
	} else if(lua_type(L, -1) == LUA_TFUNCTION) { val->tp='f'; }
	else if(lua_type(L, -1) == LUA_TUSERDATA) { val->tp='u'; }
	else if(lua_type(L, -1) == LUA_TTHREAD) { val->tp='T'; }
	else if(lua_type(L, -1) == LUA_TLIGHTUSERDATA) { val->tp='l'; }
	else { STOP(1, "unreachable block"); }
lb_RTN:;
	return v->inner_flg;
}







lhh_t* lhh_new(void){
	lua_State *L = luaL_newstate();
	; STOP(L==NULL, "fatal err");
	lhh_t* res = (lhh_t*)calloc(1, sizeof(lhh_t) );
	; STOP(res==NULL, "memerr");
	res->L = L;
	strcpy(res->glname, LHH_GLNAME);
	lua_newtable(L);
//	lua_pushnumber(L, 0);
//	lua_newtable(L);
//	lua_rawset(L, -3);
	lua_setglobal(L, LHH_GLNAME);
	return res;
}
void lhh_free(lhh_t* obj){
	lua_close(obj->L);
	*obj=(lhh_t){0};
	free(obj);
}

//#define lhh_set(obj, key, val, ...)	lhh_set_impl(0, obj, #key, key, #val, val, __VA_ARGS__+0)
//#define lhh_setd(obj, key, val, ...)	lhh_set_impl(1, obj, #key, key, #val, val, __VA_ARGS__+0)
int lhh_set_impl(int dmode, lhh_t* obj, const char* s, ...){
	lua_t* L = obj->L;
	va_list vp;
	va_start(vp, s);
//
//	lhh_preset(obj);
	lua_getglobal(L, obj->glname);
	char* p = (char*)s;
	//key
//dbg(dmode, lua_gettop(L) );
	if(p[0]=='"') { lua_pushstring(L, va_arg(vp, char*) ); }
	else if(p[0]=='&') { lua_pushstring(L, *va_arg(vp, char**) );}	//&系は二重なので細工
	else if(dmode){ lua_pushnumber(L, va_arg(vp, double) );}
	else{ lua_pushnumber(L, va_arg(vp, int) ); }
	//val
	p = va_arg(vp, char*);
//dbg("23", p[0]);
	if(p[0]=='"') {
		p = va_arg(vp, char*);
		int sz=0;
		if(dmode){ sz = (int)va_arg(vp, double); }
		else{ sz = va_arg(vp, int); }
		if(!sz){sz=strlen(p); }
		lua_pushlstring(L, p, sz);
	}else if(p[0]=='&') {
		p = *va_arg(vp, char**);
		int sz=0;
		if(dmode){ sz = (int)va_arg(vp, double); }
		else{ sz = va_arg(vp, int); }
		if(!sz){sz=strlen(p); }
		lua_pushlstring(L, p, sz);
	}else if(dmode){
		double num = va_arg(vp, double);
		lua_pushnumber(L, num);
	}else{
		int num = va_arg(vp, int);
		lua_pushnumber(L, num);
	}
	lua_rawset(L, -3);
	lua_pop(L, 1);
	va_end(vp);
	obj->cnt++;
	return obj->cnt;
}

int lhh_unset_impl(int dmode, lhh_t* obj, const char* s, ...){
	lua_t* L = obj->L;
	va_list vp;
	va_start(vp, s);
	char* p = (char*)s;
//
//	lhh_preset(obj);
	lua_getglobal(L, obj->glname);
	//key
	if(p[0]=='"') { lua_pushstring(L, va_arg(vp, char*) ); }
	else if(p[0]=='&') { lua_pushstring(L, *va_arg(vp, char**) );}	//&系は二重なので細工
	else if(dmode){	lua_pushnumber(L, va_arg(vp, double) );}
	else{ lua_pushnumber(L, va_arg(vp, int) ); }
	lua_pushnil(L);
	lua_rawset(L, -3);
	lua_pop(L, 1);
	obj->cnt--;
	return obj->cnt;
}

lhh_rt lhh_get_impl(int dmode, lhh_t* obj, const char* s, ...){
	lua_t* L = obj->L;
	lhh_rt res={0};
	va_list vp;
	va_start(vp, s);
	char* p = (char*)s;
//
	; STOP(obj->lpflg, "lhh_get() doesnt work in the pairloop to save stack state");
	lua_settop(L, 0);
	lua_getglobal(L, obj->glname);

//	lhh_preset(obj);
	//key
	if(p[0]=='"') { lua_pushstring(L, va_arg(vp, char*) ); }
	else if(p[0]=='&') { lua_pushstring(L, *va_arg(vp, char**) );}	//&系は二重なので細工
	else if(dmode){	lua_pushnumber(L, va_arg(vp, double) );}
	else{ lua_pushnumber(L, va_arg(vp, int) ); }
	lua_rawget(L, -2);
	lua_remove(L, 1);
	//numが先じゃないとstirngになっちゃう
	if(lua_isnumber(L, -1) ){
		res.tp='n';
		res.d = lua_tonumber(L, -1);
		res.i = (int)res.d;
	}else if(lua_isstring(L, -1) ){
		size_t sz;
		res.s = lua_tolstring(L, -1, &sz);
		res.sz = (int)sz;
		res.tp='s';
	}else{
		res.tp='N';
	}
	return res;
}

/*SH_SMP
#include "lbc.h"	//SH_co*	-static liblbc.a libluajit-5.1.a -ldl -lm	*
#include <stdio.h>

int main(int argc, char** argv) {
	lbc_t* obj = lbc_new();
//	#include "mod.h"
//	#include "run.h"
//	lbc_reqbcM(obj, mod);
//	lbc_runbcM(obj, argc, argv, run);
	lbc_runstr(obj, argc, argv, "return 1,2,3" );
	lbc_rt res = lbc_tbval(obj, 0, 1);
	printf("%c / %d\n", res.tp, res.i);
	lbc_free(obj);
	return 0;
}

//SH_SMPE*/



/*
 change log
 --
2021-08-28	Momi-g	<dmy@dmy.dmy>

	* lbc.sh.c(lbc_keysv, valsv): change f(key, obj) >> f(obj, key)

2021-07-05	Momi-g	<dmy@dmy.dmy>

	* lbc.sh.c(msgp.h): omit dbg()

2021-07-05	Momi-g	<dmy@dmy.dmy>

	* lbc.sh.c(lbc_rt): rename member 'rc' >> 'inner_flg'
	(doc): fix doc. v1.1.0

2021-06-23	Momi-g	<dmy@dmy.dmy>

	* lbc.sh.c(LBC_C2P, P2C, P2S): fix directf() ffi macros

2021-05-08	Momi-g	<dmy@dmy.dmy>

	* lbc.sh.c(all): v1.0.0 release

2021-04-25	Momi-g	<dmy@dmy.dmy>

	* lbc.sh.c(-): init

*/

