/****************************************************************************/
/* The developnent of this program is partly supported by IPA.              */
/* (Infomation-Technology Promotion Agency, Japan).                         */
/****************************************************************************/

/****************************************************************************/
/*  cloopprofiler.c                                                         */
/*  Copyright : Copyright (C) 2006 ALPHA SYSTEMS INC.                       */
/*  Authors : Daisuke Abe (abeda@alpha.co.jp)                               */
/*           ALPHA SYSTEMS INC. knoppix team(knoppix@alpha.co.jp)           */
/*                                                                          */
/*  This is free software; you can redistribute it and/or modify            */
/*  it under the terms of the GNU General Public License as published by    */
/*  the Free Software Foundation; either version 2 of the License, or       */
/*  (at your option) any later version.                                     */
/*                                                                          */
/*  This software is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/*  GNU General Public License for more details.                            */
/*                                                                          */
/*  You should have received a copy of the GNU General Public License       */
/*  along with this software; if not, write to the Free Software            */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,  */
/*  USA.                                                                    */
/****************************************************************************/

/**
 * Comment for Doxygen
 * @file cloopprofiler.c
 * @author Daisuke Abe
 * @date 28/11/2005
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

///Include Files
#include <gtk/gtk.h>
#include <gdk/gdkx.h>
#include <glib.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>

#include <arpa/inet.h>

#include "cloopprofiler.h"
#include "support.h"

#include "compressed_loop.h"
#include "cloop.h"

///Data Structure
///Color Setting
//Block:Read time
static const gushort tcolor[] = { 0xffff, 0xdddd, 0xffff };
//Block:Read Frequency
static const gushort fcolor[] = { 0xffff, 0xffff, 0xcccc };
//Block:BackGround Color
static const gushort bcolor[] = { 0xbbbb, 0xbbbb, 0xbbbb };
//Frame:Now Drawing
static const gushort ncolor[] = { 0x0000, 0xffff, 0xffff };
//Frame:Default Color
static const gushort pcolor[] = { 0x0000, 0x0000, 0x0000 };
//Frame:Select
static const gushort scolor[] = { 0xffff, 0x8888, 0x8888 };
//
static const gushort tfcolor[] = { 0xffff, 0xdddd, 0xffff };
static const gushort ftcolor[] = { 0xffff, 0xffff, 0xcccc };
static const gushort tcolorr[] = { 0xffff, 0x9999, 0xffff };
static const gushort fcolorr[] = { 0xffff, 0xffff, 0x8888 };
static const gushort tfcolorr[] = { 0xffff, 0x9999, 0xffff };
static const gushort ftcolorr[] = { 0xffff, 0xffff, 0x8888 };

//Color Path
static const gushort *c_path[] = {
  tcolor,
  fcolor,
  bcolor,
  ncolor,
  pcolor,
  scolor,
  tfcolor,
  ftcolor,
  tcolorr,
  fcolorr,
  tfcolorr,
  ftcolorr,
};

//extern const char* separator;

/**
 * @brief delay_absorber - Delay reliever
 * @param[in] pointer who has all parameter
 */
void
delay_absorber (gpointer tmp)
{
  t_run_status *run = ((t_all_data *) tmp)->run;
  t_timer *timer = &run->timer;

  ///matching timer->count and run->p_timer
  timer->t_count = run->p_timer * 100;

  gtk_timeout_remove (run->d_tag);
  run->d_tag = 0;
}

/**
 * @brief reset_popup - pop up window destroyer
 * @param[in] pop up window widget
 * @param[in] event status
 * @param[in] pointer who has all parameter 
 */
void
reset_popup (GtkWidget * widget, GdkEventButton * event, gpointer tmp)
{
  t_app_status *app = ((t_all_data *) tmp)->app;
  t_run_status *run = ((t_all_data *) tmp)->run;

  t_draw *blks = &app->blks;

  t_popup *popup_list = &run->popup_list;
  t_block *pblock = NULL;

  GtkWidget *pwidget = NULL;
  GList *plist = run->load_list;

  if (popup_list == NULL)
    return;
  if (plist == NULL)
    return;

  dbg ("popup destroy\n");

  if (app->popup != NULL && widget != NULL)
    {
      dbg ("popup destroy\n");
      pwidget = app->popup;

      ///paint block to stand out
      ///paint block to stand out
      if (run->restype == READ_BLOCKS)
	{
	  pblock = popup_list->pblock;
	  draw_one_frame (blks, pblock, F_GRND, 0, tmp);
	  draw_one_block (blks, pblock, run->type, 0, tmp);
	}
      else if (run->restype == CLOOP_HEADER)
	{
	  plist = g_list_first (plist);
	  while (plist != NULL)
	    {
	      pblock = (t_block *) plist->data;

	      gint tmpsize = 0;
	      glong offset = pblock->cloop->offset;

	      while (tmpsize < pblock->cloop->size)
		{
		  draw_one_frame (blks, pblock, F_GRND, offset, tmp);
		  draw_one_block (blks, pblock, run->type, offset, tmp);
		  offset += blks->summary_cnt * SECTOR_SIZE;
		  tmpsize += blks->summary_cnt * SECTOR_SIZE;
		}
	      plist = g_list_next (plist);
	    }
	}

      gtk_widget_destroy (GTK_WIDGET (pwidget));
      app->popup = NULL;
    }
  draw_update (blks);
  thread_flush ();
}

/**
 * @brief check_draw_area - first step on drawarea initalize 
 * @param[in] pointer who contains all pointer
 */
void
check_draw_area (gpointer tmp)
{
  t_app_status *app = ((t_all_data *) tmp)->app;
  t_draw *blks = &app->blks;

  t_run_status *run = ((t_all_data *) tmp)->run;
  t_result *result = &run->result;
  t_clpimg *cloop = &run->cloop;

  gint total_bcount = 0;
  gint cnt, i;

  ///Resource Type:read_blocks log
  if (run->restype == READ_BLOCKS)
    {
      dbg ("restype=READ_BLOCKS\n");
      total_bcount = result->total_blocks;
      ///blks->summary_cnt = input_data
      blks->summary_cnt = app->summary.cnt;
    }
  ///Resouce Type:cloop_header
  else if (run->restype == CLOOP_HEADER)
    {
      dbg ("restype=CLOOP_HEADER\n");
      total_bcount = cloop->total_size / SECTOR_SIZE;
      cnt = 1;
      for (i = 0; i < app->summary.cnt; i++)
	cnt *= 2;

      blks->summary_cnt = cnt;
      dbg ("summary_cnt:%d\n", blks->summary_cnt);
    }

  ///Calc drawarea width
  blks->width = blks->scrl_win->allocation.width - 2;

  ///Calc rectangles that draw one line
  blks->draw_per_line = (blks->width / BLK_WIDTH) - MARGIN;
  if (blks->draw_per_line % BLK_PER_LINE_BASE != 0)
    blks->draw_per_line -= (blks->draw_per_line % BLK_PER_LINE_BASE);

  ///Calc data blocks that contain one line
  blks->blks_per_line = blks->summary_cnt * blks->draw_per_line;

  dbg ("draw_per_line:%d\n", blks->draw_per_line);
  dbg ("blks_per_line:%d\n", blks->blks_per_line);

  ///Calc drawarea height
  blks->height = BLK_HEIGHT * (total_bcount / blks->blks_per_line) + MARGIN * 4;

  ///Set size data to run status(use on "callbacks.c")
  run->width = blks->width;
  run->height = blks->height;

  ///Drawarea check
  if (blks->pixmap != NULL)
    {
      dbg ("blks->pixmap clear!!!\n");
      gdk_pixmap_unref (blks->pixmap);
    }

  dbg ("blks->pixmap get!!!\n");
  blks->pixmap = gdk_pixmap_new (blks->widget->window,
				 blks->width, blks->height, -1);
}

/**
 * @brief get_summarized_type - compere 2 values
 * @param[in] current value
 * @param[in] new value
 * @param[out] bigger value
 * @note no use.
 */
gint
get_summarized_type (gint cur_type, gint new_type)
{
  return (cur_type > new_type ? cur_type : new_type);
}

/**
 * @brief get_summary_cnt - get summary magunifition from spin box
 * @param[in] spin box widget
 */
void
get_summary_cnt (t_spin * summary)
{
  summary->cnt =
    gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (summary->widget));
}

/**
 * @brief get_disp_state - get display mode from combo box
 * @param[in] combo box widget
 */
void
get_disp_state (t_combo * combo)
{
  combo->s =
    (gchar *)
    gtk_entry_get_text (GTK_ENTRY ((GTK_COMBO (combo->widget))->entry));

  ///NULL check.
  if (combo->s[0] == '\0')
    combo->s = NULL;
}

/**
 * @brief build_one_block build a cloop block data from read_blocks log
 * @param[in] one line in "read_blocks"
 * @param[out] a block data
 */
t_block *
build_one_block (gchar * buffer)
{

  t_block *tmp_blk;
  gint top, bottom, count;
  gint bnum = 0;
  gchar *data_buf;
  gchar tmpbuf[BUFLEN];

  if(buffer[0]=='\t')return NULL;
  
  strcpy (tmpbuf, buffer);
  tmp_blk = (t_block *) malloc (sizeof (t_block));
  if (tmp_blk == NULL)
    {
      fprintf (stderr, "build_one_block:malloc failed.\n");
      abort();
      //exit (1);
    }

  tmp_blk->extract_time = 1;

  top = bottom = count = 0;

  count = 0;
  bnum++;

  data_buf = strtok (tmpbuf, SEP);
  tmp_blk->read_no = atoi (data_buf);
  while(data_buf != NULL)
  //while (count < 5)
    {
      data_buf = strtok (NULL, SEP);
      if(data_buf == NULL)
	{
	  if(count < 4) return NULL;
	  tmp_blk->extract_time = 0;
	  break;
	}
       switch(count)
	{
	  ///Block Number
	case 0:
	  tmp_blk->blk_no = atoi (data_buf);
	  break;
	case 1:
	  tmp_blk->size = atoi (data_buf);
	  break;
	case 2:
	  tmp_blk->utc_time = atof (data_buf);
	  break;
	case 3:
	  tmp_blk->ref_time = atof (data_buf);
	  break;
	case 4:
	  tmp_blk->extract_time = atof (data_buf);
	  
	  break;
	}
      count++;
    }
  ///Zero Clear block draw time
  tmp_blk->draw_time = 0;

  return tmp_blk;
}

/**
 * @brief make_result - g_foreach function:sum result data from block data
 * @param[in] block data
 * @param[in] read_blocks_result
 */
void
make_result (gpointer data, gpointer user_data)
{
  t_block *block = (t_block *) data;
  t_result *result = (t_result *) user_data;

  result->total_size += block->size;
  result->total_read_time += block->ref_time;
  result->total_freq++;

  if (block->draw_freq == 0)
    {
      result->total_ublocks++;
      result->total_ublock_size += block->size;
    }

}

/**
 * @brief build_result_data - build satisical data from read_blocks log
 * @param[in] pointer who contains all parameters
 * @note TODO!:Rifactaring
 *
 */
void
build_result_data (gpointer tmp)
{
  t_app_status *app = ((t_all_data *) tmp)->app;
  t_text *appstate = &app->state;
  GtkWidget *s_widget = (GtkWidget *) appstate->widget;
  GtkWidget *tb = (GtkWidget *) appstate->text_buffer;

  t_run_status *run = ((t_all_data *) tmp)->run;
  t_result *result = &run->result;

  GList *head, *tail;
  GString *state = NULL;

  state = g_string_sized_new (8192);

  if (appstate->gstring != NULL)
    g_string_free (appstate->gstring, TRUE);
  appstate->gstring = state;

  g_list_foreach (run->blk_list, make_result, result);

  head = g_list_first (run->blk_list);
  tail = g_list_last (run->blk_list);

  result->start_time = ((t_block *) head->data)->utc_time;
  dbg ("start time:%lf\n", result->start_time);
  result->total_time =
    ((t_block *) tail->data)->utc_time - ((t_block *) head->data)->utc_time;
  dbg ("end time:%lf\n", ((t_block *) tail->data)->utc_time);
  dbg ("total time:%lf\n", result->total_time);

  result->cloop_total_size = 64 * result->total_blocks / 1024.0;
  result->total_size_as_kb = result->total_size / 1024.0;
  result->total_ublock_size /= 1024.0;

  if (result->total_read_time > 0)
    result->trance_rate = result->total_size / result->total_read_time;
  else
    result->trance_rate = result->total_size / 1;
  
  result->read_time_average = result->total_time / result->total_freq;
  result->read_size_average =
    result->total_size / result->total_freq / 1024.0;
  result->ublock_size_average =
    result->total_ublock_size / result->total_ublocks / 1024.0;

  g_string_sprintfa (state, "***cloop status***\n\n");

#if 0
  if (result->is_opt != NULL)
    g_string_sprintfa (state, "optimized                   \t\t: %s\n",
		       result->is_opt);
#endif /* 0 */
  if (result->is_opt)
    g_string_sprintfa (state, "optimized                   \t\t: TRUE\n");
  else if (!result->is_opt)
    g_string_sprintfa (state, "optimized                   \t\t: FALSE\n");

  //dbg ("mount real cloop : %s\n", run->is_cloop);
  if (run->is_cloop)
    g_string_sprintfa (state, "real cloop mount            \t\t: TRUE\n");
  else if (!run->is_cloop)
    g_string_sprintfa (state, "real cloop mount            \t\t: FALSE\n");

#if 0
  g_string_sprintfa (state, "real cloop mount            \t\t: %s\n",
		     run->is_cloop);
#endif /* 0 */

 g_string_sprintfa (state, "SECTOR SIZE                 \t\t: %d\n\n",
		     SECTOR_SIZE);


#if 0
  for (i = 1; i < (sizeof (result_path) / sizeof (result_path[0])); i++)
    {
      dbg ("result_path:%s", result_path[i]);
      //dbg("result_data:%f.1\n", *result_data[i]);
      //dbg(result_path[i],result_data[i]);
      g_string_sprintfa (state, result_path[i], (int) *result_data[i]);
    }
#endif

  //g_string_sprintfa (state, "cloop device id\t\t\t: %d\n",
  //	     result->cloop_device);
  g_string_sprintfa (state, "read jiffies\t\t\t: %.3f\n",
		     result->read_jiffies);
  g_string_sprintfa (state, "cloop total blocks\t\t\t: %d\n",
		     result->total_blocks);
  g_string_sprintfa (state, "total cloop size [MB]\t\t: %d\n\n",
		     result->cloop_total_size);

  g_string_sprintfa (state, "total read freq\t\t\t: %d\n",
		     result->total_freq);
  g_string_sprintfa (state, "total read size [KB]\t\t: %d\n",
		     result->total_size_as_kb);
  g_string_sprintfa (state, "read size average [KB]\t\t: %.2f\n",
		     result->read_size_average);
  g_string_sprintfa (state, "total unique block\t\t\t: %d\n",
		     result->total_ublocks);
  g_string_sprintfa (state, "total unique block size [KB]\t\t: %d\n",
		     result->total_ublock_size);
  g_string_sprintfa (state, "unique size average [KB]\t\t: %.2f\n\n",
		     result->ublock_size_average);

  g_string_sprintfa (state, "total read time [s]\t\t\t: %.3f\n",
		     result->total_read_time);
  g_string_sprintfa (state, "total exec time [s]\t\t\t: %.3f\n",
		     result->total_time);
  g_string_sprintfa (state, "read time average [s]\t\t: %.3f\n",
		     result->read_time_average);
  g_string_sprintfa (state, "total trance rate [KB/s]\t\t: %.3f\n",
		     result->trance_rate);

  dbg ("set text.\n");
  gtk_text_buffer_set_text (GTK_TEXT_BUFFER (tb), state->str, -1);
  gtk_text_view_set_buffer (GTK_TEXT_VIEW (s_widget), GTK_TEXT_BUFFER (tb));

  gtk_widget_set_sensitive (GTK_WIDGET (app->resourcemenu_item[0]), TRUE);
}

/**
 * @brief analyze_cloop - analyze cloop header
 * @param[in] pointer who has all parameters
 */
void
analyze_cloop (gpointer tmp)
{
  t_app_status *app = ((t_all_data *) tmp)->app;
  t_draw *blks = &app->blks;
  t_run_status *run = ((t_all_data *) tmp)->run;
  gchar *cloop_file = run->cloopname;
  t_clpimg *cloop = &run->cloop;
  t_result *result = &run->result;

  GList *list = NULL;

  int cloop_fd;
  struct cloop_head header;
  struct block_info *blk_info_tbl = NULL;

  unsigned int total_offsets;
  unsigned int offset;
  loff_t *offsets = NULL;
  int i;

  dbg ("analyze cloop.\n");

  list = run->blk_list;
  if (list == NULL)
    {
      fprintf (stderr, "you must read READ_BLOCKS before read cloop!\n");
      return;
    }

  if (regular_file_check (cloop_file) == 0)
    {
      return;
    }

  /// cloop open
  dbg ("cloop file:%s\n", cloop_file);
  cloop_fd = open (cloop_file, O_RDONLY);
  if (cloop_fd < 0)
    {
      dialog_message("%s %s",cloop_file,ERR_FOPEN);
      //dialog_message ("cloop file open error");
      return;
    }

  /// header read
  if (read (cloop_fd, &header, sizeof (struct cloop_head))
      != sizeof (struct cloop_head))
    {
      dialog_message("%s %s",cloop_file,ERR_FOPEN);
      close(cloop_fd);
      return;
    }

  if (header.preamble[0x0B] != 'V' || header.preamble[0x0C] != '2')
    {
      dialog_message ("%s %s",cloop_file,ERR_FORMAT);
      close(cloop_fd);
      return;
    }

  dbg ("header:\n%s", header.preamble);

  ///ntohl : exchange Bytes network order to local order
  total_offsets = ntohl (header.num_blocks) + 1;

  ///No Optimized Cloop
  if (header.preamble[0x10] != 'O' || header.preamble[0x11] != 'P'
      || header.preamble[0x12] != 'T')
    {
      result->is_opt=FALSE;
      dbg ("non optimized image.\n");

      /// offsets read
      offsets = malloc (sizeof (loff_t) * total_offsets);
      if (!offsets)
	{
	  fprintf (stderr, "%s\n",ERR_MEMORY);
	  abort();
	}

      dbg ("total_offsets:%d\n", total_offsets);

      if (read (cloop_fd, offsets, (sizeof (loff_t) * total_offsets))
	  != (sizeof (loff_t) * total_offsets))
	{
	  dialog_message("%s",ERR_OFFSET);
	  close(cloop_fd);
	  return;
	}

      cloop->first_offset = be64_to_cpu (offsets[0]);
      cloop->total_size =
	be64_to_cpu (offsets[total_offsets - 1]) - be64_to_cpu (offsets[0]);
      cloop->total_offsets = be64_to_cpu (offsets[total_offsets - 1]);

      dbg ("marge for read_blocks list.\n");
      run->tmp_list = g_list_first (run->blk_list);

      while (1)
	{
	  for (i = 0; i < total_offsets; i++)
	    {
	      if (i == ((t_block *) list->data)->blk_no)
		{
		  if (((t_block *) list->data)->cloop == NULL)
		    ((t_block *) list->data)->cloop = malloc (sizeof (t_cloop));
		  if (((t_block *) list->data)->cloop == NULL)
		    {
		      fprintf (stderr, "%s\n",ERR_MEMORY);
		      exit (1);
		    }
		  memset (((t_block *) list->data)->cloop, '\0',
			  sizeof (t_cloop));

		  ((t_block *) list->data)->cloop->no = i;
		  ((t_block *) list->data)->cloop->size = be64_to_cpu (offsets[i + 1]) - be64_to_cpu (offsets[i]);
		  ((t_block *) list->data)->cloop->offset = be64_to_cpu (offsets[i]);

		  break;
		}
	    }
	  if(i==total_offsets)
	    {
	      dialog_message("%s",ERR_OFFSET);
	      close(cloop_fd);
	      return;
	    }

	  if (list->next != NULL)
	    list = g_list_next (list);
	  else
	    break;

	}
      free (offsets);
    }
  ///Optimized Cloop
  else
    {
      result->is_opt=TRUE;
      dbg ("optimized image.\n");

      dbg ("Malloc.\n");
      blk_info_tbl = malloc (sizeof (struct block_info) * total_offsets);
      if (!blk_info_tbl)
	{
	  fprintf (stderr, "%s\n",ERR_MEMORY);
	  close(cloop_fd);
	  abort();
	}

      if (read(cloop_fd, blk_info_tbl, sizeof (struct block_info) * total_offsets) != sizeof (struct block_info) * total_offsets)
	{
	  dialog_message("%s",ERR_BLKTBL);
	  close(cloop_fd);
	  return;
	}

      ///Get Offsets 
      offset = sizeof (header) + sizeof (loff_t) * total_offsets;
      dbg ("Malloc.\n");
      offsets = malloc (sizeof (loff_t) * total_offsets);
      if (!offsets)
	{
	  fprintf (stderr, "%s\n",ERR_MEMORY);
	  close(cloop_fd);
	  abort();
	}

      ///Calc total offsets
      for (i = 0; i < total_offsets; i++)
	{
	  offsets[i] = offset;
	  offset += ntohl (blk_info_tbl[i].size);
	}

      ///Sum parameters
      cloop->first_offset = sizeof (header) + sizeof (loff_t) * total_offsets;
      cloop->total_size = offset - sizeof (header) + sizeof (loff_t) * total_offsets;
      cloop->total_offsets = offset;

      list = g_list_first (run->blk_list);

      while (1)
	{
	  for (i = 0; i < total_offsets; i++)
	    {
	      if (i == ((t_block *) list->data)->blk_no)
		{
		  if (((t_block *) list->data)->cloop == NULL)
		    ((t_block *) list->data)->cloop =
		      malloc (sizeof (t_cloop));
		  if (((t_block *) list->data)->cloop == NULL)
		    {
		      fprintf (stderr, "%s\n",ERR_MEMORY);
		      abort();
		    }
		  memset (((t_block *) list->data)->cloop, '\0',
			  sizeof (t_cloop));

		  ((t_block *) list->data)->cloop->no = i;
		  ((t_block *) list->data)->cloop->size = ntohl (blk_info_tbl[i].size);
		  ((t_block *) list->data)->cloop->offset = be64_to_cpu (blk_info_tbl[i].offset);
		  break;
		}
	    }

	  if (list->next != NULL)
	    list = g_list_next (list);
	  else
	    break;
	}
      free (offsets);
    }

  list = g_list_first (run->blk_list);

  dbg ("first offset:%ld\n", cloop->first_offset);
  dbg ("total size:%ld\n", cloop->total_size);
  dbg ("total offset:%d\n", cloop->total_offsets);

  //return size;
  close (cloop_fd);

  run->is_cloop = TRUE;

  gtk_widget_set_sensitive (app->wizard_menu, TRUE);
  //gtk_widget_set_sensitive (app->optwiz_menu, TRUE);
  gtk_widget_set_sensitive (GTK_WIDGET (app->resourcemenu_item[1]), TRUE);
  clearance_field (tmp);
  //play_label_change (tmp);
  check_draw_area (tmp);
  build_result_data (tmp);
  draw_initalize (blks->summary_cnt, tmp);
  //draw_view (tmp, EV_STOP);

  thread_flush ();
}

gint
chops_read_log(gpointer tmp)
{
  FILE *fp = NULL;

  t_run_status *run = ((t_all_data *) tmp)->run;
  t_result *result = &run->result;
  gchar *filename = run->filename;

  gchar *tmpbuf = NULL;
  gchar *tmpstr = NULL;

  t_block *tmp_blk = NULL;
  t_block *buf_blk = NULL;
  gint select_tag;
  gint count = 0;
  gint len = 0;
  gint i;
  
  if(!file_size_check(filename))
    {
      return -1;
    }

  fp=fopen(filename,"r");
  if(!fp)
    {
      dialog_message("%s %s",filename,ERR_FOPEN);
      return -1;
    }

  tmpbuf = malloc (sizeof (gchar) * BUFLEN);
  if (tmpbuf == NULL)
    {
      fprintf (stderr, "%s\n",ERR_MEMORY);
      fclose(fp);
      abort();
    }
  memset (tmpbuf, '\0', sizeof (gchar) * BUFLEN);
  tmpstr = malloc (sizeof (gchar) * BUFLEN);
  if (tmpstr == NULL)
    {
      fprintf (stderr, "%s\n",ERR_MEMORY);
      fclose(fp);
      abort();
    }
  memset (tmpstr, '\0', sizeof (gchar) * BUFLEN);

  ///while file end.
  while (fgets (tmpbuf, BUFLEN, fp)!= 0)
    {
      len = strlen(tmpbuf);
      tmpbuf[len-1]='\0';
      
      for(i=0;i<len;i++)
	if(!(isdigit(tmpbuf[i])||tmpbuf[i]=='.'||tmpbuf[i]==' '||tmpbuf[i]=='\t'))break;
      
      if(i==len-1)
	{
	  gint sp = 0;

	  while(tmpbuf[sp]==' ' || tmpbuf[sp]=='\t')sp++;

	  if (!(isdigit(tmpbuf[sp])))continue;
	  else
	    {
	      select_tag = 0;
	      tmp_blk = build_one_block (tmpbuf);
	      if(tmp_blk == NULL)
		{
		  free(tmpbuf);fclose(fp);
		  dialog_message("%s %s",filename,ERR_NODATA);
		  return 0;
		}

	      tmp_blk->draw_freq = 0;
	      tmp_blk->cloop = NULL;

	      g_list_foreach (run->blk_list, count_read_freq, tmp_blk);

	      if (tmp_blk->read_no != 0)
		tmp_blk->draw_time = tmp_blk->utc_time - buf_blk->utc_time;
	      else
		tmp_blk->draw_time = tmp_blk->ref_time;

	      run->blk_list = g_list_append (run->blk_list, tmp_blk);
	      count++;
	    }
	  buf_blk = tmp_blk;
	}
      ///other parametars
      else
	{
	  ///cut on ':'
	  tmpstr = strtok (tmpbuf, ":");
	  if (tmpbuf == NULL)
	    {
	      dialog_message("%s %s",filename,ERR_NODATA);
	      fclose (fp);
	      return 0;
	    }
	  if (strncmp (tmpstr, "cloop", 5) == 0)
	    {
	    }
	  else if (strncmp (tmpstr, "now", 3) == 0)
	    {
	      tmpstr = strtok (NULL, ":");
	      result->read_jiffies = atof (tmpstr);
	    }
	  else if (strncmp (tmpstr, "total", 5) == 0)
	    {
	      tmpstr = strtok (NULL, ":");
	      result->total_blocks = atoi (tmpstr);
	    }
	  else if (strncmp (tmpstr, "file", 4 ) == 0)
	    {
	      tmpstr = strtok (NULL,":");
	      if(strcmp(tmpstr,"optimized") == 0)
		{
		  run->is_cloop = TRUE;
		}
	      else
		{
		  run->is_cloop = FALSE;
		}
	    }
	}
    }
  
  if(tmp_blk == NULL)
    {
      dialog_message("%s %s",filename,ERR_NODATA);
      return 0;
    }

  tmp_blk->draw_time = 0;

  fclose(fp);
  free (tmpbuf);
  return count;
}


/**
 * @brief analyze_read_log - analyze "read_blocks" log data
 * @param[in] pointer who contains all parameters
 */
void
analyze_read_log (gpointer tmp)
{
  t_app_status *app = ((t_all_data *) tmp)->app;
  t_scale *scale = &app->scale;
  t_draw *blks = &app->blks;

  t_text *appstate = &app->state;
  GtkWidget *s_widget = (GtkWidget *) appstate->widget;
  GtkWidget *tb = (GtkWidget *) appstate->text_buffer;

  t_run_status *run = ((t_all_data *) tmp)->run;
  t_result *result = &run->result;
  t_timer *timer = &run->timer;
  gchar *filename = run->filename;

  gint count;

  run->blk_list = NULL;
  result->is_opt = FALSE;

  if (regular_file_check (filename) == 0)
    {
      goto exit_1;
    }

  ///clearance data
  if (run->blk_list)
    {
      ///Stop Play
      if (timer->timer_tag == 0)
	stop_timer (tmp);
      reset_timer (tmp);

      scale->adjust->upper = 100;

      ///Run Status Clear if it exists.
      if (run != NULL)
	{
	  if (run->blk_list != NULL)
	    g_list_free (run->blk_list);
	  if (run->tmp_list != NULL)
	    g_list_free (run->tmp_list);

	  memset (run, '\0', sizeof (t_run_status));
	  memset (result, '\0', sizeof (t_result));
	}
    }

  ///block list check.
  if (run->blk_list != NULL)
    {
      dbg ("block list clear.\n");
      g_list_free (run->blk_list);
      g_list_free (run->tmp_list);
    }

  count = chops_read_log(tmp);
  if(count<=0) goto exit_2;

  run->tmp_list = g_list_first(run->blk_list);
  timer->top_block_jiffies = ((t_block*)run->tmp_list->data)->utc_time;
  //timer->top_block_jiffies = 0;
  

  dbg ("analyze over.\n");
  run->b_loc = 0;
  run->is_cloop = FALSE;

  gtk_widget_set_sensitive (app->cloopfile_menu, TRUE);
  gtk_widget_set_sensitive (GTK_WIDGET (app->resourcemenu_item[1]), FALSE);
  gtk_widget_set_sensitive (GTK_WIDGET (app->wizard_menu), FALSE);

  //on_rblog_menu_activate (NULL, tmp);

  clearance_field (tmp);
  //play_label_change (tmp);
  check_draw_area (tmp);
  draw_initalize (blks->summary_cnt, tmp);

  build_result_data (tmp);

  return;

 exit_2:
 exit_1:
  memset(&run->result,0,sizeof(run->result));
  check_draw_area(tmp);
  draw_initalize (blks->summary_cnt, tmp);

  gtk_text_buffer_set_text (GTK_TEXT_BUFFER (tb), "", -1);
  gtk_text_view_set_buffer (GTK_TEXT_VIEW (s_widget), GTK_TEXT_BUFFER (tb));

  gtk_widget_set_sensitive (app->wizard_menu, FALSE);
  gtk_widget_set_sensitive (app->cloopfile_menu, FALSE);

  return;
}

/**
 * @brief draw_update - update draw field
 * @param[in] a pointer that point for draw field
 */
void
draw_update (t_draw * blks)
{
  GdkRectangle rect;
  rect.x = 0;
  rect.y = 0;
  rect.width = blks->width;
  rect.height = blks->height;

  gtk_widget_queue_draw (blks->widget);
}

/**
 * @brief append_one_list - append one block data to CList
 * @param[in] pointer who contains all parameters
 * @param[in] pointer that point a data block
 */
void
append_one_list (gpointer all, t_block * block)
{
  t_app_status *app = ((t_all_data *) all)->app;

  GtkWidget *clist = app->list.widget;

  gchar *p[INFO_CNT];
  gint i;

  for (i = 0; i < INFO_CNT; i++)
    {
      p[i] = malloc (sizeof (gchar) * 16);
      if (p[i] == NULL)
	{
	  fprintf (stderr, "%s\n",ERR_MEMORY);
	  abort();
	  //exit (1);
	}
      memset (p[i], '\0', sizeof (gchar) * 16);
    }
  sprintf (p[0], "%d", block->read_no);
  sprintf (p[1], "%d", block->blk_no);
  sprintf (p[2], "%0.3f", block->ref_time);
  sprintf (p[3], "%d", block->draw_freq);
  if (block->cloop != NULL)
   sprintf (p[4], "%ld", block->cloop->offset);
  sprintf (p[5], "%d", block->size);	//size
  if (block->ref_time > 0)
    sprintf (p[6], "%0.3f", block->size / 1000 / block->ref_time);
  else if (block->draw_freq > 0)
    sprintf (p[6], "Read From Cache.");
  else
    sprintf (p[6], "%0.3f", block->size / 0.1);
  gtk_clist_append (GTK_CLIST (clist), p);

}

/**
 * @brief draw_one_frame - draw frame of one block
 * @param[in] pointer to a draw area
 * @param[in] pointer to a block data
 * @param[in] draw type(e.f. "READ TIME")
 * @param[in] offset
 * @param[in] pointer who contains all parameters
 */
void
draw_one_frame (t_draw * blks,
		t_block * block, gint type, glong offset, gpointer tmp)
{
  t_run_status *run = ((t_all_data *) tmp)->run;

  gint x = 0, y = 0;

  gint blk_no = block->blk_no;
  gchar restype = run->restype;
  gint drawperline = blks->draw_per_line;
  gint bnum_width = blks->bnum_width;

  ///when resource type is "read_blocks" log
  if (restype == READ_BLOCKS)
    {
      x =
	((blk_no / blks->summary_cnt) % drawperline) * FRM_WIDTH +
	MARGIN * 2 + bnum_width + MARGIN_2 -
	((blk_no / blks->summary_cnt) % drawperline);
      y =
	((blk_no / blks->summary_cnt) / drawperline) * FRM_HEIGHT + MARGIN +
	1 - (blk_no / blks->summary_cnt / drawperline);
    }
  ///when resource type is "cloop header"
  else if (restype == CLOOP_HEADER)
    {
      x =
	((offset / (blks->summary_cnt * SECTOR_SIZE)) % drawperline) *
	FRM_WIDTH + MARGIN * 2 + bnum_width + MARGIN_2 -
	((offset / (blks->summary_cnt * SECTOR_SIZE)) % drawperline);
      y =
	((offset / (blks->summary_cnt * SECTOR_SIZE)) / drawperline) *
	FRM_HEIGHT + MARGIN + 1 -
	(offset / (blks->summary_cnt * SECTOR_SIZE)) / drawperline;
    }

  gdk_draw_rectangle (blks->pixmap,
		      set_color (blks->b_gc, blks->bk_color[type], 0),
		      FALSE, x - 1, y - 1, FRM_WIDTH - 1, FRM_HEIGHT - 1);
}

/**
 * @brief draw_one_block - draw frame of one block
 * @param[in] pointer to a draw area
 * @param[in] pointer to a block data
 * @param[in] draw type
 * @param[in] data offset of a block data on the cloop  block device
 * @param[in] pointer who contains all parameters
 */
void
draw_one_block (t_draw * blks,
		t_block * block, gint type, glong offset, gpointer tmp)
{
  t_run_status *run = ((t_all_data *) tmp)->run;

  gint blk_no = block->blk_no;
  gchar restype = run->restype;
  gint drawperline = blks->draw_per_line;
  gint bnum_width = blks->bnum_width;

  t_clpimg *cloop = NULL;

  GdkPoint points[3];

  gint x = 0, y = 0;
  gint rate = 0;
  gint rtype = 0;

  if (restype == READ_BLOCKS)
    {
      x =
	((blk_no / blks->summary_cnt) % drawperline) * FRM_WIDTH +
	MARGIN * 2 + bnum_width + MARGIN_2 -
	(blk_no / blks->summary_cnt) % drawperline;
      y =
	((blk_no / blks->summary_cnt) / drawperline) * FRM_HEIGHT + MARGIN -
	(blk_no / blks->summary_cnt) / drawperline;
    }

  ///when resource type is "cloop header".
  else if (restype == CLOOP_HEADER)
    {
      cloop = &run->cloop;

      x =
	((offset / (blks->summary_cnt * SECTOR_SIZE)) % drawperline) *
	FRM_WIDTH + MARGIN * 2 + bnum_width + MARGIN_2 -
	(offset / (blks->summary_cnt * SECTOR_SIZE)) % drawperline;
      y =
	((offset / (blks->summary_cnt * SECTOR_SIZE)) / drawperline) *
	FRM_HEIGHT + MARGIN -
	(offset / (blks->summary_cnt * SECTOR_SIZE)) / drawperline;
    }
  ///Color Select
  if (type == R_FREQ || type == R_FREQ_R || type == R_FREQT
      || type == R_FREQT_R)
    {
      rate = block->draw_freq * FREQ_RATE;
    }
  else if (type == R_TIME || type == R_TIME_R || type == R_TIMEF
	   || type == R_TIMEF_R)
    {
      rate = block->ref_time * TIME_RATE;
    }

  ///If draw type needs the color for triangle field.
  if (type == R_FREQT_R)
    {
      type = R_FREQ_R;
      rtype = R_TIME_R;
    }
  else if (type == R_TIMEF_R)
    {
      type = R_TIME_R;
      rtype = R_FREQ_R;
    }

  if (rate < run->rate[blk_no / blks->summary_cnt])
    rate = run->rate[blk_no / blks->summary_cnt];
  else
    run->rate[blk_no / blks->summary_cnt] = rate;

  if (type == B_GRND)
    rate = 0;

  gdk_draw_rectangle (blks->pixmap,
		      set_color (blks->b_gc, blks->bk_color[type], rate),
		      TRUE, x, y + 1, BLK_WIDTH - 1, BLK_HEIGHT - 1);

  rate = 0;

  if (type == R_TIMEF || type == R_FREQT || rtype == R_TIME_R
      || rtype == R_FREQ_R)
    {
      if (rtype == R_FREQ_R)
	{
	  rate = block->draw_freq * FREQ_RATE;
	}
      else if (rtype == R_TIME_R)
	{
	  rate = block->ref_time * TIME_RATE;
	}
      else if (type == R_TIMEF)
	{
	  rtype = R_FREQ;
	  rate = block->draw_freq * FREQ_RATE;
	}
      else if (type == R_FREQT)
	{
	  rtype = R_TIME;
	  rate = block->ref_time * TIME_RATE;
	}

      points[0].x = x + BLK_WIDTH - 1;
      points[0].y = y;
      points[1].x = x;
      points[1].y = y + BLK_HEIGHT;
      points[2].x = x + BLK_WIDTH - 1;
      points[2].y = y + BLK_HEIGHT;

      gdk_draw_polygon (blks->pixmap,
			set_color (blks->b_gc, blks->bk_color[rtype], rate),
			TRUE, points, 3);
    }
}

/**
 * @brief rollback_blocks - repaint drawn blocks.
 * @param[in] draw scale
 * @param[in] pointer who contains all parameters
 */
void
rollback_blocks (gint summary_cnt, gpointer tmp)
{
  t_app_status *app = ((t_all_data *) tmp)->app;
  t_draw *blks = &app->blks;

  t_run_status *run = ((t_all_data *) tmp)->run;
  gchar restype = run->restype;

  t_block *tblock;
  gint count = 0;
  gint tmpsize = 0;
  gulong offset = 0;

  gint *first_draw = run->first_draw;

  GList *list = run->blk_list;

  if (list != NULL)
    {
      tblock = (t_block *) list->data;

      ///run->b_loc : last draw block point
      while (count != run->b_loc)
	{
	  if (first_draw[tblock->blk_no / blks->summary_cnt] == 0)
	    first_draw[tblock->blk_no / blks->summary_cnt] = tblock->blk_no;

	  run->p_timer += tblock->draw_time;
	  {
	    tmpsize = 0;
	    offset = 0;

	    if (restype == READ_BLOCKS)
	      {
		draw_one_block (blks, tblock, run->type, 0, tmp);
	      }
	    else if (restype == CLOOP_HEADER)
	      {
		offset = tblock->cloop->offset;

		while (tmpsize < tblock->cloop->size)
		  {
		    draw_one_block (blks, tblock, run->type, offset, tmp);
		    offset += summary_cnt * SECTOR_SIZE;
		    tmpsize += summary_cnt * SECTOR_SIZE;
		  }
	      }
	    run->size += tblock->size;
	  }
	  append_one_list (tmp, tblock);

	  run->b_edge = count;

	  list = g_list_next (list);
	  if (list != NULL)
	    tblock = (t_block *) list->data;

	  count++;
	}
    }
  draw_update (blks);
}

/**
 * @brief draw_initalize - Initalize draw field
 * @param[in] draw scale
 * @param[in] pointer who contains all parameters
 */
void
draw_initalize (gint summary_cnt, gpointer tmp)
{
  t_app_status *app = ((t_all_data *) tmp)->app;
  t_draw *blks = &app->blks;
  GtkStyle *style = blks->widget->style;

  t_run_status *run = ((t_all_data *) tmp)->run;
  t_result *result = &run->result;
  t_clpimg *cloop = &run->cloop;
  gchar restype = run->restype;

  gint type;
  guint block_count = 0;
  guint dbcount = 0;
  guint bn = 0;
  guint cnt = 0;
  guint bcnt = 0;

  gint x, y;
  guint i;

  gchar bnum[10 + 1] = "";

  blks->gc = style->bg_gc[GTK_STATE_NORMAL];

  if (restype == READ_BLOCKS)
    block_count = result->total_blocks;
  else if (restype == CLOOP_HEADER)
    block_count = cloop->total_size / SECTOR_SIZE;

  type = run->type;
  gint rate = 1;

  x = MARGIN;
  y = MARGIN;
  
  blks->startx = 0;
  blks->starty = 0;
  blks->endx = 0;
  blks->endy = 0;

  //Initalize Draw Area
  gtk_drawing_area_size (GTK_DRAWING_AREA (blks->widget),
			 blks->width, blks->height);

  gdk_draw_rectangle (blks->pixmap,
		      blks->gc, TRUE, 0, 0, blks->width, blks->height);

  blks->gc = style->black_gc;
  blks->b_gc = set_color (blks->b_gc, blks->bk_color[B_GRND], rate);

  if(block_count == 0)return;

  /// while Draw Base Blocks are finished.
  for (i = 0; i < block_count; i++, cnt++)
    {
      ///New Line.
      if (bn % blks->blks_per_line == 0)
	{
	  ///Block No Tag.
	  if (restype == READ_BLOCKS)
	    sprintf (bnum, "%7u", bn);
	  else if (restype == CLOOP_HEADER)
	    sprintf (bnum, "%7u", bn * SECTOR_SIZE / 1024);

	  x = 0;
	  pango_layout_set_text (blks->font->layout, bnum, -1);
	  gdk_draw_layout (blks->pixmap, style->black_gc, x, y + 2,
			   blks->font->layout);
	  x += 50;

	  if (blks->startx == 0)
	    blks->startx = x;
	  if (blks->starty == 0)
	    blks->starty = y;
	}
      ///count % summary_cnt = 0
      if (cnt == summary_cnt)
	{
	  gdk_draw_rectangle (blks->pixmap,
			      blks->gc,
			      FALSE, x, y, FRM_WIDTH - 1, FRM_HEIGHT - 1);

	  gdk_draw_rectangle (blks->pixmap,
			      blks->b_gc,
			      TRUE,
			      x + 1, y + 1, BLK_WIDTH - 1, BLK_HEIGHT - 1);

	  x += FRM_WIDTH - 1;
	  cnt = 0;
	  bcnt++;
	  dbcount++;
	}
      ///When a block at finish in line
      if (bn % blks->blks_per_line == blks->blks_per_line - 1)
	{
	  ///Block Count isn't match "blks->draw_per_line"
	  if (bcnt != blks->draw_per_line)
	    {
	      gdk_draw_rectangle (blks->pixmap,
				  blks->gc,
				  FALSE, x, y, FRM_WIDTH - 1, FRM_HEIGHT - 1);
	      gdk_draw_rectangle (blks->pixmap,
				  blks->b_gc,
				  TRUE,
				  x + 1, y + 1,
				  BLK_WIDTH - 1, BLK_HEIGHT - 1);
	    }

	  ///Reset Parameters
	  if (blks->endx == 0)
	    blks->endx = x + FRM_WIDTH;
	  bcnt = 0;
	  x = 0;
	  y += FRM_HEIGHT - 1;
	}
      bn++;
    }

  ///runrate memory.
  if (run->rate == NULL)
    {
      run->rate = malloc (sizeof (gint) * dbcount);
      run->first_draw = malloc (sizeof (gint) * dbcount);
    }
  else
    {
      run->rate = realloc (run->rate, sizeof (gint) * dbcount);
      run->first_draw = malloc (sizeof (gint) * dbcount);
    }

  if (run->rate == NULL || run->first_draw == NULL)
    {
      fprintf (stderr, "%s\n",ERR_MEMORY);
      abort();
    }
  memset (run->rate, 0, sizeof (gint) * dbcount);
  memset (run->rate, 0, sizeof (gint) * dbcount);

  run->p_timer = 0;
  run->size = 0;

  blks->endy = y + FRM_HEIGHT;

  dbg ("field init done.\n");

  rollback_blocks (summary_cnt, tmp);

  dbg ("block field start->end :(%d,%d)->(%d,%d).\n", blks->startx,
       blks->starty, blks->endx, blks->endy);
}


/**
 * @brief - stop_timer - stop draw animation
 * @param[in] pointer who contains all parameters
 */
void
stop_timer (gpointer tmp)
{
  t_run_status *run = ((t_all_data *) tmp)->run;
  t_timer *timer = &run->timer;

  if(timer->timer_tag!=0)
    gtk_timeout_remove (timer->timer_tag);

}

/**
 * @brief reset_timer - reset animation at first
 * @param[in] pointer who contains all parameters
 */
void
reset_timer (gpointer tmp)
{
  t_app_status *app = ((t_all_data *) tmp)->app;
  t_draw *blks = &app->blks;

  t_run_status *run = ((t_all_data *) tmp)->run;
  t_timer *timer = &run->timer;

  timer->min = 0;
  timer->sec = 0;
  timer->msec = 0;

  timer->t_count = 0;

  ///Sum Timer
  timer->min = (timer->t_count+timer->top_block_jiffies) / 6000;
  timer->sec = (timer->t_count+timer->top_block_jiffies) / 100 - timer->min * 60;
  timer->msec = (timer->t_count+timer->top_block_jiffies) - (timer->sec * 100 + timer->min * 6000);

#if 0
  timer->min = timer->t_count / 6000;
  timer->sec = timer->t_count / 100 - timer->min * 60;
  timer->msec = timer->t_count - (timer->sec * 100 + timer->min * 6000);
#endif

  if (timer->timer_tag != 0)
    stop_timer (tmp);

  timer->timer_tag = 0;

  if (timer->t_string == NULL)
    {
      timer->t_string = malloc (sizeof (gchar) * 10);
      if (timer->t_string == NULL)
	abort();
	//exit (1);
      memset (timer->t_string, '\0', sizeof (gchar) * 10);
    }

  sprintf (timer->t_string, "%02d'%02d''%02d", timer->min, timer->sec,
	   timer->msec);

  draw_update (blks);
}

/**
 * @brief draw_timer - runnning animation function
 * @param@[in] pointer who contains all parameters
 */
void
draw_timer (gpointer tmp)
{
  t_app_status *app = ((t_all_data *) tmp)->app;
  t_scale *scale = &app->scale;
  GtkWidget *tlabel = app->time.widget;

  t_run_status *run = ((t_all_data *) tmp)->run;
  t_play *play = &run->play;
  t_timer *timer = &run->timer;
  t_result *result = &run->result;

  t_block *block = NULL;
  int time = 0;

  if (timer->timer_tag != 0)
    gtk_timeout_remove (timer->timer_tag);

  ///Get String for Timer (xx:xx:xx)
  if (timer->t_string == NULL)
    {
      timer->t_string = malloc (sizeof (gchar) * 10);
      if (timer->t_string == NULL)
	abort();
      memset (timer->t_string, '\0', sizeof (gchar) * 10);
    }

  if (play->frame > 0)
    {
      timer->t_count++;
    }
  else if (play->frame < 0)
    {
      if (timer->t_count > 0)
	timer->t_count--;
      else
	timer->t_count = 0;
    }

  ///Sum Timer
  timer->min = (timer->t_count+timer->top_block_jiffies) / 6000;
  timer->sec = (timer->t_count+timer->top_block_jiffies) / 100 - timer->min * 60;
  timer->msec = (timer->t_count+timer->top_block_jiffies) - (timer->sec * 100 + timer->min * 6000);

#if 0
  timer->min = timer->t_count / 6000;
  timer->sec = timer->t_count / 100 - timer->min * 60;
  timer->msec = timer->t_count - (timer->sec * 100 + timer->min * 6000);
#endif /* 0 */

  sprintf (timer->t_string, "%02d'%02d''%02d", timer->min, timer->sec,
	   timer->msec);
  gtk_label_set (GTK_LABEL (tlabel), timer->t_string);

  ///draw blocks 
  if (run->tmp_list != NULL && run->tmp_list->data != NULL
      && play->frame != 0)
    {
      block = (t_block *) run->tmp_list->data;
      if (play->frame > 0 && block != NULL)
	{
	  if (block->progress == FALSE)
	    {
	      run->p_timer += block->draw_time;
	      block->progress = TRUE;
	    }
	  if ((run->p_timer * 1000) < (timer->t_count * 10) && block != NULL)
	    draw_blocks (tmp);
	}
      else
	{
	  if (block->progress == TRUE && block != NULL)
	    {
	      run->p_timer -= block->draw_time;
	      block->progress = FALSE;
	    }
	  if ((run->p_timer * 1000) > (timer->t_count * 10) && block != NULL)
	    draw_blocks (tmp);
	}
    }

  if(run->tmp_list!=NULL)
    {
      ///Next Block
      ///Play Backward
      if (play->frame < 0 && play->is_play == TRUE)
	timer->timer_tag =
	  gtk_timeout_add (-10 / play->frame, (GtkFunction) draw_timer,
			   (gpointer) tmp);
      
      ///Play Forward
      else if(play->frame > 0 && play->is_play == TRUE)
	timer->timer_tag =
	  gtk_timeout_add (10 / play->frame, (GtkFunction) draw_timer,
			   (gpointer) tmp);
    }
  else
    {
      play_label_change(tmp);
      timer->t_count = result->read_jiffies*1000 + result->total_time * 100;
    }

  //time = timer->t_count /(result->total_time*100) * 100;
  time = timer->t_count / result->total_time;

  ///Timer Widget Update!!
  gtk_adjustment_set_value (GTK_ADJUSTMENT (scale->adjust), time);

}

/**
 * @brief draw_view - generalization to drawing
 * @param[in] pointer who contains all parameters
 * @param[in] event type
 */
void
draw_view (gpointer tmp, gint ev_type)
{
  t_app_status *app = ((t_all_data *) tmp)->app;
  t_draw *blks;
  blks = &app->blks;
  //blks->summary_cnt=app->summary.cnt;
  //check_draw_area(tmp);

  switch (ev_type)
    {
    case EV_STOP:
    case EV_SUMMARY:
      //draw_initalize(app->summary.cnt,tmp);
      draw_initalize (blks->summary_cnt, tmp);
      break;
    case EV_PLAY:
      break;
    }
}

/**
 * @brief draw_blocks - control to drawing
 * @param[in] pointer who contains all parameters
 */
void
draw_blocks (t_all_data * all)
{
  t_app_status *app = ((t_all_data *) all)->app;
  GtkWidget *slabel = app->size.widget;
  t_draw *blks = &app->blks;
  GtkStyle *style = blks->widget->style;

  t_run_status *run = ((t_all_data *) all)->run;
  t_play *play = &run->play;
  t_block *block = (t_block *) run->tmp_list->data;
  t_block *pblock = NULL;
  t_timer *timer = &run->timer;
  t_result *result = &run->result;
  gint *first_draw = run->first_draw;

  GList *nlist = NULL;
  gint type = 0;		// = run->type;
  gint rate = 1;
  gdouble d_canceler = 0;

  gint tmpsize = 0;
  gulong offset = 0;

  blks->gc = style->bg_gc[GTK_STATE_NORMAL];
  
  play->is_next = TRUE;
  
  if (app->size.string == NULL)
    {
      app->size.string = malloc (sizeof (gchar) * 16);
      if (app->size.string == NULL)
	abort();
	//exit (1);
      memset (app->size.string, '\0', sizeof (gchar) * 16);
    }

  ///Play,Forward.
  if (play->frame > 0)
    {
      if (first_draw[block->blk_no / blks->summary_cnt] == 0)
	{
	  first_draw[block->blk_no / blks->summary_cnt] = block->blk_no;
	}
      if (run->tmp_list->next == NULL)
	{
	  play->is_next = FALSE;
	  d_canceler = block->draw_time;
	}
      else
	{
	  play->is_next = TRUE;
	  nlist = g_list_next (run->tmp_list);
	}

      if (run->tmp_list->prev != NULL)
	pblock = (t_block *) (run->tmp_list->prev->data);

      ///rewrite last draw block.
      run->b_loc++;
      run->size += block->size;

      if (run->b_loc > run->b_edge)
	run->b_edge = run->b_loc;

      if (run->p_timer * 1000 - timer->t_count * 10 > 0 && run->d_tag == 0)
	run->d_tag =
	  gtk_timeout_add ((1000 * block->draw_time / play->frame),
			   (GtkFunction) delay_absorber, (gpointer) all);

      run->load_list = g_list_append (run->load_list, block);

      type = run->type;
    }
  ///rewind
  else if (play->frame < 0)
    {
      ///if block list at last.
      if (run->tmp_list->prev == NULL)
	{
	  play->frame = 0;
	  play->is_play = FALSE;
	  play->is_pause = FALSE;

	  timer->t_count = result->read_jiffies*100;

	  if (timer->timer_tag != 0)
	    stop_timer (all);
	  reset_timer (all);
	  play->is_next = FALSE;

	  play_label_change (all);
	  timer->t_count = result->read_jiffies*100;

	  run->size = 0;
	  sprintf (app->size.string, "%d", run->size);
	  gtk_label_set (GTK_LABEL (slabel), app->size.string);

	  draw_update (blks);
	  return;
	}
      ///if list has more previous element.
      else
	{
	  nlist = g_list_previous (run->tmp_list);
	  block = (t_block *) nlist->data;

	  if (run->tmp_list->next != NULL)
	    pblock = (t_block *) (run->tmp_list->next->data);

	  run->b_loc--;

	  if (first_draw[block->blk_no / blks->summary_cnt] == block->blk_no)
	    {
	      type = B_GRND;
	      rate = 0;
	    }
	  else
	    {
	      type = run->type;
	      rate = run->rate[block->blk_no / blks->summary_cnt];
	    }

	  run->size -= block->size;

	  if (run->p_timer >= (block->utc_time - result->start_time))
	    {
	      run->p_timer = (block->utc_time - result->start_time);
	      run->d_tag =
		gtk_timeout_add ((1000 * block->draw_time / play->frame),
				 (GtkFunction) delay_absorber,
				 (gpointer) all);
	    }
	  run->load_list = g_list_remove (run->load_list, block);
	  play->is_next = TRUE;
	}
    }
  ///when re-draw flag is on.
  if (play->r_draw)
    {
      draw_initalize (blks->summary_cnt, all);
      //draw_view (all, EV_STOP);
      play->r_draw = FALSE;
    }

  sprintf (app->size.string, "%d", run->size);
  gtk_label_set (GTK_LABEL (slabel), app->size.string);

  if (run->restype == READ_BLOCKS)
    {
      if (play->frame > 0)
	draw_one_block (blks, block, run->type, 0, all);
      else
	draw_one_block (blks, block, B_GRND, 0, all);

      if (pblock != NULL)
	{
	  draw_one_frame (blks, pblock, F_GRND, 0, all);
	  draw_one_frame (blks, block, F_NOW, 0, all);
	}
    }
  else if (run->restype == CLOOP_HEADER)
    {
      tmpsize = 0;
      offset = 0;
      offset = block->cloop->offset;

      if (pblock != NULL)
	{
	  draw_one_frame (blks, pblock, F_GRND, pblock->cloop->offset, all);
	  draw_one_frame (blks, block, F_NOW, offset, all);
	}

      while (tmpsize < block->cloop->size)
	{
	  if (play->frame > 0)
	    draw_one_block (blks, block, run->type, offset, all);
	  else
	    draw_one_block (blks, block, B_GRND, offset, all);

	  offset += blks->summary_cnt * SECTOR_SIZE;
	  tmpsize += blks->summary_cnt * SECTOR_SIZE;
	}
    }

  draw_update (blks);

  if (play->frame > 0 && run->b_loc == run->b_edge)
    append_one_list (all, block);

  run->tmp_list = nlist;

  ///when list has no element for next
  if (play->is_next == FALSE)
    {
      dbg ("draw over.\n");
      //what is "100"?
      //timer->t_count = result->read_jiffies*100 + run->result.total_time * 100;
      //timer->t_count = run->result.extract_time_sum*100;

      draw_timer (all);
      //stop_timer(all);

      play->frame = 0;
      play->is_play = FALSE;
      play->is_pause = FALSE;

      run->size = result->total_size;
      sprintf (app->size.string, "%d", run->size);
      gtk_label_set (GTK_LABEL (slabel), app->size.string);

      play_label_change (all);

      draw_update (blks);
      return;
    }
}

/**
 * @brief pause_playing - pause playing
 * @param[in] pointer who contains all parameters
 */
void
pause_playing (gpointer tmp)
{
  stop_timer (tmp);
}

/**
 * @brief init_draw_area - Initalize draw area.
 * @param[in] event type
 * @param[in] pointer to widget status
 * @param[out] Return "TRUE" when init successful.
 */
gboolean
init_draw_area (gint ev_type, gpointer p)
{
  t_draw *blks = &((t_app_status *) p)->blks;
  GtkScrolledWindow *win;
  GtkAdjustment *adj;

  win = GTK_SCROLLED_WINDOW (blks->scrl_win);
  adj = gtk_scrolled_window_get_vadjustment (win);
  gtk_adjustment_set_value (adj, 0.0);

  if (blks->pixmap)
    {
      draw_update (blks);
    }
  else
    {
      return FALSE;
    }

  draw_update (blks);

  return TRUE;
}

/**
 * @brief init_gui Initalize Widget
 * @param[in] pointer to widget status
 * @param[in] pointer to run status
 * @param[in] event type
 */
void
init_gui (t_app_status * app, t_run_status * run, gint ev_type)
{
  gint i, max;

  if (ev_type == EV_PLAY)
    {
      gtk_clist_clear (GTK_CLIST (app->list.widget));
    }
  i = max = 0;

  gtk_widget_set_sensitive (GTK_WIDGET (app->resourcemenu_item[0]), FALSE);
  gtk_widget_set_sensitive (GTK_WIDGET (app->resourcemenu_item[1]), FALSE);

  if (!init_draw_area (ev_type, app))

    return;

}

/**
 * @brief main - Main routine
 * @param[in] if this value is not "2", then it is amusing somewhere. 
 * @param[in] "read_blocks"log filename
 * @param[out] A trivial joke degree isn't meant in this value.
 */
int
main (int argc, char *argv[])
{
  /***  ***/
  //value set
  t_all_data all;
  t_app_status app;
  t_run_status run;
  int i;

  memset (&all, '\0', sizeof (t_all_data));
  memset (&app, '\0', sizeof (t_app_status));
  memset (&run, '\0', sizeof (t_run_status));

  all.app = &app;
  all.run = &run;

  //GTK
  gtk_set_locale ();

  g_thread_init (NULL);
  gdk_threads_init ();

  gtk_init (&argc, &argv);

  app.window = create_window (&all);

  //ѿåƥ
  for (i = 0; i < MAX_VIEW; i++)
    {
      app.blks.bk_color[i].red = (gushort) c_path[i][0];
      app.blks.bk_color[i].green = (gushort) c_path[i][1];
      app.blks.bk_color[i].blue = (gushort) c_path[i][2];
    }

  if (app.blks.b_gc == NULL)
    {
      app.blks.b_gc = gdk_gc_new (app.blks.widget->window);
    }
  app_font_init (&app, FONT);

  run.type = R_TIME;
  run.b_loc = 0;
  run.p_timer = 0;
  run.delay = 0;

  run.play.is_play = FALSE;
  run.play.is_pause = FALSE;
  run.play.frame = 0;

  get_summary_cnt (&app.summary);
  reset_timer (&all);  
  init_gui (&app, &run, EV_STOP);

  if(access("/proc/cloop/reset_read_blocks",F_OK) == 0)
    gtk_widget_set_sensitive(app.reset_menu,TRUE);

  if(run.play.is_play)dbg("play.\n");
  if(run.play.is_pause)dbg("pause.\n");
  dbg("play.frame:%d\n",run.play.frame);

  gdk_threads_enter ();
  gtk_main ();
  gdk_threads_leave ();


  return 0;
}
