/************************************************************
* Copyright (C) 2008 Masahiko SAWAI All Rights Reserved. 
************************************************************/

package say.wiiremote;

import java.io.IOException;

public class WiiRemote
{
	public static final int MAX_DEVICES = 32;

	public static final int MASK_BUTTON_TWO    = 0x0001;
	public static final int MASK_BUTTON_ONE    = 0x0002;
	public static final int MASK_BUTTON_B      = 0x0004;
	public static final int MASK_BUTTON_A      = 0x0008;
	public static final int MASK_BUTTON_MINUS  = 0x0010;
	public static final int MASK_BUTTON_HOME   = 0x0080;
	public static final int MASK_BUTTON_LEFT   = 0x0100;
	public static final int MASK_BUTTON_RIGHT  = 0x0200;
	public static final int MASK_BUTTON_DOWN   = 0x0400;
	public static final int MASK_BUTTON_UP     = 0x0800;
	public static final int MASK_BUTTON_PLUS   = 0x1000;
	public static final int MASK_BUTTONS = (
		MASK_BUTTON_TWO|MASK_BUTTON_ONE| 
		MASK_BUTTON_B|MASK_BUTTON_A| 
		MASK_BUTTON_MINUS|MASK_BUTTON_HOME| 
		MASK_BUTTON_LEFT|MASK_BUTTON_RIGHT| 
		MASK_BUTTON_DOWN|MASK_BUTTON_UP| 
		MASK_BUTTON_PLUS);

	public static final int MASK_LED_1    = 0x0001;
	public static final int MASK_LED_2    = 0x0002;
	public static final int MASK_LED_3    = 0x0004;
	public static final int MASK_LED_4    = 0x0008;
	public static final int MASK_LEDS = (
		MASK_LED_1|MASK_LED_2|
		MASK_LED_3|MASK_LED_4);

	public static final int FUNCTION_CONTINUOUS = 0;
	public static final int FUNCTION_MOTION     = 1;
	public static final int FUNCTION_IR         = 2;
	public static final int FUNCTION_SPEAKER    = 3;

	/* read write data */
	public static final int DATA_FORCE_FEEDBACK      = 0;
	public static final int DATA_LEDS                = 1;
	public static final int DATA_SPEAKER_FORMAT      = 2;
	public static final int DATA_SPEAKER_VOLUME      = 3;
	public static final int DATA_SPEAKER_SAMPLE_RATE = 4;
	/* read only data */
	public static final int DATA_BUTTONS             = 5;
	public static final int DATA_MOTION_X            = 6;
	public static final int DATA_MOTION_Y            = 7;
	public static final int DATA_MOTION_Z            = 8;
	public static final int DATA_IR1_FOUND           = 9;
	public static final int DATA_IR1_SIZE            = 10;
	public static final int DATA_IR1_X               = 11;
	public static final int DATA_IR1_Y               = 12;
	public static final int DATA_IR2_FOUND           = 13;
	public static final int DATA_IR2_SIZE            = 14;
	public static final int DATA_IR2_X               = 15;
	public static final int DATA_IR2_Y               = 16;
	public static final int DATA_IR_FOUND            = 17;
	public static final int DATA_IR_SIZE             = 18;
	public static final int DATA_IR_X                = 19;
	public static final int DATA_IR_Y                = 20;
	public static final int DATA_BATTERY_LEVEL       = 21; /* 0 - 200(?) */

	public static final int IO_TIMEOUT = -2;
	public static final int IO_ERROR   = -1;
	public static final int IO_SUCCESS = 0;

	private static final WiiRemote[] WIIREMOTES = new WiiRemote[MAX_DEVICES];

	static
	{
		System.loadLibrary("wiiremote-java");

		// init WIIREMOTES
		for (int i = 0;i < MAX_DEVICES;i++)
		{
			WIIREMOTES[i] = new WiiRemote(i);
		}
	} 

	private static int getButtonMaskFromButtonName(String buttonNameString)
	{
		int buttonMask = 0x00;
		String str = buttonNameString.toUpperCase();

		if (str.indexOf('2') >= 0) buttonMask |= MASK_BUTTON_TWO;
		if (str.indexOf('1') >= 0) buttonMask |= MASK_BUTTON_ONE;
		if (str.indexOf('B') >= 0) buttonMask |= MASK_BUTTON_B;
		if (str.indexOf('A') >= 0) buttonMask |= MASK_BUTTON_A;
		if (str.indexOf('-') >= 0) buttonMask |= MASK_BUTTON_MINUS;
		if (str.indexOf('H') >= 0) buttonMask |= MASK_BUTTON_HOME;
		if (str.indexOf('L') >= 0) buttonMask |= MASK_BUTTON_LEFT;
		if (str.indexOf('R') >= 0) buttonMask |= MASK_BUTTON_RIGHT;
		if (str.indexOf('D') >= 0) buttonMask |= MASK_BUTTON_DOWN;
		if (str.indexOf('U') >= 0) buttonMask |= MASK_BUTTON_UP;
		if (str.indexOf('+') >= 0) buttonMask |= MASK_BUTTON_PLUS;

		return buttonMask;
	}

	public static boolean checkButtonAny(int buttonState, String buttonMaskString)
	{
		return checkButtonAny(buttonState, getButtonMaskFromButtonName(buttonMaskString));
	}

	public static boolean checkButtonAll(int buttonState, String buttonMaskString)
	{
		return checkButtonAll(buttonState, getButtonMaskFromButtonName(buttonMaskString));
	}

	public static boolean checkButtonAny(int buttonState, int buttonMask)
	{
		return ((buttonState & buttonMask) != 0);
	}

	public static boolean checkButtonAll(int buttonState, int buttonMask)
	{
		return ((buttonState & buttonMask) == buttonMask);
	}

	/**
	* Get Wii Remote devices.
	*
	* @return  an array of Wii Remote devices.
	**/
	public static WiiRemote[] getWiiRemotes()
	{
		int number = getNumWiiRemote();
		WiiRemote[] wiiRemotes = new WiiRemote[number];

		for (int i = 0;i < number;i++)
		{
			wiiRemotes[i] = WIIREMOTES[i];
		}

		return wiiRemotes;
	}


	/**
	* Initialize this library system. 
	* It is necessary to call this function before
	* the call of any other functions. 
	**/
	public static native void init() throws IOException; 

	/**
	* Quit library system.
	* You should call this method before stop the program. 
	**/
	public static native void quit() throws IOException;

	/**
	* Update all opened Wii Remote devices.
	*
	* @see #getState
	**/
	public static native void updateAll() throws IOException;

	/**
	* Checks whether there is input data from the Wii Remote device.
	* The state is updated if there is data. 
	*
	* @return IO_ERROR, IO_TIMEOUT, or updated device index.
	* @see #getState
	**/
	public static native int poll() throws IOException;

	/**
	* Get number of Wii Remote devices.
	*
	* @return the number of Wii Remote devices.
	**/
	public static native int getNumWiiRemote();

	/**
	* Get Wii Remote device at specified index.
	* If an illegal value is passed to <code>deviceIndex</code>,
	* an exception is occurred.
	*
	* @param  deviceIndex  the index of device.
	*                       a valid range is 0 to WiiRemote#getNumWiiRemote()-1.
	* @return an instance of WiiRemote.
	**/
	public static native WiiRemote getWiiRemoteAt(int deviceIndex)
		throws IOException;

	private int deviceIndex;

	private WiiRemote(int deviceIndex)
	{
		this.deviceIndex = deviceIndex;
	}

	/**
	* Check button state.
	* 
	*
	* @param buttonMaskString  an button mask specified as string.
	*                          You can use following character.
	* <ul>
	* <li>'2'</li>
	* <li>'1'</li>
	* <li>'B'</li>
	* <li>'A'</li>
	* <li>'-'</li>
	* <li>'H'</li>
	* <li>'L'</li>
	* <li>'R'</li>
	* <li>'D'</li>
	* <li>'U'</li>
	* <li>'+'</li>
	* </ul>
	* @return  return true, if one of specified buttons is pressed. 
	**/
	public boolean checkButtonAny(String buttonMaskString)
	{
		return checkButtonAny(this.getState(DATA_BUTTONS), buttonMaskString);
	}

	/**
	* Check button state.
	* 
	* @param buttonMaskString  an button mask specified as string.
	*                          You can use following character.
	* <ul>
	* <li>'2'</li>
	* <li>'1'</li>
	* <li>'B'</li>
	* <li>'A'</li>
	* <li>'-'</li>
	* <li>'H'</li>
	* <li>'L'</li>
	* <li>'R'</li>
	* <li>'D'</li>
	* <li>'U'</li>
	* <li>'+'</li>
	* </ul>
	* @return  return true, if all of specified buttons is pressed. 
	**/
	public boolean checkButtonAll(String buttonMaskString)
	{
		return checkButtonAll(this.getState(DATA_BUTTONS), buttonMaskString);
	}

	/**
	* Check button state.
	* 
	* @param buttonMask  an button mask specified as string.
	*                          You can use following character.
	* @return  return true, if one of specified buttons is pressed. 
	**/
	public boolean checkButtonAny(int buttonMask)
	{
		return checkButtonAny(this.getState(DATA_BUTTONS), buttonMask);
	}

	/**
	* Check button state.
	* 
	* @param buttonMask  an button mask specified as string.
	*                          You can use following character.
	* @return  return true, if all of specified buttons is pressed. 
	**/
	public boolean checkButtonAll(int buttonMask)
	{
		return checkButtonAll(this.getState(DATA_BUTTONS), buttonMask);
	}

	/**
	* Open connection to Wii Remote
	**/
	public native void open() throws IOException;

	/**
	* Checks whether the connection to Wii Remote has opened. 
	*
	* @return  return true if device is opened.
	**/
	public native boolean isOpened();

	/**
	* Close connection to Wii Remote.
	**/
	public native void close() throws IOException;

	/**
	* Update the state of the Wii Remote device. 
	**/
	public native void update() throws IOException;

	/**
	* Checks whether the specified function is enabled. 
	*
	* @return  return true if the specified function is enabled.
	**/
	public native boolean isEnabled(int functionType);

	/**
	* Enable or disable the specified function. 
	**/
	public native void setEnabled(int functionType, boolean value)
		throws IndexOutOfBoundsException, IOException;

	/**
	* Get device state from WiiRemote.
	*
	* @param dataType  the type of data.
	* @return  the state of the device.
	**/
	public native int getState(int dataType);

	/**
	* Sets the state of the Wii Remote device. 
	*
	* @param dataType  the type of data.
	* @param value     the state of the device.
	**/
	public native void setState(int dataType, int value)
		throws IndexOutOfBoundsException, IOException;

	/**
	* Play sound.
	**/
	public native void playSound(byte[] data, int offset, int length)
		throws IOException;

	/**
	* Play sound.
	**/
	public void playSound(byte[] data)
		throws IOException
	{
		this.playSound(data, 0, data.length);
	}

	/**
	* Writes data to the specified memory area. 
	**/
	public native void writeDataToMemory(int address,
		byte[] data, int dataSize) throws IOException;

}
