#!/bin/bash
# _/_/_/_/ _/_/_/_/ _/_/_/_/ _/_/_/_/ _/_/_/_/ _/_/_/_/ _/_/_/_/ _/_/_/_/
# Name          : heartbeat_pcsmoni.sh
# Description   : Heartbeat のプロセス監視を行う
#                 -- heartbeat関連プロセスが異常終了、あるいは
#                    プロセスリスタートがかかっている場合に、
#                    master control processをkillする。
#                    (watchdog経由でノードをrebootする)
# Author        : 2006.10.19 h.yamauchi
# Editor        : 2007.02.09 t.fukada
# Editor        : 2007.03.01 m.takeuchi
# Source        : /etc/ha.d/monitoring/heartbeat_pcsmoni.sh
# OS  Version   : RHEL ES4 Update4
# HB2 Version   : 2.0.8.p1
#
# License: GNU General Public License (GPL)
# Copyright (c) 2008 NIPPON TELEGRAPH AND TELEPHONE CORPORATION
# _/_/_/_/ _/_/_/_/ _/_/_/_/ _/_/_/_/ _/_/_/_/ _/_/_/_/ _/_/_/_/ _/_/_/_/

HA_DIR=/etc/ha.d
RUNDIR=/var/run

# サービス操作情况ファイル
SERVICE_STATE=${RUNDIR}/heartbeat/start_stop_state

# 共通で生存を監視するHeartbeatプロセス
MONITORING_PROCESS=(
"heartbeat: FIFO reader"
"heartbeat: write: bcast eth2"
"heartbeat: read: bcast eth2"
"heartbeat: write: bcast eth4"
"heartbeat: read: bcast eth4"
"heartbeat: write: ping"
"heartbeat: read: ping"
"heartbeat/ccm"
"heartbeat/cib"
"heartbeat/lrmd"
"heartbeat/crmd"
"heartbeat/stonithd"
"heartbeat/attrd"
"heartbeat/mgmtd"
)

# リスタート監視対象のHeartbeatプロセス(DC)
MONITORING_DC_PROCESS=(
"heartbeat/tengine"
"heartbeat/pengine"
)

# リスタート監視対象のHeartbeatプロセス(DC/非DC共通)
MONITORING_RESTART_CHK_PROCESS=(ccm cib lrmd crmd stonithd attrd mgmtd)

# Heartbeatプロセスモニタ プロセス
KILLING_SELF_PROCESS=(heartbeat_pcsmoni.sh)

# 監視異常時のHeartBeatプロセスのKILL対象指定
KILLING_PROCESS=(
"heartbeat: master control process"
)

# 開始待ち時間(Default:30)
MON_HB_START_WAIT_TIME=30

# モニタリング間隔(Default:60)
MON_HB_MONITORING_WAIT_TIME=30

# ログファイル
MON_HB_MONITORING_HBLOG=/var/log/heartbeat_process_monitor.log

# for ocf_log
#. /usr/lib64/heartbeat/ocf-shellfuncs
#HALOG_FILE=/var/log/ha-log

# HBプロセスモニタリングディレクトリ
BASE_DIR=${HA_DIR}/monitoring

# プロセスリスタート判定用PIDディレクトリ
WORK_PID_DIR=${RUNDIR}/heartbeat/rsctmp

# ログメッセージ
LPPID="Heartbeat Process Monitor($$)"
INF="INFO"
ERR="ERROR"
WAN="WARNING"
ILOG001="${INF}: monitoring [start]"
ILOG002="${INF}: monitoring [beginning waiting]"
ILOG003="${INF}: monitoring [end]"
ILOG006="${INF}: monitoring [process is alive]"
ELOG001="${ERR}: target monitor process is down -- "
ELOG002="${ERR}: target monitor process is restarted -- "
WLOG001="${WAN}: killing heartbeat process -- "

#
# ロギング
#
logging(){
    echo `date` ${LPPID} ${1} >> ${MON_HB_MONITORING_HBLOG}

    # for ocf_log
    #ocf_log debug ${LPPID} ${1} >> ${HALOG_FILE} 2>&1
}

#
# モニタ初期化
#
heartbeat_monitor_start(){
    #
    # モニタリングのスクリプトが実行中の場合は停止する
    #
    for KILL_SELF_PROC in "${KILLING_SELF_PROCESS[@]}"
    do
        ps -ef | grep "${KILL_SELF_PROC}" | grep -v grep | \
          awk -v PID=$$ '$2!=PID {print $2}' | xargs kill -9 > /dev/null 2>&1
    done

    #
    # heartbeatからのシグナルを拒否
    #
    trap '' 1 2 3 15

    #
    logging "${ILOG002} ${MON_HB_START_WAIT_TIME}sec"
    sleep $MON_HB_START_WAIT_TIME
}

#
# モニタリング対象プロセスの生存確認（共通プロセス）
#
isalive_process(){

    ISALIVE_FLG=1

    # モニタリング対象プロセスの生存を確認する
    for MON_PROC in "${MONITORING_PROCESS[@]}"
    do
        pgrep -fl "${MON_PROC}" > /dev/null 2>&1
        PRESULT=$?
        if [ ${PRESULT} -ne 0 ]; then
            K_FLG=`cat ${SERVICE_STATE}`
            if [ ${K_FLG} = "start" ]; then
                logging "${ELOG001} ${MON_PROC}"
            fi
            ISALIVE_FLG=0
            break
        fi
    done

    if [ ${ISALIVE_FLG} -eq 1 ]; then
        # 対象プロセスがリスタートされている場合は、異常処理を実施
        for MON_PROC in "${MONITORING_RESTART_CHK_PROCESS[@]}"
        do
            isProcess_restart ${MON_PROC}
            PRESULT=$?
            if [ ${PRESULT} -ne 0 ]; then
                ISALIVE_FLG=0
                break
            fi
        done
    fi
    return ${ISALIVE_FLG}
}

#
# モニタリング対象プロセスの生存確認（DCプロセス）
#
isalive_process_dc(){

    ISALIVE_FLG=1

    # モニタリング対象プロセス(DC)の生存を確認する
    for MON_PROC in "${MONITORING_DC_PROCESS[@]}"
    do
        pgrep -fl "${MON_PROC}" > /dev/null 2>&1
        PRESULT=$?
        if [ ${PRESULT} -ne 0 ]; then
            K_FLG=`cat ${SERVICE_STATE}`
            if [ ${K_FLG} = "start" ]; then
                logging "${ELOG001} ${MON_PROC}"
            fi
            ISALIVE_FLG=0
            break
        fi
    done

    if [ ${ISALIVE_FLG} -eq 1 ]; then
        # pengine,tengineがリスタートされている場合は、異常処理を実施
        for MON_PROC in "${MONITORING_DC_PROCESS[@]}"
        do
            isProcess_restart ${MON_PROC}
            PRESULT=$?
            if [ ${PRESULT} -ne 0 ]; then
                ISALIVE_FLG=0
                break
            fi
        done
    fi

    return ${ISALIVE_FLG}
}

#
# 特定プロセスのリスタート判定処理
# (crmd, pengine, tengineは異常リスタートでPIDが変わるがそのままではMONOTORが効かなくなる為)
#
isProcess_restart(){
    PRESULT=0

    # 対象プロセス初期化
    TARGET_PROC=$1

    # 作業PIDファイルを初期化
    PID_FILE=${WORK_PID_DIR}/${TARGET_PROC}.pid

    # 現在の対象プロセスのPIDを取得する
    NOW_PID=`ps -ef | grep heartbeat | grep ${TARGET_PROC} | grep -v grep | awk '{print $2}'`
    if [ -z ${NOW_PID} ]; then
        # PIDが取れない場合は処理しない
        :
    else
        if [ -f ${PID_FILE} ]; then
            # 以前のPENGINEのPIDを取得する
            PREV_PID=`cat ${PID_FILE}`

            # 以前のPIDと現在のPIDに違いがあれば、対象プロセスのrestartは異常とみなす
            if [ ${PREV_PID} -ne ${NOW_PID} ]; then
                # サービス状況を読み込む
                K_FLG=`cat ${SERVICE_STATE}`
                if [ ${K_FLG} = "start" ]; then
                    # 異常によるrestartなのでログを出力する
                    #
                    logging "${ELOG002} ${TARGET_PROC}"
                fi
                PRESULT=1
            else
                # 同じPIDの場合は正常
                :
            fi
        else
            # PIDファイルが存在しない場合は、初期PIDファイルを作成する
            echo ${NOW_PID} > ${PID_FILE}
        fi
    fi

    return ${PRESULT}
}

#
# HeartBeat ProcessのＫＩＬＬ
#
heartbeat_killall(){
    # HeartBeat PROCESS系のKILL
    for KILL_PROC in "${KILLING_PROCESS[@]}"
    do
        pgrep -fl "${KILL_PROC}" > /dev/null 2>&1
        RESULT=$?
        if [ ${RESULT} -eq 0 ]; then
            KILL_PID=`pgrep -fl "${KILL_PROC}" | awk '{print $1}'`
            logging "${WLOG001} ${KILL_PROC}(pid:${KILL_PID})"
            echo ${KILL_PID} | xargs kill -9 > /dev/null 2>&1
        fi
    done
}

#
# DCノードかどうかの確認
#
isdc(){

    # DCノード名は、crmadminだとpengineがダウンしたときに
    # コマンドが復帰しなくなってしまうためcrm_monで取得する
    #DCNODE=`/usr/sbin/crmadmin -D | awk '{print $4}'`
    DCNODE=`crm_mon -1 | awk '{if($1=="Current") print $3}' | tr [:upper:] [:lower:]`
    MYNODE=`uname -n | tr [:upper:] [:lower:]`
    if [ ${DCNODE} = ${MYNODE} ]; then
        # DCノード
        ISDC=1
    else
        ISDC=0
    fi
    return ${ISDC}
}

#
# モニタリング
#
heartbeat_monitor(){

    heartbeat_monitor_start
    logging "${ILOG001}"

    #
    # Monitoringループ
    #
    while [ 1 ]
    do
        # DCノードかどうかチェックする
        isdc
        ISDC=$?

        # モニタリング対象プロセスの生存を確認する
        isalive_process $MONITORING_PROCESS[@]
        ISALIVE_FLG=$?

        if [ $ISALIVE_FLG -eq 1 ]; then
            if [ $ISDC -eq 1 ]; then
                # モニタリング対象プロセスの生存を確認する(DC)
                isalive_process_dc
                ISALIVE_FLG=$?
            fi
        fi

        if [ $ISALIVE_FLG -eq 0 ]; then

            # モニタリング対象プロセスが存在していない場合

            K_FLG=`cat ${SERVICE_STATE}`
            if [ ${K_FLG} = "start" ]; then

                # HeartBeat PROCESS系のKILL
                heartbeat_killall

                # この後watchdogによるreboot

            fi
            break

        else

            # モニタリング対象プロセスが存在している場合
            if [ $ISDC -eq 1 ]; then
                logging "${ILOG006}(DC)"
            else
                logging "${ILOG006}"
            fi

        fi

        K_FLG=`cat ${SERVICE_STATE}`
        if [ ${K_FLG} = "stop" ]; then

            # サービスが停止された場合モニタ終了
            break
        fi

        #
        # モニタリング間隔の間、スリープ
        #
        sleep $MON_HB_MONITORING_WAIT_TIME
    done

    logging "${ILOG003}"
    exit 0
}

#
#
#
heartbeat_usage(){
    echo "Usage : $KILLING_SELF_PROCESS { monitor | killall } " 1>&2
    return 1
}

#
#
#
case $1 in
    monitor)    heartbeat_monitor
        exit $? ;;
    killall)    heartbeat_killall
        exit $? ;;
    *)          heartbeat_usage
        exit $? ;;
esac
exit 0
