/**
 * @file  Clipboard.cpp
 * @brief Nbv{[hNX.
 *
 * @author JIN
 *
 * Copyright (C) 2008- JIN All rights reserved.
 */
#include "StdAfx.h"
#include "Clipboard.h"
#include "MiscUtil.h"

namespace GenericUtility {

namespace {

#define ARRAY_SIZE(x) (sizeof(x)/sizeof((x)[0]))

/// eLXgp̃tH[}bg
#ifdef _UNICODE
static const UINT TEXT_FORMAT = CF_UNICODETEXT;
#else	// #ifdef _UNICODE
static const UINT TEXT_FORMAT = CF_TEXT;
#endif	// #ifdef _UNICODE

}	// anonymous namespace

/////////////////////////////////////////////////////////////////////////////

CClipboard::CClipboard() : m_bOpen(FALSE)
{
}

CClipboard::~CClipboard()
{
	Close();
}

bool CClipboard::Open(bool bEmpty, CWnd* pWnd)
{
	Close();

	// I[v
	m_bOpen = ::OpenClipboard(pWnd->GetSafeHwnd());
	if (!m_bOpen) {
		return false;
	}

	// ɂ
	if (bEmpty) {
		VERIFY(::EmptyClipboard());
	}

	return true;
}

void CClipboard::Close()
{
	if (m_bOpen) {
		VERIFY(::CloseClipboard());
		m_bOpen = FALSE;
	}
}

std::vector<UINT> CClipboard::EnumFormats() const
{
	std::vector<UINT> formats;

	if (!m_bOpen) {
		return formats;
	}

	UINT format = 0;
	while (format = ::EnumClipboardFormats(format)) {
		formats.push_back(format);
	}

	return formats;
}

boost::optional<UINT> CClipboard::GetPriorityFormat(const UINT *paFormatPriorityList, int cFormats)
{
	if (!m_bOpen) {
		return boost::optional<UINT>();
	}

	int ret = ::GetPriorityClipboardFormat(const_cast<UINT*>(paFormatPriorityList), cFormats);
	if (ret < 0) {
		return boost::optional<UINT>();
	}

	return boost::optional<UINT>(ret);
}

CGlobalLockPtr CClipboard::GetGlobalMemoryData(UINT uFormat) const
{
	if (!m_bOpen) {
		return CGlobalLockPtr();
	}

	// Nbv{[hw̃tH[}bgŎ擾
	HGLOBAL hGlobal = ::GetClipboardData(uFormat);
	if (!hGlobal) {
		return CGlobalLockPtr();
	}

	return CGlobalLockPtr(new CGlobalLock(hGlobal));
}

bool CClipboard::SetBitmapData(CBitmap& bitmap)
{
	if (!m_bOpen) {
		return false;
	}

	if (!::SetClipboardData(CF_BITMAP, bitmap)) {
		return false;
	}

	// f^b`
	bitmap.Detach();
	return true;
}

bool CClipboard::SetTextData(LPCTSTR lpszText)
{
	if (!m_bOpen) {
		return false;
	}

	// m
	CGlobalAlloc ga(lpszText);
	if (!ga) {
		return false;
	}

	// Nbv{[hɐݒ肷
	if (!::SetClipboardData(TEXT_FORMAT, ga)) {
		return false;
	}

	// f^b`
	ga.Detach();
	return true;
}

CString CClipboard::GetTextData() const
{
	// Nbv{[hf[^擾
	CGlobalLockPtr gl = GetGlobalMemoryData(TEXT_FORMAT);
	if (!gl) {
		return CString();
	}

	// ɕϊ
	return CString((LPCTSTR)*gl);
}

bool CClipboard::SetFileNameData(const std::vector<CString>& fileNames) const
{
	if (!m_bOpen) {
		return false;
	}

	// m
	CGlobalAlloc ga(fileNames);
	if (!ga) {
		return false;
	}

	// Nbv{[hɐݒ肷
	if (!::SetClipboardData(CF_HDROP, ga)) {
		return false;
	}

	// f^b`
	ga.Detach();
	return true;
}

std::vector<CString> CClipboard::GetFileNameData() const
{
	std::vector<CString> fileNames;

	// Nbv{[hf[^擾
	HDROP hDrop = (HDROP)::GetClipboardData(CF_HDROP);
	if (!hDrop) {
		return fileNames;
	}

	// t@C
	UINT nCount = ::DragQueryFile(hDrop, ~0, NULL, 0);
	if (nCount == 0) {
		return fileNames;
	}

	fileNames.reserve(nCount);
	for (UINT i = 0; i < nCount; ++i) {
		TCHAR szFileName[MAX_PATH];
		if (::DragQueryFile(hDrop, i, szFileName, MAX_PATH)) {
			fileNames.push_back(szFileName);
		}
	}

	return fileNames;
}

/////////////////////////////////////////////////////////////////////////////

bool CClipboard::SetBitmap(CBitmap& bitmap, CWnd* pWnd)
{
	// Nbv{[h
	CClipboard cb;
	// I[vċɂ
	if (!cb.Open(true, pWnd)) {
		return false;
	}
	// rbg}bvf[^ݒ肷
	return cb.SetBitmapData(bitmap);
}

bool CClipboard::SetText(LPCTSTR lpszText, CWnd* pWnd)
{
	// Nbv{[h
	CClipboard cb;
	// I[vċɂ
	if (!cb.Open(true, pWnd)) {
		return false;
	}
	// eLXgf[^ݒ肷
	return cb.SetTextData(lpszText);
}

CString CClipboard::GetText(CWnd* pWnd, size_t retry)
{
	// Nbv{[h
	CClipboard cb;

	// I[vɐǂ
	bool succeeded = false;
	for (size_t i = 0; i < retry + 1; ++i) {
		// I[v
		if (cb.Open(false, pWnd)) {
			// 
			succeeded = true;
			break;
		}
		::Sleep(0);
	}
	// I[vs
	if (!succeeded) {
		return CString();
	}

	// eLXg擾T|[gĂtH[}bg
	static const UINT SUPPORTED_FORMAT[] = {
		// eLXg
		TEXT_FORMAT,
		// t@C
		CF_HDROP,
	};

	// LȃtH[}bg擾
	boost::optional<UINT> format = cb.GetPriorityFormat(SUPPORTED_FORMAT, ARRAY_SIZE(SUPPORTED_FORMAT));
	// LȃtH[}bgȂ
	if (!format) {
		return CString();
	}

	switch (*format) {
		case TEXT_FORMAT:
			// eLXgf[^擾
			return cb.GetTextData();
		case CF_HDROP:
			// t@Cs؂Ŏ擾
			return ConcatenateStrings(cb.GetFileNameData(), _T("\r\n"));
		default:
			return CString();
	}
}

bool CClipboard::SetFileNames(const std::vector<CString>& fileNames, CWnd* pWnd)
{
	// Nbv{[h
	CClipboard cb;
	// I[vċɂ
	if (!cb.Open(true, pWnd)) {
		return false;
	}
	// t@Cݒ肷
	return cb.SetFileNameData(fileNames);
}

/////////////////////////////////////////////////////////////////////////////

CClipboardChain::CClipboardChain(CWnd* pWnd)
	: m_pWnd(pWnd)
	, m_hNextCBChain(NULL)
{
	// Nbv{[hr[Aݒ肷
	if (m_pWnd) {
		m_hNextCBChain = m_pWnd->SetClipboardViewer();
	}
}

CClipboardChain::~CClipboardChain()
{
	// Nbv{[hr[A
	if (m_pWnd) {
		m_pWnd->ChangeClipboardChain(m_hNextCBChain);
	}
}

void CClipboardChain::OnDrawClipboard()
{
	if (m_hNextCBChain != NULL) {
		::SendMessage(m_hNextCBChain, WM_DRAWCLIPBOARD, 0, 0);
	}
}

void CClipboardChain::OnChangeCbChain(HWND hWndRemove, HWND hWndAfter)
{
	// ̃Nbv{[hr[Aւ
	if(m_hNextCBChain == hWndRemove) {
		m_hNextCBChain = hWndAfter;
	}
	// ̃Nbv{[hr[Aɂ̂܂ܓ]
	else if (m_hNextCBChain != NULL) {
		::SendMessage(m_hNextCBChain, WM_CHANGECBCHAIN, (WPARAM)hWndRemove, (LPARAM)hWndAfter);
	}
}

}	// namespace GenericUtility
