package org.gamelet.middleware.canvas.renderer;

import javax.microedition.lcdui.Font;
import javax.microedition.lcdui.Graphics;
import javax.microedition.lcdui.Image;

import org.gamelet.middleware.canvas.BaseCanvas;
import org.gamelet.middleware.util.FPSCounter;

/**
 * 
 * <p>レンダリングスレッドで利用されるレンダラの抽象実装</p>
 * <p>描画処理を実装するには{@link #mainRender(Graphics)}を実装します。<br>
 * <code>#mainRender(Graphics)</code>は{@link #renderContents}内で呼ばれ、
 * スレッドの休止処理やFPSカウンタに応じた描画管理などを<code>AbstractRenderer</code>が行います。
 * </p>
 * 
 * <p>レンダラでは以下の管理も行います。</p>
 * 
 * <ul>
 * <li>バックバッファ</li>
 * <li>フォント設定</li>
 * <li>色設定</li>
 * <li>装飾設定</li>
 * <li>特殊な描画</li>
 * <li>FSPの管理</li>
 * </ul>
 * 
 * <hr>
 * 
 * <h3>バックバッファ</h3>
 * 
 * <p>バックバッファの管理ではバックバッファ用の
 * {@link javax.microedition.lcdui.Image Image}と、その描画用
 * {@link javax.microedition.lcdui.Graphics Graphics}を管理しています。</p>
 *
 * <hr>
 * 
 * <h3>フォント設定</h3>
 * <p>フォントの情報は現在のフォントと１つ前に変更されたフォントの２種類を持つことが出来ます。</p>
 * 
 * <hr>
 *
 *<h3>色設定</h3>
 * <p>背景色と前景色を保持します。前景色はフォントの色となります。</p>
 * <hr>
 * 
 * <h3>装飾設定</h3>
 * 袋文字と影付き文字の装飾情報を保持します。
 * <hr>
 * 
 * <h3>特殊な描画</h3>
 * 
 * <p>袋文字と影付き文字を描画できます。</p>
 * 
 * <hr>
 * 
 * <h3>FPSの管理</h3>
 * <p>コンストラクタに{@link org.gamelet.middleware.util.FPSCounter FPSCounter}を渡した場合、
 * カウンタに設定されたFPSの通りに描画更新されます。</p>
 * 
 * <hr>
 * 
 * <h3>mainRenderの定型文</h3>
 * <pre class="code">
 * protected void mainRender(Graphics g){
 * 
 * 	g.setFont(getCurrentFont());
 * 	g.setColor(fgColor);
 * 
 * 	this.renderBackground(g, 0, 0, width, height);
 * 	
 * 	//描画コード
 * 
 * 	canvas.flushGraphics();
 * }
 * </pre>
 * <p>g.setFont(getCurrentFont())とg.setColor(fgColor)はフォントとフォントの色の設定。
 * widthとheightは背景の塗りつぶすサイズ、
 * canvas.flushGraphics()は最後に描画グラフィックスをフラッシュするコード。
 * これらを忘れると内容が正しく描画されないので注意</p>
 * 
 * <hr>
 *  * <p>詳しくは以下を参照してください。</p>
 * 
 * @see #bgColor
 * @see #fgColor
 * @see #decEdgeColor
 * @see #decShadowColor
 * @see #decShadowLv
 * @see #getBGI()
 * @see #getBGI(int, int)
 * @see #getGraphics()
 * @see #getGraphics(int, int)
 * @see #getCurrentFont()
 * @see #setCurrentFont(Font)
 * @see #getOldFont()
 * @see #textStting(Font, int, int, int, int, int)
 * @see org.gamelet.middleware.util.FPSCounter
 * 
 * @author tasogare
 *
 */
public abstract class AbstractRenderer implements Runnable {

	/**
	 * 背景色
	 */
	public int bgColor = 0xffffff; 

	/**
	 * 前景色
	 */
	public int fgColor = 0x000000;

	/**
	 * 影の位置
	 */
	public int decShadowLv = 1;

	/**
	 * 影の色
	 */
	public int decShadowColor = 0x000000;

	/**
	 * 袋文字の縁取りの色
	 */
	public int decEdgeColor = 0x000000;

	protected BaseCanvas canvas;

	/*
	 * テキスト設定
	 */

	protected FPSCounter counter;
	private boolean mFPS;

	/**
	 * レンダラの状態
	 */
	protected String state;

	/*
	 * レンダラの状態変数
	 */

	/**
	 * レンダラの走っている状態。
	 * この状態のときは仕事をしている
	 */
	public static final String STATE_RUNNING = "st_run";

	/**
	 * レンダラの安定している状態。
	 * この状態のときはレンダラは特に何もやっていない。
	 */
	public static final String STATE_STABLE = "st_sta";

	/**
	 * レンダラの休止している状態。
	 * この状態のときは仕事を休止し、再び走っている状態になったとき仕事を再開する。
	 */
	public static final String STATE_SUSPEND = "st_sus";

	private RendererStateListener rendererStateListener;

	protected long interval = 1000;

	//background image
	private Image bgi;

	/*
	 * フォント装飾
	 */

	private Graphics bg;

	/**
	 * 現在のフォント
	 */
	private Font fntFont = Font.getDefaultFont( );

	/**
	 * 変更前のフォント
	 */
	private Font fntOldFont = fntFont;

	/**
	 * 固定フレームレートで描画するレンダラを作成する
	 * @param canvas 描画先キャンバス
	 */
	public AbstractRenderer(BaseCanvas canvas){
		this.canvas = canvas;
		this.counter = null;
		mFPS = false;
		state = STATE_STABLE;
	}

	/**
	 * 可変フレームレートで描画するレンダラを作成する
	 * @param canvas 描画先キャンバス
	 * @param counter FPSカウンタ
	 */
	public AbstractRenderer(BaseCanvas canvas, FPSCounter counter){
	    	this(canvas);
		this.counter = counter;
		mFPS=true;
	}

	/**
	 * 
	 * 文字を描画する。
	 * 
	 * @param g 描画先
	 * @param ch 描画文字
	 * @param x 塗りつぶす座標 - X
	 * @param y 塗りつぶす座標 - Y
	 * @param anchor アンカー
	 * 
	 * @exception IllegalArgumentException アンカーが不正の場合
	 */
	public void drawChar(
		Graphics g, char ch, int x, int y, int anchor)
		throws IllegalArgumentException{

		g.setColor(fgColor);
		g.drawChar(ch, x, y, anchor);
	}

	/**
	 * 
	 * 袋文字を描画する。
	 * 
	 * @param g 描画先
	 * @param ch 描画文字
	 * @param x 塗りつぶす座標 - X
	 * @param y 塗りつぶす座標 - Y
	 * @param anchor アンカー
	 * 
	 * @exception IllegalArgumentException アンカーが不正の場合
	 */
	public void drawCharEdged(
		Graphics g, char ch, int x, int y, int anchor)
		throws IllegalArgumentException{

		g.setColor(decEdgeColor);

		g.drawChar(ch, x-1, y-1, anchor);
		g.drawChar(ch, x, y-1, anchor);
		g.drawChar(ch, x+1, y-1, anchor);
		g.drawChar(ch, x-1, y, anchor);
		g.drawChar(ch, x+1, y, anchor);
		g.drawChar(ch, x-1, y+1, anchor);
		g.drawChar(ch, x, y+1, anchor);
		g.drawChar(ch, x+1, y+1, anchor);

		g.setColor(fgColor);
		g.drawChar(ch, x, y, anchor);
	}

	/**
	 * 
	 * 文字配列を描画する。
	 * 
	 * @param g 描画先
	 * @param data 描画文字配列
	 * @param x 塗りつぶす座標 - X
	 * @param y 塗りつぶす座標 - Y
	 * @param anchor アンカー
	 * 
	 * @exception ArrayIndexOutOfBoundsException <code>offset</code>と<code>length</code>が無効な範囲の場合
	 * @exception NullPointerException <code>data</code>が<code>null</code>の場合
	 * @exception IllegalArgumentException アンカーが不正の場合
	 */
	public void drawChars(
		Graphics g, char[] data, int offset, int length, int x, int y,
		int anchor) throws ArrayIndexOutOfBoundsException,
		IllegalArgumentException, NullPointerException
	{
		g.setColor(fgColor);
		g.drawChars(data, offset, length, x, y, anchor);
	}

	/**
	 * 
	 * 袋文字配列を描画する。
	 * 
	 * @param g 描画先
	 * @param data 描画文字配列
	 * @param x 塗りつぶす座標 - X
	 * @param y 塗りつぶす座標 - Y
	 * @param anchor アンカー
	 * 
	 * @exception ArrayIndexOutOfBoundsException <code>offset</code>と<code>length</code>が無効な範囲の場合
	 * @exception NullPointerException <code>data</code>が<code>null</code>の場合
	 * @exception IllegalArgumentException アンカーが不正の場合
	 */
	public void drawCharsEdged(
		Graphics g, char[] data, int offset, int length, int x, int y,
		int anchor) throws ArrayIndexOutOfBoundsException,
		IllegalArgumentException, NullPointerException
	{

		g.setColor(decEdgeColor);

		g.drawChars(data, offset, length, x-1, y-1, anchor);
		g.drawChars(data, offset, length, x, y-1, anchor);
		g.drawChars(data, offset, length, x+1, y-1, anchor);
		g.drawChars(data, offset, length, x-1, y, anchor);
		g.drawChars(data, offset, length, x+1, y, anchor);
		g.drawChars(data, offset, length, x-1, y+1, anchor);
		g.drawChars(data, offset, length, x, y+1, anchor);
		g.drawChars(data, offset, length, x+1, y+1, anchor);

		g.setColor(fgColor);
		g.drawChars(data, offset, length, x, y, anchor);
	}

	/**
	 * 
	 * 影付き文字を描画する。
	 * 
	 * @param g 描画先
	 * @param ch 描画文字
	 * @param x 塗りつぶす座標 - X
	 * @param y 塗りつぶす座標 - Y
	 * @param anchor アンカー
	 * 
	 * @exception IllegalArgumentException アンカーが不正の場合
	 */
	public void drawCharShadowd(
		Graphics g, char ch, int x, int y, int anchor)
		throws IllegalArgumentException{

		g.setColor(decShadowColor);
		g.drawChar(ch, x + decShadowLv, y + decShadowLv, anchor);

		g.setColor(fgColor);
		g.drawChar(ch, x, y, anchor);
	}

	/**
	 * 
	 * 影付き文字配列を描画する。
	 * 
	 * @param g 描画先
	 * @param data 描画文字配列
	 * @param x 塗りつぶす座標 - X
	 * @param y 塗りつぶす座標 - Y
	 * @param anchor アンカー
	 * 
	 * @exception ArrayIndexOutOfBoundsException <code>offset</code>と<code>length</code>が無効な範囲の場合
	 * @exception NullPointerException <code>data</code>が<code>null</code>の場合
	 * @exception IllegalArgumentException アンカーが不正の場合
	 */
	public void drawCharsShadowd(
		Graphics g, char[] data, int offset, int length, int x, int y,
		int anchor) throws ArrayIndexOutOfBoundsException,
		IllegalArgumentException, NullPointerException
	{

		g.setColor(decShadowColor);
		g.drawChars(data, offset, length, x + decShadowLv, y + decShadowLv,
			anchor);

		g.setColor(fgColor);
		g.drawChars(data, offset, length, x, y, anchor);
	}

	/**
	 * 
	 * 文字列を描画する。
	 * 
	 * @param g 描画先
	 * @param mess 描画文字列
	 * @param x 塗りつぶす座標 - X
	 * @param y 塗りつぶす座標 - Y
	 * @param anchor アンカー
	 * 
	 * @exception NullPointerException <code>mess</code>が<code>null</code>の場合
	 * @exception IllegalArgumentException アンカーが不正の場合
	 */
	public void drawString(
		Graphics g, String mess, int x, int y, int anchor)
		throws NullPointerException, IllegalArgumentException
	{
		g.setColor(fgColor);
		g.drawString(mess, x, y, anchor);
	}

	/**
	 * 
	 * 袋文字列を描画する。
	 * 
	 * @param g 描画先
	 * @param mess 描画文字列
	 * @param x 塗りつぶす座標 - X
	 * @param y 塗りつぶす座標 - Y
	 * @param anchor アンカー
	 * 
	 * @exception NullPointerException <code>mess</code>が<code>null</code>の場合
	 * @exception IllegalArgumentException アンカーが不正の場合
	 */
	public void drawStringEdged(
		Graphics g, String mess, int x, int y, int anchor)
		throws NullPointerException, IllegalArgumentException
	{

		g.setColor(decEdgeColor);

		g.drawString(mess, x-1, y-1, anchor);
		g.drawString(mess, x, y-1, anchor);
		g.drawString(mess, x+1, y-1, anchor);
		g.drawString(mess, x-1, y, anchor);
		g.drawString(mess, x+1, y, anchor);
		g.drawString(mess, x-1, y+1, anchor);
		g.drawString(mess, x, y+1, anchor);
		g.drawString(mess, x+1, y+1, anchor);

		g.setColor(fgColor);
		g.drawString(mess, x, y, anchor);
	}

	/**
	 * 
	 * 影付き文字列を描画する。
	 * 
	 * @param g 描画先
	 * @param mess 描画文字列
	 * @param x 塗りつぶす座標 - X
	 * @param y 塗りつぶす座標 - Y
	 * @param anchor アンカー
	 * 
	 * @exception NullPointerException <code>mess</code>が<code>null</code>の場合
	 * @exception IllegalArgumentException アンカーが不正の場合
	 */
	public void drawStringShadowd(
		Graphics g, String mess, int x, int y, int anchor)
		throws NullPointerException, IllegalArgumentException
	{

		g.setColor(decShadowColor);
		g.drawString(mess, x + decShadowLv, y + decShadowLv, anchor);

		g.setColor(fgColor);
		g.drawString(mess, x, y, anchor);
	}

	/**
	 * 
	 * 袋部分文字列を描画する。
	 * 
	 * @param g 描画先
	 * @param mess 描画文字列
	 * @param x 塗りつぶす座標 - X
	 * @param y 塗りつぶす座標 - Y
	 * @param anchor アンカー
	 * 
	 * @exception StringIndexOutOfBoundsException <code>offset</code>と<code>len</code>が無効な範囲の場合
	 * @exception NullPointerException <code>mess</code>が<code>null</code>の場合
	 * @exception IllegalArgumentException アンカーが不正の場合
	 */
	public void drawSubstringEdged(
		Graphics g, String mess, int offset, int len, int x, int y, int anchor)
		throws StringIndexOutOfBoundsException, NullPointerException,
		IllegalArgumentException
	{

		g.setColor(decEdgeColor);

		g.drawSubstring(mess, offset, len, x-1, y-1, anchor);
		g.drawSubstring(mess, offset, len, x, y-1, anchor);
		g.drawSubstring(mess, offset, len, x+1, y-1, anchor);
		g.drawSubstring(mess, offset, len, x-1, y, anchor);
		g.drawSubstring(mess, offset, len, x+1, y, anchor);
		g.drawSubstring(mess, offset, len, x-1, y+1, anchor);
		g.drawSubstring(mess, offset, len, x, y+1, anchor);
		g.drawSubstring(mess, offset, len, x+1, y+1, anchor);

		g.setColor(fgColor);
		g.drawSubstring(mess, offset, len, x, y, anchor);
	}

	/**
	 * 
	 * 影付き部分文字列を描画する。
	 * 
	 * @param g 描画先
	 * @param mess 描画文字列
	 * @param x 塗りつぶす座標 - X
	 * @param y 塗りつぶす座標 - Y
	 * @param anchor アンカー
	 * 
	 * @exception StringIndexOutOfBoundsException <code>offset</code>と<code>len</code>が無効な範囲の場合
	 * @exception NullPointerException <code>mess</code>が<code>null</code>の場合
	 * @exception IllegalArgumentException アンカーが不正の場合
	 */
	public void drawSubstringShadowd(
		Graphics g, String mess, int offset, int len, int x, int y, int anchor)
		throws StringIndexOutOfBoundsException, NullPointerException,
		IllegalArgumentException
	{

		g.setColor(decShadowColor);
		g.drawSubstring(mess, offset, len, x + decShadowLv, y + decShadowLv,
			anchor);

		g.setColor(fgColor);
		g.drawSubstring(mess, offset, len, x, y, anchor);
	}

	/**
	 * 
	 * バックグラウンドイメージを取得します。
	 * バックグラウンドイメージは<code>null</code>の可能性があります。
	 * 
	 * @return バックグラウンドイメージ
	 */
	public final Image getBGI() {
		return bgi;
	}

	/**
	 * 新たなバックグラウンドイメージを生成して、
	 * バックグラウンドイメージを取得します。
	 * 
	 * @param width 新しいバックグラウンドイメージの幅
	 * @param height 新しいバックグラウンドイメージの高さ
	 * @return バックグラウンドイメージ
	 */
	public final Image getBGI(
		int width, int height)
	{
		bgi = Image.createImage(width, height);
		return bgi;
	}

	/**
	 * 
	 * 現在のフォントを返す。
	 * 
	 * @return 現在のフォント
	 */
	public final Font getCurrentFont() {
		return fntFont;
	}

	/**
	 * FPSカウンタを取得します。
	 * @return FPSカウンタ
	 */
	public FPSCounter getFPSCounter(){
		return counter;
	}

	/**
	 * バックグラウンドイメージの描画用 <code>Graphics</code> を取得します。
	 * @return 描画用 <code>Graphics</code>
	 * @exception NullPointerException バックグラウンドイメージが<code>null</code>の場合
	 */
	public final Graphics getGraphics() throws NullPointerException {
		return (bgi != null) ? bgi.getGraphics( ) : null;
	}

	/**
	 * 新たなバックグラウンドイメージを生成して、
	 * その描画用 <code>Graphics</code> を取得します。
	 * 
	 * @param width 新しいバックグラウンドイメージの幅
	 * @param height 新しいバックグラウンドイメージの高さ
	 * @return 新たな描画用 <code>Graphics</code>
	 */
	public final Graphics getGraphics(
		int width, int height)
	{
		bgi = Image.createImage(width, height);
		return ( bg = bgi.getGraphics( ) );
	}

	/**
	 * レンダリングスレッドのインターバルを取得します。
	 * @return インターバル値
	 */
	public final long getInterval() {
		return interval;
	}

	/**
	 * 一つ前のフォントを返す。
	 * 
	 * @return 一つ前のフォント
	 */
	public final Font getOldFont() {
		return fntOldFont;
	}

	/**
	 * 背景を現在の背景色で塗りつぶす。
	 * 
	 * @param g 描画先
	 * @param x 塗りつぶす座標 - X
	 * @param y 塗りつぶす座標 - Y
	 * @param width 塗りつぶす幅
	 * @param height 塗りつぶす高さ
	 */
	public final void renderBackground(
		Graphics g, int x, int y, int width, int height)
	{
		g.setColor(bgColor);
		g.fillRect(x, y, width, height);
	}

	/**
	 * <p>内容の描画と休止処理を行います。</p>
	 * 
	 * <p>FPSカウンタが存在するなら可変フレームレート、
	 * 存在しないなら固定フレームレートとなります。</p>
	 * 
	 * <p>固定フレームレートの場合、
	 * レンダリングスレッドのインターバル値を直接指定してスレッドを休止させます。</p>
	 * 
	 * @param g 描画グラフィックス
	 */
	public final void renderContents(Graphics g){

	    setState(STATE_RUNNING);

	    Thread thisThread = Thread.currentThread();
	    while (canvas.getAnimator( ) == thisThread) {
		if(mFPS){
		    getFPSCounter( ).update( );
		}

		System.out.println("running");

		beforeMainRender(g);
		mainRender(g);
		afterMainRender(g);

		if(mFPS){
		    long e = getFPSCounter( ).elapsed( );
		    System.out.println("elapsed " + e + " ms");
		    System.out.println("FPS : " + getFPSCounter( ).getNowFPS());
		    setInterval(e);
		}

		renderSleepOrWait( );
	    }
	}

	/**
	 * 必要に応じてレンダリングスレッドを休止か停止状態にします。
	 *
	 */
	public final void renderSleepOrWait() {
		try {

			Thread.sleep(interval);

			synchronized ( this ) {
				while ( AbstractRenderer.STATE_SUSPEND.equals(state) ) wait( );
				System.gc( );
			}

		} catch ( InterruptedException e ) {
			System.out.println(e.getMessage( ));
		}
	}

	/**
	 * レンダリングスレッドに渡され実行されます。
	 */
	public abstract void run();

	/**
	 * @param bgi 設定する bgi
	 */
	public final void setBGI(
		Image bgi)
	{
		this.bgi = bgi;
	}

	/**
	 * 
	 * 現在のフォントを変更します。
	 * 以前に設定されていたフォントは{@link #getOldFont()}で取得できます。
	 * 
	 * @param fntFont 設定するフォント
	 */
	public final void setCurrentFont(
		Font fntFont)
	{
		fntOldFont = this.fntFont;
		this.fntFont = fntFont;
	}

	/**
	 * �レンダリングスレッドのインターバルを設定します。
	 * 
	 * @param interval インターバル値
	 */
	public final void setInterval(
		long interval)
	{
		this.interval = interval;
	}

	/**
	 * レンダラの状態を設定します。ユーザー定義の状態を扱うときはこのメソッドをオーバーライドします。
	 * @param state 設定する状態
	 * @throws IllegalStateException 無効な状態が設定された場合
	 */
	public void setState(String state) throws IllegalStateException{

	    if(STATE_STABLE.equals(state)){

		this.state = state;
		System.out.println(" crrent state : " + state);

		if(getRendererStateListener() != null){
		    getRendererStateListener().changeStable();
		}

	    }else if(STATE_RUNNING.equals(state)){

		this.state = state;
		System.out.println(" crrent state : " + state);

		if(getRendererStateListener() != null){
		    getRendererStateListener().changeRunning();
		}

	    }else if(STATE_SUSPEND.equals(state)){

		this.state = state;
		System.out.println(" crrent state : " + state);

		if(getRendererStateListener() != null){
		    getRendererStateListener().changeSuspend();
		}

	    }
	}

	/**
	 * レンダラの状態を取得します。
	 * @return レンダラの状態
	 */
	public String getState(){
	    return state;
	}

	/**
	 * レンダラステートリスナを設定します。
	 * @param listnerer リスナ
	 */
	public void setRendererStateListener(RendererStateListener listnerer){
	    rendererStateListener = listnerer;
	}

	/**
	 * レンダラステートリスナを設定します。
	 * @return レンダラステートリスナ
	 */
	public RendererStateListener getRendererStateListener(){
	    return rendererStateListener;
	}

	/**
	 * 
	 * テキスト設定を初期化する。
	 * 
	 * @param font 新しいフォント
	 * @param bgColor 新しい背景色
	 * @param fgColor 新しい前景色
	 * @param decShadowLv 新しい影付き文字の影の位置
	 * @param decShadowColor 新しい影付き文字の影の色
	 * @param decEdgeColor 新しい袋文字の縁取りの色
	 */
	public final void textStting(
		Font font, int bgColor, int fgColor, int decShadowLv,
		int decShadowColor, int decEdgeColor)
	{
		setCurrentFont(font);
		this.bgColor = bgColor;
		this.fgColor = fgColor;

		this.decShadowLv = decShadowLv;
		this.decShadowColor = decShadowColor;

		this.decEdgeColor = decEdgeColor;
	}


	/**
	 * レンダリングスレッドのサスペンド状態を反転します。
	 *
	 */
	public final synchronized void toggleSuspend() {

		boolean rendererRunning = !( AbstractRenderer.STATE_SUSPEND.equals(state) );
		if(rendererRunning){
		    setState(STATE_SUSPEND);
		}else{
		    setState(STATE_RUNNING);
		}

		if ( !rendererRunning ) notify( );
	}

	/**
	 * 実際に内容を描画します。
	 * @param g 描画グラフィックス
	 */
	protected abstract void mainRender(Graphics g);

	/**
	 * {@link #mainRender(Graphics)}が呼ばれる前に呼ばれる。
	 */
	protected abstract void beforeMainRender(Graphics g);

	/**
	 * {@link #mainRender(Graphics)}が呼ばれた後に呼ばれる。
	 */
	protected abstract void afterMainRender(Graphics g);

//	public abstract void keyPressed(int keyCode);
//
//	public abstract void keyRepeated(int keyCode);
//
//	public abstract void keyReleased(int keyCode);
//
//	public abstract void pointerPressed(int x, int y);
//
//	public abstract void pointerReleased(int x, int y);
//
//	public abstract void pointerDragged(int x, int y);
}
