package org.lixm.optional.v11.namespace;

/**
 * <p>
 * 名前空間を処理するクラスのためのインターフェイスです。
 * </p>
 * 
 * <p>
 * このインターフェイスでは名前空間URIとプレフィックスの関連付けおよび、 関連付けられたそれらの取得方法の実装を強制します。
 * </p>
 * 
 * <p>
 * このクラスにに関連付けられる名前空間URIは文字列で表されます。<br>
 * プレフィックスは<code>Object</code>型で表現されているので 任意の型を名前空間URIに関連付けることができます。
 * </p>
 * 
 * <p>
 * それは単純な例では<code>String</code>型となるでしょう。
 * </p>
 * 
 * @author tasogare
 * @version 1.1.0
 * @since 1.1.0
 * 
 */
public interface NamespaceBinding {

    public static final String NS_DEFAULT_PREFIX = "";
    public static final String NS_XML_URI = "http://www.w3.org/XML/1998/namespace";
    public static final String NS_XML_PREFIX = "xml";
    public static final String NS_XMLNS_ATTRIBUTE_URI = "http://www.w3.org/2000/xmlns/";
    public static final String NS_XMLNS_ATTRIBUTE = "xmlns";
    public static final String NS_W3C_XML_SCHEMA_URI = "http://www.w3.org/2001/XMLSchema";
    public static final String NS_W3C_XML_SCHEMA_INSTANCE_URI = "http://www.w3.org/2001/XMLSchema-instance";
    public static final String NS_W3C_XPATH_DATATYPE_URI = "http://www.w3.org/2003/11/xpath-datatypes";
    public static final String NS_RELAXNG_URI = "http://relaxng.org/ns/structure/1.0";

    /**
     * <p>
     * プレフィックスを名前空間URIへ関連付けます。
     * </p>
     * <p>
     * プレフィックスは例えば単純に文字列であったり、そのプレフィックスを持つ要素、 属性に対するアクションテーブルかもしれません。
     * </p>
     * 
     * <p>
     * XMLや名前空間の仕様で定められているプレフィックス、 名前空間URIを変更しようとした場合{@linkplain IllegalArgumentException}を投げます。
     * </p>
     * 
     * @param prefix
     *                プレフィックスを表す任意のObject
     * @param namespaceURI
     *                名前空間URIを表す文字列
     * @return 以前に関連付けられていたプレフィックス。なければnull。
     * @throws IllegalArgumentException
     *                 XMLや名前空間の仕様で決められているプレフィックス、 名前空間URIを変更しようとした場合
     * 
     * @see #NS_DEFAULT_PREFIX
     * @see #NS_XML_PREFIX
     * @see #NS_XML_URI
     * @see #NS_XMLNS_ATTRIBUTE
     * @see #NS_XMLNS_ATTRIBUTE_URI
     */
    public Object put(Object prefix, String namespaceURI)
	    throws IllegalArgumentException;

    /**
     * <p>
     * プレフィックスから関連する名前空間URIを取得します。
     * </p>
     * 
     * @param prefix
     *                プレフィックスを表す任意のObject
     * @return 名前空間URIを表す文字列
     * @throws IllegalArgumentException
     *                 prefixがnullの場合
     */
    public String getNamespaceURI(Object prefix)
	    throws IllegalArgumentException;

    /**
     * 名前空間URIから関連するプレフィックスを取得します。 候補が複数ある場合の戻り値は実装依存です。
     * 
     * @param namespaceURI
     *                名前空間URIを表す文字列
     * @return プレフィックスを表す任意のObject
     * @throws IllegalArgumentException
     */
    public Object getPrefix(String namespaceURI)
	    throws IllegalArgumentException;

    /**
     * 名前空間URIから関連するプレフィックスを配列として取得します。 候補が一つの場合配列の長さは1です。
     * 
     * @param namespaceURI
     *                名前空間URIを表す文字列
     * @return プレフィックスを表す任意のObject配列
     * @throws IllegalArgumentException
     */
    public Object[] getPrefixes(String namespaceURI)
	    throws IllegalArgumentException;
}
