﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using NetMonitor.AmfFormat;
using NetMonitor.LoW.StaticData;
using NetMonitor.View;

namespace NetMonitor.LoW
{
	struct LBossDamageLog
	{
		public Chara chara;
		public int damage;
		public LBossDamageLog(int damage)
		{
			chara = MainData.Me;
			this.damage = damage;
		}

		public static LBossDamageLog FromAmfObject(AmfObjectData obj, DateTime update)
		{
			LBossDamageLog result = new LBossDamageLog();
			foreach (AmfObjectMember mem in obj.Members)
			{
				switch (mem.Key)
				{
					case "chara_data": result.chara = MainData.GetChara((AmfObjectData)mem.Value, update); break;
					case "damage": result.damage = mem.Value.GetIntValue(); break;
				}
			}
			return result;
		}
	}
	enum LBossInfo
	{
		None = 0,
		/// <summary>
		/// サーバーからのデータが不完全でcsvデータもなく推測もできない
		/// </summary>
		Unknown,
		/// <summary>
		/// サーバーから送られてきたデータと静的データを合わせたデータ
		/// </summary>
		Calculated,
		/// <summary>
		/// サーバーから送られてきた完全データ/LBossは一部のデータが欠けているので推論するしかない
		/// </summary>
		//Complete,
	}
	class LBoss:IComparable<LBoss>
	{
		private LBoss(string name, int max_hp)
		{
			this.lboss_name = name;
			this.max_hp = max_hp;
			Doubt = false;
		}
		private LBoss(LBossStatic data)
		{
			StaticData = data;
			Info = LBossInfo.Calculated;
			lboss_name = data.name;
			Doubt = data.Doubt;
			lboss_rank = data.lboss_rank;
			req_damage = data.req_damage;
			req_force_base = data.req_force;
			max_hp = data.max_hp;
			Param = data.Param;
			limit = TimeSpan.FromMinutes(data.limit);
		}
		public LBossStatic StaticData;
		public LBossInfo Info;
		public Chara owner;
		public string lboss_name;
		public int hp;
		public int req_force;//現実に必要なフォース量戦友なら1/3済み
		/// <summary>
		/// おそらくユニークID
		/// </summary>
		public int ld_id;
		public string lb_id;//同名敵ごとに共通のID？
		public bool Doubt;
		public int my_damage;
		public DateTime timelimit;
		public DateTime lastUpdate;
		public bool myself;
		public bool friend_flg;
		public int sec;
		public bool readRecentDamage;
		public readonly List<LBossDamageLog> recent_damage = new List<LBossDamageLog>();

		public readonly int lboss_rank;
		public readonly int req_force_base;
		public readonly int max_hp;
		public readonly int req_damage;
		public readonly BParameter Param;
		public readonly TimeSpan limit;
		public static int GetLBossID(AmfObjectData obj)
		{
			return obj.GetMember("ld_id").GetIntValue();
		}

		public static LBoss FromAmfObject(AmfObjectData obj, DateTime update)
		{
			LBoss result = null;
			LBossStatic data = null;
			string name = null;
			int max_hp = 0;
			AmfObjectData lbossObj = obj.GetMember("lboss") as AmfObjectData;
			if (lbossObj == null)
				goto unknown;
			AmfStringData nameObj = lbossObj.GetMember("lboss_name") as AmfStringData;
			if (nameObj == null)
				goto unknown;
			name = nameObj.GetStrValue();
			max_hp = lbossObj.GetMember("max_hp").GetIntValue();
			if (string.IsNullOrEmpty(name) || max_hp <= 0)
				goto unknown;
			data = LoWDataStatic.GetLBossData(name, max_hp);
			if (data == null)
				goto unknown;
			result = new LBoss(data);
			result.UpdateDate(obj, update);

			return result;
		unknown:
			if (string.IsNullOrEmpty(name))
				name = "(名称不明)";
			result = new LBoss(name, max_hp);
			if (!string.IsNullOrEmpty(name) && max_hp > 0)
				result.UpdateDate(obj, update);
			result.Info = LBossInfo.Unknown;
			result.lastUpdate = update;
			result.Doubt = true;
			return result;
		}

		public void UpdateDate(AmfObjectData obj, DateTime update)
		{
			lastUpdate = update;
			bool hasDamageData = false;
			foreach (AmfObjectMember mem in obj.Members)
			{
				switch (mem.Key)
				{
					case "chara_data": owner = MainData.GetChara((AmfObjectData)mem.Value, update); break;
					case "ld_id": ld_id = mem.Value.GetIntValue(); break;
					case "timelimit": timelimit = mem.Value.GetDateTimeValue(); break;
					case "sec": sec = mem.Value.GetIntValue(); break;
					case "myself": myself = mem.Value.GetBooleanValue(); break;
					case "friend_flg": friend_flg = mem.Value.GetBooleanValue(); break;
					case "lboss":
						{
							AmfObjectData lbossObj = (AmfObjectData)mem.Value;
							foreach (AmfObjectMember lbossMem in lbossObj.Members)
							{
								switch (lbossMem.Key)
								{
									case "lboss_rank":
										if (lboss_rank != lbossMem.Value.GetIntValue())
											Doubt = true;
										break;
									case "max_hp":
										if (max_hp != lbossMem.Value.GetIntValue())
											Doubt = true;
										break;
									//case "lboss_name":
									case "hp": hp = lbossMem.Value.GetIntValue(); break;
									case "req_force": req_force = lbossMem.Value.GetIntValue(); break;
									case "lb_id": lb_id = lbossMem.Value.GetStrValue(); break;
								}
							}
						}
						break;
					case "recent_damage":
						{
							readRecentDamage = true;
	 						hasDamageData = true;
							recent_damage.Clear();
							if (mem.Value is AmfNullData)
								break;
							AmfArrayData array = (AmfArrayData)mem.Value;
							foreach(AmfData data in array.Array)
								recent_damage.Add(LBossDamageLog.FromAmfObject((AmfObjectData)data, update));
						}
						break;
				}
			}
			if (hasDamageData)
			{
				int total_damage = 0;
				my_damage = 0;
				foreach (LBossDamageLog log in recent_damage)
				{
					total_damage += log.damage;
					if(log.chara.myself)
						my_damage += log.damage;
				}
				
				if (max_hp - total_damage != hp)
					Doubt = true;
			}
			if (Doubt && StaticData != null && !StaticData.Doubt)
			{
				NMConsole.TextEnqueue("データ不整合:LBOSS." + lboss_name, TextType.Warnning);
				StaticData.Doubt = true;
			}

		}
		
		public void AddDamageLog(int damage)
		{
			recent_damage.Add(new LBossDamageLog(damage));
			my_damage += damage;
			hp -= damage;
		}


		public string NameWithAbl
		{
			get
			{
				if (Param.ability == BattleAbility.None)
					return lboss_name;
				if (Param.ability == BattleAbility.ImmuneToPhysical)
					return lboss_name + "(物理無効)";
				if (Param.ability == BattleAbility.ImmuneToMagic)
					return lboss_name + "(魔法無効)";
				return lboss_name;
			}
		}


		public int CompareTo(LBoss other)
		{
			return ld_id.CompareTo(other.ld_id);
		}
	}
}
