<?php
require_once('sonots.class.php');
require_once('toc.class.php'); // to get title of a page

/**
 * Meta Information of A Page
 *
 * This class is a structure having meta information members for a page. 
 * Setter functions try to retrieve meta information of the page automatically.
 * Use direct member access to get meta information or to set information. 
 *
 * This class can also be thought as a namespace having collections 
 * of functions to obtain page meta information. 
 * Use static PluginSonotsMetapage::metaname() for this purpose. 
 *
 * @package    PluginSonots
 * @license    http://www.gnu.org/licenses/gpl.html GPL v2
 * @author     sonots <http://lsx.sourceforge.jp/>
 * @version    $Id: metapage.class.php, v 1.6 2008-06-10 11:14:46 sonots $
 * @require    sonots     v 1.9
 * @require    toc        v 1.3
 */
class PluginSonotsMetapage
{
    /**
     * Pagename
     *
     * @var string
     */
    var $page;
    /**
     * Relative Path from Current Dir
     *
     * @var string
     */
    var $relname;
    /**
     * How to read (Kanji) or Alias
     *
     * @var string
     */
    var $reading;
    /**
     * Directory tree depth (Relative Depth)
     *
     * @var int
     */
    var $depth;
    /**
     * Timestamp
     *
     * @var int
     */
    var $timestamp;
    /**
     * Modified Date in date format
     *
     * @var string
     */
    var $date;
    /**
     * New! (Modified within 1 day or 3 days, etc)
     *
     * @var string
     */
    var $new;
    /**
     * Whether newpage or not
     *
     * @var boolean
     */
    var $newpage;
    /**
     * Whether leaf on directory tree or not
     *
     * @var boolean
     */
    var $leaf;
    /**
     * Existence
     *
     * @var boolean
     */
    var $exist = TRUE;
    /**
     * link string to be shown in link
     *
     * @var string
     */
    var $linkstr;
    /**
     * link to be shown in html
     *
     * @var string
     */
    var $link;
    /**
     * extra information to be shown in html
     *
     * @var string
     */
    var $info = '';
    /**
     * Popular Count
     *
     * @var int
     */
    var $popular;
    /**
     * Table of Contents
     *
     * @var array
     */
    var $toc;
    /**
     * TITLE of the page
     *
     * @var string
     */
    var $title;
    /**
     * First Heading String of the page
     *
     * @var string
     */
    var $firsthead;

    /**
     * constructor
     *
     * @access public
     * @param string $page
     */
    function PluginSonotsMetapage($page)
    {
        $this->page = $page;
        $this->relname = $page;
    }

    ////////////// member functions //////////
    /**
     * Set relative name property of this page automatically
     *
     * @access public
     * @param string $currdir
     * @uses relname
     */
    function set_relname($currdir = '')
    {
        $this->relname = PluginSonotsMetapage::relname($this->page, $currdir);
    }
    /**
     * Set depth property of this page automatically
     *
     * @access public
     * @uses detph
     */
    function set_depth()
    {
        $this->depth = PluginSonotsMetapage::depth($this->relname);
    }
    /**
     * Set reading (how to read Kanji) property of this page automatically
     *
     * @access public
     * @uses reading
     */
    function set_reading()
    {
        $this->reading = PluginSonotsMetapage::reading($this->page);
    }
    /**
     * Set local filename property of this page automatically
     *
     * @access public
     * @uses filename
     */
    function set_filename()
    {
        $this->filename = PluginSonotsMetapage::filename($this->page);
    }
    /**
     * Set timestamp property of this page automatically
     *
     * @access public
     * @uses timestamp
     */
    function set_timestamp()
    {
        $this->timestamp = PluginSonotsMetapage::timestamp($this->page);
    }
    /**
     * Set modified date property of this page automatically
     *
     * @access public
     * @uses date
     */
    function set_date()
    {
        if (! isset($this->timestamp)) $this->set_timestamp();
        $this->date = PluginSonotsMetapage::date($this->timestamp);
    }
    /**
     * Set New! property of this page automatically
     *
     * @access public
     * @uses newdate
     */
    function set_new()
    {
        if (! isset($this->timestamp)) $this->set_timestamp();
        $this->new = PluginSonotsMetapage::newdate($this->timestamp);
    }
    /**
     * Set info property of this page automatically
     *
     * @access public
     * @param array $order array of new,date specifying order.
     */
    function set_info($order = array()) 
    {
        if (! $this->exist) {
            $this->info = '';
            return;
        }
        $info = '';
        foreach ($order as $elem) {
            switch ($elem) {
            case 'date':
                if (! isset($this->date)) $this->set_date();
                $info .= '<span class="comment_date">' . $this->date . '</span>';
                break;
            case 'new':
                if (! isset($this->new)) $this->set_new();
                $info .= $this->new;
                break;
            }
        }
        $this->info = $info; //PluginSonotsMetapage::info($info);
    }
    /**
     * Set newpage property of this page automatically
     *
     * @access public
     */
    function set_newpage()
    {
        $this->newpage = PluginSonotsMetapage::newpage($this->page);
    }
    /**
     * Set leaf property of this page
     *
     * leaf can not be determined item by item
     *
     * @access public
     * @param boolean $leaf
     */
    function set_leaf($leaf)
    {
        $this->leaf = $leaf;
    }
    /**
     * Set exist property of this page automatically
     *
     * @access public
     */
    function set_exist()
    {
        $this->exist = PluginSonotsMetapage::exist($this->page);
    }
    /**
     * Set linkstr property of this page automatically
     *
     * @access public
     * @param string $optlinkstr
     * @uses linkstr
     */
    function set_linkstr($optlinkstr = 'relative')
    {
        $currdir = substr($this->page, 0, strlen($this->page) - strlen($this->relname));
        $this->linkstr = PluginSonotsMetapage::linkstr($this->page, $optlinkstr, $currdir);
    }
     
    /**
     * Set link property of this page automatically
     *
     * @access public
     * @param string $optlinkstr
     * @param string $optlink
     * @uses linkstr
     * @uses link
     */
    function set_link($optlinkstr = 'relative', $optlink = 'page')
    {
        $this->set_linkstr($optlinkstr);
        $this->link = ($this->exist) ? 
            PluginSonotsMetapage::link($this->page, $this->linkstr, $optlink) : $linkstr;
    }
    /**
     * Set popular property of this page automatically
     *
     * @access public
     * @param string $when
     * @uses popular
     */
    function set_popular($when = 'today')
    {
        $this->popular = PluginSonotsMetapage::popular($this->page, $when);
    }
    /**
     * Set Table of Contents property of this page automatically
     *
     * @access public
     * @param boolean $usecache
     * @uses toc
     */
    function set_toc($usecache = true)
    {
        $this->toc = PluginSonotsMetapage::toc($this->page, $usecache);
    }
    /**
     * Set title property of this page automatically
     *
     * @access public
     * @param boolean $usecache
     * @uses title
     */
    function set_title($usecache = true)
    {
        $this->title = PluginSonotsMetapage::title($this->page, $usecache);
    }
    /**
     * Set firsthead property of this page automatically
     *
     * @access public
     * @param boolean $usecache
     * @uses firsthead
     */
    function set_firsthead($usecache = true)
    {
        $this->firsthead = PluginSonotsMetapage::firsthead($this->page, $usecache);
    }

    //////////////////// static functions /////////////
    /**
     * Get relative path of a page (lower path only)
     *
     * @access public
     * @static
     * @param string $page
     * @param string $currdir
     * @return string $relname
     */
    function relname($page, $currdir = '')
    {
        if ($currdir == '') {
            return $page;
        } else {
            $currdirlen = strlen($currdir);
            if ($currdir{$currdirlen-1} !== '/') {
                ++$currdirlen; //Add strlen('/')
            }
            return substr($page, $currdirlen);
        }
    }
    /**
     * Get directory depth of a path
     *
     * @access public
     * @static
     * @param string $path
     * @return int $depth
     */
    function depth($path)
    {
        return substr_count($path, '/') + 1;
    }

    /**
     * Get reading of a page
     *
     * @access public
     * @static
     * @param string $page pagename
     * @return string reading
     * @uses sonots::get_readings
     */
    function reading($page)
    {
        $readings = sonots::get_readings((array)$page);
        return current($readings);
    }
    /**
     * Get local filename of a page
     *
     * @access public
     * @static
     * @param string $page
     * @return string
     */
    function filename($page)
    {
        return get_filename($page);
    }
    /**
     * Get timestamp of a page
     *
     * @access public
     * @static
     * @param string $page
     * @return string
     */
    function timestamp($page)
    {
        return get_filetime($page);
    }
    /**
     * Get date format of a timestamp
     *
     * @access public
     * @static
     * @param int $timestamp
     * @return string
     */
    function date($timestamp)
    {
        return format_date($timestamp);
    }
    /**
     * Get New! of a page
     *
     * @access public
     * @static
     * @param int $timestamp
     * @return string
     */
    function newdate($timestamp)
    {
        // ToDo: Implementing by myself to get more flexibility
        $date = PluginSonotsMetapage::date($timestamp);
        return do_plugin_inline('new', 'nodate', $date);
    }

    /**
     * Get newpage information of a page
     *
     * @access public
     * @static
     * @param string $page pagename
     * @return boolean
     * @uses sonots::is_newpage
     */
    function newpage($page)
    {
        return sonots::is_newpage($page);
    }

    /**
     * Get existence of a page
     *
     * @access public
     * @static
     * @param string $page
     * @return boolean
     */
    function exist($page)
    {
        return is_page($page);
    }

    /**
     * Get string used in html link
     *
     * @access public
     * @static
     * @param string $page pagename
     * @param string $option option
     *  - name|absolute|page|pagename: pagename (absolute path)
     *  - basename         : page basename
     *  - title            : TITLE: of page
     *  - headline         : The first headline of a page
     *  - relative|relname : relative pagename from currdir
     * @param string $currdir (required for 'relative')
     * @return string
     * @uses sonots::get_basename
     * @uses title
     * @uses firsthead
     * @uses relname
     */
    function linkstr($page, $option = 'relative', $currdir = '')
    {
        switch ($option) {
        case 'name':
        case 'absolute':
        case 'page':
        case 'pagename':
        default:
            $linkstr = htmlspecialchars($page);
            break;
        case 'basename':
            $linkstr = htmlspecialchars(sonots::get_basename($page));
            break;
        case 'title':
            $title = PluginSonotsMetapage::title($page);
            $linkstr = strip_htmltag(make_link($title));
            break;
        case 'headline':
            $firsthead = PluginSonotsMetapage::firsthead($page);
            $linkstr = strip_htmltag(make_link($firsthead));
            break;
        case 'relname':
        case 'relative':
            $linkstr = htmlspecialchars(PluginSonotsMetapage::relname($page, $currdir));
            break;
        }
        return $linkstr;
    }

    /**
     * Get link of a page
     * 
     * @access public
     * @static
     * @param string $page pagename
     * @param string $linkstr linkstr
     * @param string $option option
     * - page   : link to page
     * - anchor : link to anchor
     * - off    : no link
     * @return string
     * @uses sonots::make_pagelink_nopg
     * @uses sonots::make_pageanchor
     */
    function link($page, $linkstr, $option)
    {
        switch ($option) {
        case 'page':
            $link = sonots::make_pagelink_nopg($page, $linkstr);
            break;
        case 'anchor':
            $anchor = sonots::make_pageanchor($page);
            $link = sonots::make_pagelink_nopg('', $linkstr, '#' . $anchor);
            break;
        case 'off':
            $link = $linkstr;
            break;
        }
        return $link;
    }

    /**
     * Get number of popular
     *
     * @access public
     * @static
     * @param string $page pagename
     * @param string $when 'total' or 'today' or 'yesterday' or 'recent'
     * @return int
     */
    function popular($page, $when)
    {
        static $localtime, $toay, $yesterday;
        if (! isset($localtime)) {
            if (function_exists('set_timezone')) { // plus
                list($zone, $zonetime) = set_timezone(DEFAULT_LANG);
                $localtime = UTIME + $zonetime;
                $today = gmdate('Y/m/d', $localtime);
                $yesterday = gmdate('Y/m/d', gmmktime(0,0,0, gmdate('m',$localtime), gmdate('d',$localtime)-1, gmdate('Y',$localtime)));
            } else {
                $localtime = ZONETIME + UTIME;
                $today = get_date('Y/m/d'); // == get_date('Y/m/d', UTIME) == date('Y/m/d, ZONETIME + UTIME);
                $yesterday = get_date('Y/m/d', mktime(0,0,0, date('m',$localtime), date('d',$localtime)-1, date('Y',$localtime)));
            }
        }
        $counterfile = COUNTER_DIR . encode($page) . '.count';
        if (is_readable($counterfile)) {
            $lines = file($counterfile);
            $lines = array_map('rtrim', $lines);
            list($total_count, $date, $today_count, $yesterday_count, $ip) = $lines;
        } else {
            return 0;
        }
        $popular = 0;
        switch ($when) {
        case 'total':
            $popular = $total_count;
            break;
        case 'today':
            if ($date == $today) {
                $popular = $today_count;
            }
            break;
        case 'yesterday':
            if ($date == $today) {
                $popular = $yesterday_count;
            } elseif ($date == $yesterday) {
                $popular = $today_count;
            }
            break;
        case 'recent':
            if ($date == $today) {
                $popular = $today_count + $yesterday_count;
            } elseif ($date == $yesterday) {
                $popular = $today_count;
            }
            break;
        }
        return $popular;
    }
    /**
     * Get Table of Contents of a page
     * 
     * @access public
     * @static
     * @param string $page
     * @param boolean $usecache use toc cache or not
     * @return PluginSonotsToc
     * @uses PluginSonotsToc
     */
    function toc($page, $usecache = true)
    {
        return new PluginSonotsToc($page, $usecache);
    }
    /**
     * Get TITLE of the page
     * 
     * @access public
     * @static
     * @param string $page
     * @param boolean $usecache use toc cache or not
     * @return string TITLE
     * @uses PluginSonotsToc
     */
    function title($page, $usecache = true)
    {
        $toc = new PluginSonotsToc($page, $usecache);
        return $toc->get_title();
    }
    /**
     * Get First Headline String of the page
     * 
     * @access public
     * @static
     * @param string $page
     * @param boolean $usecache use toc cache or not
     * @return string first heading string
     * @uses PluginSonotsToc
     */
    function firsthead($page, $usecache = true)
    {
        $toc = new PluginSonotsToc($page, $usecache);
        return $toc->get_firsthead();
    }
        
}

?>
